// SPDX-License-Identifier: GPL-2.0
/*
 * es9080_core.c
 *
 * ES9080 multi dac codec core driver
 *
 * Copyright 2021,2022 Sony Corporation
 */
#include <linux/clk.h>
#include <linux/delay.h>
#include <linux/device.h>
#include <linux/module.h>
#include <linux/pm_runtime.h>
#include <linux/gpio.h>
#include <linux/gpio/consumer.h>
#include <linux/i2c.h>
#include <linux/regmap.h>
#include <linux/regulator/consumer.h>
#include <sound/pcm_params.h>
#include <sound/soc.h>
#include <sound/tlv.h>
#include "es9080.h"

#define ES9080_MAIN_DAC_ADDR	(0x48)
#define ES9080_SUB_DAC_ADDR	(0x49)
#define ES9080_CHANNELS		(16)
#define ES9080_SPLIT_CH		(8)
#define ES9080_VOL_MIN		(0)
#define ES9080_VOL_MAX		(255)
#define ES9080_MAX_SYSCLK	(50000000)

#define WAKE_BUSY_SEC		(10)
#define I2C_ERR_RETRY		(16)
#define CHNL_MAIN		(1)
#define CHNL_SUB		(2)
#define CHNL_BOTH		(3)

#define	MSG_LEN			(32)

/* Delay until resync execution[jiffies] */
#define RESYNC_DELAY		(0)
/* Resync execution reference time[usec] */
#define RESYNC_TIME_US		(1100000)

#define ES9080_NUM_SUPPLIES 3
static const char *const es9080_supply_names[ES9080_NUM_SUPPLIES] = {
	"dvdd",
	"avdd",
	"vcca"
};

struct es9080_priv {
	struct i2c_client *client;		/* I2C(0x4C) for PLL */
	struct i2c_client *client_m;		/* I2C(0x48) for Main DAC */
	struct i2c_client *client_s;		/* I2C(0x49) for Sub DAC*/
	struct gpio_desc *gpio_ce_m;		/* Chip enable Main DAC */
	struct gpio_desc *gpio_ce_s;		/* Chip enable Sub DAC */
	struct gpio_desc *gpio_dsdpcm;		/* DSD/PCM switch out */
	struct gpio_desc *gpio_socmute;		/* Hardware-mute out */
	struct clk *mclk;
	struct regulator_bulk_data supplies[ES9080_NUM_SUPPLIES];
	unsigned int pcm_ch[ES9080_CHANNELS];	/* Enable-ch map of PCM */
	unsigned int dsd_ch[ES9080_CHANNELS];	/* Enable-ch map of DSD */
	unsigned int automute_time;		/* Automute time */
	unsigned int automute_level;		/* Automute on level */
	unsigned int automute_off_level;	/* Automute off level */
	bool mclk_x2;				/* use 49.1MHz mclk */
	bool pll_bypass;			/* use pll-bypass mode */
	bool self_pm;				/* pm control by self */
	bool master_only;			/* Only dac master mode */
	bool mono_volume;			/* Volume control of 2ch */

	struct snd_soc_codec *codec;

	unsigned long sysclk;
	unsigned int dai_fmt;
	unsigned int frm_byt;
	unsigned int rate;
	unsigned int master;
	unsigned int inpcfg;
	unsigned int channels;
	unsigned int volume[ES9080_CHANNELS];
	unsigned int volume_dsd[ES9080_CHANNELS];
	unsigned int regbuf[2][ES9080_REG_SIZE];
	bool active;
	struct delayed_work dwork;		/* Workqueue */
};

static struct es9080_priv *es9080_data;	/* For access from extern module */

/* initial pll settings */

static const struct reg_default es9080_pll_init[] = {
	{ ES9080_REG_PLL_REGISTER8,		0x10 },
	{ ES9080_REG_PLL_REGISTER7_2,		0xC8 },
	{ ES9080_REG_PLL_REGISTER1,		0x03 },
	{ ES9080_REG_PLL_REGISTER2,		0x5B },
	{ ES9080_REG_PLL_REGISTER3,		0x05 },
	{ ES9080_REG_PLL_REGISTER5,		0x43 },
	{ ES9080_REG_PLL_REGISTER6_0,		0x00 },
	{ ES9080_REG_PLL_REGISTER6_1,		0x00 },
	{ ES9080_REG_PLL_REGISTER6_2,		0x04 },
	{ ES9080_REG_PLL_REGISTER7_0,		0x41 },
	{ ES9080_REG_PLL_REGISTER7_1,		0x08 },
	{ ES9080_REG_PLL_REGISTER4,		0x7D },
	{ ES9080_REG_PLL_REGISTER8,		0x30 },
	{ ES9080_REG_PLL_REGISTER4,		0x7F },
	{ ES9080_REG_PLL_REGISTER4,		0x7F },
	{ ES9080_REG_PLL_REGISTER4,		0x7F },
	{ ES9080_REG_PLL_REGISTER4,		0x7F },
};

static const struct reg_default es9080_pll_mckx2[] = {
	{ ES9080_REG_PLL_REGISTER8,		0x10 },
	{ ES9080_REG_PLL_REGISTER7_2,		0xC8 },
	{ ES9080_REG_PLL_REGISTER1,		0x03 },
	{ ES9080_REG_PLL_REGISTER2,		0x5B },
	{ ES9080_REG_PLL_REGISTER3,		0x05 },
	{ ES9080_REG_PLL_REGISTER5,		0x43 },
	{ ES9080_REG_PLL_REGISTER6_0,		0x00 },
	{ ES9080_REG_PLL_REGISTER6_1,		0x00 },
	{ ES9080_REG_PLL_REGISTER6_2,		0x04 },
	{ ES9080_REG_PLL_REGISTER7_0,		0x81 },
	{ ES9080_REG_PLL_REGISTER7_1,		0x08 },
	{ ES9080_REG_PLL_REGISTER4,		0x7D },
	{ ES9080_REG_PLL_REGISTER8,		0x30 },
	{ ES9080_REG_PLL_REGISTER4,		0x7F },
	{ ES9080_REG_PLL_REGISTER4,		0x7F },
	{ ES9080_REG_PLL_REGISTER4,		0x7F },
	{ ES9080_REG_PLL_REGISTER4,		0x7F },
};

static const struct reg_default es9080_pll_bypass[] = {
	{ ES9080_REG_PLL_REGISTER7_2,		0x40 },
	{ ES9080_REG_PLL_REGISTER1,		0x03 },
	{ ES9080_REG_PLL_REGISTER2,		0xC3 },
};

static const struct reg_default es9080_pll_main[] = {
	{ ES9080_REG_PLL_REGISTER1,		0x03 },
};

static const struct reg_default es9080_pll_sub[] = {
	{ ES9080_REG_PLL_REGISTER1,		0x07 },
};

/* initial DAC settings */

static const struct reg_default es9080_reg_default[] = {
	{ ES9080_REG_CLK_ENB,			0xFF },
	{ ES9080_REG_SYS_MODE_CFG,		0x01 },
	{ ES9080_REG_DAC_CFG,			0x07 },
	{ ES9080_REG_MASTER_CLK_CFG,		0x07 },
	{ ES9080_REG_ANALOG_DAC_ENB,		0xFF },
	{ ES9080_REG_CP_CLK_DIV,		0x30 },
	{ ES9080_REG_ANALOG_CTRL_DELAY,		0xBB },
	{ ES9080_DIGITAL_CTRL_OVERRIDE,		0x3C },
	{ ES9080_REG_FORCE_PLL_LOCK,		0x00 },
	{ ES9080_REG_GPIO56_CFG,		0x50 },
	{ ES9080_REG_GPIO_INP_ENB,		0x3F },
	{ ES9080_REG_INP_CFG,			0x00 },
	{ ES9080_REG_MASTER_MODE_CFG,		0x01 },
	{ ES9080_REG_TDM_CFG1,			0x01 },
	{ ES9080_REG_TDM_CFG2,			0x01 },
	{ ES9080_REG_TDM_CFG3,			0x00 },
	{ ES9080_REG_TDM_CH1_CFG,		0x00 },
	{ ES9080_REG_TDM_CH2_CFG,		0x01 },
	{ ES9080_REG_TDM_CH3_CFG,		0x10 },
	{ ES9080_REG_TDM_CH4_CFG,		0x11 },
	{ ES9080_REG_TDM_CH5_CFG,		0x20 },
	{ ES9080_REG_TDM_CH6_CFG,		0x21 },
	{ ES9080_REG_TDM_CH7_CFG,		0x30 },
	{ ES9080_REG_TDM_CH8_CFG,		0x31 },
	{ ES9080_REG_RESYNC_CFG,		0x00 },
	{ ES9080_REG_MUTE_CTRL,			0xFF },
	{ ES9080_REG_IIR_CFG,			0xE4 },
	{ ES9080_REG_THD_COMP_C2_CH1357_0,	0x00 },
	{ ES9080_REG_THD_COMP_C2_CH1357_1,	0x01 },
	{ ES9080_REG_THD_COMP_C3_CH1357_0,	0x01 },
	{ ES9080_REG_THD_COMP_C3_CH1357_1,	0x01 },
	{ ES9080_REG_THD_COMP_C2_CH2468_0,	0x00 },
	{ ES9080_REG_THD_COMP_C2_CH2468_1,	0x01 },
	{ ES9080_REG_THD_COMP_C3_CH2468_0,	0x10 },
	{ ES9080_REG_THD_COMP_C3_CH2468_1,	0x01 },
	{ ES9080_REG_AUTOMUTE_ENB,		0x00 },
	{ ES9080_REG_AUTOMUTE_TIME_0,		0x00 },
	{ ES9080_REG_AUTOMUTE_TIME_1,		0x80 },
	{ ES9080_REG_AUTOMUTE_LEVEL_0,		0x00 },
	{ ES9080_REG_AUTOMUTE_LEVEL_1,		0x00 },
	{ ES9080_REG_AUTOMUTE_OFF_LEVEL_0,	0x00 },
	{ ES9080_REG_AUTOMUTE_OFF_LEVEL_1,	0x00 },
	{ ES9080_REG_VOLUME1,			0x00 },
	{ ES9080_REG_VOLUME2,			0x00 },
	{ ES9080_REG_VOLUME3,			0x00 },
	{ ES9080_REG_VOLUME4,			0x00 },
	{ ES9080_REG_VOLUME5,			0x00 },
	{ ES9080_REG_VOLUME6,			0x00 },
	{ ES9080_REG_VOLUME7,			0x00 },
	{ ES9080_REG_VOLUME8,			0x00 },
	{ ES9080_REG_VOLUME_MONO_CTRL,		0x40 },
	{ ES9080_REG_NSMOD_DITHER_CFG,		0x54 },
	{ ES9080_REG_NSMOD_DITHER_CH12,		0x44 },
	{ ES9080_REG_NSMOD_DITHER_CH34,		0x44 },
	{ ES9080_REG_NSMOD_DITHER_CH56,		0x44 },
	{ ES9080_REG_NSMOD_DITHER_CH78,		0x44 },
	{ ES9080_REG_SYS_CFG,			0x02 },
};

/* DAC filter settings */

static const struct reg_default es9080_reg_pcm[] = {
	{ ES9080_REG_NSMOD_DITHER_CFG,		0x54 },
	{ ES9080_REG_NSMOD_DITHER_CH12,		0x44 },
	{ ES9080_REG_NSMOD_DITHER_CH34,		0x44 },
	{ ES9080_REG_NSMOD_DITHER_CH56,		0x44 },
	{ ES9080_REG_NSMOD_DITHER_CH78,		0x44 },
};

static const struct reg_default es9080_reg_dsd[] = {
	{ ES9080_REG_NSMOD_DITHER_CFG,		0x54 },
	{ ES9080_REG_NSMOD_DITHER_CH12,		0x44 },
	{ ES9080_REG_NSMOD_DITHER_CH34,		0x44 },
	{ ES9080_REG_NSMOD_DITHER_CH56,		0x44 },
	{ ES9080_REG_NSMOD_DITHER_CH78,		0x44 },
};

/* DAC i2C control functions */

static int es9080_write_reg(struct i2c_client *i2c,
				unsigned int addr, unsigned int dat)
{
	unsigned char buf[2];
	int ret = 0, rty;

	buf[0] = (unsigned char)addr;	/* Reg.Addr. */
	buf[1] = (unsigned char)dat;	/* Reg.Data */
	rty = 0;
	while (rty <= I2C_ERR_RETRY) {
		DEBUG("send(0x%02x): 0x%02x=0x%02x (%d)\n",
				i2c->addr, buf[0], buf[1], rty);
		ret = i2c_master_send(i2c, buf, 2);
		if (ret >= 0)
			break;
		usleep_range(500, 550);
		rty++;
	}
	if (ret < 0) {
		dev_err(&i2c->dev, "%s() failed(%d)\n", __func__, ret);
		return ret;
	}
	return 0;
}

#ifdef ES9080_RBREG
static int es9080_read_reg(struct i2c_client *i2c,
				unsigned int addr, unsigned int *dat)
{
	struct i2c_msg msg[2];
	unsigned char buf[2];
	int ret = 0, rty;

	buf[0] = addr;		/* Reg.Addr. */
	msg[0].addr = i2c->addr;
	msg[0].flags = 0;	/* write */
	msg[0].len = 1;
	msg[0].buf = &buf[0];
	msg[1].addr = i2c->addr;
	msg[1].flags = 1;	/* read */
	msg[1].len = 1;
	msg[1].buf = &buf[1];
	rty = 0;
	while (rty <= I2C_ERR_RETRY) {
		ret = i2c_transfer(i2c->adapter, msg, 2);
		if (ret >= 0)
			break;
		usleep_range(500, 550);
		rty++;
	}
	if (ret < 0) {
		dev_err(&i2c->dev, "%s() failed(%d)\n", __func__, ret);
		return ret;
	}
	DEBUG("receive(0x%02x): 0x%02x=0x%02x (%d)\n",
					i2c->addr, addr, buf[1], rty);
	*dat = (unsigned int)buf[1];
	return 0;
}
#endif

static int es9080_send_regs(struct i2c_client *i2c,
				const struct reg_default data[], int size)
{
	int ret = 0, i;

	for (i = 0; i < size; i++) {
		ret = es9080_write_reg(i2c, data[i].reg, data[i].def);
		if (ret < 0)
			break;
	}
	return ret;
}

#ifdef ES9080_RBREG
static int es9080_dump_regs(struct i2c_client *i2c,
				unsigned char addr, size_t size)
{
	unsigned int dat;
	int ret = 0, i;

	for (i = 0; i < size; i++) {
		ret = es9080_read_reg(i2c, addr + i, &dat);
		if (ret < 0)
			break;
	}
	return ret;
}
#endif

static int es9080_wake_busy(struct i2c_client *i2c)
{
	struct i2c_msg msg[2];
	unsigned char buf[2];
	int bsy, ret = 0;

	buf[0] = 0xE1;		/* Reg.Addr. */
	msg[0].addr = i2c->addr;
	msg[0].flags = 0;	/* write */
	msg[0].len = 1;
	msg[0].buf = &buf[0];
	msg[1].addr = i2c->addr;
	msg[1].flags = 1;	/* read */
	msg[1].len = 1;
	msg[1].buf = &buf[1];
	bsy = (WAKE_BUSY_SEC * 1000) / 100;
	while ((bsy--) > 0) {
		ret = i2c_transfer(i2c->adapter, msg, 2);
		if (ret >= 0)
			break;
		usleep_range(100000, 101000);
	}
	if (ret < 0) {
		dev_err(&i2c->dev, "%s() failed(%d)\n", __func__, ret);
		return ret;
	}
	DEBUG("chip_id=0x%02x (%d)\n", buf[1], bsy);
	return 0;
}

int es9080_regist_client(struct i2c_client *i2c)
{
	int ret = 0;

	TRACE("enter\n");
	if (!i2c)
		return -EINVAL;
	if (!es9080_data)
		return -EPROBE_DEFER;

	switch (i2c->addr) {
	case ES9080_MAIN_DAC_ADDR:
		es9080_data->client_m = i2c;
		break;
	case ES9080_SUB_DAC_ADDR:
		if (es9080_data->gpio_ce_m && es9080_data->gpio_ce_s) {
			es9080_data->client_s = i2c;
		} else {
			dev_info(&i2c->dev,
				"DAC not available (0x%x)\n", i2c->addr);
			return -EADDRNOTAVAIL;
		}
		break;
	default:
		dev_err(&i2c->dev, "unknown DAC address(0x%x)\n", i2c->addr);
		return -EFAULT;
	}
	TRACE("done\n");
	return ret;
}
EXPORT_SYMBOL_GPL(es9080_regist_client);

void es9080_free_client(struct i2c_client *i2c)
{
	TRACE("enter\n");
	if (!i2c)
		return;
	if (!es9080_data)
		return;

	switch (i2c->addr) {
	case ES9080_MAIN_DAC_ADDR:
		es9080_data->client_m = NULL;
		break;
	case ES9080_SUB_DAC_ADDR:
		es9080_data->client_s = NULL;
		break;
	default:
		dev_err(&i2c->dev, "unknown DAC address(0x%x)\n", i2c->addr);
	}
	TRACE("done\n");
}
EXPORT_SYMBOL_GPL(es9080_free_client);

static void es9080_setup_regs(struct es9080_priv *priv,
				const struct reg_default data[], int size)
{
	int i;

	for (i = 0; i < size; i++) {
		priv->regbuf[0][data[i].reg] = data[i].def;
		priv->regbuf[1][data[i].reg] = data[i].def;
	}
}

static int es9080_write_bits(struct es9080_priv *priv,
		unsigned int reg, unsigned int mask, unsigned int val,
		unsigned int chnl)
{
	int ret = 0;
	unsigned int dat;

	if (!priv->active)
		return 0;
	if ((chnl & CHNL_MAIN) && (priv->client_m)) {
		dat = priv->regbuf[0][reg];
		dat &= ~mask;
		dat |= (val & mask);
		if (priv->regbuf[0][reg] != dat) {
			ret = es9080_write_reg(priv->client_m, reg, dat);
			if (ret)
				return ret;
			priv->regbuf[0][reg] = dat;
		}
	}
	if ((chnl & CHNL_SUB) && (priv->client_s)) {
		dat = priv->regbuf[1][reg];
		dat &= ~mask;
		dat |= (val & mask);
		if (priv->regbuf[1][reg] != dat) {
			ret = es9080_write_reg(priv->client_s, reg, dat);
			if (ret)
				return ret;
			priv->regbuf[1][reg] = dat;
		}
	}
	return ret;
}

static int es9080_write(struct es9080_priv *priv,
		unsigned int reg, unsigned int val,
		unsigned int chnl)
{
	int ret = 0;

	if (!priv->active)
		return 0;
	if ((chnl & CHNL_MAIN) && (priv->client_m)) {
		if (priv->regbuf[0][reg] != val) {
			ret = es9080_write_reg(priv->client_m, reg, val);
			if (ret)
				return ret;
			priv->regbuf[0][reg] = val;
		}
	}
	if ((chnl & CHNL_SUB) && (priv->client_s)) {
		if (priv->regbuf[1][reg] != val) {
			ret = es9080_write_reg(priv->client_s, reg, val);
			if (ret)
				return ret;
			priv->regbuf[1][reg] = val;
		}
	}
	return ret;
}

static int es9080_initial(struct es9080_priv *priv)
{
	int ret;

	if (priv->gpio_ce_m)
		gpiod_set_value_cansleep(priv->gpio_ce_m, 0);
	if (priv->gpio_ce_s)
		gpiod_set_value_cansleep(priv->gpio_ce_s, 0);
	usleep_range(10000, 10100);

	/* Initial sub DAC pll */
	if (priv->gpio_ce_m && priv->gpio_ce_s) {
		gpiod_set_value_cansleep(priv->gpio_ce_s, 1);
		usleep_range(2000, 2020);
		/* PLL registres initial */
		/* MCLK input is 49.14MHz Bypass */
		if (priv->pll_bypass)
			ret = es9080_send_regs(priv->client,
					es9080_pll_bypass,
					ARRAY_SIZE(es9080_pll_bypass));
		/* MCLK input is 49.14MHz */
		else if (priv->mclk_x2)
			ret = es9080_send_regs(priv->client,
					es9080_pll_mckx2,
					ARRAY_SIZE(es9080_pll_mckx2));
		/* MCLK input is 24.52MHz */
		else
			ret = es9080_send_regs(priv->client,
					es9080_pll_init,
					ARRAY_SIZE(es9080_pll_init));
		if (ret)
			goto i2c_out;

		/* Set sub i2c address */
		ret = es9080_send_regs(priv->client,
				es9080_pll_sub,
				ARRAY_SIZE(es9080_pll_sub));
		if (ret)
			goto i2c_out;
		usleep_range(2000, 2020);
	}

	/* Initial main DAC pll */
	if (priv->gpio_ce_m)
		gpiod_set_value_cansleep(priv->gpio_ce_m, 1);
	usleep_range(2000, 2020);
	/* PLL registres initial */
	/* MCLK input is 49.14MHz Bypass */
	if (priv->pll_bypass)
		ret = es9080_send_regs(priv->client,
				es9080_pll_bypass,
				ARRAY_SIZE(es9080_pll_bypass));
	/* MCLK input is 49.14MHz */
	else if (priv->mclk_x2)
		ret = es9080_send_regs(priv->client,
				es9080_pll_mckx2,
				ARRAY_SIZE(es9080_pll_mckx2));
	/* MCLK input is 24.52MHz */
	else
		ret = es9080_send_regs(priv->client,
				es9080_pll_init,
				ARRAY_SIZE(es9080_pll_init));
	if (ret)
		goto i2c_out;

	/* Set main i2c address */
	ret = es9080_send_regs(priv->client,
			es9080_pll_main,
			ARRAY_SIZE(es9080_pll_main));
	if (ret)
		goto i2c_out;
	usleep_range(2000, 2020);

i2c_out:
	return ret;
}

static void es9080_disconnect(struct es9080_priv *priv)
{
	if (priv->gpio_ce_m)
		gpiod_set_value_cansleep(priv->gpio_ce_m, 0);
	if (priv->gpio_ce_s)
		gpiod_set_value_cansleep(priv->gpio_ce_s, 0);
}

static int es9080_gain_setup(struct es9080_priv *priv, unsigned int fmt)
{
	int index, steps, vol, ret = 0;

	if (priv->mono_volume)
		steps = 2;
	else
		steps = 1;

	for (index = 0; index < ES9080_CHANNELS; index += steps) {
		if (fmt == SND_SOC_DAIFMT_I2S)
			vol = priv->volume[index];
		else
			vol = priv->volume_dsd[index];

		if (index < ES9080_SPLIT_CH)
			ret |= es9080_write(priv,
				ES9080_REG_VOLUME1 + index,
				vol, CHNL_MAIN);
		else
			ret |= es9080_write(priv,
				ES9080_REG_VOLUME1 + index - ES9080_SPLIT_CH,
				vol, CHNL_SUB);
	}
	return ret;
}

static int es9080_filter_setup(struct es9080_priv *priv, unsigned int fmt)
{
	int i, ret = 0;

	if (fmt == SND_SOC_DAIFMT_I2S) {
		for (i = 0; i < ARRAY_SIZE(es9080_reg_pcm); i++)
			ret |= es9080_write(priv,
				es9080_reg_pcm[i].reg,
				es9080_reg_pcm[i].def,
				CHNL_BOTH);
	} else {
		for (i = 0; i < ARRAY_SIZE(es9080_reg_dsd); i++)
			ret |= es9080_write(priv,
				es9080_reg_dsd[i].reg,
				es9080_reg_dsd[i].def,
				CHNL_BOTH);
	}
	return ret;
}

static int es9080_automute(struct es9080_priv *priv)
{
	int ret = 0;

	if (priv->automute_time >= 1
	 && priv->automute_time <= 0x7FF
	 && priv->automute_level >= 1
	 && priv->automute_off_level >= 1) {
		ret |= es9080_write(priv,
			ES9080_REG_AUTOMUTE_ENB,
			0xFF, CHNL_BOTH);

		DEBUG("automute_time=%u\n", priv->automute_time);
		ret |= es9080_write(priv,
			ES9080_REG_AUTOMUTE_TIME_0,
			priv->automute_time & 0xFF,
			CHNL_BOTH);
		ret |= es9080_write(priv,
			ES9080_REG_AUTOMUTE_TIME_1,
			((priv->automute_time >> 8) & 0x07) | 0x80, CHNL_BOTH);

		DEBUG("automute_level=%u\n", priv->automute_level);
		ret |= es9080_write(priv,
			ES9080_REG_AUTOMUTE_LEVEL_0,
			priv->automute_level & 0xFF, CHNL_BOTH);
		ret |= es9080_write(priv,
			ES9080_REG_AUTOMUTE_LEVEL_1,
			(priv->automute_level >> 8) & 0xFF, CHNL_BOTH);

		DEBUG("automute_off_level=%u\n", priv->automute_off_level);
		ret |= es9080_write(priv,
			ES9080_REG_AUTOMUTE_OFF_LEVEL_0,
			priv->automute_off_level & 0xFF, CHNL_BOTH);
		ret |= es9080_write(priv,
			ES9080_REG_AUTOMUTE_OFF_LEVEL_1,
			(priv->automute_off_level >> 8) & 0xFF, CHNL_BOTH);
	} else {
		DEBUG("automute disable\n");
		ret |= es9080_write(priv,
			ES9080_REG_AUTOMUTE_ENB,
			0x00, CHNL_BOTH);
	}

	return ret;
}

#ifdef ES9080_AUTOMUTE_SYSFS
static int automute_time_set(const char *buf, const struct kernel_param *kp)
{
	unsigned int res;
	int ret;

	ret = kstrtouint(buf, 0, &res);
	if (ret)
		return ret;
	es9080_data->automute_time = res;
	es9080_automute(es9080_data);
	return ret;
}
static int automute_time_get(char *buf, const struct kernel_param *kp)
{
	return snprintf(buf, MSG_LEN, "%u\n", es9080_data->automute_time);
}
static struct kernel_param_ops automute_time_ops = {
	.set = automute_time_set,
	.get = automute_time_get,
};
module_param_cb(automute_time, &automute_time_ops, NULL, 0644);

static int automute_level_set(const char *buf, const struct kernel_param *kp)
{
	unsigned int res;
	int ret;

	ret = kstrtouint(buf, 0, &res);
	if (ret)
		return ret;
	es9080_data->automute_level = res;
	es9080_automute(es9080_data);
	return ret;
}
static int automute_level_get(char *buf, const struct kernel_param *kp)
{
	return snprintf(buf, MSG_LEN, "%u\n", es9080_data->automute_level);
}
static struct kernel_param_ops automute_level_ops = {
	.set = automute_level_set,
	.get = automute_level_get,
};
module_param_cb(automute_level, &automute_level_ops, NULL, 0644);

static int automute_offlev_set(const char *buf, const struct kernel_param *kp)
{
	unsigned int res;
	int ret;

	ret = kstrtouint(buf, 0, &res);
	if (ret)
		return ret;
	es9080_data->automute_off_level = res;
	es9080_automute(es9080_data);
	return ret;
}
static int automute_offlev_get(char *buf, const struct kernel_param *kp)
{
	return snprintf(buf, MSG_LEN, "%u\n", es9080_data->automute_off_level);
}
static struct kernel_param_ops automute_offlev_ops = {
	.set = automute_offlev_set,
	.get = automute_offlev_get,
};
module_param_cb(automute_off_level, &automute_offlev_ops, NULL, 0644);
#endif

/* power-control functions */

static int es9080_power_on(struct device *dev)
{
	struct es9080_priv *priv = dev_get_drvdata(dev);
	int ret;

	TRACE("enter\n");
	if (priv->mclk) {
		ret = clk_prepare_enable(priv->mclk);
		if (ret) {
			dev_err(dev, "failed to enable mclk(%d)\n", ret);
			goto err_out;
		}
		priv->sysclk = clk_get_rate(priv->mclk);
	}

	ret = regulator_bulk_enable(ARRAY_SIZE(priv->supplies), priv->supplies);
	if (ret) {
		dev_err(dev, "failed to enable supplies(%d)\n", ret);
		goto err_clk;
	}

	ret = es9080_initial(priv);
	if (ret) {
		dev_err(dev, "failed to initial device(%d)\n", ret);
		goto err_pwr;
	}

	if (priv->client_m) {
		ret = es9080_wake_busy(priv->client_m);
		if (ret)
			goto err_pwr;

		ret = es9080_send_regs(priv->client_m, es9080_reg_default,
					    ARRAY_SIZE(es9080_reg_default));
		if (ret)
			goto err_pwr;
	}
	if (priv->client_s) {
		ret = es9080_wake_busy(priv->client_s);
		if (ret)
			goto err_pwr;

		ret = es9080_send_regs(priv->client_s, es9080_reg_default,
					    ARRAY_SIZE(es9080_reg_default));
		if (ret)
			goto err_pwr;
	}

	/* Set default */
	es9080_setup_regs(priv, es9080_reg_default,
				ARRAY_SIZE(es9080_reg_default));
	priv->dai_fmt = SND_SOC_DAIFMT_I2S;
	priv->frm_byt = 4;
	priv->rate = 48000;
	priv->channels = 2;
	priv->inpcfg = 0x00;
	priv->active = true;

	if (priv->pll_bypass)
		es9080_write(priv, ES9080_REG_FORCE_PLL_LOCK, 0x80, CHNL_BOTH);

	if (priv->mono_volume)
		es9080_write(priv,
				ES9080_REG_VOLUME_MONO_CTRL, 0x60, CHNL_BOTH);

	ret = es9080_gain_setup(priv, priv->dai_fmt);
	if (ret)
		goto err_pwr;

	ret = es9080_filter_setup(priv, priv->dai_fmt);
	if (ret)
		goto err_pwr;

	ret = es9080_automute(priv);
	if (ret)
		goto err_pwr;

	if (priv->master_only)
		es9080_write(priv, ES9080_REG_INP_CFG, 0x10, CHNL_MAIN);

#ifdef ES9080_RBREG
	if (priv->client_m)
		es9080_dump_regs(priv->client_m, 0x00, 0xA5);
	if (priv->client_s)
		es9080_dump_regs(priv->client_s, 0x00, 0xA5);
#endif
	TRACE("done\n");
	return 0;

err_pwr:
	regulator_bulk_disable(ARRAY_SIZE(priv->supplies), priv->supplies);
err_clk:
	if (priv->mclk)
		clk_disable_unprepare(priv->mclk);
err_out:
	TRACE("error(%d)\n", ret);
	return ret;
}

static void es9080_power_off(struct device *dev)
{
	struct es9080_priv *priv = dev_get_drvdata(dev);

	TRACE("enter\n");
	priv->active = false;

	es9080_disconnect(priv);

	regulator_bulk_disable(ARRAY_SIZE(priv->supplies), priv->supplies);
	if (priv->mclk)
		clk_disable_unprepare(priv->mclk);
	TRACE("done\n");
}

/* codec driver functions */

static int es9080_volume_info(struct snd_kcontrol *kctl,
				struct snd_ctl_elem_info *uinfo)
{
	TRACE("enter\n");
	uinfo->type = SNDRV_CTL_ELEM_TYPE_INTEGER;
	uinfo->count = ES9080_CHANNELS;
	uinfo->value.integer.min = ES9080_VOL_MIN;
	uinfo->value.integer.max = ES9080_VOL_MAX;
	TRACE("done\n");
	return 0;
}

static int es9080_volume_get(struct snd_kcontrol *kctl,
				struct snd_ctl_elem_value *uval)
{
	unsigned int index, value;

	TRACE("enter\n");
	for (index = 0; index < ES9080_CHANNELS; index++) {
		if (es9080_data->dai_fmt == SND_SOC_DAIFMT_I2S)
			value = es9080_data->volume[index];
		else
			value = es9080_data->volume_dsd[index];
		uval->value.integer.value[index] = value;
		DEBUG("got vol(%u)=%u\n", index, value);
	}
	TRACE("done\n");
	return 0;
}

static int es9080_volume_put(struct snd_kcontrol *kctl,
				struct snd_ctl_elem_value *uval)
{
	unsigned int index, value;
	int ret = 0;

	TRACE("enter\n");
	for (index = 0; index < ES9080_CHANNELS; index++) {
		value = (unsigned int)(uval->value.integer.value[index]);
		if (value > ES9080_VOL_MAX) {
			DEBUG("invalid vol(%d)=%d\n", index, value);
			value = ES9080_VOL_MAX;
		}
		if (es9080_data->dai_fmt == SND_SOC_DAIFMT_I2S) {
			es9080_data->volume[index] = value;
			DEBUG("set vol(%d)=%d\n", index, value);
		} else {
			es9080_data->volume_dsd[index] = value;
			DEBUG("set vol_dsd(%d)=%d\n", index, value);
		}
		if (index < ES9080_SPLIT_CH)
			ret = es9080_write(es9080_data,
				ES9080_REG_VOLUME1 + index,
				value, CHNL_MAIN);
		else
			ret = es9080_write(es9080_data,
				ES9080_REG_VOLUME1 + index - ES9080_SPLIT_CH,
				value, CHNL_SUB);
	}
	TRACE("done\n");
	return ret;
}

static const struct snd_kcontrol_new es9080_snd_controls[] = {
	{
		.iface = SNDRV_CTL_ELEM_IFACE_MIXER,
		.name = "DAC Output Gain",
		.index = 0,
		.info = es9080_volume_info,
		.get = es9080_volume_get,
		.put = es9080_volume_put,
		.private_value = 0
	},
};

static const struct snd_soc_dapm_widget es9080_dapm_widgets[] = {
	SND_SOC_DAPM_DAC("DAC1", "Playback", SND_SOC_NOPM, 0, 0),
	SND_SOC_DAPM_DAC("DAC2", "Playback", SND_SOC_NOPM, 0, 0),
	SND_SOC_DAPM_DAC("DAC3", "Playback", SND_SOC_NOPM, 0, 0),
	SND_SOC_DAPM_DAC("DAC4", "Playback", SND_SOC_NOPM, 0, 0),
	SND_SOC_DAPM_DAC("DAC5", "Playback", SND_SOC_NOPM, 0, 0),
	SND_SOC_DAPM_DAC("DAC6", "Playback", SND_SOC_NOPM, 0, 0),
	SND_SOC_DAPM_DAC("DAC7", "Playback", SND_SOC_NOPM, 0, 0),
	SND_SOC_DAPM_DAC("DAC8", "Playback", SND_SOC_NOPM, 0, 0),
	SND_SOC_DAPM_DAC("DAC9", "Playback", SND_SOC_NOPM, 0, 0),
	SND_SOC_DAPM_DAC("DAC10", "Playback", SND_SOC_NOPM, 0, 0),
	SND_SOC_DAPM_DAC("DAC11", "Playback", SND_SOC_NOPM, 0, 0),
	SND_SOC_DAPM_DAC("DAC12", "Playback", SND_SOC_NOPM, 0, 0),

	SND_SOC_DAPM_OUTPUT("AOUT1L"),
	SND_SOC_DAPM_OUTPUT("AOUT1R"),
	SND_SOC_DAPM_OUTPUT("AOUT2L"),
	SND_SOC_DAPM_OUTPUT("AOUT2R"),
	SND_SOC_DAPM_OUTPUT("AOUT3L"),
	SND_SOC_DAPM_OUTPUT("AOUT3R"),
	SND_SOC_DAPM_OUTPUT("AOUT4L"),
	SND_SOC_DAPM_OUTPUT("AOUT4R"),
	SND_SOC_DAPM_OUTPUT("AOUT5L"),
	SND_SOC_DAPM_OUTPUT("AOUT5R"),
	SND_SOC_DAPM_OUTPUT("AOUT6L"),
	SND_SOC_DAPM_OUTPUT("AOUT6R"),
	SND_SOC_DAPM_OUTPUT("AOUT7L"),
	SND_SOC_DAPM_OUTPUT("AOUT7R"),
	SND_SOC_DAPM_OUTPUT("AOUT8L"),
	SND_SOC_DAPM_OUTPUT("AOUT8R"),
	SND_SOC_DAPM_OUTPUT("AOUT9L"),
	SND_SOC_DAPM_OUTPUT("AOUT9R"),
	SND_SOC_DAPM_OUTPUT("AOUT10L"),
	SND_SOC_DAPM_OUTPUT("AOUT10R"),
	SND_SOC_DAPM_OUTPUT("AOUT11L"),
	SND_SOC_DAPM_OUTPUT("AOUT11R"),
	SND_SOC_DAPM_OUTPUT("AOUT12L"),
	SND_SOC_DAPM_OUTPUT("AOUT12R"),
};

static const struct snd_soc_dapm_route es9080_dapm_routes[] = {
	/* Playback */
	{ "AOUT1L", NULL, "DAC1" },
	{ "AOUT1R", NULL, "DAC1" },
	{ "AOUT2L", NULL, "DAC2" },
	{ "AOUT2R", NULL, "DAC2" },
	{ "AOUT3L", NULL, "DAC3" },
	{ "AOUT3R", NULL, "DAC3" },
	{ "AOUT4L", NULL, "DAC4" },
	{ "AOUT4R", NULL, "DAC4" },
	{ "AOUT5L", NULL, "DAC5" },
	{ "AOUT5R", NULL, "DAC5" },
	{ "AOUT6L", NULL, "DAC6" },
	{ "AOUT6R", NULL, "DAC6" },
	{ "AOUT7L", NULL, "DAC7" },
	{ "AOUT7R", NULL, "DAC7" },
	{ "AOUT8L", NULL, "DAC8" },
	{ "AOUT8R", NULL, "DAC8" },
	{ "AOUT9L", NULL, "DAC9" },
	{ "AOUT9R", NULL, "DAC9" },
	{ "AOUT10L", NULL, "DAC10" },
	{ "AOUT10R", NULL, "DAC10" },
	{ "AOUT11L", NULL, "DAC11" },
	{ "AOUT11R", NULL, "DAC11" },
	{ "AOUT12L", NULL, "DAC12" },
	{ "AOUT12R", NULL, "DAC12" },
};

static void es9080_shutdown(struct snd_pcm_substream *substream,
			 struct snd_soc_dai *dai)
{
	struct snd_soc_codec *codec = dai->codec;
	struct es9080_priv *priv = snd_soc_codec_get_drvdata(codec);

	TRACE("enter\n");
	if (priv->master && !priv->master_only)
		es9080_write(priv, ES9080_REG_INP_CFG,
			priv->inpcfg & ES9080_INPUT_SEL_MASK, CHNL_MAIN);
	TRACE("done\n");
}

static int es9080_set_dai_sysclk(struct snd_soc_dai *dai,
				  int clk_id, unsigned int freq, int dir)
{
	struct snd_soc_codec *codec = dai->codec;
	struct es9080_priv *priv = snd_soc_codec_get_drvdata(codec);
	int ret;

	TRACE("enter\n");
#ifdef ES9080_MAX_SYSCLK
	if (freq > ES9080_MAX_SYSCLK)
		return -EINVAL;
#endif
	if (priv->mclk) {
		DEBUG("clk_set_rate(%u)\n", freq);
		ret = clk_set_rate(priv->mclk, freq);
		if (ret) {
			dev_err(codec->dev, "failed mclk set rate(%d)\n", ret);
			return ret;
		}
	}

	priv->sysclk = freq;
	TRACE("done\n");
	return 0;
}

static int es9080_set_dai_fmt(struct snd_soc_dai *dai,
			       unsigned int format)
{
	struct snd_soc_codec *codec = dai->codec;
	struct es9080_priv *priv = snd_soc_codec_get_drvdata(codec);
	u32 fmt, mdecfg, inpcfg;

	TRACE("enter\n");
	DEBUG("Format=%u\n", format);
	switch (format & SND_SOC_DAIFMT_MASTER_MASK) {
	case SND_SOC_DAIFMT_CBM_CFS:
	case SND_SOC_DAIFMT_CBM_CFM:
		priv->master = 1;
		break;
	case SND_SOC_DAIFMT_CBS_CFS:
		priv->master = 0;
		break;
	default:
		dev_err(codec->dev, "unsupported dai format(0x%x)\n", format);
		return -EINVAL;
	}
	switch (format & SND_SOC_DAIFMT_FORMAT_MASK) {
	case SND_SOC_DAIFMT_I2S:
		DEBUG("I2S Mode\n");
		fmt = SND_SOC_DAIFMT_I2S;
		mdecfg = 0x01;
		inpcfg = 0x10;
		if (priv->gpio_dsdpcm)
			gpiod_set_value_cansleep(priv->gpio_dsdpcm, 0);
		break;
	case SND_SOC_DAIFMT_PDM:	/* DSD */
		DEBUG("DSD Mode\n");
		fmt = SND_SOC_DAIFMT_PDM;
		mdecfg = 0x02;
		inpcfg = 0x24;
		if (priv->gpio_dsdpcm)
			gpiod_set_value_cansleep(priv->gpio_dsdpcm, 1);
		break;
	default:
		dev_err(codec->dev, "unsupported dai format(0x%x)\n", format);
		return -EINVAL;
	}
	priv->dai_fmt = fmt;
	es9080_gain_setup(priv, priv->dai_fmt);
	es9080_filter_setup(priv, priv->dai_fmt);
	es9080_write(priv, ES9080_REG_SYS_MODE_CFG, mdecfg, CHNL_BOTH);
	priv->inpcfg = inpcfg;
	if (priv->master_only)
		es9080_write(priv, ES9080_REG_INP_CFG,
				inpcfg, CHNL_MAIN);
	else
		es9080_write(priv, ES9080_REG_INP_CFG,
				inpcfg & ES9080_INPUT_SEL_MASK, CHNL_MAIN);
	es9080_write(priv, ES9080_REG_INP_CFG,
			inpcfg & ES9080_INPUT_SEL_MASK, CHNL_SUB);
	TRACE("done\n");
	return 0;
}

static int es9080_hw_params(struct snd_pcm_substream *substream,
			     struct snd_pcm_hw_params *params,
			     struct snd_soc_dai *dai)
{
	struct snd_soc_codec *codec = dai->codec;
	struct es9080_priv *priv = snd_soc_codec_get_drvdata(codec);
	int rate = params_rate(params);
	unsigned int channels = params_channels(params);
	snd_pcm_format_t format = params_format(params);
	unsigned int fmt, dsdpcm, mdecfg, inpcfg;
	unsigned int fs_div, enb_2x;

	TRACE("enter\n");
	DEBUG("Format=%d\n", format);
	switch (format) {
	case SNDRV_PCM_FORMAT_S16_LE:
		priv->frm_byt = 2;
		dsdpcm = 0;
		break;
	case SNDRV_PCM_FORMAT_S24_LE:
		priv->frm_byt = 4;
		dsdpcm = 0;
		break;
	case SNDRV_PCM_FORMAT_S32_LE:
		priv->frm_byt = 4;
		dsdpcm = 0;
		break;
	case SNDRV_PCM_FORMAT_DSD_U8:
		priv->frm_byt = 1;
		dsdpcm = 1;
		break;
	case SNDRV_PCM_FORMAT_DSD_U16_LE:
	case SNDRV_PCM_FORMAT_DSD_U16_BE:
		priv->frm_byt = 2;
		dsdpcm = 1;
		break;
	case SNDRV_PCM_FORMAT_DSD_U32_LE:
	case SNDRV_PCM_FORMAT_DSD_U32_BE:
		priv->frm_byt = 4;
		dsdpcm = 1;
		break;
	default:
		dev_err(codec->dev, "unsupported dai format(%d)\n", format);
		return -EINVAL;
	}
	if (dsdpcm) {
		DEBUG("DSD Mode (U%u)\n", priv->frm_byt * 8);
		fmt = SND_SOC_DAIFMT_PDM;
		mdecfg = 0x02;
		inpcfg = 0x24;
		if (priv->gpio_dsdpcm)
			gpiod_set_value_cansleep(priv->gpio_dsdpcm, 1);
	} else {
		DEBUG("I2S Mode (S%u)\n", priv->frm_byt * 8);
		fmt = SND_SOC_DAIFMT_I2S;
		mdecfg = 0x01;
		inpcfg = 0x10;
		if (priv->gpio_dsdpcm)
			gpiod_set_value_cansleep(priv->gpio_dsdpcm, 0);
	}
	priv->dai_fmt = fmt;
	es9080_gain_setup(priv, priv->dai_fmt);
	es9080_filter_setup(priv, priv->dai_fmt);
	es9080_write(priv, ES9080_REG_SYS_MODE_CFG, mdecfg, CHNL_BOTH);
	priv->inpcfg = inpcfg;
	if (priv->master_only)
		es9080_write(priv, ES9080_REG_INP_CFG,
				inpcfg, CHNL_MAIN);
	else
		es9080_write(priv, ES9080_REG_INP_CFG,
				inpcfg & ES9080_INPUT_SEL_MASK, CHNL_MAIN);
	es9080_write(priv, ES9080_REG_INP_CFG,
			inpcfg & ES9080_INPUT_SEL_MASK, CHNL_SUB);

	DEBUG("Rate=%d\n", rate);
	if (priv->dai_fmt == SND_SOC_DAIFMT_I2S) {
		switch (rate) {		/* SYS_CLK=24.576|22.5792MHz */
		case 32000:
		case 44100:
		case 48000:
			fs_div = 7;	/* SYS_CLK/8 */
			enb_2x = 0;
			break;
		case 88200:
		case 96000:
			fs_div = 3;	/* SYS_CLK/4 */
			enb_2x = 0;
			break;
		case 176400:
		case 192000:
			fs_div = 1;	/* SYS_CLK/2 */
			enb_2x = 0;
			break;
		case 352800:
		case 384000:
			fs_div = 0;	/* SYS_CLK/1 */
			enb_2x = 0;
			break;
		case 705600:
		case 768000:
			fs_div = 0;	/* SYS_CLK/1 */
			enb_2x = 1;	/* 2x Enable */
			break;
		default:
			dev_err(codec->dev, "unsupported pcm rate(%d)\n", rate);
			return -EINVAL;
		}
		/* Setup PCM mode */
		DEBUG("PCM Div=%d+1 2x=%d\n", fs_div, enb_2x);
	} else if (priv->dai_fmt == SND_SOC_DAIFMT_PDM) {
		switch (rate * priv->frm_byt) {
		case 352800:		/* DSD64 2.8MHz */
			fs_div = 7;	/* SYS_CLK/8 */
			enb_2x = 0;
			break;
		case 705600:		/* DSD128 5.6MHz */
			fs_div = 3;	/* SYS_CLK/4 */
			enb_2x = 0;
			break;
		case 1411200:		/* DSD256 11.3MHz */
			fs_div = 1;	/* SYS_CLK/2 */
			enb_2x = 0;
			break;
		case 2822400:		/* DSD512 22.6MHz */
			fs_div = 0;	/* SYS_CLK/1 */
			enb_2x = 0;
			break;
		default:
			dev_err(codec->dev, "unsupported dsd rate(%d)\n", rate);
			return -EINVAL;
		}
		/* Setup DSD mode */
		DEBUG("DSD Div=%d+1 2x=%d\n", fs_div, enb_2x);
	} else {
		dev_err(codec->dev, "unselected dai format\n");
		return -EINVAL;
	}
	priv->rate = rate;
	es9080_write_bits(priv, ES9080_REG_DAC_CFG,
				ES9080_SELECT_IDAC_NUM_MASK,
				fs_div << ES9080_SELECT_IDAC_NUM_SHIFT,
				CHNL_BOTH);
	es9080_write_bits(priv, ES9080_REG_SYS_CFG,
				ES9080_ENABLE_2X_MODE_MASK,
				enb_2x << ES9080_ENABLE_2X_MODE_SHIFT,
				CHNL_BOTH);
	if (priv->master || priv->master_only) {
		es9080_write_bits(priv, ES9080_REG_MASTER_CLK_CFG,
					ES9080_SELECT_MENC_NUM_MASK,
					fs_div << ES9080_SELECT_MENC_NUM_SHIFT,
					CHNL_MAIN);
		es9080_write(priv, ES9080_REG_INP_CFG, inpcfg, CHNL_MAIN);
	}

	DEBUG("Channels=%d\n", channels);
	priv->channels = channels;

	TRACE("done\n");
	return 0;
}

static int es9080_hw_free(struct snd_pcm_substream *substream,
			struct snd_soc_dai *dai)
{
	struct snd_soc_codec *codec = dai->codec;
	struct es9080_priv *priv = snd_soc_codec_get_drvdata(codec);

	TRACE("enter\n");
	if (priv->dai_fmt == SND_SOC_DAIFMT_I2S)
		cancel_delayed_work_sync(&priv->dwork);

	es9080_gain_setup(priv, SND_SOC_DAIFMT_I2S);
	es9080_filter_setup(priv, SND_SOC_DAIFMT_I2S);
	TRACE("done\n");
	return 0;
}

static int es9080_digital_mute(struct snd_soc_dai *dai, int mute)
{
	struct snd_soc_codec *codec = dai->codec;
	struct es9080_priv *priv = snd_soc_codec_get_drvdata(codec);
	unsigned int chidx, actch, dacm, dacs;

	TRACE("enter\n");
	DEBUG("HW-Mute=%d\n", mute);
	if (priv->gpio_socmute)
		gpiod_set_value_cansleep(priv->gpio_socmute, mute);

	if (mute) {
		es9080_write(priv, ES9080_REG_MUTE_CTRL, 0xff, CHNL_BOTH);
	} else {
		if (priv->channels < 0)
			chidx = 0;
		else if (priv->channels > ES9080_CHANNELS)
			chidx = ES9080_CHANNELS - 1;
		else
			chidx = priv->channels - 1;

		if (priv->dai_fmt == SND_SOC_DAIFMT_PDM)
			actch = priv->dsd_ch[chidx];
		else
			actch = priv->pcm_ch[chidx];
		dacm = (~actch & 0x00ff);
		dacs = (~actch & 0xff00) >> 8;
		es9080_write(priv, ES9080_REG_MUTE_CTRL, dacm, CHNL_MAIN);
		es9080_write(priv, ES9080_REG_MUTE_CTRL, dacs, CHNL_SUB);
		DEBUG("MuteCtrl m=0x%02x s=0x%02x\n", dacm, dacs);
	}
	TRACE("done\n");
	return 0;
}

static int es9080_trigger(struct snd_pcm_substream  *substream,
				int cmd, struct snd_soc_dai *dai)
{
	struct snd_soc_codec *codec = dai->codec;
	struct es9080_priv *priv = snd_soc_codec_get_drvdata(codec);

	TRACE("enter\n");
	if (cmd == SNDRV_PCM_TRIGGER_START
	 && priv->dai_fmt == SND_SOC_DAIFMT_I2S)
		schedule_delayed_work(&priv->dwork, RESYNC_DELAY);
	TRACE("done\n");
	return 0;
}

static void es9080_resync(struct work_struct *p_work)
{
	struct delayed_work *p_data;
	struct es9080_priv *priv;
	long twocycus;

	TRACE("enter\n");
	p_data = container_of(p_work, struct delayed_work, work);
	priv = container_of(p_data, struct es9080_priv, dwork);

	/* Waiting for 1 sampling or more[us] */
	twocycus = RESYNC_TIME_US / priv->rate;
	es9080_write(priv, ES9080_REG_RESYNC_CFG, 0x10, CHNL_BOTH);
	usleep_range(twocycus, twocycus + 2);
	es9080_write(priv, ES9080_REG_RESYNC_CFG, 0x0F, CHNL_BOTH);
	usleep_range(twocycus, twocycus + 2);
	es9080_write(priv, ES9080_REG_RESYNC_CFG, 0x00, CHNL_BOTH);
	TRACE("done\n");
}

static int es9080_codec_probe(struct snd_soc_codec *codec)
{
	struct es9080_priv *priv = snd_soc_codec_get_drvdata(codec);
	struct device *dev = &priv->client->dev;
#ifndef CONFIG_PM
	int ret;
#endif
	TRACE("enter\n");
	priv->codec = codec;
#ifdef CONFIG_PM
	if (priv->self_pm)
		pm_runtime_forbid(dev);
#else
	ret = es9080_power_on(dev);
	if (ret)
		retun ret;
#endif
	TRACE("done\n");
	return 0;
}

static int es9080_codec_remove(struct snd_soc_codec *codec)
{
	struct es9080_priv *priv = snd_soc_codec_get_drvdata(codec);
	struct device *dev = &priv->client->dev;

	TRACE("enter\n");
#ifdef CONFIG_PM
	if (priv->self_pm)
		pm_runtime_allow(dev);
#else
	es9080_power_off(dev);
#endif
	TRACE("done\n");
	return 0;
}

static const struct snd_soc_dai_ops es9080_dai_ops = {
	.set_sysclk	= es9080_set_dai_sysclk,
	.set_fmt	= es9080_set_dai_fmt,
	.hw_params	= es9080_hw_params,
	.hw_free	= es9080_hw_free,
	.digital_mute	= es9080_digital_mute,
	.shutdown	= es9080_shutdown,
	.trigger	= es9080_trigger,
};

static struct snd_soc_dai_driver es9080_dai_driver = {
	.name = "es9080-dai",
	.playback = {
		.stream_name = "Playback",
		.channels_min = 1,
		.channels_max = 16,
		.rates = SNDRV_PCM_RATE_KNOT,
		.formats = SNDRV_PCM_FMTBIT_S16_LE |
			   SNDRV_PCM_FMTBIT_S24_LE |
			   SNDRV_PCM_FMTBIT_S32_LE |
			   SNDRV_PCM_FMTBIT_DSD_U8 |
			   SNDRV_PCM_FMTBIT_DSD_U16_LE |
			   SNDRV_PCM_FMTBIT_DSD_U16_BE |
			   SNDRV_PCM_FMTBIT_DSD_U32_LE |
			   SNDRV_PCM_FMTBIT_DSD_U32_BE,
	},
	.ops = &es9080_dai_ops,
};

static const struct snd_soc_codec_driver es9080_codec_driver = {
	.probe = es9080_codec_probe,
	.remove = es9080_codec_remove,
	.component_driver = {
		.controls		= es9080_snd_controls,
		.num_controls		= ARRAY_SIZE(es9080_snd_controls),
		.dapm_widgets		= es9080_dapm_widgets,
		.num_dapm_widgets	= ARRAY_SIZE(es9080_dapm_widgets),
		.dapm_routes		= es9080_dapm_routes,
		.num_dapm_routes	= ARRAY_SIZE(es9080_dapm_routes)
	},
};

/* i2c device driver functions */

static int es9080_probe(struct i2c_client *i2c, const struct i2c_device_id *id)
{
	struct es9080_priv *priv;
	struct device *dev = &i2c->dev;
	struct device_node *np = dev->of_node;
	unsigned int val;
	int ret, i;

	TRACE("enter(0x%x)\n", i2c->addr);
	priv = devm_kzalloc(dev, sizeof(struct es9080_priv), GFP_KERNEL);
	if (priv == NULL) {
		dev_err(dev, "failed memory allocate\n");
		return -ENOMEM;
	}
	priv->client = i2c;
	priv->client_m = NULL;
	priv->client_s = NULL;
	priv->gpio_ce_m = NULL;
	priv->gpio_ce_s = NULL;
	priv->gpio_dsdpcm = NULL;
	priv->gpio_socmute = NULL;
	priv->mclk = NULL;
	priv->mclk_x2 = false;
	priv->pll_bypass = false;
	priv->self_pm = false;
	/* Set default */
	priv->dai_fmt = SND_SOC_DAIFMT_I2S;
	priv->active = false;
	for (i = 0; i < ES9080_CHANNELS; i++) {
		priv->pcm_ch[i] = 0xFFFF;
		priv->dsd_ch[i] = 0x3F3F;
	}
	i2c_set_clientdata(i2c, priv);

	priv->gpio_ce_m = devm_gpiod_get(dev, "ce_main", GPIOD_OUT_LOW);
	if (IS_ERR(priv->gpio_ce_m)) {
		ret = PTR_ERR(priv->gpio_ce_m);
		if (ret != -ENOENT) {
			dev_err(dev, "get ce_main-gpios failed(%d)\n", ret);
			return ret;
		}
		dev_dbg(dev, "ce_main-gpio unused(%d)\n", ret);
		priv->gpio_ce_m = NULL;
	}

	priv->gpio_ce_s = devm_gpiod_get(dev, "ce_sub", GPIOD_OUT_LOW);
	if (IS_ERR(priv->gpio_ce_s)) {
		ret = PTR_ERR(priv->gpio_ce_s);
		if (ret != -ENOENT) {
			dev_err(dev, "get ce_sub-gpios failed(%d)\n", ret);
			return ret;
		}
		dev_dbg(dev, "ce_sub-gpio unused(%d)\n", ret);
		priv->gpio_ce_s = NULL;
	}

	priv->gpio_dsdpcm = devm_gpiod_get(dev, "dsdpcm", GPIOD_OUT_LOW);
	if (IS_ERR(priv->gpio_dsdpcm)) {
		ret = PTR_ERR(priv->gpio_dsdpcm);
		if (ret != -ENOENT) {
			dev_err(dev, "get dsdpcm-gpios failed(%d)\n", ret);
			return ret;
		}
		dev_dbg(dev, "dsdpcm-gpio unused(%d)\n", ret);
		priv->gpio_dsdpcm = NULL;
	}

	priv->gpio_socmute = devm_gpiod_get(dev, "socmute", GPIOD_OUT_LOW);
	if (IS_ERR(priv->gpio_socmute)) {
		ret = PTR_ERR(priv->gpio_socmute);
		if (ret != -ENOENT) {
			dev_err(dev, "get socmute-gpios failed(%d)\n", ret);
			return ret;
		}
		dev_dbg(dev, "socmute-gpio unused(%d)\n", ret);
		priv->gpio_socmute = NULL;
	}

	priv->mclk = devm_clk_get(dev, "mclk");
	if (IS_ERR(priv->mclk)) {
		ret = PTR_ERR(priv->mclk);
		if (ret != -ENOENT) {
			dev_err(dev, "get mclk-clocks failed(%d)\n", ret);
			return ret;
		}
		dev_dbg(dev, "mclk-clocks unused(%d)\n", ret);
		priv->mclk = NULL;
	}

	for (i = 0; i < ARRAY_SIZE(priv->supplies); i++)
		priv->supplies[i].supply = es9080_supply_names[i];

	ret = devm_regulator_bulk_get(dev,
		ARRAY_SIZE(priv->supplies), priv->supplies);
	if (ret) {
		if (ret == -EPROBE_DEFER)
			dev_warn(dev, "defer with power-supplies(%d)\n", ret);
		else
			dev_err(dev, "failed power-supplies get(%d)\n", ret);
		return ret;
	}

	for (i = 0; i < ES9080_CHANNELS; i++) {
		if (of_property_read_u32_index(np, "gain", i, &val))
			break;
		priv->volume[i] = val;
	}
	for (i = 0; i < ES9080_CHANNELS; i++) {
		if (of_property_read_u32_index(np, "gain-dsd", i, &val))
			break;
		priv->volume_dsd[i] = val;
	}
	for (i = 0; i < ES9080_CHANNELS; i++) {
		if (of_property_read_u32_index(np, "pcm-ch", i, &val))
			break;
		priv->pcm_ch[i] = val;
	}
	for (i = 0; i < ES9080_CHANNELS; i++) {
		if (of_property_read_u32_index(np, "dsd-ch", i, &val))
			break;
		priv->dsd_ch[i] = val;
	}
	if (!of_property_read_u32(np, "automute-time", &val))
		priv->automute_time = val;
	if (!of_property_read_u32(np, "automute-level", &val))
		priv->automute_level = val;
	if (!of_property_read_u32(np, "automute-off-level", &val))
		priv->automute_off_level = val;
	priv->mclk_x2 = of_property_read_bool(np, "mclk_x2");
	priv->pll_bypass = of_property_read_bool(np, "pll_bypass");
	priv->self_pm = of_property_read_bool(np, "self_pm_ctrl");
	priv->master_only = of_property_read_bool(np, "master_only");
	priv->mono_volume = of_property_read_bool(np, "mono_volume");

	/* Create delayed Workqueue */
	INIT_DELAYED_WORK(&priv->dwork, es9080_resync);

	ret = snd_soc_register_codec(dev, &es9080_codec_driver,
						&es9080_dai_driver, 1);
	if (ret) {
		dev_err(dev, "failed to register codec(%d)\n", ret);
		return ret;
	}

#ifdef CONFIG_PM
	pm_runtime_enable(dev);
#endif
	es9080_data = priv;
	TRACE("done(0x%x)\n", i2c->addr);
	return 0;
}

static int es9080_remove(struct i2c_client *i2c)
{
	TRACE("enter(0x%x)\n", i2c->addr);
	cancel_delayed_work_sync(&es9080_data->dwork);
#ifdef CONFIG_PM
	pm_runtime_disable(&i2c->dev);
#endif
	snd_soc_unregister_codec(&i2c->dev);
	es9080_data = NULL;
	TRACE("done(0x%x)\n", i2c->addr);
	return 0;
}

#ifdef CONFIG_PM
static int es9080_rt_resume(struct device *dev)
{
	int ret;

	TRACE("enter\n");
	ret = es9080_power_on(dev);
	TRACE("done\n");
	return ret;
}

static int es9080_rt_suspend(struct device *dev)
{
	TRACE("enter\n");
	es9080_power_off(dev);
	TRACE("done\n");
	return 0;
}

const struct dev_pm_ops es9080_pm_ops = {
	SET_RUNTIME_PM_OPS(es9080_rt_suspend, es9080_rt_resume, NULL)
};
#endif

static const struct i2c_device_id es9080_id[] = {
	{ "es9080", },
	{ }
};
MODULE_DEVICE_TABLE(i2c, es9080_id);

static const struct of_device_id es9080_of_match[] = {
	{ .compatible = "ess,es9080", },
	{ }
};
MODULE_DEVICE_TABLE(of, es9080_of_match);

static struct i2c_driver es9080_core_driver = {
	.probe		= es9080_probe,
	.remove		= es9080_remove,
	.id_table	= es9080_id,
	.driver		= {
		.name	= "es9080",
		.of_match_table = es9080_of_match,
#ifdef CONFIG_PM
		.pm		= &es9080_pm_ops,
#endif
	},
};

module_i2c_driver(es9080_core_driver);
MODULE_DESCRIPTION("es9080 multi dac codec core driver");
MODULE_AUTHOR("Sony Corporation");
MODULE_LICENSE("GPL v2");
