// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright 2021 Sony Corporation.
 */

#include <linux/module.h>
#include <linux/platform_device.h>
#include <sound/soc.h>
#include <sound/soc-dai.h>
#include <linux/delay.h>

#include <sound/pcm5102a_mute.h>

static BLOCKING_NOTIFIER_HEAD(pcm5102a_notify_list);

int register_pcm5102a_notifier(struct notifier_block *nb)
{
	return blocking_notifier_chain_register(&pcm5102a_notify_list, nb);
}
EXPORT_SYMBOL_GPL(register_pcm5102a_notifier);

int unregister_pcm5102a_notifier(struct notifier_block *nb)
{
	return blocking_notifier_chain_unregister(&pcm5102a_notify_list, nb);
}
EXPORT_SYMBOL_GPL(unregister_pcm5102a_notifier);

static int pcm5102a_mute_trigger(struct snd_pcm_substream *substream,
			      int cmd, struct snd_soc_dai *dai)
{
	if (cmd == SNDRV_PCM_TRIGGER_STOP)
		blocking_notifier_call_chain(&pcm5102a_notify_list, 1, NULL);

	return 0;
}

static int pcm5102a_mute_digital_mute(struct snd_soc_dai *dai, int mute)
{
	if (mute == 1)
		usleep_range(4000, 4100);

	return 0;
}

static const struct snd_soc_dai_ops pcm5102a_mute_dai_ops = {
	.trigger = pcm5102a_mute_trigger,
	.digital_mute = pcm5102a_mute_digital_mute,
};

static struct snd_soc_dai_driver pcm5102a_mute_dai_driver = {
	.name = "pcm5102a_mute-dai",
	.playback = {
		.stream_name = "Playback",
		.channels_min = 1,
		.channels_max = 16,
		.rates = SNDRV_PCM_RATE_8000_192000,
		.formats = SNDRV_PCM_FMTBIT_S16_LE |
			   SNDRV_PCM_FMTBIT_S24_LE |
			   SNDRV_PCM_FMTBIT_S32_LE,
	},
	.ops = &pcm5102a_mute_dai_ops,
};

static const struct snd_soc_codec_driver pcm5102a_mute_driver = {
};

static int pcm5102a_mute_probe(struct platform_device *pdev)
{
	return snd_soc_register_codec(&pdev->dev, &pcm5102a_mute_driver,
				      &pcm5102a_mute_dai_driver, 1);
}

static int pcm5102a_mute_remove(struct platform_device *pdev)
{
	snd_soc_unregister_codec(&pdev->dev);

	return 0;
}

#ifdef CONFIG_OF
static const struct of_device_id pcm5102a_mute_device_id[] = {
	{ .compatible = "sony,pcm5102a_mute" },
	{}
};
MODULE_DEVICE_TABLE(of, pcm5102a_mute_device_id);
#endif

static struct platform_driver pcm5102a_mute_platform_driver = {
	.driver = {
		.name = "pcm5102a_mute",
		.owner = THIS_MODULE,
#if defined(CONFIG_OF)
		.of_match_table = of_match_ptr(pcm5102a_mute_device_id),
#endif
	},
	.probe	= pcm5102a_mute_probe,
	.remove = pcm5102a_mute_remove,
};
module_platform_driver(pcm5102a_mute_platform_driver);

/* Module information */
MODULE_DESCRIPTION("pcm5102a_mute codec driver");
MODULE_AUTHOR("Sony Corporation");
MODULE_LICENSE("GPL v2");
