/*
 * Copyright 2017 NXP
 * Copyright 2019,2020,2021,2022 Sony Corporation
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#include <linux/module.h>
#include <sound/soc.h>
#include <sound/pcm_params.h>
#include <linux/of_device.h>
#include <linux/clk.h>
#include <linux/clk-provider.h>
#include <linux/gpio/consumer.h>
#include <linux/of_device.h>
#include <linux/of_gpio.h>
#include <linux/regmap.h>
#include <linux/mfd/syscon.h>
#include <misc/sw_pll.h>
#include <sound/sony-asoc-card.h>
#include <linux/pm_runtime.h>
#include <linux/mutex.h>

#include "fsl_sai.h"
#include "fsl_spdif.h"

#define CLK_8K_FREQ    24576000
#define CLK_11K_FREQ   22579200

#define DEFAULT_DAIFMT	SND_SOC_DAIFMT_I2S

uint    pcm_dai_format_clock_master = SND_SOC_DAIFMT_CBS_CFS;
module_param_named(pcm_clock_master, pcm_dai_format_clock_master, uint, S_IWUSR | S_IRUGO);
uint    dsd_dai_format_clock_master = SND_SOC_DAIFMT_CBM_CFM;
module_param_named(dsd_clock_master, dsd_dai_format_clock_master, uint, S_IWUSR | S_IRUGO);

struct sony_sw_pll {
	u8 reg_gpr;
	u8 reg_value;
	u8 reg_bit;
};

struct clock_gate {
        u16 reg_offset;
        u8 reg_value;
        u8 reg_bit;
};
struct sony_multi_data {
	struct snd_soc_card card;
	struct snd_soc_dai_link dai[8];
	unsigned int slots;
	unsigned int slot_width;
	struct snd_soc_dai *tx_sub_dai;
	struct clk *clk_apll;
	struct clk *clk_ext2;
	struct clk *clk_src_main;
	struct clk *clk_src_sub;
	struct clk *clk_root_sub;
	int hdmi_switch;
	int dsd_enable;
	int mute_gpio;
	int dsd_direct;
	int playback_count;
	int capture_count;
	bool hdmi_flag_previous;
	bool hdmi_flag;
	unsigned long mclk_freq;
	struct regmap *gpr;
	struct regmap *clock;
	struct clock_gate sai2;
	bool sai2_root_clk_control;
	bool sw_pll_enable;
	struct sony_sw_pll common[2];
	struct sony_sw_pll hdmi;
	struct sony_sw_pll spdif;
	struct sony_sw_pll sai6;
	struct sony_sw_pll sai5;
	unsigned int playback_daifmt;
	struct pinctrl *pinctrl;
	struct pinctrl_state *sai2_active;
	struct pinctrl_state *sai2_inactive;
	struct mutex sub_dai_mclk_mutex;
	bool sub_dai_mclk_enable;
};

static BLOCKING_NOTIFIER_HEAD(sony_asoc_card_notify_list);

int register_sony_asoc_card_notifier(struct notifier_block *nb)
{
    return blocking_notifier_chain_register(&sony_asoc_card_notify_list, nb);
}
EXPORT_SYMBOL_GPL(register_sony_asoc_card_notifier);

int unregister_sony_asoc_card_notifier(struct notifier_block *nb)
{
    return blocking_notifier_chain_unregister(&sony_asoc_card_notify_list, nb);
}
EXPORT_SYMBOL_GPL(unregister_sony_asoc_card_notifier);

static BLOCKING_NOTIFIER_HEAD(sony_asoc_zone_notify_list);

int register_sony_asoc_zone_notifier(struct notifier_block *nb)
{
	return blocking_notifier_chain_register(&sony_asoc_zone_notify_list,
						nb);
}
EXPORT_SYMBOL_GPL(register_sony_asoc_zone_notifier);

int unregister_sony_asoc_zone_notifier(struct notifier_block *nb)
{
	return blocking_notifier_chain_unregister(&sony_asoc_zone_notify_list,
						  nb);
}
EXPORT_SYMBOL_GPL(unregister_sony_asoc_zone_notifier);

static int sw_pll_set_rate(struct clk * clk, unsigned long rate, struct sony_multi_data *data)
{
	int ret;

	if (data->sai2_root_clk_control)
		regmap_update_bits(data->clock, data->sai2.reg_offset,
                        0x1 << data->sai2.reg_bit,
                        0x0 << data->sai2.reg_bit);

#ifdef CONFIG_SONY_SW_PLL
        sw_pll_get_dsm_lock();
        ret = clk_set_rate(clk, rate);
        sw_pll_get_dsm_unlock();
#else
	ret = clk_set_rate(clk, rate);
#endif
	if (data->sai2_root_clk_control)
		regmap_update_bits(data->clock, data->sai2.reg_offset,
			0x1 << data->sai2.reg_bit,
			data->sai2.reg_value << data->sai2.reg_bit);
	return ret;
}

static bool imx_is_dsd(struct snd_pcm_hw_params *params)
{
        snd_pcm_format_t format = params_format(params);

        switch (format) {
        case SNDRV_PCM_FORMAT_DSD_U8:
        case SNDRV_PCM_FORMAT_DSD_U16_LE:
        case SNDRV_PCM_FORMAT_DSD_U16_BE:
        case SNDRV_PCM_FORMAT_DSD_U32_LE:
        case SNDRV_PCM_FORMAT_DSD_U32_BE:
                return true;
        default:
                return false;
        }
}

static int imx_clk_src_parent_change(struct sony_multi_data *data, struct clk *clk_parent)
{
	int ret = 0;

	if (!clk_parent)
		return -1;

	if (clk_parent == data->clk_apll)
		ret = sw_pll_set_rate(data->clk_apll, data->mclk_freq * 16, data);

	if (data->hdmi_flag_previous && data->hdmi_switch)
		gpio_set_value(data->hdmi_switch, 0);

	if (data->clk_src_main)
		ret = clk_set_parent(data->clk_src_main, clk_parent);

	if (data->clk_src_sub)
		ret = clk_set_parent(data->clk_src_sub, data->clk_ext2);

	if (data->hdmi_flag && data->hdmi_switch)
		gpio_set_value(data->hdmi_switch, 1);

	return ret;
}

static const struct sai_fs_map {
	unsigned int rmin;
	unsigned int rmax;
	unsigned int wmin;
	unsigned int wmax;
} fs_map[] = {
	/* Normal, < 32kHz */
	{ .rmin = 8000,   .rmax = 24000,  .wmin = 1024, .wmax = 1024, },
	/* Normal, 32kHz */
	{ .rmin = 32000,  .rmax = 32000,  .wmin = 512,  .wmax = 512, },
	/* Normal */
	{ .rmin = 44100,  .rmax = 48000,  .wmin = 512,  .wmax = 512,  },
	/* Double */
	{ .rmin = 88200,  .rmax = 96000,  .wmin = 256,  .wmax = 256,  },
	/* Quad */
	{ .rmin = 176400, .rmax = 192000, .wmin = 128,  .wmax = 128,  },
	/* Oct */
	{ .rmin = 352800, .rmax = 384000, .wmin = 64,   .wmax = 64,  },
	/* Hex */
	{ .rmin = 705600, .rmax = 768000, .wmin = 32,   .wmax = 32,   },
	/* x32 */
	{ .rmin = 1411200, .rmax = 1411200, .wmin = 16,   .wmax = 16,   },
};

static const u32 sai_rates[] = {
	8000, 11025, 16000, 22050,
	32000, 44100, 48000, 88200,
	96000, 176400, 192000, 352800,
	384000, 705600, 768000, 1411200,
};

static const u32 sai_channels[] = {
	1, 2, 4, 6, 8, 10, 12, 14, 16,
};

static unsigned long sony_sai_get_mclk_rate(struct snd_pcm_substream *substream,
					struct snd_pcm_hw_params *params)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct device *dev = rtd->card->dev;
	unsigned int rate = params_rate(params);
	unsigned int width = 2 * params_physical_width(params);
	int i;

	for (i = 0; i < ARRAY_SIZE(fs_map); i++) {
		if (rate >= fs_map[i].rmin && rate <= fs_map[i].rmax) {
			width = max(width, fs_map[i].wmin);
			width = min(width, fs_map[i].wmax);
			return rate * width;
		}
	}

	dev_err(dev, "illegal rate (%u)\n", rate);

	return 0;
}

static int imx_hdmi_spdif_hw_params(struct snd_pcm_substream *substream,
                struct snd_pcm_hw_params *params)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct device *dev = rtd->card->dev;
	int ret = 0;
	u64 rate = params_rate(params);
	unsigned int freq;
	struct snd_soc_card *card = rtd->card;
	struct sony_multi_data *data = snd_soc_card_get_drvdata(card);
	int ext2_clk_rate;

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		freq = do_div(rate, 8000) ? CLK_11K_FREQ : CLK_8K_FREQ;
		ret = snd_soc_dai_set_sysclk(rtd->cpu_dai, STC_TXCLK_SPDIF_ROOT,
			freq, SND_SOC_CLOCK_OUT);
		if (ret)
			dev_err(dev, "failed to set cpu sysclk: %d\n", ret);
	}

	if (substream->stream == SNDRV_PCM_STREAM_CAPTURE) {
		data->mclk_freq = sony_sai_get_mclk_rate(substream, params);
		if (data->sw_pll_enable) {
			data->hdmi_flag_previous = data->hdmi_flag;
			data->hdmi_flag = true;
			ret = sw_pll_set_rate(data->clk_apll, data->mclk_freq * 16, data);
			if (ret)
				dev_err(dev, "fail to clk_set_rate\n");

			ret = clk_set_parent(data->clk_src_sub, data->clk_ext2);
			if (ret)
				dev_err(dev, "fail to change to set ext2\n");

			ext2_clk_rate = clk_get_rate(data->clk_ext2);
			ret = clk_set_rate(data->clk_root_sub, ext2_clk_rate);
			if (ret)
                                dev_err(dev, "clk_root_sub : fail to clk_set_rate\n");

			if (data->hdmi_switch)
				gpio_set_value(data->hdmi_switch, data->hdmi_flag);
		} else {
			data->hdmi_flag_previous = data->hdmi_flag;
			data->hdmi_flag = true;
			ret = imx_clk_src_parent_change(data, data->clk_apll);
		}

		if (ret < 0)
			dev_warn(dev, "failed to set parent %s: %d\n",
				__clk_get_name(data->clk_apll), ret);
	}

	if (data->sw_pll_enable) {
		regmap_update_bits(data->gpr, data->spdif.reg_gpr,
				0x1F << data->spdif.reg_bit,
				data->spdif.reg_value << data->spdif.reg_bit);
	}

#ifdef CONFIG_SONY_SW_PLL
	sw_pll_set_fixed(0);
	sw_pll_set_mode(SW_PLL_MODE_OPT);
#endif
	fsl_spdif_set_spdif_switch(SPDIF_SWITCH_MODE_HDMI);

	return ret;
}

static int imx_spdif_hw_params(struct snd_pcm_substream *substream,
                struct snd_pcm_hw_params *params)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct device *dev = rtd->card->dev;
	int ret = 0;
	u64 rate = params_rate(params);
	unsigned int freq;
	struct snd_soc_card *card = rtd->card;
	struct sony_multi_data *data = snd_soc_card_get_drvdata(card);

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		freq = do_div(rate, 8000) ? CLK_11K_FREQ : CLK_8K_FREQ;
		ret = snd_soc_dai_set_sysclk(rtd->cpu_dai, STC_TXCLK_SPDIF_ROOT,
			freq, SND_SOC_CLOCK_OUT);
		if (ret)
			dev_err(dev, "failed to set cpu sysclk: %d\n", ret);
	}

	if (substream->stream == SNDRV_PCM_STREAM_CAPTURE) {
		data->mclk_freq = sony_sai_get_mclk_rate(substream, params);
		if (data->sw_pll_enable) {
			data->hdmi_flag_previous = data->hdmi_flag;
			data->hdmi_flag = false;

			ret = sw_pll_set_rate(data->clk_apll, data->mclk_freq * 16, data);
			if (ret)
				dev_err(dev, "fail to clk_set_rate\n");

			if (data->hdmi_switch)
				gpio_set_value(data->hdmi_switch, data->hdmi_flag);
		} else {
			data->hdmi_flag_previous = data->hdmi_flag;
			data->hdmi_flag = false;
			ret = imx_clk_src_parent_change(data, data->clk_apll);
		}
		if (ret < 0)
			dev_warn(dev, "failed to set parent %s: %d\n",
				__clk_get_name(data->clk_apll), ret);
	}

	if (data->sw_pll_enable) {
		regmap_update_bits(data->gpr, data->spdif.reg_gpr,
				0x1F << data->spdif.reg_bit,
				data->spdif.reg_value << data->spdif.reg_bit);
	}

#ifdef CONFIG_SONY_SW_PLL
	sw_pll_set_fixed(0);
	sw_pll_set_mode(SW_PLL_MODE_OPT);
#endif
	fsl_spdif_set_spdif_switch(SPDIF_SWITCH_MODE_OPT);

	return ret;
}

static int imx_spdif_monitor_hw_params(struct snd_pcm_substream *substream,
				struct snd_pcm_hw_params *params)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_card *card = rtd->card;
	struct sony_multi_data *data = snd_soc_card_get_drvdata(card);

	if (substream->stream == SNDRV_PCM_STREAM_CAPTURE) {
		data->hdmi_flag_previous = data->hdmi_flag;
		data->hdmi_flag = false;
		if (data->hdmi_switch)
			gpio_set_value(data->hdmi_switch, data->hdmi_flag);
	}

	fsl_spdif_set_spdif_switch(SPDIF_SWITCH_MODE_OPT);

	return 0;
}

static int sony_sai_dai_set_fmt(struct snd_pcm_substream *substream,
				struct snd_soc_dai *cpu_dai, bool is_dsd)
{
	struct sony_multi_data *data =
		snd_soc_card_get_drvdata(
		((struct snd_soc_pcm_runtime *)substream->private_data)->card);
	struct fsl_sai *sai;
	unsigned int fmt = SND_SOC_DAIFMT_NB_NF | DEFAULT_DAIFMT;
	int dir = FSL_FMT_TRANSMITTER;
	int ret;

	switch (substream->stream) {
	case SNDRV_PCM_STREAM_PLAYBACK:
		if (is_dsd)
			fmt = SND_SOC_DAIFMT_NB_NF | SND_SOC_DAIFMT_PDM;
		else
			fmt = SND_SOC_DAIFMT_NB_NF | data->playback_daifmt;
		dir = FSL_FMT_TRANSMITTER;
		break;
	case SNDRV_PCM_STREAM_CAPTURE:
                if (is_dsd)
			fmt = SND_SOC_DAIFMT_NB_NF | SND_SOC_DAIFMT_PDM;
                else
			fmt = SND_SOC_DAIFMT_NB_NF | DEFAULT_DAIFMT;

		dir = FSL_FMT_RECEIVER;
		break;
	default:
		break;
	}

	sai = snd_soc_dai_get_drvdata(cpu_dai);
	if (sai->masterflag[dir])
		fmt = (fmt & (~SND_SOC_DAIFMT_MASTER_MASK)) |
					sai->masterflag[dir];
	ret = fsl_sai_set_dai_fmt_tr(cpu_dai, fmt, dir);

	return ret;
}


static int sony_sai_dynamic_clock_master_cpu_dai_set_fmt(struct snd_pcm_substream *substream,
				struct snd_soc_dai *cpu_dai, bool is_dsd)
{
	struct sony_multi_data *data =
		snd_soc_card_get_drvdata(
		((struct snd_soc_pcm_runtime *)substream->private_data)->card);
	unsigned int fmt = SND_SOC_DAIFMT_NB_NF | DEFAULT_DAIFMT;
	int dir = FSL_FMT_TRANSMITTER;
	int ret;

	switch (substream->stream) {
	case SNDRV_PCM_STREAM_PLAYBACK:
		if (is_dsd)
			fmt = SND_SOC_DAIFMT_NB_NF | SND_SOC_DAIFMT_PDM | dsd_dai_format_clock_master;
		else
			fmt = SND_SOC_DAIFMT_NB_NF | data->playback_daifmt | pcm_dai_format_clock_master;
		dir = FSL_FMT_TRANSMITTER;
		break;
	case SNDRV_PCM_STREAM_CAPTURE:
                if (is_dsd)
			fmt = SND_SOC_DAIFMT_NB_NF | SND_SOC_DAIFMT_PDM | dsd_dai_format_clock_master;
                else
			fmt = SND_SOC_DAIFMT_NB_NF | DEFAULT_DAIFMT | pcm_dai_format_clock_master;

		dir = FSL_FMT_RECEIVER;
		break;
	default:
		break;
	}

	ret = fsl_sai_set_dai_fmt_tr(cpu_dai, fmt, dir);

	return ret;
}

static int sony_sai_dynamic_clock_master_codec_dai_set_fmt(struct snd_pcm_substream *substream,
				struct snd_soc_dai *codec_dai, bool is_dsd)
{
	struct sony_multi_data *data =
		snd_soc_card_get_drvdata(
		((struct snd_soc_pcm_runtime *)substream->private_data)->card);
	unsigned int fmt = SND_SOC_DAIFMT_NB_NF | DEFAULT_DAIFMT;
	int ret;
        struct snd_soc_pcm_runtime *rtd = substream->private_data;
        struct device *dev = rtd->card->dev;

	switch (substream->stream) {
	case SNDRV_PCM_STREAM_PLAYBACK:
		if (is_dsd)
			fmt = SND_SOC_DAIFMT_NB_NF | SND_SOC_DAIFMT_PDM | dsd_dai_format_clock_master;
		else
			fmt = SND_SOC_DAIFMT_NB_NF | data->playback_daifmt | pcm_dai_format_clock_master;
		break;
	case SNDRV_PCM_STREAM_CAPTURE:
		dev_err(dev, "No support codec dai fmt\n");
		return (-1);
		break;
	default:
		break;
	}

	ret = snd_soc_dai_set_fmt(codec_dai, fmt);

	return ret;
}

static int sony_sai_hw_params(struct snd_pcm_substream *substream,
                struct snd_pcm_hw_params *params)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_dai *cpu_dai = rtd->cpu_dai;
	struct snd_soc_card *card = rtd->card;
	struct device *dev = card->dev;
	unsigned int channels = params_channels(params);
	struct sony_multi_data *data = snd_soc_card_get_drvdata(card);
	struct snd_soc_dai *sub_dai = data->tx_sub_dai;
	struct clk *clk_parent = data->clk_apll;
	int ret;
	bool is_dsd = imx_is_dsd(params);
	struct fsl_sai *sub_sai;
	int slots;
	int slot_width;

	data->mclk_freq = sony_sai_get_mclk_rate(substream, params);
	if (substream->stream == SNDRV_PCM_STREAM_CAPTURE) {
		data->hdmi_flag_previous = data->hdmi_flag;
		data->hdmi_flag = false;
#ifdef CONFIG_SONY_SW_PLL
		//analog in
		sw_pll_set_fixed(1);
		sw_pll_set_mode(SW_PLL_MODE_NONE);
#endif
	} else if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK
		&& data->capture_count == 0) {
		data->hdmi_flag_previous = data->hdmi_flag;
#ifdef CONFIG_SONY_SW_PLL
		//file playback
		sw_pll_set_fixed(1);
		sw_pll_set_mode(SW_PLL_MODE_NONE);
#endif
	}

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		if (data->dsd_direct)
			gpio_set_value(data->dsd_direct, 0);
	}

	if (is_dsd) {
		channels = 1;
		slots = 1;
		slot_width = params_width(params);
		if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK)
			if (data->dsd_enable)
				gpio_set_value(data->dsd_enable, 1);
	} else {
		slots = 2;
		slot_width = params_physical_width(params);
		if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK)
			if (data->dsd_enable)
				gpio_set_value(data->dsd_enable, 0);
	}

	ret = sony_sai_dai_set_fmt(substream, cpu_dai, is_dsd);
	if (ret) {              
		dev_err(dev, "failed to set cpu dai fmt: %d\n", ret);
		return ret;
	}

	ret = snd_soc_dai_set_tdm_slot(cpu_dai,
				BIT(channels) - 1, BIT(channels) - 1,
				slots, slot_width);
	if (ret) {
		dev_err(dev, "failed to set cpu dai tdm slot: %d\n", ret);
		return ret;
	}

	if (clk_parent) {
		if (data->sw_pll_enable) {

			if (substream->stream != SNDRV_PCM_STREAM_PLAYBACK ||
				data->capture_count == 0) {
					ret = sw_pll_set_rate(data->clk_apll, data->mclk_freq * 16, data);
					if (ret)
						dev_err(dev, "fail to clk_set_rate\n");
			}

                        if (data->hdmi_switch)
                                gpio_set_value(data->hdmi_switch, data->hdmi_flag);
		} else {
			ret = imx_clk_src_parent_change(data, data->clk_apll);
		}

		if (ret < 0)
			dev_warn(dev, "failed to set parent %s: %d\n",
				__clk_get_name(clk_parent), ret);
	}

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		/* fixed clock rate */
		ret = snd_soc_dai_set_sysclk(cpu_dai, FSL_SAI_CLK_MAST1, 0,
						SND_SOC_CLOCK_OUT);
		if (ret < 0)
			dev_err(dev, "failed to set cpu dai mclk1 rate (%lu): %d\n",
				data->mclk_freq, ret);
	}

	if (sub_dai) {
		ret = sony_sai_dai_set_fmt(substream, sub_dai, is_dsd);
		if (ret) {              
			dev_err(dev, "failed to set sub dai fmt: %d\n", ret);
			return ret;
		}

		ret = snd_soc_dai_set_tdm_slot(sub_dai,
					BIT(channels) - 1, BIT(channels) - 1,
					slots, slot_width);
		if (ret) {
			dev_err(dev, "failed to set sub dai tdm slot: %d\n", ret);
			return ret;
		}

		mutex_lock(&data->sub_dai_mclk_mutex);

		if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
			if (!data->sub_dai_mclk_enable) {
				sub_sai = snd_soc_dai_get_drvdata(sub_dai);
				if (sub_sai->keep_mclk_out_rate)
					clk_disable_unprepare(
					sub_sai->mclk_clk[FSL_SAI_CLK_MAST1]);
				ret = snd_soc_dai_set_sysclk(sub_dai, FSL_SAI_CLK_MAST1, data->mclk_freq,
							SND_SOC_CLOCK_OUT);
				data->sub_dai_mclk_enable = true;
				if (sub_sai->keep_mclk_out_rate)
					ret |= clk_prepare_enable(
					sub_sai->mclk_clk[FSL_SAI_CLK_MAST1]);
			} else
				ret = 0;
			if (ret < 0) {
				dev_err(dev, "failed to set sub dai mclk1 rate (%lu): %d\n",
					data->mclk_freq, ret);
				mutex_unlock(&data->sub_dai_mclk_mutex);
				return ret;
			}
		}

		if (sub_dai->driver->ops && sub_dai->driver->ops->hw_params) {
			ret = sub_dai->driver->ops->hw_params(substream, 
							      params,
							      sub_dai);
			if (ret < 0) 
				dev_err(sub_dai->dev, "can't set %s params: %d\n",
					sub_dai->name, ret);
		}
		mutex_unlock(&data->sub_dai_mclk_mutex);

		if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
			if (!IS_ERR_OR_NULL(data->pinctrl) &&
				!IS_ERR_OR_NULL(data->sai2_active)){
				ret = pinctrl_select_state(data->pinctrl, data->sai2_active);
				if (ret) {
					dev_err(sub_dai->dev,
					"failed to set proper pins state: %d\n", ret);
					return ret;
				}
			}
		}
	}
	return ret;
}

static int sony_sai_zone_hw_params(struct snd_pcm_substream *substream,
                struct snd_pcm_hw_params *params)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_dai *cpu_dai = rtd->cpu_dai;
	struct snd_soc_card *card = rtd->card;
	struct device *dev = card->dev;
	unsigned int channels = params_channels(params);
	struct sony_multi_data *data = snd_soc_card_get_drvdata(card);
	struct snd_soc_dai *sub_dai = data->tx_sub_dai;
	struct clk *clk_parent = data->clk_apll;
	int ret;
	bool is_dsd = imx_is_dsd(params);
	struct fsl_sai *sub_sai;
	int slots;
	int slot_width;

	if ((substream->stream == SNDRV_PCM_STREAM_CAPTURE) || is_dsd ) {
		dev_err(dev, "No support in case of zone_out\n");
		return -EPERM;
	}

	data->mclk_freq = sony_sai_get_mclk_rate(substream, params);
	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK
		&& data->capture_count == 0) {
		data->hdmi_flag_previous = data->hdmi_flag;
#ifdef CONFIG_SONY_SW_PLL
		//file playback
		sw_pll_set_fixed(1);
		sw_pll_set_mode(SW_PLL_MODE_NONE);
#endif
	}

	slots = 2;
	slot_width = params_physical_width(params);

	ret = sony_sai_dai_set_fmt(substream, cpu_dai, is_dsd);
	if (ret) {              
		dev_err(dev, "failed to set cpu dai fmt: %d\n", ret);
		return ret;
	}

	ret = snd_soc_dai_set_tdm_slot(cpu_dai,
				BIT(channels) - 1, BIT(channels) - 1,
				slots, slot_width);
	if (ret) {
		dev_err(dev, "failed to set cpu dai tdm slot: %d\n", ret);
		return ret;
	}

	if (clk_parent) {
		if (data->sw_pll_enable) {

			if (substream->stream != SNDRV_PCM_STREAM_PLAYBACK ||
				data->capture_count == 0) {
					ret = sw_pll_set_rate(data->clk_apll, data->mclk_freq * 16, data);
					if (ret)
						dev_err(dev, "fail to clk_set_rate\n");
			}

                        if (data->hdmi_switch)
                                gpio_set_value(data->hdmi_switch, data->hdmi_flag);
		} else {
			ret = imx_clk_src_parent_change(data, data->clk_apll);
		}

		if (ret < 0)
			dev_warn(dev, "failed to set parent %s: %d\n",
				__clk_get_name(clk_parent), ret);
	}

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		/* fixed clock rate */
		ret = snd_soc_dai_set_sysclk(cpu_dai, FSL_SAI_CLK_MAST1, 0,
						SND_SOC_CLOCK_OUT);
		if (ret < 0)
			dev_err(dev, "failed to set cpu dai mclk1 rate (%lu): %d\n",
				data->mclk_freq, ret);
	}

	if (sub_dai) {
		ret = sony_sai_dai_set_fmt(substream, sub_dai, is_dsd);
		if (ret) {              
			dev_err(dev, "failed to set sub dai fmt: %d\n", ret);
			return ret;
		}

		ret = snd_soc_dai_set_tdm_slot(sub_dai,
					BIT(channels) - 1, BIT(channels) - 1,
					slots, slot_width);
		if (ret) {
			dev_err(dev, "failed to set sub dai tdm slot: %d\n", ret);
			return ret;
		}

		mutex_lock(&data->sub_dai_mclk_mutex);

		if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
			if (!data->sub_dai_mclk_enable) {
				sub_sai = snd_soc_dai_get_drvdata(sub_dai);
				if (sub_sai->keep_mclk_out_rate)
					clk_disable_unprepare(
					sub_sai->mclk_clk[FSL_SAI_CLK_MAST1]);
				ret = snd_soc_dai_set_sysclk(sub_dai, FSL_SAI_CLK_MAST1, data->mclk_freq,
							SND_SOC_CLOCK_OUT);
				data->sub_dai_mclk_enable = true;
				if (sub_sai->keep_mclk_out_rate)
					ret |= clk_prepare_enable(
					sub_sai->mclk_clk[FSL_SAI_CLK_MAST1]);
			} else
				ret = 0;
			if (ret < 0) {
				dev_err(dev, "failed to set sub dai mclk1 rate (%lu): %d\n",
					data->mclk_freq, ret);
				mutex_unlock(&data->sub_dai_mclk_mutex);
				return ret;
			}
		}

		if (sub_dai->driver->ops && sub_dai->driver->ops->hw_params) {
			ret = sub_dai->driver->ops->hw_params(substream, 
							      params,
							      sub_dai);
			if (ret < 0) 
				dev_err(sub_dai->dev, "can't set %s params: %d\n",
					sub_dai->name, ret);
		}
		mutex_unlock(&data->sub_dai_mclk_mutex);

		if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
			if (!IS_ERR_OR_NULL(data->pinctrl) &&
				!IS_ERR_OR_NULL(data->sai2_active)){
				ret = pinctrl_select_state(data->pinctrl, data->sai2_active);
				if (ret) {
					dev_err(sub_dai->dev,
					"failed to set proper pins state: %d\n", ret);
					return ret;
				}
			}
		}
	}
	return ret;
}

static int sony_sai_dynamic_clock_master_hw_params(struct snd_pcm_substream *substream,
                struct snd_pcm_hw_params *params)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_dai *cpu_dai = rtd->cpu_dai;
	struct snd_soc_dai *codec_dai = rtd->codec_dai;
	struct snd_soc_card *card = rtd->card;
	struct device *dev = card->dev;
	unsigned int channels = params_channels(params);
	struct sony_multi_data *data = snd_soc_card_get_drvdata(card);
	struct snd_soc_dai *sub_dai = data->tx_sub_dai;
	struct clk *clk_parent = data->clk_apll;
	int ret;
	bool is_dsd = imx_is_dsd(params);
	struct fsl_sai *sub_sai;
	int slots;
	int slot_width;

	data->mclk_freq = sony_sai_get_mclk_rate(substream, params);
	if (substream->stream == SNDRV_PCM_STREAM_CAPTURE) {
		data->hdmi_flag_previous = data->hdmi_flag;
		data->hdmi_flag = false;
	} else if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK
		&& data->capture_count == 0) {
		data->hdmi_flag_previous = data->hdmi_flag;
	}

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		if (data->dsd_direct)
			gpio_set_value(data->dsd_direct, 0);
	}

	if (is_dsd) {
		channels = 1;
		slots = 1;
		slot_width = params_width(params);
		if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK)
			if (data->dsd_enable)
				gpio_set_value(data->dsd_enable, 1);
	} else {
		slots = 2;
		slot_width = params_physical_width(params);
		if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK)
			if (data->dsd_enable)
				gpio_set_value(data->dsd_enable, 0);
	}

	ret = sony_sai_dynamic_clock_master_cpu_dai_set_fmt(substream, cpu_dai, is_dsd);
	if (ret) {
		dev_err(dev, "failed to set cpu dai slave fmt: %d\n", ret);
		return ret;
	}

	ret = sony_sai_dynamic_clock_master_codec_dai_set_fmt(substream, codec_dai, is_dsd);
	if (ret) {
		dev_err(dev, "failed to set codec dai slave fmt: %d\n", ret);
		return ret;
	}

	ret = snd_soc_dai_set_tdm_slot(cpu_dai,
				BIT(channels) - 1, BIT(channels) - 1,
				slots, slot_width);
	if (ret) {
		dev_err(dev, "failed to set cpu dai tdm slot: %d\n", ret);
		return ret;
	}

	if (clk_parent) {
		if (data->sw_pll_enable) {

			if (substream->stream != SNDRV_PCM_STREAM_PLAYBACK ||
				data->capture_count == 0) {
					ret = sw_pll_set_rate(data->clk_apll, data->mclk_freq * 16, data);
					if (ret)
						dev_err(dev, "fail to clk_set_rate\n");
			}

                        if (data->hdmi_switch)
                                gpio_set_value(data->hdmi_switch, data->hdmi_flag);
		} else {
			ret = imx_clk_src_parent_change(data, data->clk_apll);
		}

		if (ret < 0)
			dev_warn(dev, "failed to set parent %s: %d\n",
				__clk_get_name(clk_parent), ret);
	}

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		/* fixed clock rate */
		ret = snd_soc_dai_set_sysclk(cpu_dai, FSL_SAI_CLK_MAST1, 0,
						SND_SOC_CLOCK_OUT);
		if (ret < 0)
			dev_err(dev, "failed to set cpu dai mclk1 rate (%lu): %d\n",
				data->mclk_freq, ret);
	}

	if (sub_dai) {
		ret = sony_sai_dai_set_fmt(substream, sub_dai, is_dsd);
		if (ret) {
			dev_err(dev, "failed to set sub dai fmt: %d\n", ret);
			return ret;
		}

		ret = snd_soc_dai_set_tdm_slot(sub_dai,
					BIT(channels) - 1, BIT(channels) - 1,
					slots, slot_width);
		if (ret) {
			dev_err(dev, "failed to set sub dai tdm slot: %d\n", ret);
			return ret;
		}

		mutex_lock(&data->sub_dai_mclk_mutex);

		if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
			if (!data->sub_dai_mclk_enable) {
				sub_sai = snd_soc_dai_get_drvdata(sub_dai);
				if (sub_sai->keep_mclk_out_rate)
					clk_disable_unprepare(
					sub_sai->mclk_clk[FSL_SAI_CLK_MAST1]);
				ret = snd_soc_dai_set_sysclk(sub_dai, FSL_SAI_CLK_MAST1, data->mclk_freq,
							SND_SOC_CLOCK_OUT);
				data->sub_dai_mclk_enable = true;
				if (sub_sai->keep_mclk_out_rate)
					ret |= clk_prepare_enable(
					sub_sai->mclk_clk[FSL_SAI_CLK_MAST1]);
			} else
				ret = 0;

			if (ret < 0) {
				dev_err(dev, "failed to set sub dai mclk1 rate (%lu): %d\n",
					data->mclk_freq, ret);
				mutex_unlock(&data->sub_dai_mclk_mutex);
				return ret;
			}
		}

		if (sub_dai->driver->ops && sub_dai->driver->ops->hw_params) {
			ret = sub_dai->driver->ops->hw_params(substream, 
							      params,
							      sub_dai);
			if (ret < 0)
				dev_err(sub_dai->dev, "can't set %s params: %d\n",
					sub_dai->name, ret);
		}

		mutex_unlock(&data->sub_dai_mclk_mutex);

		if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
			if (!IS_ERR_OR_NULL(data->pinctrl) &&
				!IS_ERR_OR_NULL(data->sai2_active)){
				ret = pinctrl_select_state(data->pinctrl, data->sai2_active);
				if (ret) {
					dev_err(sub_dai->dev,
					"failed to set proper pins state: %d\n", ret);
					return ret;
				}
			}
		}
	}
	return ret;
}

static int sony_sai_direct_hw_params(struct snd_pcm_substream *substream,
                struct snd_pcm_hw_params *params)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_dai *cpu_dai = rtd->cpu_dai;
	struct snd_soc_card *card = rtd->card;
	struct device *dev = card->dev;
	unsigned int channels = params_channels(params);
	struct sony_multi_data *data = snd_soc_card_get_drvdata(card);
	struct snd_soc_dai *sub_dai = data->tx_sub_dai;
	struct clk *clk_parent = data->clk_apll;
	int ret;
	bool is_dsd = imx_is_dsd(params);
	struct fsl_sai *sub_sai;
	int slots;
	int slot_width;

	data->mclk_freq = sony_sai_get_mclk_rate(substream, params);
	if (substream->stream == SNDRV_PCM_STREAM_CAPTURE) {
		data->hdmi_flag_previous = data->hdmi_flag;
		data->hdmi_flag = false;
	} else if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK
		&& data->capture_count == 0) {
		data->hdmi_flag_previous = data->hdmi_flag;
	}

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		if (data->dsd_direct)
			gpio_set_value(data->dsd_direct, 1);
	}

	if (is_dsd) {
		channels = 1;
		slots = 1;
		slot_width = params_width(params);
		if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK)
			if (data->dsd_enable)
				gpio_set_value(data->dsd_enable, 1);
	} else {
		slots = 2;
		slot_width = params_physical_width(params);
		if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK)
			if (data->dsd_enable)
				gpio_set_value(data->dsd_enable, 0);
	}

	ret = sony_sai_dai_set_fmt(substream, cpu_dai, is_dsd);
	if (ret) {              
		dev_err(dev, "failed to set cpu dai fmt: %d\n", ret);
		return ret;
	}

	ret = snd_soc_dai_set_tdm_slot(cpu_dai,
				BIT(channels) - 1, BIT(channels) - 1,
				slots, slot_width);
	if (ret) {
		dev_err(dev, "failed to set cpu dai tdm slot: %d\n", ret);
		return ret;
	}

	if (clk_parent) {
		if (data->sw_pll_enable) {

			if (substream->stream != SNDRV_PCM_STREAM_PLAYBACK ||
				data->capture_count == 0) {
					ret = sw_pll_set_rate(data->clk_apll, data->mclk_freq * 16, data);
					if (ret)
						dev_err(dev, "fail to clk_set_rate\n");
			}

                        if (data->hdmi_switch)
                                gpio_set_value(data->hdmi_switch, data->hdmi_flag);
		} else {
			ret = imx_clk_src_parent_change(data, data->clk_apll);
		}

		if (ret < 0)
			dev_warn(dev, "failed to set parent %s: %d\n",
				__clk_get_name(clk_parent), ret);
	}

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		/* fixed clock rate */
		ret = snd_soc_dai_set_sysclk(cpu_dai, FSL_SAI_CLK_MAST1, 0,
						SND_SOC_CLOCK_OUT);
		if (ret < 0)
			dev_err(dev, "failed to set cpu dai mclk1 rate (%lu): %d\n",
				data->mclk_freq, ret);
	}

	if (sub_dai) {
		ret = sony_sai_dai_set_fmt(substream, sub_dai, is_dsd);
		if (ret) {              
			dev_err(dev, "failed to set sub dai fmt: %d\n", ret);
			return ret;
		}

		ret = snd_soc_dai_set_tdm_slot(sub_dai,
					BIT(channels) - 1, BIT(channels) - 1,
					slots, slot_width);
		if (ret) {
			dev_err(dev, "failed to set sub dai tdm slot: %d\n", ret);
			return ret;
		}

		mutex_lock(&data->sub_dai_mclk_mutex);

		if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
			if (!data->sub_dai_mclk_enable) {
				sub_sai = snd_soc_dai_get_drvdata(sub_dai);
				if (sub_sai->keep_mclk_out_rate)
					clk_disable_unprepare(
					sub_sai->mclk_clk[FSL_SAI_CLK_MAST1]);
				ret = snd_soc_dai_set_sysclk(sub_dai, FSL_SAI_CLK_MAST1, data->mclk_freq,
							SND_SOC_CLOCK_OUT);
				data->sub_dai_mclk_enable = true;
				if (sub_sai->keep_mclk_out_rate)
					ret |= clk_prepare_enable(
					sub_sai->mclk_clk[FSL_SAI_CLK_MAST1]);
			} else
				ret = 0;

			if (ret < 0) {
				dev_err(dev, "failed to set sub dai mclk1 rate (%lu): %d\n",
					data->mclk_freq, ret);
				mutex_unlock(&data->sub_dai_mclk_mutex);
				return ret;
			}
		}

		if (sub_dai->driver->ops && sub_dai->driver->ops->hw_params) {
			ret = sub_dai->driver->ops->hw_params(substream,
						params,
						sub_dai);
			if (ret < 0)
				dev_err(sub_dai->dev, "can't set %s params: %d\n",
					sub_dai->name, ret);
		}

		mutex_unlock(&data->sub_dai_mclk_mutex);

		if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
			if (!IS_ERR_OR_NULL(data->pinctrl) &&
				!IS_ERR_OR_NULL(data->sai2_active)){
				ret = pinctrl_select_state(data->pinctrl, data->sai2_active);
				if (ret) {
					dev_err(sub_dai->dev,
					"failed to set proper pins state: %d\n", ret);
					return ret;
				}
			}
		}
	}
	return ret;
}

static int sony_hdmi_sai_hw_params(struct snd_pcm_substream *substream,
                struct snd_pcm_hw_params *params)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_dai *cpu_dai = rtd->cpu_dai;
	struct snd_soc_card *card = rtd->card;
	struct device *dev = card->dev;
	unsigned int channels = params_channels(params);
	struct sony_multi_data *data = snd_soc_card_get_drvdata(card);
	struct snd_soc_dai *sub_dai = data->tx_sub_dai;
	struct clk *clk_parent = data->clk_apll;
	int ret;
	int ext2_clk_rate;
	bool is_dsd = imx_is_dsd(params);
	int slots;
	int slot_width;

	data->mclk_freq = sony_sai_get_mclk_rate(substream, params);
	if (substream->stream == SNDRV_PCM_STREAM_CAPTURE) {
		data->hdmi_flag_previous = data->hdmi_flag;
		data->hdmi_flag = true;
	} else if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK
		&& data->capture_count == 0) {
		data->hdmi_flag_previous = data->hdmi_flag;
	}

	if (is_dsd) {
		channels = 1;
		slots = 1;
		slot_width = params_width(params);
	} else {
		slots = 2;
		slot_width = params_physical_width(params);
	}

	ret = sony_sai_dai_set_fmt(substream, cpu_dai, is_dsd);
	if (ret) {
		dev_err(dev, "failed to set cpu dai fmt: %d\n", ret);
		return ret;
	}

	ret = snd_soc_dai_set_tdm_slot(cpu_dai,
			BIT(channels) - 1, BIT(channels) - 1,
			slots, slot_width);
	if (ret) {
		dev_err(dev, "failed to set cpu dai tdm slot: %d\n", ret);
		return ret;
	}

	if (data->sw_pll_enable) {
		ret = sw_pll_set_rate(data->clk_apll, data->mclk_freq * 16, data);
		if (ret)
			dev_err(dev, "fail to clk_set_rate\n");

		if (data->hdmi_flag) {
			ret = clk_set_parent(data->clk_src_sub, data->clk_ext2);
			if (ret)
				dev_err(dev, "fail to change to set ext2\n");

			ext2_clk_rate = clk_get_rate(data->clk_ext2);
			ret = clk_set_rate(data->clk_root_sub, ext2_clk_rate);
			if (ret)
				dev_err(dev, "clk_root_sub: fail to change to set ext2\n");
		}

		if (data->hdmi_switch)
			gpio_set_value(data->hdmi_switch, data->hdmi_flag);
	} else {
		ret = imx_clk_src_parent_change(data, data->clk_apll);
	}

	if (ret < 0)
		dev_warn(dev, "failed to set parent %s: %d\n",
			__clk_get_name(clk_parent), ret);

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		/* fixed clock rate */
		ret = snd_soc_dai_set_sysclk(cpu_dai, FSL_SAI_CLK_MAST1, 0,
					SND_SOC_CLOCK_OUT);

		if (ret < 0)
			dev_err(dev, "failed to set cpu dai mclk1 rate (%lu): %d\n",
				data->mclk_freq, ret);
	}

	if (sub_dai) {
		ret = sony_sai_dai_set_fmt(substream, sub_dai, is_dsd);
		if (ret) {
			dev_err(dev, "failed to set sub dai fmt: %d\n", ret);
			return ret;
		}

		ret = snd_soc_dai_set_tdm_slot(sub_dai,
					BIT(2) - 1, BIT(2) - 1,
					slots, slot_width);
		if (ret) {
			dev_err(dev, "failed to set sub dai tdm slot: %d\n", ret);
			return ret;
		}

		if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
			mutex_lock(&data->sub_dai_mclk_mutex);
			if (!data->sub_dai_mclk_enable) {
				ret = snd_soc_dai_set_sysclk(sub_dai, FSL_SAI_CLK_MAST1, data->mclk_freq,
						SND_SOC_CLOCK_OUT);
				data->sub_dai_mclk_enable = true;
			} else
				ret = 0;
			mutex_unlock(&data->sub_dai_mclk_mutex);
			if (ret < 0) {
				dev_err(dev, "failed to set sub dai mclk1 rate (%lu): %d\n",
					data->mclk_freq, ret);
				return ret;
			}
		}

		if (sub_dai->driver->ops && sub_dai->driver->ops->hw_params) {
			ret = sub_dai->driver->ops->hw_params(substream,
							params,
							sub_dai);
			if (ret < 0)
				dev_err(sub_dai->dev, "can't set %s params: %d\n",
					sub_dai->name, ret);
		}
	}

	if (data->sw_pll_enable) {
		regmap_update_bits(data->gpr, data->hdmi.reg_gpr,
				0x1F << data->hdmi.reg_bit,
				data->hdmi.reg_value << data->hdmi.reg_bit);
	}

#ifdef CONFIG_SONY_SW_PLL
	sw_pll_set_fixed(0);
	sw_pll_set_mode(SW_PLL_MODE_I2S);
#endif
	return ret;
}

static int sony_sai_hw_free(struct snd_pcm_substream *substream)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_card *card = rtd->card;
	struct sony_multi_data *data = snd_soc_card_get_drvdata(card);
	struct snd_soc_dai *sub_dai = data->tx_sub_dai;
	int ret;

	mutex_lock(&data->sub_dai_mclk_mutex);

	if (sub_dai && sub_dai->driver->ops && sub_dai->driver->ops->hw_free) {
		ret = sub_dai->driver->ops->hw_free(substream, sub_dai);
		if (ret < 0) {
			dev_err(sub_dai->dev, "sub_dai hw_free failed.(%d)\n", ret);
			mutex_unlock(&data->sub_dai_mclk_mutex);
			return ret;
		}
	}

	mutex_unlock(&data->sub_dai_mclk_mutex);

	return 0;
}

static int sony_sai_hdmi_hw_free(struct snd_pcm_substream *substream)
{
        struct snd_soc_pcm_runtime *rtd = substream->private_data;
        struct snd_soc_card *card = rtd->card;
        struct sony_multi_data *data = snd_soc_card_get_drvdata(card);
        struct snd_soc_dai *sub_dai = data->tx_sub_dai;
        int ret;

        if (sub_dai && sub_dai->driver->ops && sub_dai->driver->ops->hw_free) {

                ret = sub_dai->driver->ops->hw_free(substream, sub_dai);
                if (ret < 0) {
                        dev_err(sub_dai->dev, "sub_dai hw_free failed.(%d)\n", ret);
                        return ret;
                }
        }
        return 0;
}

static int sony_sai_startup(struct snd_pcm_substream *substream)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	static struct snd_pcm_hw_constraint_list constraint_rates;
	static struct snd_pcm_hw_constraint_list constraint_channels;
	int ret;
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_card *card = rtd->card;
	struct sony_multi_data *data = snd_soc_card_get_drvdata(card);
	struct snd_soc_dai *sub_dai = data->tx_sub_dai;
	ktime_t start_time = ktime_get();

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		data->playback_count++;
	}
	if (substream->stream == SNDRV_PCM_STREAM_CAPTURE) {
		if (data->capture_count > 0)
			return -1;
		data->capture_count++;
	}

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK && sub_dai)
		pm_runtime_get_sync(sub_dai->dev);

	if (sub_dai && sub_dai->driver->ops && sub_dai->driver->ops->startup) {
		ret = sub_dai->driver->ops->startup(substream, sub_dai);
		if (ret < 0) {
			dev_dbg(sub_dai->dev, "sub_dai startup failed.(%d)\n", ret);
		}
	}

	constraint_channels.list = sai_channels;
	constraint_channels.count = ARRAY_SIZE(sai_channels);
	constraint_rates.list = sai_rates;
	constraint_rates.count = ARRAY_SIZE(sai_rates);

	ret = snd_pcm_hw_constraint_list(runtime, 0, SNDRV_PCM_HW_PARAM_CHANNELS,
						&constraint_channels);
	if (ret)
		goto err_out;

	ret = snd_pcm_hw_constraint_list(runtime, 0, SNDRV_PCM_HW_PARAM_RATE,
						&constraint_rates);
	if (ret)
		goto err_out;

	dev_dbg(sub_dai->dev, "sony_sai_startup() : %lld nSec\n",
		ktime_get() - start_time);

	return 0;

err_out:
	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK && sub_dai) {
		pm_runtime_mark_last_busy(sub_dai->dev);
		pm_runtime_put_autosuspend(sub_dai->dev);
	}

	return ret;
}

static int sony_sai_zone_startup(struct snd_pcm_substream *substream)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	static struct snd_pcm_hw_constraint_list constraint_rates;
	static struct snd_pcm_hw_constraint_list constraint_channels;
	int ret;
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_card *card = rtd->card;
	struct sony_multi_data *data = snd_soc_card_get_drvdata(card);
	struct snd_soc_dai *sub_dai = data->tx_sub_dai;
	ktime_t start_time = ktime_get();

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		data->playback_count++;
	}
	if (substream->stream == SNDRV_PCM_STREAM_CAPTURE) {
		if (data->capture_count > 0)
			return -1;
		data->capture_count++;
	}

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK && sub_dai)
		pm_runtime_get_sync(sub_dai->dev);

	if (sub_dai && sub_dai->driver->ops && sub_dai->driver->ops->startup) {
		ret = sub_dai->driver->ops->startup(substream, sub_dai);
		if (ret < 0) {
			dev_dbg(sub_dai->dev, "sub_dai startup failed.(%d)\n", ret);
		}
	}

	constraint_channels.list = sai_channels;
	constraint_channels.count = ARRAY_SIZE(sai_channels);
	constraint_rates.list = sai_rates;
	constraint_rates.count = ARRAY_SIZE(sai_rates);

	ret = snd_pcm_hw_constraint_list(runtime, 0, SNDRV_PCM_HW_PARAM_CHANNELS,
					&constraint_channels);
	if (ret)
		goto err_out;

	ret = snd_pcm_hw_constraint_list(runtime, 0, SNDRV_PCM_HW_PARAM_RATE,
					&constraint_rates);
	if (ret)
		goto err_out;

	if (gpio_is_valid(data->mute_gpio)) {
		gpio_set_value_cansleep(data->mute_gpio, 1);
	}
	dev_dbg(sub_dai->dev, "sony_sai_zone_startup() : %lld nSec\n",
		ktime_get() - start_time);

	return 0;

err_out:
	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK && sub_dai) {
		pm_runtime_mark_last_busy(sub_dai->dev);
		pm_runtime_put_autosuspend(sub_dai->dev);
	}

	return ret;
}

static int sony_sai_startup_sw_pll(struct snd_pcm_substream *substream)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	static struct snd_pcm_hw_constraint_list constraint_rates;
	static struct snd_pcm_hw_constraint_list constraint_channels;
	int ret;
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_card *card = rtd->card;
	struct sony_multi_data *data = snd_soc_card_get_drvdata(card);
	struct snd_soc_dai *sub_dai = data->tx_sub_dai;

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK)
	{
		if (data->playback_count > 0)
			return -1;
		data->playback_count++;
	}
	if (substream->stream == SNDRV_PCM_STREAM_CAPTURE)
	{
		if (data->capture_count > 0)
			return -1;
		data->capture_count++;
	}

	if (sub_dai && sub_dai->driver->ops && sub_dai->driver->ops->startup)
	{
		ret = sub_dai->driver->ops->startup(substream, sub_dai);
		if (ret < 0)
		{
			dev_dbg(sub_dai->dev, "sub_dai startup failed.(%d)\n", ret);
		}
	}

	constraint_channels.list = sai_channels;
	constraint_channels.count = ARRAY_SIZE(sai_channels);
	constraint_rates.list = sai_rates;
	constraint_rates.count = ARRAY_SIZE(sai_rates);

	ret = snd_pcm_hw_constraint_list(runtime, 0, SNDRV_PCM_HW_PARAM_CHANNELS,
					 &constraint_channels);
	if (ret)
		return ret;

	ret = snd_pcm_hw_constraint_list(runtime, 0, SNDRV_PCM_HW_PARAM_RATE,
					 &constraint_rates);
	if (ret)
		return ret;

	return 0;
}

static int sony_sai_startup_spdif_monitor(struct snd_pcm_substream *substream)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	static struct snd_pcm_hw_constraint_list constraint_rates;
	static struct snd_pcm_hw_constraint_list constraint_channels;
	int ret;

	constraint_channels.list = sai_channels;
	constraint_channels.count = ARRAY_SIZE(sai_channels);
	constraint_rates.list = sai_rates;
	constraint_rates.count = ARRAY_SIZE(sai_rates);

	ret = snd_pcm_hw_constraint_list(runtime, 0, SNDRV_PCM_HW_PARAM_CHANNELS,
						&constraint_channels);
	if (ret)
		goto err_out;

	ret = snd_pcm_hw_constraint_list(runtime, 0, SNDRV_PCM_HW_PARAM_RATE,
						&constraint_rates);
	if (ret)
		goto err_out;

	return 0;

err_out:
	return ret;
}

static void sony_sai_shutdown(struct snd_pcm_substream *substream)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_card *card = rtd->card;
	struct sony_multi_data *data = snd_soc_card_get_drvdata(card);
	struct snd_soc_dai *sub_dai = data->tx_sub_dai;
	int ret;

	if (sub_dai && sub_dai->driver->ops && sub_dai->driver->ops->shutdown) {
		sub_dai->driver->ops->shutdown(substream, sub_dai);
	}

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK && sub_dai) {
		pm_runtime_mark_last_busy(sub_dai->dev);
		pm_runtime_put_autosuspend(sub_dai->dev);

		if (!IS_ERR_OR_NULL(data->pinctrl) &&
			!IS_ERR_OR_NULL(data->sai2_inactive) &&
			data->playback_count == 1) {
			ret = pinctrl_select_state(data->pinctrl, data->sai2_inactive);
			if (ret) {
				dev_err(sub_dai->dev,
				"failed to set proper pins state: %d\n", ret);
				return;
			}
                }
	}

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {

		if (data->dsd_enable)
			gpio_set_value(data->dsd_enable, 0);

		if (data->dsd_direct)
			gpio_set_value(data->dsd_direct, 0);

		data->playback_count--;
		if (data->capture_count == 0) {
			mutex_lock(&data->sub_dai_mclk_mutex);
			data->sub_dai_mclk_enable = false;
			mutex_unlock(&data->sub_dai_mclk_mutex);
		}
	}
	if (substream->stream == SNDRV_PCM_STREAM_CAPTURE)
		data->capture_count--;
}

static void sony_sai_zone_shutdown(struct snd_pcm_substream *substream)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_card *card = rtd->card;
	struct sony_multi_data *data = snd_soc_card_get_drvdata(card);
	struct snd_soc_dai *sub_dai = data->tx_sub_dai;
	int ret;

	if (sub_dai && sub_dai->driver->ops && sub_dai->driver->ops->shutdown) {
		sub_dai->driver->ops->shutdown(substream, sub_dai);
	}

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK && sub_dai) {
		pm_runtime_mark_last_busy(sub_dai->dev);
		pm_runtime_put_autosuspend(sub_dai->dev);

		if (!IS_ERR_OR_NULL(data->pinctrl) &&
			!IS_ERR_OR_NULL(data->sai2_inactive) &&
			data->playback_count == 1) {
			ret = pinctrl_select_state(data->pinctrl, data->sai2_inactive);
			if (ret) {
				dev_err(sub_dai->dev,
				"failed to set proper pins state: %d\n", ret);
				return;
			}
		}
	}

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		data->playback_count--;
		if (data->capture_count == 0) {
			mutex_lock(&data->sub_dai_mclk_mutex);
			data->sub_dai_mclk_enable = false;
			mutex_unlock(&data->sub_dai_mclk_mutex);
		}
	}
	if (substream->stream == SNDRV_PCM_STREAM_CAPTURE)
		data->capture_count--;

	if (gpio_is_valid(data->mute_gpio)) {
		gpio_set_value_cansleep(data->mute_gpio, 0);
	}
}

static void sony_sai_shutdown_sw_pll(struct snd_pcm_substream *substream)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_card *card = rtd->card;
	struct sony_multi_data *data = snd_soc_card_get_drvdata(card);
	struct snd_soc_dai *sub_dai = data->tx_sub_dai;

	if (substream->stream == SNDRV_PCM_STREAM_CAPTURE) {
#ifdef CONFIG_SONY_SW_PLL
		sw_pll_set_fixed(1);
		sw_pll_set_mode(SW_PLL_MODE_NONE);
#endif
		if (data->sw_pll_enable) {
			regmap_update_bits(data->gpr, data->sai6.reg_gpr,
				0x1F << data->sai6.reg_bit,
				data->sai6.reg_value << data->sai6.reg_bit);
		}
	}

	if (sub_dai && sub_dai->driver->ops && sub_dai->driver->ops->shutdown)
	{
		sub_dai->driver->ops->shutdown(substream, sub_dai);
	}

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK)
		data->playback_count--;
	if (substream->stream == SNDRV_PCM_STREAM_CAPTURE)
		data->capture_count--;
}

static int sony_sai_trigger(struct snd_pcm_substream *substream, int cmd)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_pcm_runtime *runtime = substream->runtime;
	struct snd_soc_card *card = rtd->card;
	struct sony_multi_data *data = snd_soc_card_get_drvdata(card);
	struct snd_soc_dai *sub_dai = data->tx_sub_dai;
	int ret;
	
	if (sub_dai && sub_dai->driver->ops && sub_dai->driver->ops->trigger) {
		ret = sub_dai->driver->ops->trigger(substream, cmd, sub_dai);
		if (ret < 0) {
			dev_err(sub_dai->dev, "%s failed to trigger %d (%d)\n",
				sub_dai->name, cmd, ret);
			return ret;
		}
	}

	if (cmd == SNDRV_PCM_TRIGGER_START) {
		if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK)
			blocking_notifier_call_chain(&sony_asoc_card_notify_list, 0, NULL);

		/* silence the buffer that has been processed by hw */
		runtime->silence_threshold = 0;
		runtime->silence_size = runtime->boundary;
	}

	return 0;
}

static int sony_sai_zone_trigger(struct snd_pcm_substream *substream, int cmd)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_pcm_runtime *runtime = substream->runtime;
	struct snd_soc_card *card = rtd->card;
	struct sony_multi_data *data = snd_soc_card_get_drvdata(card);
	struct snd_soc_dai *sub_dai = data->tx_sub_dai;
	int ret;

	if (sub_dai && sub_dai->driver->ops && sub_dai->driver->ops->trigger) {
		ret = sub_dai->driver->ops->trigger(substream, cmd, sub_dai);
		if (ret < 0) {
			dev_err(sub_dai->dev, "%s failed to trigger %d (%d)\n",
				sub_dai->name, cmd, ret);
			return ret;
		}
	}

	if (cmd == SNDRV_PCM_TRIGGER_START) {
		if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK)
			blocking_notifier_call_chain(
					&sony_asoc_zone_notify_list, 0, NULL);

		/* silence the buffer that has been processed by hw */
		runtime->silence_threshold = 0;
		runtime->silence_size = runtime->boundary;
	}

	return 0;
}

static struct snd_soc_ops sony_sai_ops = {
	.hw_params = sony_sai_hw_params,
	.hw_free = sony_sai_hw_free,
	.trigger = sony_sai_trigger,
	.startup = sony_sai_startup,
	.shutdown = sony_sai_shutdown,
};

static struct snd_soc_ops sony_sai_dynamic_clock_master_ops = {
        .hw_params = sony_sai_dynamic_clock_master_hw_params,
        .hw_free = sony_sai_hw_free,
        .trigger = sony_sai_trigger,
        .startup = sony_sai_startup,
        .shutdown = sony_sai_shutdown,
};

static struct snd_soc_ops sony_sai_zone_ops = {
	.hw_params = sony_sai_zone_hw_params,
	.hw_free = sony_sai_hw_free,
	.trigger = sony_sai_zone_trigger,
	.startup = sony_sai_zone_startup,
	.shutdown = sony_sai_zone_shutdown,
};

static struct snd_soc_ops sony_hdmi_sai_ops = {
	.hw_params = sony_hdmi_sai_hw_params,
	.hw_free = sony_sai_hdmi_hw_free,
	.trigger = sony_sai_trigger,
	.startup = sony_sai_startup_sw_pll,
	.shutdown = sony_sai_shutdown_sw_pll,
};

static struct snd_soc_ops imx_spdif_ops = {
	.hw_params = imx_spdif_hw_params,
	.startup = sony_sai_startup_sw_pll,
	.shutdown = sony_sai_shutdown_sw_pll,
};

static struct snd_soc_ops imx_hdmi_spdif_ops = {
	.hw_params = imx_hdmi_spdif_hw_params,
	.startup = sony_sai_startup_sw_pll,
	.shutdown = sony_sai_shutdown_sw_pll,
};

static struct snd_soc_ops sony_sai_direct_ops = {
        .hw_params = sony_sai_direct_hw_params,
        .hw_free = sony_sai_hw_free,
        .trigger = sony_sai_trigger,
        .startup = sony_sai_startup,
        .shutdown = sony_sai_shutdown,
};


static struct snd_soc_ops sony_sai_monitor_ops = {
	.hw_params = imx_spdif_monitor_hw_params,
	.startup = sony_sai_startup_spdif_monitor,
};

static struct snd_soc_dai_link sony_soc_dai[] = {
	{
	.name = "sai-multi-out",
	.stream_name = "sai-multi-out",
	.codec_dai_name = "tas5825-dai",
	.codec_name = "tas5825",
	.ops = &sony_sai_ops,
	.playback_only = 1,
	},
	{
	.name = "sai-analog-in",
	.stream_name = "sai-analog-in",
	.codec_dai_name = "snd-soc-dummy-dai",
	.codec_name = "snd-soc-dummy",
	.ops = &sony_sai_ops,
	.capture_only = 1,
	},
	{
	.name = "sai-hdmi-in",
	.stream_name = "sai-hdmi-in",
	.codec_dai_name = "snd-soc-dummy-dai",
	.codec_name = "snd-soc-dummy",
	.ops = &sony_hdmi_sai_ops,
	.capture_only = 1,
	},
        {
        .name = "S/PDIF OPT",
        .stream_name = "S/PDIF OPT",
        .codec_dai_name = "snd-soc-dummy-dai",
        .codec_name = "snd-soc-dummy",
        .ops = &imx_spdif_ops,
        .capture_only = 1,
        },
        {
        .name = "S/PDIF HDMI",
        .stream_name = "S/PDIF HDMI",
        .codec_dai_name = "snd-soc-dummy-dai",
        .codec_name = "snd-soc-dummy",
        .ops = &imx_hdmi_spdif_ops,
        .capture_only = 1,
        },
	{
	.name = "S/PDIF monitor",
	.stream_name = "S/PDIF monitor",
	.codec_dai_name = "snd-soc-dummy-dai",
	.codec_name = "snd-soc-dummy",
	.ops = &sony_sai_monitor_ops,
	.capture_only = 1,
	},
	{
	.name = "Direct dsd native out",
	.stream_name = "Direct out",
	.codec_dai_name = "snd-soc-dummy-dai",
	.codec_name = "snd-soc-dummy",
	.ops = &sony_sai_direct_ops,
	.playback_only = 1,
	},
	{
	.name = "Zone out",
	.stream_name = "Zone out",
	.codec_dai_name = "pcm5102a_mute-dai",
	.codec_name = "pcm5102a_mute",
	.ops = &sony_sai_zone_ops,
	.playback_only = 1,
	},
};


static int sony_asoc_card_probe(struct platform_device *pdev)
{
	int ret;
	struct device_node *cpu_np = NULL, *cpu_np_sub = NULL, *zone_np = NULL;
	struct device_node *spdif_np = NULL, *sw_pll_node = NULL, *clock_node = NULL;
	struct platform_device *cpu_pdev, *spdif_pdev, *zone_pdev;
	struct sony_multi_data *priv;
	struct snd_soc_dai_link_component cpu_dai_component;
	u32 out_val[19];
	int hdmi_switch, dsd_enable, dsd_direct;
	u32 clock[4];
	phandle phandle;
	int i;
	char *codec, *codec_dai;

	pr_debug("%s()\n", __func__);

	priv = devm_kzalloc(&pdev->dev, sizeof(*priv), GFP_KERNEL);
	if (!priv)
		return -ENOMEM;

	cpu_np = of_parse_phandle(pdev->dev.of_node, "audio-cpu", 0);
	if (!cpu_np) {
		dev_err(&pdev->dev, "cpu_np: audio dai phandle missing or invalid\n");
		ret = -EINVAL;
		goto fail;
	}

	cpu_pdev = of_find_device_by_node(cpu_np);
	if (!cpu_pdev) {
		dev_err(&pdev->dev, "failed to find SAI platform device\n");
		ret = -EINVAL;
		goto fail;
	}

	spdif_np = of_parse_phandle(pdev->dev.of_node, "audio-cpu", 2);
	if (!spdif_np) {
		dev_err(&pdev->dev, "spdif_np:audio dai phandle missing or invalid\n");
		ret = -EINVAL;
		goto fail;
	}

	spdif_pdev = of_find_device_by_node(spdif_np);
	if (!spdif_pdev) {
		dev_err(&pdev->dev, "failed to find S/PDIF platform device\n");
		ret = -EINVAL;
		goto fail;
	}

	zone_pdev = NULL;
	zone_np = of_parse_phandle(pdev->dev.of_node, "audio-cpu", 3);
	if (zone_np) {
		zone_pdev = of_find_device_by_node(zone_np);
		if (!zone_pdev) {
			dev_err(&pdev->dev, "failed to find Zone platform device\n");
			ret = -EINVAL;
			goto fail;
		}
	}

	memcpy(priv->dai, sony_soc_dai,
		sizeof(struct snd_soc_dai_link) * ARRAY_SIZE(priv->dai));

	priv->dai[0].cpu_dai_name = dev_name(&cpu_pdev->dev);
	priv->dai[0].platform_of_node = cpu_np;

	priv->dai[1].cpu_dai_name = dev_name(&cpu_pdev->dev);
	priv->dai[1].platform_of_node = cpu_np;

	priv->dai[2].cpu_dai_name = dev_name(&cpu_pdev->dev);
	priv->dai[2].platform_of_node = cpu_np;

	priv->dai[3].cpu_dai_name = dev_name(&spdif_pdev->dev);
	priv->dai[3].platform_of_node = spdif_np;

	priv->dai[4].cpu_dai_name = dev_name(&spdif_pdev->dev);
	priv->dai[4].platform_of_node = spdif_np;

	priv->dai[5].cpu_dai_name = dev_name(&spdif_pdev->dev);
	priv->dai[5].platform_of_node = spdif_np;

	priv->dai[6].cpu_dai_name = dev_name(&cpu_pdev->dev);
	priv->dai[6].platform_of_node = cpu_np;


	priv->card.num_links = 7;

	if (zone_pdev) {
		priv->dai[priv->card.num_links].cpu_dai_name = dev_name(&zone_pdev->dev);
		priv->dai[priv->card.num_links].platform_of_node = zone_np;
		priv->card.num_links++;
	}

	ret = of_property_read_string(pdev->dev.of_node,
			"multi-out-codec-dai", (const char **)&codec_dai);
	if (!ret) {
		ret = of_property_read_string(pdev->dev.of_node,
			"multi-out-codec", (const char **)&codec);
		if (!ret) {
			priv->dai[0].codec_dai_name = codec_dai;
			priv->dai[0].codec_name = codec;
			priv->dai[0].ops = &sony_sai_dynamic_clock_master_ops;

			/* Direct dsd native out is the same setting as sai-multi-out */
			priv->dai[6].codec_dai_name = codec_dai;
			priv->dai[6].codec_name = codec;
		}
	}

	priv->card.dai_link = priv->dai;
	priv->card.dev = &pdev->dev;
	priv->card.owner = THIS_MODULE;
	ret = snd_soc_of_parse_card_name(&priv->card, "model");
	if (ret)
		goto fail;

	snd_soc_card_set_drvdata(&priv->card, priv);

	ret = snd_soc_register_card(&priv->card);
	if (ret) {
		dev_err(&pdev->dev, "failed to find SAI platform device\n");
		goto fail;
	}

	priv->clk_apll = devm_clk_get(&pdev->dev, "apll");
	if (IS_ERR(priv->clk_apll)) {
		dev_err(&pdev->dev, "failed to get apll clock: %ld\n",
				PTR_ERR(priv->clk_apll));
		priv->clk_apll = NULL;
	}

	priv->clk_ext2 = devm_clk_get(&pdev->dev, "ext2");
	if (IS_ERR(priv->clk_ext2)) {
		dev_err(&pdev->dev, "failed to get ext2 clock: %ld\n",
				PTR_ERR(priv->clk_ext2));
		priv->clk_ext2 = NULL;
	}

	priv->clk_src_main = devm_clk_get(&pdev->dev, "sai2_src");
	if (IS_ERR(priv->clk_src_main)) {
		dev_err(&pdev->dev, "failed to get sai1 src clock: %ld\n",
				PTR_ERR(priv->clk_src_main));
		priv->clk_src_main = NULL;
	}

	/* get sub cpu_dai */
	priv->tx_sub_dai = NULL;
	cpu_np_sub = of_parse_phandle(pdev->dev.of_node, "audio-cpu", 1);
	if (!cpu_np_sub) {
		dev_err(&pdev->dev, "sub audio-cpu dai phandle missing or invalid\n");
		goto no_sub_dai;
	}
	cpu_pdev = of_find_device_by_node(cpu_np_sub);
	if (!cpu_pdev) {
		dev_err(&pdev->dev, "failed to find sub SAI platform device\n");
		ret = -EINVAL;
		goto fail;
	}
	cpu_dai_component.dai_name = dev_name(&cpu_pdev->dev);
	cpu_dai_component.of_node = NULL;
	cpu_dai_component.name = NULL;
	priv->tx_sub_dai = snd_soc_find_dai(&cpu_dai_component);
	if (!priv->tx_sub_dai) {
		dev_err(&pdev->dev, "failed to get sub dai by dai_name.\n");
	}

	priv->clk_src_sub = devm_clk_get(&pdev->dev, "sai1_src");
	if (IS_ERR(priv->clk_src_sub)) {
		dev_err(&pdev->dev, "failed to get sai2 src clock: %ld\n",
				PTR_ERR(priv->clk_src_sub));
		priv->clk_src_sub = NULL;
	}

	priv->clk_root_sub = devm_clk_get(&pdev->dev, "sai1_root");
	if (IS_ERR(priv->clk_root_sub)) {
		dev_err(&pdev->dev, "failed to get sai1 root clock: %ld\n",
				PTR_ERR(priv->clk_root_sub));
		priv->clk_root_sub = NULL;
	}

	hdmi_switch = of_get_named_gpio(pdev->dev.of_node, "hdmi-switch-gpios", 0);
	if (gpio_is_valid(hdmi_switch)) {
		gpio_direction_output(hdmi_switch, 0);
		priv->hdmi_switch = hdmi_switch;
	}

        dsd_enable = of_get_named_gpio(pdev->dev.of_node, "dsd-enable-gpios", 0);
        if (gpio_is_valid(dsd_enable)) {
                gpio_direction_output(dsd_enable , 0);
                priv->dsd_enable = dsd_enable;
        }

	priv->mute_gpio = of_get_named_gpio(pdev->dev.of_node, "mute-gpio", 0);
	if (gpio_is_valid(priv->mute_gpio)) {
		ret = devm_gpio_request_one(&pdev->dev, priv->mute_gpio,
				GPIOF_OUT_INIT_LOW, "zone_mute");
		if (ret) {
			dev_err(&pdev->dev, "failed to get mute gpio\n");
			goto fail;
		}
	}

	dsd_direct = of_get_named_gpio(pdev->dev.of_node, "dsd-direct-gpios", 0);
	if (gpio_is_valid(dsd_direct)) {
		gpio_direction_output(dsd_direct, 0);
		priv->dsd_direct = dsd_direct;
	}

	ret = of_property_read_u32_array(pdev->dev.of_node, "sony_sw_pll", out_val, 19);
	if (ret) {
		dev_dbg(&pdev->dev, "no sony_sw_pll property\n");
	}
	else {
		phandle = *out_val;
		sw_pll_node = of_find_node_by_phandle(phandle);

		if (!sw_pll_node)
			dev_dbg(&pdev->dev, "sw_pll; could not find gpr node by phandle\n");

		priv->gpr = syscon_node_to_regmap(sw_pll_node);
		if (IS_ERR(priv->gpr))
			dev_dbg(&pdev->dev, "sw_pll: could not find gpr regmap\n");

		for (i = 0; i < 2; i++) {
			priv->common[i].reg_gpr = out_val[3 * i + 1];
			priv->common[i].reg_value = out_val[3 * i + 2];
			priv->common[i].reg_bit = out_val[3 * i + 3];
		}

		priv->spdif.reg_gpr = out_val[3 * 2 + 1];
		priv->spdif.reg_value = out_val[3 * 2 + 2];
		priv->spdif.reg_bit = out_val[3 * 2 + 3];

		priv->hdmi.reg_gpr = out_val[3 * 3 + 1];
		priv->hdmi.reg_value = out_val[3 * 3 + 2];
		priv->hdmi.reg_bit = out_val[3 * 3 + 3];

		priv->sai6.reg_gpr = out_val[3 * 4 + 1];
		priv->sai6.reg_value = out_val[3 * 4 + 2];
		priv->sai6.reg_bit = out_val[3 * 4 + 3];

		priv->sai5.reg_gpr = out_val[3 * 5 + 1];
		priv->sai5.reg_value = out_val[3 * 5 + 2];
		priv->sai5.reg_bit = out_val[3 * 5 + 3];

		regmap_update_bits(priv->gpr, priv->common[0].reg_gpr,
				0x1F << priv->common[0].reg_bit,
				priv->common[0].reg_value << priv->common[0].reg_bit);

		regmap_update_bits(priv->gpr, priv->common[1].reg_gpr,
				0x1F << priv->common[1].reg_bit,
				priv->common[1].reg_value << priv->common[1].reg_bit);

		regmap_update_bits(priv->gpr, priv->sai6.reg_gpr,
				0x1F << priv->sai6.reg_bit,
				priv->sai6.reg_value << priv->sai6.reg_bit);

		regmap_update_bits(priv->gpr, priv->sai5.reg_gpr,
				0x1F << priv->sai5.reg_bit,
				priv->sai5.reg_value << priv->sai5.reg_bit);

		of_node_put(sw_pll_node);

		priv->sw_pll_enable = true;
	}

	ret = of_property_read_u32(pdev->dev.of_node, "playback-daifmt", &priv->playback_daifmt);
	if(ret)
		priv->playback_daifmt = DEFAULT_DAIFMT;

	if (of_property_read_bool(pdev->dev.of_node, "playback-dsd-force-dac-slave"))
		dsd_dai_format_clock_master = SND_SOC_DAIFMT_CBS_CFS;

	ret = of_property_read_u32_array(pdev->dev.of_node, "sai2_root_clk_control", clock, 4);
	if (ret) {
		dev_dbg(&pdev->dev, "no sai2_root_clk_control\n");
	} else {
		phandle = *clock;
		clock_node = of_find_node_by_phandle(phandle);
		if (!clock_node)
			dev_dbg(&pdev->dev, "Could not find clock_node by phandle\n");
		priv->clock = syscon_node_to_regmap(clock_node);
		if (IS_ERR(priv->clock))
			dev_dbg(&pdev->dev, "Could not find clock regmap\n");
		priv->sai2.reg_offset = clock[1];
		priv->sai2.reg_value = clock[2];
		priv->sai2.reg_bit = clock[3];
		of_node_put(clock_node);
		priv->sai2_root_clk_control = true;
	}
	
	priv->pinctrl = devm_pinctrl_get(&pdev->dev);
	if (!IS_ERR_OR_NULL(priv->pinctrl)){
		priv->sai2_active =
			pinctrl_lookup_state(priv->pinctrl, "sai2_active");
		priv->sai2_inactive =
			pinctrl_lookup_state(priv->pinctrl, "sai2_inactive");
	}

	mutex_init(&priv->sub_dai_mclk_mutex);
	priv->sub_dai_mclk_enable = false;

no_sub_dai:
	return 0;	

fail:
	if (cpu_np)
		of_node_put(cpu_np);
	if (cpu_np_sub)
		of_node_put(cpu_np_sub);

	return ret; 
}

static int sony_asoc_card_remove(struct platform_device *pdev)
{
	struct snd_soc_card *card = platform_get_drvdata(pdev);

	return snd_soc_unregister_card(card);
}

#ifdef CONFIG_OF
static const struct of_device_id sony_asoc_card_dt_match[] = {
	{.compatible = "fsl,sony-asoc-card",},
	{}
};
#endif

static struct platform_driver sony_asoc_card = {
	.driver = {
		.name = "sony-asoc-card",
		.owner = THIS_MODULE,
		#ifdef CONFIG_OF
		.of_match_table = sony_asoc_card_dt_match,
		#endif
	},
	.probe = sony_asoc_card_probe,
	.remove = sony_asoc_card_remove
};

module_platform_driver(sony_asoc_card);

/* Module information */
MODULE_DESCRIPTION("sony multi sai driver");
MODULE_LICENSE("GPL");
MODULE_ALIAS("sony multi sai card");

