/*
 * Copyright (c) 2011-2019, 2021 The Linux Foundation. All rights reserved.
 *
 * Previously licensed under the ISC license by Qualcomm Atheros, Inc.
 *
 *
 * Permission to use, copy, modify, and/or distribute this software for
 * any purpose with or without fee is hereby granted, provided that the
 * above copyright notice and this permission notice appear in all
 * copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE
 * AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

/*
 * This file was originally distributed by Qualcomm Atheros, Inc.
 * under proprietary terms before Copyright ownership was assigned
 * to the Linux Foundation.
 */


/** ------------------------------------------------------------------------- *
    ------------------------------------------------------------------------- *
    \file csrApi.h

    Exports and types for the Common Scan and Roaming Module interfaces.
========================================================================== */
#ifndef CSRAPI_H__
#define CSRAPI_H__

#ifdef WLAN_FEATURE_FILS_SK
#include "lim_fils_defs.h"
#endif
#include "sirApi.h"
#include "sirMacProtDef.h"
#include "csrLinkList.h"

typedef enum
{
    eCSR_AUTH_TYPE_NONE,    //never used
    // MAC layer authentication types
    eCSR_AUTH_TYPE_OPEN_SYSTEM,
    eCSR_AUTH_TYPE_SHARED_KEY,
    eCSR_AUTH_TYPE_SAE,
    eCSR_AUTH_TYPE_AUTOSWITCH,

    // Upper layer authentication types
    eCSR_AUTH_TYPE_WPA,
    eCSR_AUTH_TYPE_WPA_PSK,
    eCSR_AUTH_TYPE_WPA_NONE,

    eCSR_AUTH_TYPE_RSN,
    eCSR_AUTH_TYPE_RSN_PSK,
#if defined WLAN_FEATURE_VOWIFI_11R
    eCSR_AUTH_TYPE_FT_RSN,
    eCSR_AUTH_TYPE_FT_RSN_PSK,
#endif
#ifdef FEATURE_WLAN_WAPI
    eCSR_AUTH_TYPE_WAPI_WAI_CERTIFICATE,
    eCSR_AUTH_TYPE_WAPI_WAI_PSK,
#endif /* FEATURE_WLAN_WAPI */
#ifdef FEATURE_WLAN_ESE
    eCSR_AUTH_TYPE_CCKM_WPA,
    eCSR_AUTH_TYPE_CCKM_RSN,
#endif /* FEATURE_WLAN_ESE */
#ifdef WLAN_FEATURE_11W
    eCSR_AUTH_TYPE_RSN_PSK_SHA256,
    eCSR_AUTH_TYPE_RSN_8021X_SHA256,
#endif
#ifdef WLAN_FEATURE_FILS_SK
    eCSR_AUTH_TYPE_FILS_SHA256,
    eCSR_AUTH_TYPE_FILS_SHA384,
    eCSR_AUTH_TYPE_FT_FILS_SHA256,
    eCSR_AUTH_TYPE_FT_FILS_SHA384,
#endif
    eCSR_AUTH_TYPE_OWE,
    eCSR_NUM_OF_SUPPORT_AUTH_TYPE,
    eCSR_AUTH_TYPE_FAILED = 0xff,
    eCSR_AUTH_TYPE_UNKNOWN = eCSR_AUTH_TYPE_FAILED,

}eCsrAuthType;


typedef enum
{
    eCSR_ENCRYPT_TYPE_NONE,
    eCSR_ENCRYPT_TYPE_WEP40_STATICKEY,
    eCSR_ENCRYPT_TYPE_WEP104_STATICKEY,

    eCSR_ENCRYPT_TYPE_WEP40,
    eCSR_ENCRYPT_TYPE_WEP104,
    eCSR_ENCRYPT_TYPE_TKIP,
    eCSR_ENCRYPT_TYPE_AES,
#ifdef FEATURE_WLAN_WAPI
    eCSR_ENCRYPT_TYPE_WPI, //WAPI
#endif /* FEATURE_WLAN_WAPI */
#ifdef FEATURE_WLAN_ESE
    eCSR_ENCRYPT_TYPE_KRK,
#ifdef WLAN_FEATURE_ROAM_OFFLOAD
    eCSR_ENCRYPT_TYPE_BTK,
#endif
#endif /* FEATURE_WLAN_ESE */
#ifdef WLAN_FEATURE_11W
    //11w BIP
    eCSR_ENCRYPT_TYPE_AES_CMAC,
#endif
    eCSR_ENCRYPT_TYPE_ANY,
    eCSR_NUM_OF_ENCRYPT_TYPE = eCSR_ENCRYPT_TYPE_ANY,

    eCSR_ENCRYPT_TYPE_FAILED = 0xff,
    eCSR_ENCRYPT_TYPE_UNKNOWN = eCSR_ENCRYPT_TYPE_FAILED,

}eCsrEncryptionType;

/*---------------------------------------------------------------------------
   Enumeration of the various Security types
---------------------------------------------------------------------------*/
typedef enum
{
    eCSR_SECURITY_TYPE_WPA,
    eCSR_SECURITY_TYPE_RSN,
#ifdef FEATURE_WLAN_WAPI
    eCSR_SECURITY_TYPE_WAPI,
#endif /* FEATURE_WLAN_WAPI */
    eCSR_SECURITY_TYPE_UNKNOWN,

}eCsrSecurityType;

typedef enum {
    eCSR_DOT11_MODE_abg = 0x0001, /* 11a/b/g only, no HT, no proprietary */
    eCSR_DOT11_MODE_11a = 0x0002,
    eCSR_DOT11_MODE_11b = 0x0004,
    eCSR_DOT11_MODE_11g = 0x0008,
    eCSR_DOT11_MODE_11n = 0x0010,
    eCSR_DOT11_MODE_11g_ONLY = 0x0020,
    eCSR_DOT11_MODE_11n_ONLY = 0x0040,
    eCSR_DOT11_MODE_11b_ONLY = 0x0080,
#ifdef WLAN_FEATURE_11AC
    eCSR_DOT11_MODE_11ac     = 0x0100,
    eCSR_DOT11_MODE_11ac_ONLY = 0x0200,
#endif
    /*
     * This is for WIFI test. It is same as eWNIAPI_MAC_PROTOCOL_ALL
     * except when it starts IBSS in 11B of 2.4GHz
     * It is for CSR internal use
     */
    eCSR_DOT11_MODE_AUTO = 0x0400,

    eCSR_NUM_PHY_MODE = 16, /* specify the number of maximum bits for phyMode */

} eCsrPhyMode;

typedef tANI_U8 tCsrBssid[VOS_MAC_ADDR_SIZE];

typedef enum {
    eCSR_BSS_TYPE_INFRASTRUCTURE,
    eCSR_BSS_TYPE_INFRA_AP,       /* Soft AP */
    eCSR_BSS_TYPE_IBSS,           /* IBSS network, will NOT start */
    eCSR_BSS_TYPE_START_IBSS,     /* IBSS network, will start if no partner */
    eCSR_BSS_TYPE_WDS_AP,         /* BT-AMP AP */
    eCSR_BSS_TYPE_WDS_STA,        /* BT-AMP station */
    eCSR_BSS_TYPE_NDI,            /* NAN data path interface */
    eCSR_BSS_TYPE_ANY,            /* any BSS type (IBSS or Infrastructure) */
} eCsrRoamBssType;



typedef enum {
    eCSR_SCAN_REQUEST_11D_SCAN = 1,
    eCSR_SCAN_REQUEST_FULL_SCAN,
    eCSR_SCAN_IDLE_MODE_SCAN,
    eCSR_SCAN_HO_BG_SCAN, // bg scan request in NRT & RT Handoff sub-states
    eCSR_SCAN_HO_PROBE_SCAN, // directed probe on an entry from the candidate list
    eCSR_SCAN_HO_NT_BG_SCAN, // bg scan request in NT  sub-state
    eCSR_SCAN_P2P_DISCOVERY,

    eCSR_SCAN_SOFTAP_CHANNEL_RANGE,
    eCSR_SCAN_P2P_FIND_PEER,
}eCsrRequestType;

typedef enum {
    eCSR_SCAN_RESULT_GET = 0,
    eCSR_SCAN_RESULT_FLUSH = 1,     //to delete all cached scan results
}eCsrScanResultCmd;

typedef enum
{
    eCSR_SCAN_SUCCESS,
    eCSR_SCAN_FAILURE,
    eCSR_SCAN_ABORT,
   eCSR_SCAN_FOUND_PEER,
}eCsrScanStatus;

/* Reason to abort the scan
 * The reason can used later to decide whether to update the scan results
 * to upper layer or not
 */
typedef enum
{
    eCSR_SCAN_ABORT_DEFAULT = 1,
    eCSR_SCAN_ABORT_DUE_TO_BAND_CHANGE, //Scan aborted due to band change
}eCsrAbortReason;

typedef enum
{
    eCSR_BW_20MHz_VAL = 20,
    eCSR_BW_40MHz_VAL = 40,
    eCSR_BW_80MHz_VAL = 80,
    eCSR_BW_160MHz_VAL = 160
}eCSR_BW_Val;

typedef enum
{
   eCSR_INI_SINGLE_CHANNEL_CENTERED = 0,
   eCSR_INI_DOUBLE_CHANNEL_HIGH_PRIMARY,
   eCSR_INI_DOUBLE_CHANNEL_LOW_PRIMARY,
#ifdef WLAN_FEATURE_11AC
   eCSR_INI_QUADRUPLE_CHANNEL_20MHZ_LOW_40MHZ_CENTERED,
   eCSR_INI_QUADRUPLE_CHANNEL_20MHZ_CENTERED_40MHZ_CENTERED,
   eCSR_INI_QUADRUPLE_CHANNEL_20MHZ_HIGH_40MHZ_CENTERED,
   eCSR_INI_QUADRUPLE_CHANNEL_20MHZ_LOW_40MHZ_LOW,
   eCSR_INI_QUADRUPLE_CHANNEL_20MHZ_HIGH_40MHZ_LOW,
   eCSR_INI_QUADRUPLE_CHANNEL_20MHZ_LOW_40MHZ_HIGH,
   eCSR_INI_QUADRUPLE_CHANNEL_20MHZ_HIGH_40MHZ_HIGH,
#endif
   eCSR_INI_CHANNEL_BONDING_STATE_MAX
}eIniChanBondState;


#define CSR_RSN_PMKID_SIZE          16
#define CSR_RSN_MAX_PMK_LEN         48
#define CSR_MAX_PMKID_ALLOWED       32
#define CSR_WEP40_KEY_LEN       5
#define CSR_WEP104_KEY_LEN      13
#define CSR_TKIP_KEY_LEN        32
#define CSR_AES_KEY_LEN         16
#define CSR_MAX_TX_POWER        ( WNI_CFG_CURRENT_TX_POWER_LEVEL_STAMAX )
#define CSR_MAX_RSC_LEN          16
#ifdef FEATURE_WLAN_WAPI
#define CSR_WAPI_BKID_SIZE          16
#define CSR_MAX_BKID_ALLOWED        16
#define CSR_WAPI_KEY_LEN        32
#define CSR_MAX_KEY_LEN         ( CSR_WAPI_KEY_LEN )  //longest one is for WAPI
#else
#define CSR_MAX_KEY_LEN         ( CSR_TKIP_KEY_LEN )  //longest one is for TKIP
#endif /* FEATURE_WLAN_WAPI */
#ifdef FEATURE_WLAN_ESE
#define CSR_KRK_KEY_LEN 16
#endif
/* Cache ID length */
#define CACHE_ID_LEN 2

typedef struct tagCsrChannelInfo
{
    tANI_U8 numOfChannels;
    tANI_U8 *ChannelList;   //it will be an array of channels
}tCsrChannelInfo, *tpCsrChannelInfo;

typedef struct tagCsrSSIDInfo
{
   tSirMacSSid     SSID;
   tANI_BOOLEAN    handoffPermitted;
   tANI_BOOLEAN    ssidHidden;
}tCsrSSIDInfo;

typedef struct tagCsrSSIDs
{
    tANI_U32 numOfSSIDs;
    tCsrSSIDInfo *SSIDList;   //To be allocated for array of SSIDs
}tCsrSSIDs;

typedef struct tagCsrBSSIDs
{
    tANI_U32 numOfBSSIDs;
    tCsrBssid *bssid;
}tCsrBSSIDs;

typedef struct tagCsrStaParams
{
    tANI_U16   capability;
    tANI_U8    extn_capability[SIR_MAC_MAX_EXTN_CAP];
    tANI_U8    supported_rates_len;
    tANI_U8    supported_rates[SIR_MAC_MAX_SUPP_RATES];
    tANI_U8    htcap_present;
    tSirHTCap  HTCap;
    tANI_U8    vhtcap_present;
    tSirVHTCap VHTCap;
    tANI_U8    uapsd_queues;
    tANI_U8    max_sp;
    tANI_U8    supported_channels_len;
    tANI_U8    supported_channels[SIR_MAC_MAX_SUPP_CHANNELS];
    tANI_U8    supported_oper_classes_len;
    tANI_U8    supported_oper_classes[SIR_MAC_MAX_SUPP_OPER_CLASSES];
}tCsrStaParams;

typedef struct tagCsrScanRequest
{
    tSirScanType scanType;
    tCsrBssid bssid;
    eCsrRoamBssType BSSType;
    tCsrSSIDs SSIDs;
    tCsrChannelInfo ChannelInfo;
    tANI_U32 minChnTime;    //in units of milliseconds
    tANI_U32 maxChnTime;    //in units of milliseconds
    tANI_U32 minChnTimeBtc;    //in units of milliseconds
    tANI_U32 maxChnTimeBtc;    //in units of milliseconds
    /* In units of milliseconds, ignored when not connected */
    uint32_t restTime;
    /* In units of milliseconds, ignored when not connected */
    uint32_t min_rest_time;
    /* In units of milliseconds, ignored when not connected */
    uint32_t idle_time;

    tANI_U32 uIEFieldLen;
    tANI_U8 *pIEField;
    eCsrRequestType requestType;    //11d scan or full scan
    tANI_BOOLEAN p2pSearch;
    tANI_BOOLEAN skipDfsChnlInP2pSearch;

    uint32_t enable_scan_randomization;
    uint8_t mac_addr[VOS_MAC_ADDR_SIZE];
    uint8_t mac_addr_mask[VOS_MAC_ADDR_SIZE];
    bool ie_whitelist;
    uint32_t probe_req_ie_bitmap[PROBE_REQ_BITMAP_LEN];
    uint32_t num_vendor_oui;
    struct vendor_oui *voui;
}tCsrScanRequest;

typedef struct tagCsrBGScanRequest
{
    tSirScanType scanType;
    tSirMacSSid SSID;
    tCsrChannelInfo ChannelInfo;
    tANI_U32 scanInterval;  //in units of milliseconds
    tANI_U32 minChnTime;    //in units of milliseconds
    tANI_U32 maxChnTime;    //in units of milliseconds
    tANI_U32 minChnTimeBtc;    //in units of milliseconds
    tANI_U32 maxChnTimeBtc;    //in units of milliseconds
    /* In units of milliseconds, ignored when not connected */
    uint32_t restTime;
    /* In units of milliseconds, ignored when not connected */
    uint32_t min_rest_time;
    /* In units of milliseconds, ignored when not connected */
    uint32_t idle_time;
    tANI_U32 throughputImpact;      //specify whether BG scan cares about impacting throughput  //ignored when not connected
    tCsrBssid bssid;    //how to use it?? Apple
}tCsrBGScanRequest, *tpCsrBGScanRequest;


typedef struct tagCsrScanResultInfo
{
    //Carry the IEs for the current BSSDescription. A pointer to tDot11fBeaconIEs. Maybe NULL for start BSS.
    void *pvIes;
    tAniSSID ssId;
    v_TIME_t timer; // timer is variable which is used for hidden SSID's timer value
    //This member must be the last in the structure because the end of tSirBssDescription is an
    //    array with nonknown size at this time
    tSirBssDescription BssDescriptor;
}tCsrScanResultInfo;

typedef struct tagCsrEncryptionList
{

    tANI_U32 numEntries;
    eCsrEncryptionType encryptionType[eCSR_NUM_OF_ENCRYPT_TYPE];

}tCsrEncryptionList, *tpCsrEncryptionList;

typedef struct tagCsrAuthList
{
    tANI_U32 numEntries;
    eCsrAuthType authType[eCSR_NUM_OF_SUPPORT_AUTH_TYPE];
}tCsrAuthList, *tpCsrAuthList;

#ifdef WLAN_FEATURE_VOWIFI_11R
typedef struct tagCsrMobilityDomainInfo
{
    tANI_U8 mdiePresent;
    tANI_U16 mobilityDomain;
} tCsrMobilityDomainInfo;
#endif

#ifdef FEATURE_WLAN_ESE
typedef struct tagCsrEseCckmInfo
{
    tANI_U32       reassoc_req_num;
    tANI_BOOLEAN   krk_plumbed;
    tANI_U8        krk[SIR_KRK_KEY_LEN];
#ifdef WLAN_FEATURE_ROAM_OFFLOAD
    tANI_U8        btk[SIR_BTK_KEY_LEN];
#endif
} tCsrEseCckmInfo;
#endif

#if defined(FEATURE_WLAN_ESE) && defined(FEATURE_WLAN_ESE_UPLOAD)
typedef struct tagCsrEseCckmIe
{
    tANI_U8 cckmIe[DOT11F_IE_RSN_MAX_LEN];
    tANI_U8 cckmIeLen;
} tCsrEseCckmIe;
#endif /* FEATURE_WLAN_ESE && FEATURE_WLAN_ESE_UPLOAD */

typedef struct tagCsrScanResultFilter
{
    tCsrBSSIDs BSSIDs;    //each bssid has a length of VOS_MAC_ADDR_SIZE (6)
    tCsrSSIDs SSIDs;
    tCsrChannelInfo ChannelInfo;
    tCsrAuthList authType;
    tCsrEncryptionList EncryptionType;
    //eCSR_ENCRYPT_TYPE_ANY cannot be set in multicast encryption type. If caller doesn't case,
    //put all supported encryption types in here
    tCsrEncryptionList mcEncryptionType;
    eCsrRoamBssType BSSType;
    //this is a bit mask of all the needed phy mode defined in eCsrPhyMode
    tANI_U32 phyMode;
    //If countryCode[0] is not 0, countryCode is checked independent of fCheckUnknownCountryCode
    tANI_U8 countryCode[WNI_CFG_COUNTRY_CODE_LEN];
    tANI_U8 uapsd_mask;
    /*For WPS filtering if true => auth and ecryption should be ignored*/
    tANI_BOOLEAN bWPSAssociation;
    tANI_BOOLEAN bOSENAssociation;
#if defined WLAN_FEATURE_VOWIFI
    /*For measurement reports --> if set, only SSID, BSSID and channel is considered for filtering.*/
    tANI_BOOLEAN fMeasurement;
#endif
#ifdef WLAN_FEATURE_VOWIFI_11R
    tCsrMobilityDomainInfo MDID;
#endif
    tANI_BOOLEAN p2pResult;
#ifdef WLAN_FEATURE_11W
    // Management Frame Protection
    tANI_BOOLEAN MFPEnabled;
    tANI_U8 MFPRequired;
    tANI_U8 MFPCapable;
#endif
    /* The following flag is used to distinguish the
     * roaming case while building the scan filter and
     * applying it on to the scan results. This is mainly
     * used to support whitelist ssid feature.
     */
    uint8_t scan_filter_for_roam;
    tCsrBssid bssid_hint;
#ifdef FEATURE_WLAN_DISABLE_CHANNEL_SWITCH
    tVOS_CON_MODE csrPersona;
#endif
#ifdef WLAN_FEATURE_FILS_SK
    bool realm_check;
    uint8_t fils_realm[2];
#endif
}tCsrScanResultFilter;


typedef struct sCsrChnPower_
{
  tANI_U8 firstChannel;
  tANI_U8 numChannels;
  tANI_U8 maxtxPower;
}sCsrChnPower;


typedef struct sCsrChannel_
{
    tANI_U8 numChannels;
    tANI_U8 channelList[WNI_CFG_VALID_CHANNEL_LIST_LEN];
}sCsrChannel;


typedef struct tagCsr11dinfo
{
  sCsrChannel     Channels;
  tANI_U8         countryCode[WNI_CFG_COUNTRY_CODE_LEN+1];
  //max power channel list
  sCsrChnPower    ChnPower[WNI_CFG_VALID_CHANNEL_LIST_LEN];
}tCsr11dinfo;


typedef enum
{
    eCSR_ROAM_CANCELLED = 1,
    //this mean error happens before association_start or roaming_start is called.
    eCSR_ROAM_FAILED,
    //a CSR trigger roaming operation starts, callback may get a pointer to tCsrConnectedProfile
    eCSR_ROAM_ROAMING_START,
    //a CSR trigger roaming operation is completed
    eCSR_ROAM_ROAMING_COMPLETION,
    //Connection completed status.
    eCSR_ROAM_CONNECT_COMPLETION,
    //an association or start_IBSS operation starts,
    //callback may get a pointer to tCsrRoamProfile and a pointer to tSirBssDescription
    eCSR_ROAM_ASSOCIATION_START,
    //a roaming operation is finish, see eCsrRoamResult for
    //possible data passed back
    eCSR_ROAM_ASSOCIATION_COMPLETION,
    eCSR_ROAM_DISASSOCIATED,
    eCSR_ROAM_ASSOCIATION_FAILURE,
    //when callback with this flag. callback gets a pointer to the BSS desc.
    eCSR_ROAM_SHOULD_ROAM,
    //A new candidate for PMKID is found
    eCSR_ROAM_SCAN_FOUND_NEW_BSS,
    //CSR is done lostlink roaming and still cannot reconnect
    eCSR_ROAM_LOSTLINK,
    //a link lost is detected. CSR starts roaming.
    eCSR_ROAM_LOSTLINK_DETECTED,
    //TKIP MIC error detected, callback gets a pointer to tpSirSmeMicFailureInd
    eCSR_ROAM_MIC_ERROR_IND,
    eCSR_ROAM_IBSS_IND, //IBSS indications.
    //Update the connection status, useful for IBSS: new peer added, network is active etc.
    eCSR_ROAM_CONNECT_STATUS_UPDATE,
    eCSR_ROAM_GEN_INFO,
    eCSR_ROAM_SET_KEY_COMPLETE,
    eCSR_ROAM_REMOVE_KEY_COMPLETE,
    eCSR_ROAM_IBSS_LEAVE, //IBSS indications.
    //BSS in WDS mode status indication
    eCSR_ROAM_WDS_IND,
    //BSS in SoftAP mode status indication
    eCSR_ROAM_INFRA_IND,
    eCSR_ROAM_WPS_PBC_PROBE_REQ_IND,
#ifdef WLAN_FEATURE_VOWIFI_11R
    eCSR_ROAM_FT_RESPONSE,
#endif
    eCSR_ROAM_FT_START,
    eCSR_ROAM_REMAIN_CHAN_READY,
    //this mean error happens before association_start or roaming_start is called.
    eCSR_ROAM_SESSION_OPENED,
    eCSR_ROAM_FT_REASSOC_FAILED,
#ifdef FEATURE_WLAN_LFR
    eCSR_ROAM_PMK_NOTIFY,
#endif
#ifdef FEATURE_WLAN_LFR_METRICS
    eCSR_ROAM_PREAUTH_INIT_NOTIFY,
    eCSR_ROAM_PREAUTH_STATUS_SUCCESS,
    eCSR_ROAM_PREAUTH_STATUS_FAILURE,
    eCSR_ROAM_HANDOVER_SUCCESS,
#endif
#ifdef FEATURE_WLAN_TDLS
    eCSR_ROAM_TDLS_STATUS_UPDATE,
    eCSR_ROAM_RESULT_MGMT_TX_COMPLETE_IND,
#endif
    eCSR_ROAM_DISCONNECT_ALL_P2P_CLIENTS, //Disaconnect all the clients
    eCSR_ROAM_SEND_P2P_STOP_BSS, //Stopbss triggered from SME due to different
                                 // beacon interval
#ifdef WLAN_FEATURE_11W
    eCSR_ROAM_UNPROT_MGMT_FRAME_IND,
#endif

    eCSR_ROAM_IBSS_PEER_INFO_COMPLETE,

#if defined(FEATURE_WLAN_ESE) && defined(FEATURE_WLAN_ESE_UPLOAD)
    eCSR_ROAM_TSM_IE_IND,
    eCSR_ROAM_CCKM_PREAUTH_NOTIFY,
    eCSR_ROAM_ESE_ADJ_AP_REPORT_IND,
    eCSR_ROAM_ESE_BCN_REPORT_IND,
#endif /* FEATURE_WLAN_ESE && FEATURE_WLAN_ESE_UPLOAD */

    // Radar indication from lower layers
    eCSR_ROAM_DFS_RADAR_IND,
    eCSR_ROAM_SET_CHANNEL_RSP,

    // Channel sw update notification
    eCSR_ROAM_DFS_CHAN_SW_NOTIFY,
    eCSR_ROAM_EXT_CHG_CHNL_IND,

    eCSR_ROAM_NDP_STATUS_UPDATE,
    eCSR_ROAM_UPDATE_SCAN_RESULT,
    eCSR_ROAM_SAE_COMPUTE,
}eRoamCmdStatus;


//comment inside indicates what roaming callback gets
typedef enum
{
    eCSR_ROAM_RESULT_NONE,
    //this means no more action in CSR
    //If roamStatus is eCSR_ROAM_ASSOCIATION_COMPLETION, tCsrRoamInfo's pBssDesc may pass back
    eCSR_ROAM_RESULT_FAILURE,
    //Pass back pointer to tCsrRoamInfo
    eCSR_ROAM_RESULT_ASSOCIATED,
    eCSR_ROAM_RESULT_NOT_ASSOCIATED,
    eCSR_ROAM_RESULT_MIC_FAILURE,
    eCSR_ROAM_RESULT_FORCED,
    eCSR_ROAM_RESULT_DISASSOC_IND,
    eCSR_ROAM_RESULT_DEAUTH_IND,
    eCSR_ROAM_RESULT_CAP_CHANGED,
    //This means we starts an IBSS
    //tCsrRoamInfo's pBssDesc may pass back
    eCSR_ROAM_RESULT_IBSS_STARTED,
    //START_BSS failed
    //tCsrRoamInfo's pBssDesc may pass back
    eCSR_ROAM_RESULT_IBSS_START_FAILED,
    eCSR_ROAM_RESULT_IBSS_JOIN_SUCCESS,
    eCSR_ROAM_RESULT_IBSS_JOIN_FAILED,
    eCSR_ROAM_RESULT_IBSS_CONNECT,
    eCSR_ROAM_RESULT_IBSS_INACTIVE,
    //If roamStatus is eCSR_ROAM_ASSOCIATION_COMPLETION
    //tCsrRoamInfo's pBssDesc may pass back. and the peer's MAC address in peerMacOrBssid
    //If roamStatus is eCSR_ROAM_IBSS_IND,
    //the peer's MAC address in peerMacOrBssid and a beacon frame of the IBSS in pbFrames
    eCSR_ROAM_RESULT_IBSS_NEW_PEER,
    //Peer departed from IBSS, Callback may get a pointer tSmeIbssPeerInd in pIbssPeerInd
    eCSR_ROAM_RESULT_IBSS_PEER_DEPARTED,
    //Coalescing in the IBSS network (joined an IBSS network)
    //Callback pass a BSSID in peerMacOrBssid
    eCSR_ROAM_RESULT_IBSS_COALESCED,
    //If roamStatus is eCSR_ROAM_ROAMING_START, callback may get a pointer to tCsrConnectedProfile used to connect.
    eCSR_ROAM_RESULT_IBSS_STOP,
    eCSR_ROAM_RESULT_LOSTLINK,
    eCSR_ROAM_RESULT_MIC_ERROR_UNICAST,
    eCSR_ROAM_RESULT_MIC_ERROR_GROUP,
    eCSR_ROAM_RESULT_AUTHENTICATED,
    eCSR_ROAM_RESULT_NEW_RSN_BSS,
#ifdef FEATURE_WLAN_WAPI
    eCSR_ROAM_RESULT_NEW_WAPI_BSS,
#endif /* FEATURE_WLAN_WAPI */
    // WDS started successfully
    eCSR_ROAM_RESULT_WDS_STARTED,
    // WDS start failed
    eCSR_ROAM_RESULT_WDS_START_FAILED,
    // WDS stopped
    eCSR_ROAM_RESULT_WDS_STOPPED,
    // WDS joined successfully in STA mode
    eCSR_ROAM_RESULT_WDS_ASSOCIATED,
    // A station joined WDS AP
    eCSR_ROAM_RESULT_WDS_ASSOCIATION_IND,
    // WDS join failed in STA mode
    eCSR_ROAM_RESULT_WDS_NOT_ASSOCIATED,
    // WDS disassociated
    eCSR_ROAM_RESULT_WDS_DISASSOCIATED,
    // INFRA started successfully
    eCSR_ROAM_RESULT_INFRA_STARTED,
    // INFRA start failed
    eCSR_ROAM_RESULT_INFRA_START_FAILED,
    // INFRA stopped
    eCSR_ROAM_RESULT_INFRA_STOPPED,
    // A station joining INFRA AP
    eCSR_ROAM_RESULT_INFRA_ASSOCIATION_IND,
    // A station joined INFRA AP
    eCSR_ROAM_RESULT_INFRA_ASSOCIATION_CNF,
    // INFRA disassociated
    eCSR_ROAM_RESULT_INFRA_DISASSOCIATED,
    eCSR_ROAM_RESULT_WPS_PBC_PROBE_REQ_IND,
    eCSR_ROAM_RESULT_SEND_ACTION_FAIL,
    // peer rejected assoc because max assoc limit reached. callback gets pointer to peer
    eCSR_ROAM_RESULT_MAX_ASSOC_EXCEEDED,
    //Assoc rejected due to concurrent session running on a different channel
    eCSR_ROAM_RESULT_ASSOC_FAIL_CON_CHANNEL,
#ifdef FEATURE_WLAN_TDLS
    eCSR_ROAM_RESULT_ADD_TDLS_PEER,
    eCSR_ROAM_RESULT_UPDATE_TDLS_PEER,
    eCSR_ROAM_RESULT_DELETE_TDLS_PEER,
    eCSR_ROAM_TDLS_CHECK_BMPS,
    eCSR_ROAM_RESULT_TEARDOWN_TDLS_PEER_IND,
    eCSR_ROAM_RESULT_DELETE_ALL_TDLS_PEER_IND,
    eCSR_ROAM_RESULT_LINK_ESTABLISH_REQ_RSP,
    eCSR_ROAM_RESULT_TDLS_SHOULD_DISCOVER,
    eCSR_ROAM_RESULT_TDLS_SHOULD_TEARDOWN,
    eCSR_ROAM_RESULT_TDLS_SHOULD_PEER_DISCONNECTED,
#endif

    eCSR_ROAM_RESULT_IBSS_PEER_INFO_SUCCESS,
    eCSR_ROAM_RESULT_IBSS_PEER_INFO_FAILED,

    eCSR_ROAM_RESULT_DFS_RADAR_FOUND_IND,
    eCSR_ROAM_RESULT_CHANNEL_CHANGE_SUCCESS,
    eCSR_ROAM_RESULT_CHANNEL_CHANGE_FAILURE,
    eCSR_ROAM_RESULT_DFS_CHANSW_UPDATE_SUCCESS,
    eCSR_ROAM_RESULT_DFS_CHANSW_UPDATE_FAILURE,
    eCSR_ROAM_EXT_CHG_CHNL_UPDATE_IND,

    eCSR_ROAM_RESULT_NDI_CREATE_RSP,
    eCSR_ROAM_RESULT_NDI_DELETE_RSP,
    eCSR_ROAM_RESULT_NDP_INITIATOR_RSP,
    eCSR_ROAM_RESULT_NDP_NEW_PEER_IND,
    eCSR_ROAM_RESULT_NDP_CONFIRM_IND,
    eCSR_ROAM_RESULT_NDP_INDICATION,
    eCSR_ROAM_RESULT_NDP_RESPONDER_RSP,
    eCSR_ROAM_RESULT_NDP_END_RSP,
    eCSR_ROAM_RESULT_NDP_PEER_DEPARTED_IND,
    eCSR_ROAM_RESULT_NDP_END_IND,
    /* If Scan for SSID failed to found proper BSS */
    eCSR_ROAM_RESULT_SCAN_FOR_SSID_FAILURE,
}eCsrRoamResult;



/*----------------------------------------------------------------------------
  List of link quality indications HDD can receive from SME
-----------------------------------------------------------------------------*/
typedef enum
{
 eCSR_ROAM_LINK_QUAL_MIN_IND     = -1,

 eCSR_ROAM_LINK_QUAL_POOR_IND            =  0,   /* bad link                */
 eCSR_ROAM_LINK_QUAL_GOOD_IND            =  1,   /* acceptable for voice    */
 eCSR_ROAM_LINK_QUAL_VERY_GOOD_IND       =  2,   /* suitable for voice      */
 eCSR_ROAM_LINK_QUAL_EXCELLENT_IND       =  3,   /* suitable for voice      */

 eCSR_ROAM_LINK_QUAL_MAX_IND  /* invalid value */

} eCsrRoamLinkQualityInd;

typedef enum
{
    eCSR_DISCONNECT_REASON_UNSPECIFIED = 0,
    eCSR_DISCONNECT_REASON_MIC_ERROR,
    eCSR_DISCONNECT_REASON_DISASSOC,
    eCSR_DISCONNECT_REASON_DEAUTH,
    eCSR_DISCONNECT_REASON_HANDOFF,
    eCSR_DISCONNECT_REASON_IBSS_JOIN_FAILURE,
    eCSR_DISCONNECT_REASON_IBSS_LEAVE,
    eCSR_DISCONNECT_REASON_STA_HAS_LEFT,
    eCSR_DISCONNECT_REASON_NDI_DELETE,
}eCsrRoamDisconnectReason;

typedef enum
{
    // Not associated in Infra or participating in an IBSS / Ad-hoc network.
    eCSR_ASSOC_STATE_TYPE_NOT_CONNECTED,
    // Associated in an Infrastructure network.
    eCSR_ASSOC_STATE_TYPE_INFRA_ASSOCIATED,
    // Participating in an IBSS network though disconnected (no partner stations
    // in the IBSS).
    eCSR_ASSOC_STATE_TYPE_IBSS_DISCONNECTED,
    // Participating in an IBSS network with partner stations also present
    eCSR_ASSOC_STATE_TYPE_IBSS_CONNECTED,
    // Participating in a WDS network in AP or STA mode but not connected yet
    eCSR_ASSOC_STATE_TYPE_WDS_DISCONNECTED,
    // Participating in a WDS network and connected peer to peer
    eCSR_ASSOC_STATE_TYPE_WDS_CONNECTED,
    // Participating in a Infra network in AP not yet in connected state
    eCSR_ASSOC_STATE_TYPE_INFRA_DISCONNECTED,
    // Participating in a Infra network and connected to a peer
    eCSR_ASSOC_STATE_TYPE_INFRA_CONNECTED,
    // Disconnecting with AP or stop connecting process
    eCSR_ASSOC_STATE_TYPE_INFRA_DISCONNECTING,
    /* NAN Data interface not started */
    eCSR_CONNECT_STATE_TYPE_NDI_NOT_STARTED,
    /* NAN Data inteface started */
    eCSR_CONNECT_STATE_TYPE_NDI_STARTED,
}eCsrConnectState;


// This parameter is no longer supported in the Profile.  Need to set this in the global properties
// for the adapter.
typedef enum eCSR_MEDIUM_ACCESS
{
    eCSR_MEDIUM_ACCESS_AUTO = 0,
    eCSR_MEDIUM_ACCESS_DCF,
    eCSR_MEDIUM_ACCESS_eDCF,
    eCSR_MEDIUM_ACCESS_HCF,

    eCSR_MEDIUM_ACCESS_WMM_eDCF_802dot1p,
    eCSR_MEDIUM_ACCESS_WMM_eDCF_DSCP,
    eCSR_MEDIUM_ACCESS_WMM_eDCF_NoClassify,
    eCSR_MEDIUM_ACCESS_11e_eDCF = eCSR_MEDIUM_ACCESS_eDCF,
    eCSR_MEDIUM_ACCESS_11e_HCF  = eCSR_MEDIUM_ACCESS_HCF,
}eCsrMediaAccessType;

typedef enum
{
    eCSR_TX_RATE_AUTO = 0,   // use rate adaption to determine Tx rate.

    eCSR_TX_RATE_1Mbps   = 0x00000001,
    eCSR_TX_RATE_2Mbps   = 0x00000002,
    eCSR_TX_RATE_5_5Mbps = 0x00000004,
    eCSR_TX_RATE_6Mbps   = 0x00000008,
    eCSR_TX_RATE_9Mbps   = 0x00000010,
    eCSR_TX_RATE_11Mbps  = 0x00000020,
    eCSR_TX_RATE_12Mbps  = 0x00000040,
    eCSR_TX_RATE_18Mbps  = 0x00000080,
    eCSR_TX_RATE_24Mbps  = 0x00000100,
    eCSR_TX_RATE_36Mbps  = 0x00000200,
    eCSR_TX_RATE_42Mbps  = 0x00000400,
    eCSR_TX_RATE_48Mbps  = 0x00000800,
    eCSR_TX_RATE_54Mbps  = 0x00001000,
    eCSR_TX_RATE_72Mbps  = 0x00002000,
    eCSR_TX_RATE_84Mbps  = 0x00004000,
    eCSR_TX_RATE_96Mbps  = 0x00008000,
    eCSR_TX_RATE_108Mbps = 0x00010000,
    eCSR_TX_RATE_126Mbps = 0x00020000,
    eCSR_TX_RATE_144Mbps = 0x00040000,
    eCSR_TX_RATE_168Mbps = 0x00080000,
    eCSR_TX_RATE_192Mbps = 0x00100000,
    eCSR_TX_RATE_216Mbps = 0x00200000,
    eCSR_TX_RATE_240Mbps = 0x00400000,

}eCsrExposedTxRate;

typedef enum
{
    eCSR_OPERATING_CHANNEL_ALL  = 0,
    eCSR_OPERATING_CHANNEL_AUTO = eCSR_OPERATING_CHANNEL_ALL,
    eCSR_OPERATING_CHANNEL_ANY  = eCSR_OPERATING_CHANNEL_ALL,
}eOperationChannel;

typedef enum
{
    eCSR_DOT11_FRAG_THRESH_AUTO            = -1,
    eCSR_DOT11_FRAG_THRESH_MIN             = 256,
    eCSR_DOT11_FRAG_THRESH_MAX             = 2346,
    eCSR_DOT11_FRAG_THRESH_DEFAULT         = 2000
}eCsrDot11FragThresh;


//for channel bonding for ibss
typedef enum
{
    eCSR_CB_OFF = 0,
    eCSR_CB_AUTO = 1,
    eCSR_CB_DOWN = 2,
    eCSR_CB_UP = 3,
}eCsrCBChoice;

//For channel bonding, the channel number gap is 4, either up or down. For both 11a and 11g mode.
#define CSR_CB_CHANNEL_GAP 4
#define CSR_CB_CENTER_CHANNEL_OFFSET    2
#define CSR_MAX_24GHz_CHANNEL_NUMBER ( SIR_11B_CHANNEL_END )
#define CSR_MIN_5GHz_CHANNEL_NUMBER  ( SIR_11A_CHANNEL_BEGIN )
#define CSR_MAX_5GHz_CHANNEL_NUMBER  ( SIR_11P_CHANNEL_END )

// WEP keysize (in bits)...
typedef enum
{
    eCSR_SECURITY_WEP_KEYSIZE_40  =  40,   // 40 bit key + 24bit IV = 64bit WEP
    eCSR_SECURITY_WEP_KEYSIZE_104 = 104,   // 104bit key + 24bit IV = 128bit WEP

    eCSR_SECURITY_WEP_KEYSIZE_MIN = eCSR_SECURITY_WEP_KEYSIZE_40,
    eCSR_SECURITY_WEP_KEYSIZE_MAX = eCSR_SECURITY_WEP_KEYSIZE_104,
    eCSR_SECURITY_WEP_KEYSIZE_MAX_BYTES = ( eCSR_SECURITY_WEP_KEYSIZE_MAX / 8 ),
}eCsrWEPKeySize;


// Possible values for the WEP static key ID...
typedef enum
{

    eCSR_SECURITY_WEP_STATIC_KEY_ID_MIN       =  0,
    eCSR_SECURITY_WEP_STATIC_KEY_ID_MAX       =  3,
    eCSR_SECURITY_WEP_STATIC_KEY_ID_DEFAULT   =  0,

    eCSR_SECURITY_WEP_STATIC_KEY_ID_INVALID   = -1,

}eCsrWEPStaticKeyID;

// Two extra key indicies are used for the IGTK (which is used by BIP)
#define CSR_MAX_NUM_KEY     (eCSR_SECURITY_WEP_STATIC_KEY_ID_MAX + 2 + 1)

typedef enum
{
    eCSR_SECURITY_SET_KEY_ACTION_NO_CHANGE,
    eCSR_SECURITY_SET_KEY_ACTION_SET_KEY,
    eCSR_SECURITY_SET_KEY_ACTION_DELETE_KEY,
}eCsrSetKeyAction;

typedef enum
{
    eCSR_BAND_ALL,
    eCSR_BAND_24,
    eCSR_BAND_5G,
    eCSR_BAND_MAX,
}eCsrBand;


typedef enum
{
   // Roaming because HDD requested for reassoc by changing one of the fields in
   // tCsrRoamModifyProfileFields. OR
   // Roaming because SME requested for reassoc by changing one of the fields in
   // tCsrRoamModifyProfileFields.
   eCsrRoamReasonStaCapabilityChanged,
   // Roaming because SME requested for reassoc to a different AP, as part of
   // inter AP handoff.
   eCsrRoamReasonBetterAP,
   // Roaming because SME requested it as the link is lost - placeholder, will
   // clean it up once handoff code gets in
   eCsrRoamReasonSmeIssuedForLostLink,

}eCsrRoamReasonCodes;

typedef enum
{
   eCsrRoamWmmAuto = 0,
   eCsrRoamWmmQbssOnly = 1,
   eCsrRoamWmmNoQos = 2,

} eCsrRoamWmmUserModeType;

typedef enum
{
   eCSR_REQUESTER_MIN = 0,
   eCSR_DIAG,
   eCSR_UMA_GAN,
   eCSR_HDD
} eCsrStatsRequesterType;

/**
 * enum csr_hi_rssi_scan_id - Parameter ids for hi rssi scan feature
 *
 * eCSR_HI_RSSI_SCAN_MAXCOUNT_ID: how many times scan can be performed
 * eCSR_HI_RSSI_SCAN_RSSI_DELTA_ID: rssi difference to trigger scan
 * eCSR_HI_RSSI_SCAN_DELAY_ID: delay in millseconds between scans
 * eCSR_HI_RSSI_SCAN_RSSI_UB_ID: rssi upper bound for scan trigger
 */
#ifdef WLAN_FEATURE_NEIGHBOR_ROAMING
enum csr_hi_rssi_scan_id {
	eCSR_HI_RSSI_SCAN_MAXCOUNT_ID,
	eCSR_HI_RSSI_SCAN_RSSI_DELTA_ID,
	eCSR_HI_RSSI_SCAN_DELAY_ID,
	eCSR_HI_RSSI_SCAN_RSSI_UB_ID
};
#endif

typedef struct tagPmkidCandidateInfo
{
    tCsrBssid BSSID;
    tANI_BOOLEAN preAuthSupported;
}tPmkidCandidateInfo;

typedef struct tagPmkidCacheInfo
{
    tCsrBssid BSSID;
    tANI_U8 PMKID[CSR_RSN_PMKID_SIZE];
    uint8_t pmk[CSR_RSN_MAX_PMK_LEN];
    uint8_t pmk_len;
    uint8_t ssid_len;
    uint8_t ssid[SIR_MAC_MAX_SSID_LENGTH];
    uint8_t cache_id[CACHE_ID_LEN];
}tPmkidCacheInfo;

#ifdef FEATURE_WLAN_WAPI
typedef struct tagBkidCandidateInfo
{
    tCsrBssid BSSID;
    tANI_BOOLEAN preAuthSupported;
}tBkidCandidateInfo;

typedef struct tagBkidCacheInfo
{
    tCsrBssid BSSID;
    tANI_U8 BKID[CSR_WAPI_BKID_SIZE];
}tBkidCacheInfo;
#endif /* FEATURE_WLAN_WAPI */

typedef struct tagCsrKeys
{
    tANI_U8 KeyLength[ CSR_MAX_NUM_KEY ];   //Also use to indicate whether the key index is set
    tANI_U8 KeyMaterial[ CSR_MAX_NUM_KEY ][ CSR_MAX_KEY_LEN ];
    tANI_U8 defaultIndex;
}tCsrKeys;

/* Following are fields which are part of tCsrRoamConnectedProfile might need
   modification dynamically once STA is up & running and this could trigger
   reassoc */
typedef struct tagCsrRoamModifyProfileFields
{
   // during connect this specifies ACs U-APSD is to be setup
   //   for (Bit0:VO; Bit1:VI; Bit2:BK; Bit3:BE all other bits are ignored).
   //  During assoc response this COULD carry confirmation of what ACs U-APSD
   // got setup for. Later if an APP looking for APSD, SME-QoS might need to
   // modify this field
   tANI_U8     uapsd_mask;
   // HDD might ask to modify this field
   tANI_U16    listen_interval;
}tCsrRoamModifyProfileFields;

typedef struct tagCsrRoamProfile
{
    //For eCSR_BSS_TYPE_WDS_AP. There must be one SSID in SSIDs.
    //For eCSR_BSS_TYPE_WDS_STA. There must be two SSIDs. Index 0 is the SSID of the WDS-AP
    //that we need to join. Index 1 is the SSID for self BSS.
    tCsrSSIDs SSIDs;
    tCsrBSSIDs BSSIDs;
    /* this is a bit mask of all the needed phy mode defined in eCsrPhyMode */
    eCsrPhyMode phyMode;
    eCsrRoamBssType BSSType;

    tCsrAuthList AuthType;
    eCsrAuthType negotiatedAuthType;
    tCsrAuthList akm_list;

    tCsrEncryptionList EncryptionType;
    //This field is for output only, not for input
    eCsrEncryptionType negotiatedUCEncryptionType;

    //eCSR_ENCRYPT_TYPE_ANY cannot be set in multicast encryption type. If caller doesn't case,
    //put all supported encryption types in here
    tCsrEncryptionList mcEncryptionType;
    //This field is for output only, not for input
    eCsrEncryptionType negotiatedMCEncryptionType;

#ifdef WLAN_FEATURE_11W
    // Management Frame Protection
    tANI_BOOLEAN MFPEnabled;
    tANI_U8 MFPRequired;
    tANI_U8 MFPCapable;
#endif

    tCsrKeys Keys;
    eCsrCBChoice CBMode; //up, down or auto
    tCsrChannelInfo ChannelInfo;
    tANI_U8 operationChannel;
    tANI_U32 vht_channel_width;
    tANI_U16 beaconInterval;    //If this is 0, SME will fill in for caller.
    // during connect this specifies ACs U-APSD is to be setup
    //   for (Bit0:VO; Bit1:VI; Bit2:BK; Bit3:BE all other bits are ignored).
    //  During assoc response this COULD carry confirmation of what ACs U-APSD got setup for
    tANI_U8 uapsd_mask;
    tANI_U32 nWPAReqIELength;   //The byte count in the pWPAReqIE
    tANI_U8 *pWPAReqIE;   //If not null, it has the IE byte stream for WPA
    tANI_U32 nRSNReqIELength;  //The byte count in the pRSNReqIE
    tANI_U8 *pRSNReqIE;     //If not null, it has the IE byte stream for RSN
#ifdef FEATURE_WLAN_WAPI
    tANI_U32 nWAPIReqIELength;   //The byte count in the pWAPIReqIE
    tANI_U8 *pWAPIReqIE;   //If not null, it has the IE byte stream for WAPI
#endif /* FEATURE_WLAN_WAPI */

    tANI_U32 nAddIEScanLength;   //The byte count in the pAddIE for scan (at the time of join)
    tANI_U8 *pAddIEScan;       //If not null, it has the IE byte stream for additional IE, which can be WSC IE and/or P2P IE
    tANI_U32 nAddIEAssocLength;   //The byte count in the pAddIE for assoc
    tANI_U8 *pAddIEAssoc;       //If not null, it has the IE byte stream for additional IE, which can be WSC IE and/or P2P IE

    tANI_U8 countryCode[WNI_CFG_COUNTRY_CODE_LEN];  //it is ignored if [0] is 0.
    /*WPS Association if true => auth and ecryption should be ignored*/
    tANI_BOOLEAN bWPSAssociation;
    tANI_BOOLEAN bOSENAssociation;
    tANI_U32 nWSCReqIELength;   //The byte count in the pWSCReqIE
    tANI_U8 *pWSCReqIE;   //If not null, it has the IE byte stream for WSC

    tANI_U8 ieee80211d;
    tANI_U8 privacy;
    tANI_BOOLEAN fwdWPSPBCProbeReq;
    tAniAuthType csr80211AuthType;
    tANI_U32 dtimPeriod;
    tANI_BOOLEAN ApUapsdEnable;
    tANI_BOOLEAN protEnabled;
    tANI_BOOLEAN obssProtEnabled;
    tANI_U16 cfg_protection;
    tANI_U8 wps_state;

#ifdef WLAN_FEATURE_VOWIFI_11R
    tCsrMobilityDomainInfo MDID;
#endif
    tVOS_CON_MODE csrPersona;

    tANI_U8 disableDFSChSwitch;
    /* addIe params */
    tSirAddIeParams        addIeParams;
    uint8_t sap_dot11mc;
    bool do_not_roam;
    uint16_t beacon_tx_rate;
    tSirMacRateSet  supported_rates;
    tSirMacRateSet  extended_rates;
    uint8_t sub20_channelwidth;
    bool force_24ghz_in_ht20;
    tCsrBssid bssid_hint;
#ifdef WLAN_FEATURE_FILS_SK
    bool fils_connection;
    struct cds_fils_connection_info *fils_con_info;
#endif
}tCsrRoamProfile;


#ifdef FEATURE_WLAN_MCC_TO_SCC_SWITCH
typedef struct tagCsrRoamHTProfile
{
    eCsrPhyMode         phymode;
    tANI_U8             htCapability;
    tANI_U8             htSupportedChannelWidthSet;
    tANI_U8             htRecommendedTxWidthSet;
    ePhyChanBondState   htSecondaryChannelOffset;
#ifdef WLAN_FEATURE_11AC
    tANI_U8             vhtCapability;
    tANI_U8             vhtTxChannelWidthSet;
    tANI_U8             apCenterChan;
    tANI_U8             apChanWidth;
#endif
}tCsrRoamHTProfile;
#endif
typedef struct tagCsrRoamConnectedProfile
{
    tSirMacSSid SSID;
    tANI_BOOLEAN    handoffPermitted;
    tANI_BOOLEAN    ssidHidden;
    tCsrBssid bssid;
    eCsrRoamBssType BSSType;
    eCsrAuthType AuthType;
    tCsrAuthList AuthInfo;
    tCsrAuthList akm_list;
    eCsrEncryptionType EncryptionType;
    tCsrEncryptionList EncryptionInfo;
    eCsrEncryptionType mcEncryptionType;
    tCsrEncryptionList mcEncryptionInfo;
    eCsrCBChoice CBMode; //up, down or auto
    tANI_U8 operationChannel;
    tANI_U32   vht_channel_width;
    tANI_U16   beaconInterval;
    tCsrKeys Keys;
    // meaningless on connect. It's an OUT param from CSR's point of view
    // During assoc response carries the ACM bit-mask i.e. what
    // ACs have ACM=1 (if any),
    // (Bit0:VO; Bit1:VI; Bit2:BK; Bit3:BE all other bits are ignored)
    tANI_U8  acm_mask;
    tCsrRoamModifyProfileFields modifyProfileFields;
    tANI_U32 nAddIEAssocLength;   //The byte count in the pAddIE for assoc
    tANI_U8 *pAddIEAssoc;       //If not null, it has the IE byte stream for additional IE, which can be WSC IE and/or P2P IE

    tSirBssDescription *pBssDesc;
    tANI_BOOLEAN   qap; //AP supports QoS
    tANI_BOOLEAN   qosConnection; //A connection is QoS enabled
#ifdef WLAN_FEATURE_VOWIFI_11R
    tCsrMobilityDomainInfo MDID;
#endif

#ifdef FEATURE_WLAN_ESE
    tCsrEseCckmInfo eseCckmInfo;
    tANI_BOOLEAN    isESEAssoc;
#endif
    tANI_U32 dot11Mode;
    tANI_U8 proxyARPService;
#ifdef FEATURE_WLAN_MCC_TO_SCC_SWITCH
    tCsrRoamHTProfile HTProfile;
#endif
#ifdef WLAN_FEATURE_11W
    /* Management Frame Protection */
    tANI_BOOLEAN MFPEnabled;
    tANI_U8 MFPRequired;
    tANI_U8 MFPCapable;
#endif
}tCsrRoamConnectedProfile;


#ifdef WLAN_FEATURE_VOWIFI_11R
typedef struct tagCsr11rConfigParams
{
    tANI_BOOLEAN   IsFTResourceReqSupported;
} tCsr11rConfigParams;
#endif

#ifdef WLAN_FEATURE_NEIGHBOR_ROAMING
typedef struct tagCsrNeighborRoamConfigParams
{

    tANI_U32       nNeighborScanTimerPeriod;
    tANI_U8        nNeighborLookupRssiThreshold;
    tANI_U8        nNeighborReassocRssiThreshold;
    tANI_U16       nNeighborScanMinChanTime;
    tANI_U16       nNeighborScanMaxChanTime;
    sCsrChannel    neighborScanChanList;
    tANI_U8        nMaxNeighborRetries;
    tANI_U16       nNeighborResultsRefreshPeriod;
    tANI_U16       nEmptyScanRefreshPeriod;
    tANI_U8        nOpportunisticThresholdDiff;
    tANI_U8        nRoamRescanRssiDiff;
    tANI_U8        nRoamBmissFirstBcnt;
    tANI_U8        nRoamBmissFinalBcnt;
    tANI_U8        nRoamBeaconRssiWeight;
    tANI_U8        delay_before_vdev_stop;
    uint32_t       nhi_rssi_scan_max_count;
    uint32_t       nhi_rssi_scan_rssi_delta;
    uint32_t       nhi_rssi_scan_delay;
    int32_t        nhi_rssi_scan_rssi_ub;
}tCsrNeighborRoamConfigParams;
#endif

/**
 * enum sta_roam_policy_dfs_mode - state of DFS mode for STA ROME policy
 * @CSR_STA_ROAM_POLICY_NONE: DFS mode attribute is not valid
 * @CSR_STA_ROAM_POLICY_DFS_ENABLED:  DFS mode is enabled
 * @CSR_STA_ROAM_POLICY_DFS_DISABLED: DFS mode is disabled
 * @CSR_STA_ROAM_POLICY_DFS_DEPRIORITIZE: Deprioritize DFS channels in scanning
 */
enum sta_roam_policy_dfs_mode {
	CSR_STA_ROAM_POLICY_NONE,
	CSR_STA_ROAM_POLICY_DFS_ENABLED,
	CSR_STA_ROAM_POLICY_DFS_DISABLED,
	CSR_STA_ROAM_POLICY_DFS_DEPRIORITIZE
};

/**
 * struct csr_sta_roam_policy_params - sta roam policy params for station
 * @dfs_mode: tell is DFS channels needs to be skipped while scanning
 * @skip_unsafe_channels: tells if unsafe channels needs to be skip in scanning
 */
struct csr_sta_roam_policy_params {
	enum sta_roam_policy_dfs_mode dfs_mode;
	bool skip_unsafe_channels;
	uint8_t sap_operating_band;
};

typedef struct tagCsrConfigParam
{
    tANI_U32 FragmentationThreshold;
    tANI_U32 channelBondingMode24GHz;   // keep this tANI_U32. This gets converted to ePhyChannelBondState
    tANI_U32 channelBondingMode5GHz;    // in csrChangeDefaultConfigParam using convertCBIniValueToPhyCBState
    uint32_t stacbmode;
    eCsrPhyMode phyMode;
    eCsrBand eBand;
    tANI_U32 RTSThreshold;
    tANI_U32 HeartbeatThresh50;
    tANI_U32 HeartbeatThresh24;
    eCsrCBChoice cbChoice;
    eCsrBand bandCapability;     //indicate hw capability
    tANI_U32 bgScanInterval;
    tANI_U16 TxRate;
    eCsrRoamWmmUserModeType WMMSupportMode;
    tANI_U8 gStaLocalEDCAEnable;
    tANI_BOOLEAN Is11eSupportEnabled;
    tANI_BOOLEAN Is11dSupportEnabled;
    tANI_BOOLEAN Is11dSupportEnabledOriginal;
    tANI_BOOLEAN Is11hSupportEnabled;
    tANI_BOOLEAN shortSlotTime;
    tANI_BOOLEAN ProprietaryRatesEnabled;
    tANI_U8 AdHocChannel24;
    tANI_U8 AdHocChannel5G;
    tANI_U32 impsSleepTime;     //in units of seconds
    tANI_U32 nScanResultAgeCount;   //this number minus one is the number of times a scan doesn't find it before it is removed
    tANI_U32 scanAgeTimeNCNPS;  //scan result aging time threshold when Not-Connect-No-Power-Save, in seconds
    tANI_U32 scanAgeTimeNCPS;   //scan result aging time threshold when Not-Connect-Power-Save, in seconds
    tANI_U32 scanAgeTimeCNPS;   //scan result aging time threshold when Connect-No-Power-Save, in seconds,
    tANI_U32 scanAgeTimeCPS;   //scan result aging time threshold when Connect-Power-Savein seconds
    tANI_U32 nRoamingTime;  //In seconds, CSR will try this long before gives up. 0 means no roaming
    tANI_U8 bCatRssiOffset;     //to set the RSSI difference for each category
    tANI_U8 fEnableMCCMode; //to set MCC Enable/Disable mode
    bool mcc_rts_cts_prot_enable;
    bool mcc_bcast_prob_resp_enable;
    tANI_U8 fAllowMCCGODiffBI; //to allow MCC GO different B.I than STA's. NOTE: make sure if RIVA firmware can handle this combination before enabling this
                               //at the moment, this flag is provided only to pass Wi-Fi Cert. 5.1.12
    tCsr11dinfo  Csr11dinfo;
    //Whether to limit the channels to the ones set in Csr11dInfo. If true, the opertaional
    //channels are limited to the default channel list. It is an "AND" operation between the
    //default channels and the channels in the 802.11d IE.
    tANI_BOOLEAN fEnforce11dChannels;
    //Country Code Priority
    //0 = 802.11D > Country IOCTL > NV
    //1 = Country IOCTL > 802.11D > NV
    tANI_BOOLEAN fSupplicantCountryCodeHasPriority;
    //When true, AP with unknown country code won't be see.
    //"Unknown country code" means either Ap doesn't have 11d IE or we cannot
    //find a domain for the country code in its 11d IE.
    tANI_BOOLEAN fEnforceCountryCodeMatch;
    //When true, only APs in the default domain can be seen. If the Ap has "unknown country
    //code", or the domain of the country code doesn't match the default domain, the Ap is
    //not acceptable.
    tANI_BOOLEAN fEnforceDefaultDomain;

    tANI_U16 vccRssiThreshold;
    tANI_U32 vccUlMacLossThreshold;

    tANI_U32  nPassiveMinChnTime;    //in units of milliseconds
    tANI_U32  nPassiveMaxChnTime;    //in units of milliseconds
    tANI_U32  nActiveMinChnTime;     //in units of milliseconds
    tANI_U32  nActiveMaxChnTime;     //in units of milliseconds

    tANI_U32  nInitialDwellTime;      //in units of milliseconds
    bool      initial_scan_no_dfs_chnl;

    tANI_U32  disableAggWithBtc;
#ifdef WLAN_AP_STA_CONCURRENCY
    tANI_U32  nPassiveMinChnTimeConc;    //in units of milliseconds
    tANI_U32  nPassiveMaxChnTimeConc;    //in units of milliseconds
    tANI_U32  nActiveMinChnTimeConc;     //in units of milliseconds
    tANI_U32  nActiveMaxChnTimeConc;     //in units of milliseconds
    tANI_U32  nRestTimeConc;             //in units of milliseconds
#endif
    /*In units of milliseconds*/
    uint32_t       min_rest_time_conc;
    /*In units of milliseconds*/
    uint32_t       idle_time_conc;

    tANI_BOOLEAN IsIdleScanEnabled;
    //in dBm, the maximum TX power
    //The actual TX power is the lesser of this value and 11d.
    //If 11d is disable, the lesser of this and default setting.
    tANI_U8 nTxPowerCap;
    tANI_BOOLEAN allow_tpc_from_ap;
    tANI_U32  statsReqPeriodicity;  //stats request frequency from PE while in full power
    tANI_U32  statsReqPeriodicityInPS;//stats request frequency from PE while in power save
#ifdef WLAN_FEATURE_VOWIFI_11R
    tCsr11rConfigParams  csr11rConfig;
#endif
#ifdef FEATURE_WLAN_ESE
    tANI_U8   isEseIniFeatureEnabled;
#endif
#ifdef FEATURE_WLAN_LFR
    tANI_U8   isFastRoamIniFeatureEnabled;
    tANI_U8   MAWCEnabled;
#endif

#if  defined (WLAN_FEATURE_VOWIFI_11R) || defined (FEATURE_WLAN_ESE) || defined(FEATURE_WLAN_LFR)
    tANI_U8        isFastTransitionEnabled;
    tANI_U8        RoamRssiDiff;
    tANI_U8        nImmediateRoamRssiDiff;
    tANI_BOOLEAN   isWESModeEnabled;
#endif

#ifdef WLAN_FEATURE_NEIGHBOR_ROAMING
    tCsrNeighborRoamConfigParams    neighborRoamConfig;
#endif

    /* Instead of Reassoc, send ADDTS/DELTS even when ACM is off for that AC
     * This is mandated by WMM-AC certification */
    tANI_BOOLEAN addTSWhenACMIsOff;


    /*channelPowerInfoList24 has been seen corrupted. Set this flag to true trying to
    * detect when it happens. Adding this into code because we can't reproduce it easily.
    * We don't know when it happens. */
    tANI_BOOLEAN fValidateList;

    /*Customer wants to start with an active scan based on the default country code.
    * This optimization will minimize the driver load to association time.
    * Based on this flag we will bypass the initial passive scan needed for 11d
    * to determine the country code & domain */
    tANI_BOOLEAN fEnableBypass11d;

    /*Customer wants to optimize the scan time. Avoiding scans(passive) on DFS
    * channels while swipping through both bands can save some time
    * (apprx 1.3 sec) */
    tANI_U8 fEnableDFSChnlScan;

    //To enable/disable scanning 2.4Ghz channels twice on a single scan request from HDD
    tANI_BOOLEAN fScanTwice;
#ifdef WLAN_FEATURE_11AC
    tANI_U32        nVhtChannelWidth;
    tANI_U8         enableTxBF;
    tANI_U8         txBFCsnValue;
    tANI_U8         enable2x2;
    tANI_BOOLEAN    enableVhtFor24GHz;
    bool            vendor_vht_for_24ghz_sap;
    tANI_U8         enableMuBformee;
    tANI_U8         enableVhtpAid;
    tANI_U8         enableVhtGid;
#endif
    tANI_U8         enableAmpduPs;
    tANI_U8         enableHtSmps;
    tANI_U8         htSmps;

    /*
    * To enable/disable scanning only 2.4Ghz channels on first scan
    */
    tANI_BOOLEAN fFirstScanOnly2GChnl;

#if  defined (WLAN_FEATURE_VOWIFI_11R) || defined (FEATURE_WLAN_ESE) || defined(FEATURE_WLAN_LFR)
    tANI_BOOLEAN nRoamPrefer5GHz;
    tANI_BOOLEAN nRoamIntraBand;
    tANI_U8      nProbes;
    tANI_U16     nRoamScanHomeAwayTime;

#ifdef WLAN_FEATURE_ROAM_SCAN_OFFLOAD
    tANI_BOOLEAN isRoamOffloadScanEnabled;
    tANI_BOOLEAN bFastRoamInConIniFeatureEnabled;
#endif
#endif

    tANI_BOOLEAN ignorePeerErpInfo;
    tANI_U8 scanCfgAgingTime;

    tANI_U8   enableTxLdpc;

    tANI_U8 max_amsdu_num;
    tANI_U8 nSelect5GHzMargin;

    tANI_U8 isCoalesingInIBSSAllowed;

    eCsrBand  scanBandPreference;
#ifdef FEATURE_WLAN_MCC_TO_SCC_SWITCH
    tANI_U8  cc_switch_mode;
    bool     band_switch_enable;
    bool     ap_p2pgo_concurrency_enable;
    bool     ap_p2pclient_concur_enable;
    uint16_t ch_width_24g_orig;
    uint16_t ch_width_5g_orig;
#endif
    tANI_U8  allowDFSChannelRoam;
#ifdef WLAN_FEATURE_ROAM_OFFLOAD
    tANI_BOOLEAN isRoamOffloadEnabled;
#endif

    tANI_BOOLEAN obssEnabled;
    v_U8_t conc_custom_rule1;
    v_U8_t conc_custom_rule2;
    v_U8_t is_sta_connection_in_5gz_enabled;
    bool       ignore_peer_ht_opmode;
    /* 802.11p enable */
    bool       enable_dot11p;
    tANI_BOOLEAN sendDeauthBeforeCon;
    v_U16_t    pkt_err_disconn_th;
    int8_t    first_scan_bucket_threshold;
    bool    enable_fatal_event;
    uint32_t tx_aggregation_size;
    uint32_t rx_aggregation_size;
    uint32_t tx_aggr_sw_retry_threshhold_be;
    uint32_t tx_aggr_sw_retry_threshhold_bk;
    uint32_t tx_aggr_sw_retry_threshhold_vi;
    uint32_t tx_aggr_sw_retry_threshhold_vo;
    uint32_t tx_non_aggr_sw_retry_threshhold_be;
    uint32_t tx_non_aggr_sw_retry_threshhold_bk;
    uint32_t tx_non_aggr_sw_retry_threshhold_vi;
    uint32_t tx_non_aggr_sw_retry_threshhold_vo;
    bool enable_edca_params;
    uint32_t edca_vo_cwmin;
    uint32_t edca_vi_cwmin;
    uint32_t edca_bk_cwmin;
    uint32_t edca_be_cwmin;
    uint32_t edca_vo_cwmax;
    uint32_t edca_vi_cwmax;
    uint32_t edca_bk_cwmax;
    uint32_t edca_be_cwmax;
    uint32_t edca_vo_aifs;
    uint32_t edca_vi_aifs;
    uint32_t edca_bk_aifs;
    uint32_t edca_be_aifs;
    struct csr_sta_roam_policy_params sta_roam_policy_params;
    uint32_t sta_auth_retries_for_code17;
#ifdef WLAN_FEATURE_FILS_SK
    uint8_t fils_max_chan_guard_time;
#endif
#ifdef WLAN_FEATURE_SAP_TO_FOLLOW_STA_CHAN
    tANI_U32    sap_ch_switch_with_csa;
#endif//#ifdef WLAN_FEATURE_SAP_TO_FOLLOW_STA_CHAN
}tCsrConfigParam;

//Tush
typedef struct tagCsrUpdateConfigParam
{
   tCsr11dinfo  Csr11dinfo;
}tCsrUpdateConfigParam;
#ifdef WLAN_FEATURE_ROAM_OFFLOAD
#define csrRoamIsRoamOffloadEnabled(pMac)\
        (pMac->roam.configParam.isRoamOffloadEnabled)

#define DEFAULT_REASSOC_FAILURE_TIMEOUT 1000
#endif

#ifdef WLAN_FEATURE_ROAM_OFFLOAD
#define CSR_ROAM_AUTH_STATUS_CONNECTED      0x1 /** connected,
                                                    but not authenticated */
#define CSR_ROAM_AUTH_STATUS_AUTHENTICATED  0x2 /** connected
                                                    and authenticated */
#endif

typedef struct tagCsrRoamInfo
{
    tCsrRoamProfile *pProfile;  //may be NULL
    tSirBssDescription *pBssDesc;  //May be NULL
    tANI_U32 nBeaconLength; //the length, in bytes, of the beacon frame, can be 0
    tANI_U32 nAssocReqLength;   //the length, in bytes, of the assoc req frame, can be 0
    tANI_U32 nAssocRspLength;   //The length, in bytes, of the assoc rsp frame, can be 0
    tANI_U32 nFrameLength;
    tANI_U8  frameType;
    tANI_U8 *pbFrames;  //Point to a buffer contain the beacon, assoc req, assoc rsp frame, in that order
                        //user needs to use nBeaconLength, nAssocReqLength, nAssocRspLength to desice where
                        //each frame starts and ends.
    tANI_BOOLEAN fReassocReq;   //set to true if for re-association
    tANI_BOOLEAN fReassocRsp;   //set to true if for re-association
    tCsrBssid bssid;
    //Only valid in IBSS
    //this is the peers MAC address for eCSR_ROAM_RESULT_IBSS_NEW_PEER or PEER_DEPARTED
    tCsrBssid peerMac;
    tSirResultCodes statusCode;
    tANI_U32 reasonCode;    //this could be our own defined or sent from the other BSS(per 802.11 spec)
    tANI_U8  staId;         // Peer stationId when connected
    /*The DPU signatures will be sent eventually to TL to help it determine the
      association to which a packet belongs to*/
    /*Unicast DPU signature*/
    tANI_U8            ucastSig;

    /*Broadcast DPU signature*/
    tANI_U8            bcastSig;

    tANI_BOOLEAN fAuthRequired;   //FALSE means auth needed from supplicant. TRUE means authenticated(static WEP, open)
    tANI_U8 sessionId;
    tANI_U8 rsnIELen;
    tANI_U8 *prsnIE;
    tANI_U8 wapiIELen;
    tANI_U8 *pwapiIE;

    tANI_U8 addIELen;
    tANI_U8 *paddIE;

    union
    {
        tSirMicFailureInfo *pMICFailureInfo;
        tCsrRoamConnectedProfile *pConnectedProfile;
        tSirWPSPBCProbeReq *pWPSPBCProbeReq;
    } u;

    tANI_BOOLEAN wmmEnabledSta;   //set to true if WMM enabled STA
    tANI_U32 dtimPeriod;

#ifdef FEATURE_WLAN_ESE
    tANI_BOOLEAN isESEAssoc;
#ifdef FEATURE_WLAN_ESE_UPLOAD
    tSirTsmIE tsmIe;
    tANI_U32 timestamp[2];
    tANI_U16 tsmRoamDelay;
    tSirEseBcnReportRsp *pEseBcnReportRsp;
#endif /* FEATURE_WLAN_ESE_UPLOAD */
#endif

#ifdef WLAN_FEATURE_VOWIFI_11R
    tANI_BOOLEAN is11rAssoc;
#endif
    void* pRemainCtx;
    tANI_U32 rxChan;

#ifdef FEATURE_WLAN_TDLS
    tANI_U8 staType;
    bool tdls_prohibited;           /* per ExtCap in Assoc/Reassoc resp */
    bool tdls_chan_swit_prohibited; /* per ExtCap in Assoc/Reassoc resp */
#endif

    // Required for indicating the frames to upper layer
    tANI_U32 beaconLength;
    tANI_U8* beaconPtr;
    tANI_U32 assocReqLength;
    tANI_U8* assocReqPtr;

    tANI_S8 rxRssi;
    tSirSmeDfsEventInd dfs_event;
    tSirChanChangeResponse *channelChangeRespEvent;
    /* Timing and fine Timing measurement capability clubbed together */
    tANI_U8 timingMeasCap;
#ifdef WLAN_FEATURE_ROAM_OFFLOAD
    tANI_U8 roamSynchInProgress;
    tANI_U8 synchAuthStatus;
    tANI_U8 kck[SIR_KCK_KEY_LEN];
    tANI_U8 kek[SIR_KEK_KEY_LEN];
    tANI_U8 replay_ctr[SIR_REPLAY_CTR_LEN];
#endif
    tSirSmeChanInfo chan_info;
    uint8_t target_channel;

#ifdef WLAN_FEATURE_NAN_DATAPATH
    union {
        struct sme_ndp_peer_ind ndp_peer_ind_params;
        struct ndp_schedule_update_rsp ndp_sched_upd_rsp_params;
        struct ndp_end_indication_event *ndp_end_ind_params;
        struct ndp_end_rsp_event *ndp_end_rsp_params;
        struct ndp_confirm_event ndp_confirm_params;
        struct ndp_responder_rsp_event ndp_responder_rsp_params;
        struct ndp_indication_event ndp_indication_params;
        struct ndp_initiator_rsp ndp_init_rsp_params;
        struct ndi_create_rsp ndi_create_params;
        struct ndi_delete_rsp ndi_delete_params;
    } ndp;
#endif
    tDot11fIEHTCaps ht_caps;
    tDot11fIEVHTCaps vht_caps;
    tDot11fIEhs20vendor_ie hs20vendor_ie;
    tDot11fIEVHTOperation vht_operation;
    tDot11fIEHTInfo ht_operation;
    bool reassoc;
    /* Extended capabilities of STA */
    uint8_t ecsa_capable;
    bool                 ampdu;
    bool                 sgi_enable;
    bool                 tx_stbc;
    bool                 rx_stbc;
    tSirMacHTChannelWidth ch_width;
    enum sir_sme_phy_mode mode;
    uint8_t              max_supp_idx;
    uint8_t              max_ext_idx;
    uint8_t              max_mcs_idx;
    uint8_t              rx_mcs_map;
    uint8_t              tx_mcs_map;
#ifdef WLAN_FEATURE_FILS_SK
    bool is_fils_connection;
    uint16_t fils_seq_num;
    struct fils_join_rsp_params *fils_join_rsp;
#endif
#ifdef WLAN_FEATURE_SAE
    struct sir_sae_info *sae_info;
#endif
    struct sSirSmeAssocInd *owe_pending_assoc_ind;
}tCsrRoamInfo;


typedef struct tagCsrFreqScanInfo
{
    tANI_U32 nStartFreq;    //in unit of MHz
    tANI_U32 nEndFreq;      //in unit of MHz
    tSirScanType scanType;
}tCsrFreqScanInfo;


typedef struct sSirSmeAssocIndToUpperLayerCnf
{
    tANI_U16             messageType; // eWNI_SME_ASSOC_CNF
    tANI_U16             length;
    tANI_U8              sessionId;
    tSirResultCodes      statusCode;
    tSirMacAddr          bssId;      // Self BSSID
    tSirMacAddr          peerMacAddr;
    tANI_U16             aid;
    tSirMacAddr          alternateBssId;
    tANI_U8              alternateChannelId;
    tANI_U8              wmmEnabledSta;   //set to true if WMM enabled STA
    tSirRSNie            rsnIE;           // RSN IE received from peer
    tSirWAPIie           wapiIE;          // WAPI IE received from peer
    tSirAddie            addIE;           // Additional IE received from peer, which can be WSC and/or P2P IE
    tANI_U8              reassocReq;      //set to true if reassoc
    /* Timing and fine Timing measurement capability clubbed together */
    tANI_U8              timingMeasCap;
    tSirSmeChanInfo      chan_info;
    /* Extended capabilities of STA */
    uint8_t              ecsa_capable;
    uint32_t             ies_len;
    uint8_t              *ies;
    bool                 ampdu;
    bool                 sgi_enable;
    bool                 tx_stbc;
    tSirMacHTChannelWidth ch_width;
    enum sir_sme_phy_mode mode;
    bool                 rx_stbc;
    uint8_t              max_supp_idx;
    uint8_t              max_ext_idx;
    uint8_t              max_mcs_idx;
    uint8_t              rx_mcs_map;
    uint8_t              tx_mcs_map;
} tSirSmeAssocIndToUpperLayerCnf, *tpSirSmeAssocIndToUpperLayerCnf;

typedef struct tagCsrSummaryStatsInfo
{
   tANI_U32 retry_cnt[4];
   tANI_U32 multiple_retry_cnt[4];
   tANI_U32 tx_frm_cnt[4];
   //tANI_U32 num_rx_frm_crc_err; same as rx_error_cnt
   //tANI_U32 num_rx_frm_crc_ok; same as rx_frm_cnt
   tANI_U32 rx_frm_cnt;
   tANI_U32 frm_dup_cnt;
   tANI_U32 fail_cnt[4];
   tANI_U32 rts_fail_cnt;
   tANI_U32 ack_fail_cnt;
   tANI_U32 rts_succ_cnt;
   tANI_U32 rx_discard_cnt;
   tANI_U32 rx_error_cnt;
   tANI_U32 tx_byte_cnt;

}tCsrSummaryStatsInfo;

typedef struct tagCsrGlobalClassAStatsInfo
{
   tANI_U32 rx_frag_cnt;
   tANI_U32 promiscuous_rx_frag_cnt;
   //tANI_U32 rx_fcs_err;
   tANI_U32 rx_input_sensitivity;
   tANI_U32 max_pwr;
   //tANI_U32 default_pwr;
   tANI_U32 sync_fail_cnt;
   tANI_U32 tx_rate;
   tANI_U32 rx_rate;
   //mcs index for HT20 and HT40 rates
   tANI_U32  tx_mcs_index;
   tANI_U32  rx_mcs_index;
   //to defferentiate between HT20 and HT40 rates;short and long guard interval
   tANI_U32  tx_rx_rate_flags;

}tCsrGlobalClassAStatsInfo;

typedef struct tagCsrGlobalClassBStatsInfo
{
   tANI_U32 uc_rx_wep_unencrypted_frm_cnt;
   tANI_U32 uc_rx_mic_fail_cnt;
   tANI_U32 uc_tkip_icv_err;
   tANI_U32 uc_aes_ccmp_format_err;
   tANI_U32 uc_aes_ccmp_replay_cnt;
   tANI_U32 uc_aes_ccmp_decrpt_err;
   tANI_U32 uc_wep_undecryptable_cnt;
   tANI_U32 uc_wep_icv_err;
   tANI_U32 uc_rx_decrypt_succ_cnt;
   tANI_U32 uc_rx_decrypt_fail_cnt;
   tANI_U32 mcbc_rx_wep_unencrypted_frm_cnt;
   tANI_U32 mcbc_rx_mic_fail_cnt;
   tANI_U32 mcbc_tkip_icv_err;
   tANI_U32 mcbc_aes_ccmp_format_err;
   tANI_U32 mcbc_aes_ccmp_replay_cnt;
   tANI_U32 mcbc_aes_ccmp_decrpt_err;
   tANI_U32 mcbc_wep_undecryptable_cnt;
   tANI_U32 mcbc_wep_icv_err;
   tANI_U32 mcbc_rx_decrypt_succ_cnt;
   tANI_U32 mcbc_rx_decrypt_fail_cnt;

}tCsrGlobalClassBStatsInfo;

typedef struct tagCsrGlobalClassCStatsInfo
{
   tANI_U32 rx_amsdu_cnt;
   tANI_U32 rx_ampdu_cnt;
   tANI_U32 tx_20_frm_cnt;
   tANI_U32 rx_20_frm_cnt;
   tANI_U32 rx_mpdu_in_ampdu_cnt;
   tANI_U32 ampdu_delimiter_crc_err;

}tCsrGlobalClassCStatsInfo;

typedef struct tagCsrGlobalClassDStatsInfo
{
   tANI_U32 tx_uc_frm_cnt;
   tANI_U32 tx_mc_frm_cnt;
   tANI_U32 tx_bc_frm_cnt;
   tANI_U32 rx_uc_frm_cnt;
   tANI_U32 rx_mc_frm_cnt;
   tANI_U32 rx_bc_frm_cnt;
   tANI_U32 tx_uc_byte_cnt[4];
   tANI_U32 tx_mc_byte_cnt;
   tANI_U32 tx_bc_byte_cnt;
   tANI_U32 rx_uc_byte_cnt[4];
   tANI_U32 rx_mc_byte_cnt;
   tANI_U32 rx_bc_byte_cnt;
   tANI_U32 rx_byte_cnt;
   tANI_U32 num_rx_bytes_crc_ok;
   tANI_U32 rx_rate;

}tCsrGlobalClassDStatsInfo;

typedef struct tagCsrPerStaStatsInfo
{
   tANI_U32 tx_frag_cnt[4];
   tANI_U32 tx_ampdu_cnt;
   tANI_U32 tx_mpdu_in_ampdu_cnt;
} tCsrPerStaStatsInfo;

struct csr_per_chain_rssi_stats_info {
  int8 rssi[NUM_CHAINS_MAX];
  tSirMacAddr peer_mac_addr;
};

typedef struct tagCsrRoamSetKey
{
    eCsrEncryptionType encType;
    tAniKeyDirection keyDirection;    //Tx, Rx or Tx-and-Rx
    tCsrBssid peerMac;   //Peers MAC address. ALL 1's for group key
    tANI_U8 paeRole;      //0 for supplicant
    tANI_U8 keyId;  // Kye index
    tANI_U16 keyLength;  //Number of bytes containing the key in pKey
    tANI_U8 Key[CSR_MAX_KEY_LEN];
    tANI_U8 keyRsc[CSR_MAX_RSC_LEN];
} tCsrRoamSetKey;

typedef struct tagCsrRoamRemoveKey
{
    eCsrEncryptionType encType;
    tCsrBssid peerMac;   //Peers MAC address. ALL 1's for group key
    tANI_U8 keyId;  //key index
} tCsrRoamRemoveKey;

#ifdef FEATURE_WLAN_TDLS

typedef struct tagCsrLinkEstablishParams
{
    tSirMacAddr peerMac;
    tANI_U8 uapsdQueues;
    tANI_U8 maxSp;
    uint8_t qos;
    tANI_U8 isBufSta;
    tANI_U8 isOffChannelSupported;
    tANI_U8 isResponder;
    tANI_U8 supportedChannelsLen;
    tANI_U8 supportedChannels[SIR_MAC_MAX_SUPP_CHANNELS];
    tANI_U8 supportedOperClassesLen;
    tANI_U8 supportedOperClasses[SIR_MAC_MAX_SUPP_OPER_CLASSES];
}tCsrTdlsLinkEstablishParams;

typedef struct tagCsrTdlsSendMgmt
{
        tSirMacAddr peerMac;
        tANI_U8 frameType;
        tANI_U8 dialog;
        tANI_U16 statusCode;
        tANI_U8 responder;
        tANI_U32 peerCapability;
        tANI_U8 *buf;
        tANI_U8 len;

}tCsrTdlsSendMgmt;

#endif

typedef void * tScanResultHandle;

#define CSR_INVALID_SCANRESULT_HANDLE       (NULL)

#ifdef WLAN_FEATURE_ROAM_SCAN_OFFLOAD
typedef struct tagCsrHandoffRequest
{
    tCsrBssid bssid;
    tANI_U8 channel;
    tANI_U8 src;     /* To check if its a REASSOC or a FASTREASSOC IOCTL */
}tCsrHandoffRequest;
#endif

typedef enum
{
    REASSOC     = 0,
    FASTREASSOC = 1,
    CONNECT_CMD_USERSPACE = 2,
}handoff_src;

#if defined(FEATURE_WLAN_ESE) && defined(FEATURE_WLAN_ESE_UPLOAD)
typedef struct tagCsrEseBeaconReqParams
{
    tANI_U16   measurementToken;
    tANI_U8    channel;
    tANI_U8    scanMode;
    tANI_U16   measurementDuration;
} tCsrEseBeaconReqParams, *tpCsrEseBeaconReqParams;

typedef struct tagCsrEseBeaconReq
{
    tANI_U8                numBcnReqIe;
    tCsrEseBeaconReqParams bcnReq[SIR_ESE_MAX_MEAS_IE_REQS];
} tCsrEseBeaconReq, *tpCsrEseBeaconReq;
#endif /* FEATURE_WLAN_ESE && FEATURE_WLAN_ESE_UPLOAD */

struct tagCsrDelStaParams
{
    tCsrBssid peerMacAddr;
    u16 reason_code;
    u8 subtype;
};

/**
 * struct csr_set_tx_max_pwr_per_band - Req params to
 * set max tx power per band
 * @band: band for which power to be set
 * @power: power to set in dB
 */
struct csr_set_tx_max_pwr_per_band {
	eCsrBand band;
	tPowerdBm power;
};

/**
 * struct wep_update_default_key_idx: wep default key index structure
 *
 * @session_id: session ID for the connection session
 * @default_idx: default key index for wep
 *
 * structure includes sesssion id for connection and default key
 * index used for wep
 */
struct wep_update_default_key_idx {
	uint8_t session_id;
	uint8_t default_idx;
};

////////////////////////////////////////////Common SCAN starts

//void *p2 -- the second context pass in for the caller
//***what if callback is called before requester gets the scanId??
typedef eHalStatus (*csrScanCompleteCallback)(tHalHandle, void *p2,
                                              tANI_U8 sessionId,
                                              tANI_U32 scanID,
                                              eCsrScanStatus status);



///////////////////////////////////////////Common Roam starts

//pContext is the pContext passed in with the roam request
//pParam is a pointer to a tCsrRoamInfo, see definition of eRoamCmdStatus and
//   eRoamCmdResult for detail valid members. It may be NULL
//roamId is to identify the callback related roam request. 0 means unsolicit
//roamStatus is a flag indicating the status of the callback
//roamResult is the result
typedef eHalStatus (*csrRoamCompleteCallback)(void *pContext, tCsrRoamInfo *pParam, tANI_U32 roamId,
                                              eRoamCmdStatus roamStatus, eCsrRoamResult roamResult);

typedef eHalStatus (*csrRoamSessionCloseCallback)(void *pContext);

/* ---------------------------------------------------------------------------
    \fn csrRoamGetNumPMKIDCache
    \brief return number of PMKID cache entries
    \return tANI_U32 - the number of PMKID cache entries
  -------------------------------------------------------------------------------*/
//tANI_U32 csrRoamGetNumPMKIDCache(tHalHandle hHal);

/* ---------------------------------------------------------------------------
    \fn csrRoamGetPMKIDCache
    \brief return PMKID cache from CSR
    \param pNum - caller allocated memory that has the space of the number of pBuf tPmkidCacheInfo as input. Upon returned, *pNum has the
    needed or actually number in tPmkidCacheInfo.
    \param pPmkidCache - Caller allocated memory that contains PMKID cache, if any, upon return
    \return eHalStatus - when fail, it usually means the buffer allocated is not big enough
  -------------------------------------------------------------------------------*/
//eHalStatus csrRoamGetPMKIDCache(tHalHandle hHal, tANI_U32 *pNum, tPmkidCacheInfo *pPmkidCache);

//pProfile - pointer to tCsrRoamProfile
#define CSR_IS_START_IBSS(pProfile) (eCSR_BSS_TYPE_START_IBSS == (pProfile)->BSSType)
#define CSR_IS_JOIN_TO_IBSS(pProfile) (eCSR_BSS_TYPE_IBSS == (pProfile)->BSSType)
#define CSR_IS_IBSS(pProfile) ( CSR_IS_START_IBSS(pProfile) || CSR_IS_JOIN_TO_IBSS(pProfile) )
#define CSR_IS_INFRASTRUCTURE(pProfile) (eCSR_BSS_TYPE_INFRASTRUCTURE == (pProfile)->BSSType)
#define CSR_IS_ANY_BSS_TYPE(pProfile) (eCSR_BSS_TYPE_ANY == (pProfile)->BSSType)
#define CSR_IS_WDS_AP( pProfile )  ( eCSR_BSS_TYPE_WDS_AP == (pProfile)->BSSType )
#define CSR_IS_WDS_STA( pProfile ) ( eCSR_BSS_TYPE_WDS_STA == (pProfile)->BSSType )
#define CSR_IS_WDS( pProfile )  ( CSR_IS_WDS_AP( pProfile ) || CSR_IS_WDS_STA( pProfile ) )
#define CSR_IS_INFRA_AP( pProfile )  ( eCSR_BSS_TYPE_INFRA_AP == (pProfile)->BSSType )
#ifdef WLAN_FEATURE_NAN_DATAPATH
#define CSR_IS_NDI(pProfile)  (eCSR_BSS_TYPE_NDI == (pProfile)->BSSType)
#else
#define CSR_IS_NDI(pProfile)  (FALSE)
#endif

//pProfile - pointer to tCsrRoamConnectedProfile
#define CSR_IS_CONN_INFRA_AP( pProfile )  ( eCSR_BSS_TYPE_INFRA_AP == (pProfile)->BSSType )
#define CSR_IS_CONN_WDS_AP( pProfile )  ( eCSR_BSS_TYPE_WDS_AP == (pProfile)->BSSType )
#define CSR_IS_CONN_WDS_STA( pProfile ) ( eCSR_BSS_TYPE_WDS_STA == (pProfile)->BSSType )
#define CSR_IS_CONN_WDS( pProfile )  ( CSR_IS_WDS_AP( pProfile ) || CSR_IS_WDS_STA( pProfile ) )

#ifdef WLAN_FEATURE_NAN_DATAPATH
#define CSR_IS_CONN_NDI(pProfile)  (eCSR_BSS_TYPE_NDI == (pProfile)->BSSType)
#else
#define CSR_IS_CONN_NDI(pProfile)  (FALSE)
#endif



///////////////////////////////////////////Common Roam ends

#ifdef WLAN_FEATURE_SAE
#define CSR_IS_AUTH_TYPE_SAE(auth_type) \
	(eCSR_AUTH_TYPE_SAE == auth_type)
#else
#define CSR_IS_AUTH_TYPE_SAE(auth_type) (false)
#endif

/* ---------------------------------------------------------------------------
    \fn csrSetChannels
    \brief HDD calls this function to change some global settings.
    caller must set the all fields or call csrGetConfigParam to prefill the fields.
    \param pParam - caller allocated memory
    \return eHalStatus
  -------------------------------------------------------------------------------*/

eHalStatus csrSetChannels(tHalHandle hHal,  tCsrConfigParam *pParam  );

eHalStatus csrSetRegInfo(tHalHandle hHal,  tANI_U8 *apCntryCode);


//enum to string conversion for debug output
const char * get_eRoamCmdStatus_str(eRoamCmdStatus val);
const char * get_eCsrRoamResult_str(eCsrRoamResult val);
/* ---------------------------------------------------------------------------
    \fn csrSetPhyMode
    \brief HDD calls this function to set the phyMode.
    This function must be called after CFG is downloaded and all the band/mode setting already passed into
    CSR.
    \param phyMode - indicate the phyMode needs to set to. The value has to be either 0, or some bits set.
    See eCsrPhyMode for definition
    \param eBand - specify the operational band (2.4, 5 or both)
    \param pfRestartNeeded - pointer to a caller allocated space. Upon successful return, it indicates whether
    a restart is needed to apply the change
    \return eHalStatus
  -------------------------------------------------------------------------------*/
eHalStatus csrSetPhyMode(tHalHandle hHal, tANI_U32 phyMode, eCsrBand eBand, tANI_BOOLEAN *pfRestartNeeded);

void csrDumpInit(tHalHandle hHal);


/*---------------------------------------------------------------------------
  This is the type for a link quality callback to be registered with SME
  for indications
  Once the link quality has been indicated, subsequently, link indications are
  posted each time there is a CHANGE in link quality.
  *** If there is no change in link, there will be no indication ***

  The indications may be based on one or more criteria internal to SME
  such as RSSI and PER.

  \param ind - Indication being posted
  \param pContext - any user data given at callback registration.
  \return None

---------------------------------------------------------------------------*/
typedef void (* csrRoamLinkQualityIndCallback)
             (eCsrRoamLinkQualityInd  ind, void *pContext);


/*---------------------------------------------------------------------------
  This is the type for a statistics callback to be registered with SME
  for stats reporting

  Since the client requesting for the stats already know which class/type of
  stats it asked for, the callback will carry them in the rsp buffer
  (void * stats) whose size will be same as the size of requested stats &
  will be exactly in the same order requested in the stats mask from LSB to MSB

  \param stats - stats rsp buffer sent back with the report
  \param pContext - any user data given at callback registration.
  \return None

---------------------------------------------------------------------------*/
typedef void ( *tCsrStatsCallback) (void * stats, void *pContext);

/*---------------------------------------------------------------------------
  This is the type for a rssi callback to be registered with SME
  for getting rssi

  \param rssi - rssi
  \param pContext - any user data given at callback registration.
  \return None

---------------------------------------------------------------------------*/

typedef void ( *tCsrRssiCallback) (v_S7_t rssi, tANI_U32 staId, void *pContext);

#if defined(FEATURE_WLAN_ESE) && defined(FEATURE_WLAN_ESE_UPLOAD)
/*---------------------------------------------------------------------------
  This is the type for a tsm stats callback to be registered with SME
  for getting tsm stats
  \param tsmMetrics - tsmMetrics
  \param pContext - any user data given at callback registration.
  \return None
---------------------------------------------------------------------------*/
typedef void ( *tCsrTsmStatsCallback) (tAniTrafStrmMetrics tsmMetrics,
                                       tANI_U32 staId,
                                       void *pContext);
#endif /* FEATURE_WLAN_ESE && FEATURE_WLAN_ESE_UPLOAD */

/*---------------------------------------------------------------------------
  This is the type for a snr callback to be registered with SME
  for getting snr

  \param snr
  \param pContext - any user data given at callback registration.
  \return None

---------------------------------------------------------------------------*/
typedef void (*tCsrSnrCallback) (v_S7_t snr, tANI_U32 staId, void *pContext);

#ifdef WLAN_FEATURE_VOWIFI_11R
eHalStatus csrRoamIssueFTPreauthReq(tHalHandle hHal, tANI_U32 sessionId, tpSirBssDescription pBssDescription);
#endif

/*---------------------------------------------------------------------------
  This is the function to change the Band configuraiton (ALL/2.4 GHZ/5 GHZ)

  \param hHal - handle to Hal context
  \param eBand - band value
  \param sessionId - Session Identifier
  \return  eHalStatus

---------------------------------------------------------------------------*/
eHalStatus csrSetBand(tHalHandle hHal, tANI_U8 sessionId, eCsrBand eBand);

/*---------------------------------------------------------------------------
  This is the function to get the current operating band value
  \param hHal - handl to Hal context
  \return eCsrband - band value

---------------------------------------------------------------------------*/
eCsrBand csrGetCurrentBand (tHalHandle hHal);

typedef void (*csrReadyToSuspendCallback)(void *pContext, boolean suspended);

#ifdef WLAN_FEATURE_EXTWOW_SUPPORT
typedef void (*csrReadyToExtWoWCallback)(void *pContext, boolean status);
#endif

#ifdef WLAN_FEATURE_ROAM_OFFLOAD
eHalStatus csrRoamIssueFTRoamOffloadSynch(tHalHandle hHal, tANI_U32 sessionId,
                                          tSirBssDescription *pBssDescription);
#endif

/*---------------------------------------------------------------------------
  Callback to be registered with SME for getting link status
  \param status - link stats req result status
  \param pContext - any user data given at callback registration.
  \return None

---------------------------------------------------------------------------*/
typedef void (*tCsrLinkStatusCallback)(v_U8_t status, void *pContext);
typedef void (*csr_mib_stats_callback)
			(struct mib_stats_metrics *mib_stats, void *context);
/**
 * tcsr_fw_state_callback() -HDD callback registered with SME for getting
 *  firmware state
 * @context: HDD callback context
 *
 * Return: void
 */
typedef void (*tcsr_fw_state_callback)(void *context);
void csr_packetdump_timer_stop(void);
#ifdef FEATURE_WLAN_MCC_TO_SCC_SWITCH
typedef struct _session_info{
	tVOS_CON_MODE con_mode;
	eCsrBand band;
	v_U16_t och;
	v_U16_t lfreq;
	v_U16_t hfreq;
	v_U16_t cfreq;
	v_U16_t hbw;
}session_info_t;
tANI_BOOLEAN csr_find_all_session_info(
	tHalHandle hHal,
	session_info_t *session_info,
	v_U8_t * session_count);
tANI_BOOLEAN csr_find_sta_session_info(
	tHalHandle hHal,
	session_info_t *info);
tANI_BOOLEAN csr_create_sap_session_info(
	tHalHandle hHal,
	eCsrPhyMode sap_phymode,
	v_U16_t sap_ch,
	session_info_t *session_info);
#endif
struct lim_channel_status *csr_get_channel_status(
	void *p_mac, uint32_t channel_id);
void csr_clear_channel_status(void *p_mac);
#endif
