/*
 * ist3602 LCD module driver
 *
 * Copyright 2021, 2022 Sony Corporation
 */

#include <malloc.h>
#include <common.h>
#include <errno.h>
#include <command.h>
#include <dm.h>
#include <misc.h>
#include <linux/kernel.h>
#include <asm/gpio.h>
#include <asm-generic/gpio.h>
#include <ist3602.h>

DECLARE_GLOBAL_DATA_PTR;

/* Define this label when enable Debug log */
/* #define IST3602_DEBUG */

/* Define print macros */
#ifdef IST3602_DEBUG
#define DEBUG(fmt, args...) printf("%s(%d): " fmt, __func__, __LINE__, ##args)
#else
#define DEBUG(fmt, args...) {}
#endif

/* ist3602 Driver Name */
#define DEVICE_NAME		"ist3602"
#define MAX_ARG_LENGTH		(1024)

#define VOL_LIMIT		(18000)		/* Regurator voltage limit[mV] */
#define BASE_DISP_CLOCK		(750000)	/* Base display clock[Hz] */
#define DUTY_1_9		(9)		/* 1/9 Display Duty */
#define DUTY_1_17		(17)		/* 1/17 Display Duty */
#define DUTY_1_25		(25)		/* 1/25 Display Duty */
#define DUTY_1_33		(33)		/* 1/33 Display Duty */
#define DIMMER_GPIO_BID		(4)		/* Use Dimmer GPIO board id */

/* IST3602 command code 	0xBBDDD (BB=Busy[ms],DDD=Data) */
#define IST3602_DISPLAY_CLEAR	0x02001		/* (2ms) */
#define IST3602_RETURN_ZERO	0x02002		/* (2ms) */
#define IST3602_SET_ENTRY_MODE	0x00004		/* (0ms) */
#define IST3602_SET_DISPLAY	0x00008		/* (0ms) */
#define IST3602_FUNCTIONSET_0	0x00020		/* (0ms) */
#define IST3602_FUNCTIONSET_1	0x00021		/* (0ms) */
#define IST3602_FUNCTIONSET_2	0x00022		/* (0ms) */
#define IST3602_FUNCTIONSET_3	0x00023		/* (0ms) */
#define IST3602_SET_DDRAM	0x00090		/* (0ms) */
/* Command on IST=0 */
#define IST3602_SHIFT_CSR_DSP	0x00010		/* (0ms) */
#define IST3602_SET_CGRAM	0x00091		/* (0ms) */
/* Command on IST=1 */
#define IST3602_FOLLOWS_CTRL	0x00010		/* (0ms) */
#define IST3602_SET_ICONRAM	0x00040		/* (0ms) */
#define IST3602_POWER_CTRL1	0x00030		/* (0ms) */
#define IST3602_POWER_CTRL2	0x00060		/* (0ms) */
#define IST3602_V0_CTRL2	0x64070		/* (100ms) */
/* Command on IST=2 */
#define IST3602_SET_DISP_MODE	0x00010		/* (0ms) */
#define IST3602_SET_DISP_PTN	0x00060		/* (0ms) */
#define IST3602_SEL_CGRAM	0x00040		/* (0ms) */
/* Command on IST=3 */
#define IST3602_START_LINE	0x00082		/* (0ms) */
#define IST3602_CONTRAST_SET	0x00081		/* (0ms) */
#define IST3602_RGAIN_SET	0x000A0
/* Command on TestMode */
#define IST3602_IST_ENTRY	0x00088		/* (0ms) */
#define IST3602_FRAME_RATE_ADJ	0x00020		/* (0ms) */
#define IST3602_FRAME_RATE	0x000B2		/* (0ms) */
#define IST3602_OST_CLK_SEL	0x00090		/* (0ms) */
#define IST3602_OST_DIV_SEL	0x00098		/* (0ms) */
#define IST3602_EXIT_ENTRY	0x000E3		/* (0ms) */

struct ist3602_priv {
	struct gpio_desc en_gpio;
	struct gpio_desc rst_gpio;
	struct gpio_desc dmr_gpio;
	struct gpio_desc brt_gpio;
	struct gpio_desc ss_gpio;
	struct gpio_desc sclk_gpio;
	struct gpio_desc mosi_gpio;
	int width;			/* Displayed colmuns */
	int lines;			/* Displayed lines */
	int fonts;			/* Font size */
	int contrast;			/* Contrast current value */
	int ratio;			/* Regulation Ratio current value */
	int contrast_std;		/* Contrast when Dimmer OFF */
	int ratio_std;			/* Regulation Ratio when Dimmer OFF */
	int contrast_dmr;		/* Contrast when Dimmer ON */
	int ratio_dmr;			/* Regulation Ratio when Dimmer ON */
	int column_offset;		/* Column Offset */
	const char *boot_msg;		/* Message at boot time */
	int dimmer_flag;		/* Dimmer ON/OFF flag */
	bool enable_dmr_gpio;		/* Enable to control Dimmer gpio */
	int frame_rate;			/* Frame rate value */
	int bwidth;			/* Buffer size per line */
	int bias;			/* Voltage Follow Bias select */
	int booster;			/* Booster select */
	int column;			/* column pos */
	int line;			/* Line pos */
	int flag;			/* Flag(b3=Display,b2=blink,b1=cursor) */
	int ss_flag;			/* ss_gpio flags */
	int sclk_flag;			/* sclk_gpio flags */
	int mosi_flag;			/* mosi_gpio flags */
};

static const int ist3602_init[] = {
	/* b27-20:Repeat,b19-12b=Busy,b8=D/C,b7-0:Data */
	IST3602_FUNCTIONSET_0,		/* Function set (IS=00) */
	IST3602_DISPLAY_CLEAR,		/* Display Clear */
	IST3602_RETURN_ZERO,		/* Reset to zero pos */
	IST3602_SET_ENTRY_MODE | 0x02,	/* Set entry mode(Incliment) */
	IST3602_SET_DISPLAY | 0x00,	/* Display control(D=0,C=0,B=0) */
	IST3602_FUNCTIONSET_1,		/* Function set (IS=01) */
	IST3602_FOLLOWS_CTRL | 0x00,	/* Follows control(1/4 BIAS) */
	IST3602_SET_ICONRAM | 0x00,	/* Set ICON RAM Address */
	0x1F00100,			/* ICON RAM FILL(0x00 x 32) */
	IST3602_POWER_CTRL1 | 0x00,	/* Power control1(SLEEP mode) */
	IST3602_POWER_CTRL2 | 0x0F,	/* ICON/Power Control(I/R/L/F) */
	IST3602_V0_CTRL2 | 0x00,	/* V0 control Set(booster:5X) */
	IST3602_FUNCTIONSET_3,		/* Function set (IS=11) */
	IST3602_START_LINE,		/* Start line setting */
	0x000,				/* (0) */
	IST3602_FUNCTIONSET_2,		/* Function set (IS=10) */
	IST3602_SET_DISP_MODE | 0x00,	/* Set display mode(D=0,F=0) */
	IST3602_SET_DISP_PTN | 0x00,	/* Set display pattern(NRM,NOALL) */
	IST3602_SEL_CGRAM | 0x07,	/* Select CGRAM & COM/SEG direction */
	IST3602_FUNCTIONSET_0,		/* Function set (IS=00) */
	IST3602_SET_CGRAM,		/* Set CGRAM address */
	0x000,				/* (0) */
	/* Default Font */
	0x102,				/* 0x00:"   # " */
	0x105,				/* 0x00:"  # #" */
	0x0400104,			/* 0x00:"  #  " */
					/* 0x00:"  #  " */
					/* 0x00:"  #  " */
					/* 0x00:"  #  " */
					/* 0x00:"  #  " */
	0x100,
	0x0400104,			/* 0x01:"  #  " */
					/* 0x01:"  #  " */
					/* 0x01:"  #  " */
					/* 0x01:"  #  " */
					/* 0x01:"  #  " */
	0x114,				/* 0x01:"# #  " */
	0x108,				/* 0x01:" #   " */
	0x0200100,
					/* 0x02:"     " */
					/* 0x02:"     " */
	0x10A,				/* 0x02:" # # " */
	0x0100115,			/* 0x02:"# # #" */
					/* 0x02:"# # #" */
	0x10A,				/* 0x02:" # # " */
	0x0100100,			/* 0x02:"     " */

	0x11F,				/* 0x03:"#####" */
	0x111,				/* 0x03:"#   #" */
	0x112,				/* 0x03:"#  # " */
	0x114,				/* 0x03:"# #  " */
	0x118,				/* 0x03:"##   " */
	0x110,				/* 0x03:"#    " */
	0x0100100,			/* 0x03:"     " */

	/* Extra Font */
	0x11F,				/* 0x1f:"#####" */
	0x040010A,			/* 0x1f:" # # " */
					/* 0x1f:" # # " */
					/* 0x1f:" # # " */
					/* 0x1f:" # # " */
					/* 0x1f:" # # " */
	0x11F,				/* 0x1f:"#####" */
	0x0100100,
					/* 0x5C:"     " */
	0x110,				/* 0x5C:"#    " */
	0x108,				/* 0x5C:" #   " */
	0x104,				/* 0x5C:"  #  " */
	0x102,				/* 0x5C:"   # " */
	0x101,				/* 0x5C:"    #" */
	0x0100100,			/* 0x5C:"     " */

	0x10D,				/* 0x7E:" ## #" */
	0x116,				/* 0x7E:"# ## " */
	0x0500100,			/* 0x7E:"     " */
					/* 0x7E:"     " */
					/* 0x7E:"     " */
					/* 0x7E:"     " */
					/* 0x7E:"     " */

	0x111,				/* 0x84:"#   #" */
	0x10E,				/* 0x84:" ### " */
	0x0200111,			/* 0x84:"#   #" */
					/* 0x84:"#   #" */
					/* 0x84:"#   #" */
	0x10E,				/* 0x84:" ### " */
	0x111,				/* 0x84:"#   #" */
	0x100,
	IST3602_SET_DDRAM,		/* Set DDRAM address */
	0x000,				/* (0) */
};

static const int byteperline[] = { 0x60, 0x30, 0x20, 0x18 };

const int swap_char[] = {
	/* Swap Code */
	0x1F, 0x04,			/* "II"(0x04) */
	0x5C, 0x05,			/* "\" (0x05) */
	0x7E, 0x06,			/* "~" (0x06) */
	0x80, 0xE9,			/* "~!"(0xE9) */
	0x81, 0x9F,			/* "~?"(0x9F) */
	0x82, 0x12,			/* "ss"(0x12) */
	0x83, 0xE6,			/* "en"(0xE6) */
	0x84, 0x07,			/* "`o"(0x07) */
	0x85, 0xE5,			/* "fl"(0xE5) */
	0x86, 0xE4,			/* "Ct"(0xE4) */
};

static int write_device(char *data, size_t size)
{
	struct udevice *dev;
	int ret;

	ret = uclass_get_device_by_name(UCLASS_MISC, DEVICE_NAME, &dev);
	if (ret) {
		printf("uclass_get_device_by_name failed(%d)\n", ret);
		return CMD_RET_FAILURE;
	}

	ret = misc_write(dev, 0, data, size);
	if (ret) {
		printf("write failed(%d)\n", ret);
		return CMD_RET_FAILURE;
	}

	return CMD_RET_SUCCESS;
}

static int do_print(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
	char str[MAX_ARG_LENGTH];
	char *arg, c;
	int len = 0, val = 0, esc = 0;

	if (argc < 2)
		return CMD_RET_FAILURE;

	arg = argv[1];
	while (*arg != 0 && len < MAX_ARG_LENGTH) {
		c = *(arg++);
		if (esc == 1) {
			switch (c) {
			case '0':
				str[len++] = 0x00;
				esc = 0;
				break;
			case 'f':
				str[len++] = 0x0C;
				esc = 0;
				break;
			case 'h':
				str[len++] = 0x0B;
				esc = 0;
				break;
			case 'n':
				str[len++] = 0x0A;
				esc = 0;
				break;
			case 'r':
				str[len++] = 0x0D;
				esc = 0;
				break;
			case 'x':
				esc++;
				break;
			default:
				str[len++] = c;
				esc = 0;
			}
		} else if (esc > 1) {
			if (c >= '0' && c <= '9') {
				val = (val * 16) + (c - '0');
				esc++;
			} else if (c >= 'A' && c <= 'F') {
				val = (val * 16) + (c - 'A' + 10);
				esc++;
			} else if (c >= 'a' && c <= 'f') {
				val = (val * 16) + (c - 'a' + 10);
				esc++;
			} else {
				esc = 0;
			}
			if (esc == 4) {
				str[len++] = (char)(val & 0xFF);
				esc = 0;
			}
		} else if (c == '\\') {
			esc = 1;
		} else {
			str[len++] = c;
		}
	}

	return write_device(str, len);
}

static int do_init(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
	char str[2];

	str[0] = 0x1B;
	str[1] = 'I';
	return write_device(str, sizeof(str));
}

static int do_line(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
	char str[3];

	if (argc < 2)
		return CMD_RET_FAILURE;

	str[0] = 0x1B;
	str[1] = 'L';
	str[2] = *(argv[1]);
	return write_device(str, sizeof(str));
}

static int do_font(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
	char str[3];

	if (argc < 2)
		return CMD_RET_FAILURE;

	str[0] = 0x1B;
	str[1] = 'F';
	str[2] = *(argv[1]);
	return write_device(str, sizeof(str));
}

static cmd_tbl_t cmd_lcd[] = {
	U_BOOT_CMD_MKENT(print, 1, 1, do_print, "", ""),
	U_BOOT_CMD_MKENT(reset, 0, 1, do_init, "", ""),
	U_BOOT_CMD_MKENT(line, 1, 1, do_line, "", ""),
	U_BOOT_CMD_MKENT(font, 1, 1, do_font, "", ""),
};

static int do_lcd(cmd_tbl_t * cmdtp, int flag, int argc, char * const argv[])
{
	cmd_tbl_t *sel;

	if (argc < 2)
		return CMD_RET_USAGE;

	/* Strip off leading 'lcd' command argument */
	argc--;
	argv++;

	sel = find_cmd_tbl(argv[0], &cmd_lcd[0], ARRAY_SIZE(cmd_lcd));

	if (sel)
		return sel->cmd(cmdtp, flag, argc, argv);
	else
		return CMD_RET_USAGE;
}

static char lcd_help_text[] =
	"print strings      - print strings\n"
	"reset              - reset lcd panel\n"
	"line lines(1-4)    - set display line\n"
	"font fontsize(1-4) - set font size\n";

U_BOOT_CMD(
	lcd, 4, 1, do_lcd,
	"lcd sub-system",
	lcd_help_text
);

void initr_ist3602(void)
{
	int ret;
	struct udevice *dev = NULL;

	ret = uclass_get_device_by_name(UCLASS_MISC, DEVICE_NAME, &dev);
	if (ret != 0 || !dev) {
		printf("%s: Can not probe device: name=%s, ret=%d\n",
			__func__, DEVICE_NAME, ret);
	}
}

static void ist3602_send(struct ist3602_priv *priv, int val)
{
	int ret, bsy, rep, cnt, bit;

	bsy = ((val & 0x00FF000) >> 12);
	rep = ((val & 0xFF00000) >> 20) + 1;
	DEBUG("Send:0x%03x x %d\n", val, rep);
	ret = dm_gpio_set_value(&priv->ss_gpio, 1);
	for (cnt = 0; cnt < rep; cnt++) {
		for (bit = 0; bit < 9; bit++) {
			ret |= dm_gpio_set_value(&priv->mosi_gpio,
						(val >> (8 - bit)) & 1);
			udelay(1);
			ret |= dm_gpio_set_value(&priv->sclk_gpio, 1);
			udelay(1);
			ret |= dm_gpio_set_value(&priv->sclk_gpio, 0);
		}
		udelay(bsy * 1000);
	}
	udelay(1);
	ret |= dm_gpio_set_value(&priv->ss_gpio, 0);

	if (ret)
		printf("%s: gpio_set_value error(%d)\n", DEVICE_NAME, ret);
}

static void ist3602_set_mode(struct ist3602_priv *priv)
{
	/* Set Lines & Font size */
	ist3602_send(priv, IST3602_FUNCTIONSET_2);
	ist3602_send(priv, IST3602_SET_DISP_MODE
			| ((priv->lines - 1) & 3)
			| ((priv->fonts - 1) & 3) << 2);
	ist3602_send(priv, IST3602_FUNCTIONSET_0);

	priv->bwidth = byteperline[(priv->lines - 1) & 3];
}

static void ist3602_set_disp(struct ist3602_priv *priv, int syncslp)
{
	if (syncslp) {
		ist3602_send(priv, IST3602_FUNCTIONSET_1);
		if (priv->flag & 0x04)
			ist3602_send(priv, IST3602_POWER_CTRL1);
		else
			ist3602_send(priv, IST3602_POWER_CTRL1 | 0x01);
		ist3602_send(priv, IST3602_FUNCTIONSET_0);
	}
	ist3602_send(priv, IST3602_SET_DISPLAY | priv->flag);
}

static void ist3602_set_pos(struct ist3602_priv *priv)
{
	ist3602_send(priv, IST3602_SET_DDRAM);
	ist3602_send(priv, (priv->line * priv->bwidth)
		+ ((priv->column + priv->column_offset) % priv->bwidth));
}

static void ist3602_set_framerate(struct ist3602_priv *priv)
{
	int row_freq = 0;
	int framerate_ctl = 0;

	/* Row frequency = Frame rate * Duty */
	/* Frame rate control = BASE_DISP_CLOCK / Row frequency - 1 */
	switch (priv->fonts * priv->lines) {
	case 1:
		row_freq = priv->frame_rate * DUTY_1_9;
		break;
	case 2:
		row_freq = priv->frame_rate * DUTY_1_17;
		break;
	case 3:
		row_freq = priv->frame_rate * DUTY_1_25;
		break;
	case 4:
		row_freq = priv->frame_rate * DUTY_1_33;
		break;
	default:
		printf("%s: Failed to set frame rate\n", DEVICE_NAME);
		return;
	}
	framerate_ctl = BASE_DISP_CLOCK / row_freq - 1;
	DEBUG("frame rate control = %d\n", framerate_ctl);

	ist3602_send(priv, IST3602_FUNCTIONSET_3);
	ist3602_send(priv, IST3602_IST_ENTRY);
	ist3602_send(priv, IST3602_IST_ENTRY);
	ist3602_send(priv, IST3602_IST_ENTRY);
	ist3602_send(priv, IST3602_IST_ENTRY);
	ist3602_send(priv, IST3602_FRAME_RATE_ADJ | 0x08);
	ist3602_send(priv, IST3602_FRAME_RATE);
	ist3602_send(priv, framerate_ctl & 0x00FF);
	ist3602_send(priv, (framerate_ctl & 0xFF00) >> 8);
	ist3602_send(priv, IST3602_OST_CLK_SEL | 0x03);
	ist3602_send(priv, IST3602_OST_DIV_SEL | 0x01);
	ist3602_send(priv, IST3602_EXIT_ENTRY);
	ist3602_send(priv, IST3602_FUNCTIONSET_0);
}

static void ist3602_set_regulator(struct ist3602_priv *priv)
{
	int v0;

	/* RGR = 2.0 + rate * 0.5              */
	/* V0 = RGR * [(393 + CT) / 648] * 2.1 */
	v0 = ((20 + priv->ratio * 5)
		* (((393 + priv->contrast) * 2100) / 648)) / 10;
	DEBUG("regurator voltage=%d[mV]\n", v0);

	if (v0 <= VOL_LIMIT) {
		ist3602_send(priv, IST3602_FUNCTIONSET_3);
		ist3602_send(priv, IST3602_CONTRAST_SET);
		ist3602_send(priv, priv->contrast);
		ist3602_send(priv, IST3602_RGAIN_SET | priv->ratio);
		ist3602_send(priv, IST3602_FUNCTIONSET_0);
	} else {
		printf("%s:Warning: regurator-voltage over 18V\n", DEVICE_NAME);
	}
}

static void ist3602_set_bias(struct ist3602_priv *priv)
{
	ist3602_send(priv, IST3602_FUNCTIONSET_1);
	ist3602_send(priv, IST3602_FOLLOWS_CTRL | priv->bias);
	ist3602_send(priv, IST3602_FUNCTIONSET_0);
}

static void ist3602_set_booster(struct ist3602_priv *priv)
{
	ist3602_send(priv, IST3602_FUNCTIONSET_1);
	ist3602_send(priv, IST3602_V0_CTRL2 | priv->booster);
	ist3602_send(priv, IST3602_FUNCTIONSET_0);
}

static void ist3602_cls_home(struct ist3602_priv *priv, int cls)
{
	if (cls)
		ist3602_send(priv, IST3602_DISPLAY_CLEAR);
	else
		ist3602_send(priv, IST3602_RETURN_ZERO);
	priv->column = 0;
	priv->line = 0;
	if (priv->column_offset)
		ist3602_set_pos(priv);
}

static void ist3602_initial(struct ist3602_priv *priv)
{
	int i;

	if (priv->rst_gpio.dev)
		dm_gpio_set_value(&priv->rst_gpio, 1);
	if (priv->brt_gpio.dev) {
		dm_gpio_set_value(&priv->brt_gpio, 1);
		udelay(5000);
	}
	if (priv->en_gpio.dev)
		dm_gpio_set_value(&priv->en_gpio, 1);
	if (priv->rst_gpio.dev || priv->en_gpio.dev)
		udelay(10000);
	dm_gpio_set_value(&priv->ss_gpio, 0);
	dm_gpio_set_value(&priv->sclk_gpio, 0);
	if (priv->rst_gpio.dev)
		dm_gpio_set_value(&priv->rst_gpio, 0);
	udelay(10000);

	/* Initial Register Set */
	for (i = 0; i < ARRAY_SIZE(ist3602_init); i++)
		ist3602_send(priv, (int)ist3602_init[i]);

	/* Set Dimmer ON/OFF */
	if (priv->dmr_gpio.dev) {
		if (priv->enable_dmr_gpio)
			dm_gpio_set_value(&priv->dmr_gpio, priv->dimmer_flag);
		else
			dm_gpio_set_value(&priv->dmr_gpio, 0);
	}

	/* Set Contrast & Regulation Ratio */
	if (!priv->enable_dmr_gpio && priv->dimmer_flag) {
		priv->contrast = priv->contrast_dmr;
		priv->ratio = priv->ratio_dmr;
	} else {
		priv->contrast = priv->contrast_std;
		priv->ratio = priv->ratio_std;
	}
	ist3602_set_regulator(priv);

	/* Set Voltage follow Bias */
	ist3602_set_bias(priv);

	/* Set booster */
	ist3602_set_booster(priv);

	/* Set lines & font size */
	ist3602_set_mode(priv);

	/* parameter initial */
	priv->column = 0;
	priv->line = 0;
	priv->flag = 0x04;

	/* Set Frame rate */
	ist3602_set_framerate(priv);

	/* Set Home position */
	if (priv->column_offset)
		ist3602_set_pos(priv);

	/* Display ON */
	ist3602_send(priv, IST3602_SET_DISPLAY | priv->flag);
}

static void ist3602_put_char(struct ist3602_priv *priv, int c)
{
	int i;

	for (i = 0; i < ARRAY_SIZE(swap_char); i++) {
		if ((int)swap_char[i] == c) {
			c =  swap_char[i ^ 1];
			break;
		}
	}

	ist3602_send(priv, c | 0x100);
	priv->column++;
	if (priv->column == priv->width) {
		priv->line = (priv->line + 1) % priv->lines;
		priv->column = 0;
		ist3602_set_pos(priv);
	} else if (priv->column == (priv->bwidth - priv->column_offset)) {
		ist3602_set_pos(priv);
	}
}

static int ist3602_write(struct udevice *dev, int offset,
						const void *buf, int size)
{
	struct ist3602_priv *priv = dev_get_platdata(dev);
	int i, esc = 0, val = 0;
	char c, *txt = (char *)buf;

	DEBUG("enter\n");
	dm_gpio_set_dir_flags(&priv->ss_gpio, priv->ss_flag);
	dm_gpio_set_dir_flags(&priv->sclk_gpio, priv->sclk_flag);
	dm_gpio_set_dir_flags(&priv->mosi_gpio, priv->mosi_flag);

	for (i = 0; i < size; i++) {
		c = *(txt++);
		if (esc) {
			switch(c) {
			case 'I':
				ist3602_initial(priv);
				esc = 0;
				break;
			case 'D':
				priv->flag |= 0x04;
				ist3602_set_disp(priv, 1);
				esc = 0;
				break;
			case 'd':
				priv->flag &= ~0x04;
				ist3602_set_disp(priv, 1);
				esc = 0;
				break;
			case 'C':
				priv->flag |= 0x02;
				ist3602_set_disp(priv, 0);
				esc = 0;
				break;
			case 'c':
				priv->flag &= ~0x02;
				ist3602_set_disp(priv, 0);
				esc = 0;
				break;
			case 'B':
				priv->flag |= 0x01;
				ist3602_set_disp(priv, 0);
				esc = 0;
				break;
			case 'b':
				priv->flag &= ~0x01;
				ist3602_set_disp(priv, 0);
				esc = 0;
				break;
			case '+':
				priv->dimmer_flag = 0;
				if (priv->dmr_gpio.dev &&
				    priv->enable_dmr_gpio) {
					dm_gpio_set_value(&priv->dmr_gpio,
							  priv->dimmer_flag);
				} else if (!priv->enable_dmr_gpio) {
					priv->contrast = priv->contrast_std;
					priv->ratio = priv->ratio_std;
					ist3602_set_regulator(priv);
				}
				esc = 0;
				break;
			case '-':
				priv->dimmer_flag = 1;
				if (priv->dmr_gpio.dev &&
				    priv->enable_dmr_gpio) {
					dm_gpio_set_value(&priv->dmr_gpio,
							  priv->dimmer_flag);
				} else if (!priv->enable_dmr_gpio) {
					priv->contrast = priv->contrast_dmr;
					priv->ratio = priv->ratio_dmr;
					ist3602_set_regulator(priv);
				}
				esc = 0;
				break;
			case 'F':
				esc = 2;
				val = 0;
				break;
			case 'L':
				esc = 3;
				val = 0;
				break;
			case 'x':
				if (esc == 5)
					priv->line = val;
				esc = 4;
				val = 0;
				break;
			case 'y':
				if (esc == 4)
					priv->column = val;
				esc = 5;
				val = 0;
				break;
			case 'w':
				esc = 6;
				val = 0;
				break;
			case '0' ... '9':
				val = (val * 10) + (c - '0');
				if (esc == 2) {
					priv->fonts = val;
					ist3602_set_mode(priv);
					esc = 0;
				} else if (esc == 3) {
					priv->lines = val;
					ist3602_set_mode(priv);
					esc = 0;
				}
				break;
			case ';':
				if (esc == 4) {
					priv->column = val;
					ist3602_set_pos(priv);
				} else if (esc == 5) {
					priv->line = val;
					ist3602_set_pos(priv);
				} else if (esc == 6) {
					priv->width = val;
				}
				esc = 0;
				break;
			default:
				ist3602_put_char(priv, c);
				esc = 0;
			}
		} else {
			switch(c) {
			case 0x1B:
				esc = 1;
				break;
			case 0x0B:
				ist3602_cls_home(priv, 0);
				break;
			case 0x0C:
				ist3602_cls_home(priv, 1);
				break;
			case 0x0A:
				priv->line = (priv->line + 1) % priv->lines;
				priv->column = 0;
				ist3602_set_pos(priv);
				break;
			case 0x0D:
				priv->column = 0;
				ist3602_set_pos(priv);
				break;
			default:
				ist3602_put_char(priv, c);
			}
		}
	}

	dm_gpio_set_dir_flags(&priv->ss_gpio, GPIOD_IS_IN);
	dm_gpio_set_dir_flags(&priv->sclk_gpio, GPIOD_IS_IN);
	dm_gpio_set_dir_flags(&priv->mosi_gpio, GPIOD_IS_IN);
	DEBUG("done\n");
	return 0;
}

static int ist3602_read_board_id(void)
{
	int ret, board_id;
	struct udevice *dev;

	ret = uclass_get_device_by_name(UCLASS_MISC, "board_id", &dev);
	if (ret) {
		printf("uclass_get_device_by_name failed: ret=%d\n", ret);
		return ret;
	}
	ret = misc_read(dev, 0, &board_id, sizeof(board_id));
	if (ret) {
		printf("misc_read failed: ret=%d\n", ret);
		return ret;
	}

	return board_id;
}

static int ist3602_probe(struct udevice *dev)
{
	struct ist3602_priv *priv = dev_get_platdata(dev);
	int ret, val;

	/* default settings */
	priv->width = 16;
	priv->lines = 2;
	priv->fonts = 1;
	priv->contrast_std = 170;
	priv->ratio_std = 8;
	priv->contrast_dmr = 90;
	priv->ratio_dmr = 8;
	priv->column_offset = 0;
	priv->boot_msg = NULL;
	priv->frame_rate = 70;
	priv->bias = 0;
	priv->booster = 0;
	priv->dimmer_flag = 0;

	DEBUG("enter\n");
	ret = gpio_request_by_name(dev, "lcd_en-gpios", 0, &priv->en_gpio,
							GPIOD_IS_OUT);
	if (ret < 0) {
		if (ret != -ENOENT) {
			printf("%s: lcd_en-gpios not detect(%d)\n",
							DEVICE_NAME, ret);
			return ret;
		}
		priv->en_gpio.dev = NULL;
	}
	ret = gpio_request_by_name(dev, "lcd_rst-gpios", 0, &priv->rst_gpio,
							GPIOD_IS_OUT);
	if (ret < 0) {
		if (ret != -ENOENT) {
			printf("%s: lcd_rst-gpios not detect(%d)\n",
							DEVICE_NAME, ret);
			return ret;
		}
		priv->rst_gpio.dev = NULL;
	}
	ret = gpio_request_by_name(dev, "dimmer-gpios", 0, &priv->dmr_gpio,
							GPIOD_IS_OUT);
	if (ret < 0) {
		if (ret != -ENOENT) {
			printf("%s: dimmer-gpios not detect(%d)\n",
							DEVICE_NAME, ret);
			return ret;
		}
		priv->dmr_gpio.dev = NULL;
	}
	ret = gpio_request_by_name(dev, "lcd_bright-gpios", 0, &priv->brt_gpio,
							GPIOD_IS_OUT);
	if (ret < 0) {
		if (ret != -ENOENT) {
			printf("%s: lcd_bright-gpios not detect(%d)\n",
							DEVICE_NAME, ret);
			return ret;
		}
		priv->brt_gpio.dev = NULL;
	}
	ret = gpio_request_by_name(dev, "ss-gpios", 0, &priv->ss_gpio,
							GPIOD_IS_OUT);
	if (ret < 0) {
		printf("%s: ss-gpios not detect(%d)\n", DEVICE_NAME, ret);
		return ret;
	}
	priv->ss_flag = priv->ss_gpio.flags;
	ret = gpio_request_by_name(dev, "sclk-gpios", 0, &priv->sclk_gpio,
							GPIOD_IS_OUT);
	if (ret < 0) {
		printf("%s: sclk-gpios not detect(%d)\n", DEVICE_NAME, ret);
		return ret;
	}
	priv->sclk_flag = priv->sclk_gpio.flags;
	ret = gpio_request_by_name(dev, "mosi-gpios", 0, &priv->mosi_gpio,
							GPIOD_IS_OUT);
	if (ret < 0) {
		printf("%s: mosi-gpios not detect(%d)\n", DEVICE_NAME, ret);
		return ret;
	}
	priv->mosi_flag = priv->mosi_gpio.flags;

	if (dev_read_u32(dev, "width", (u32 *)&val) >= 0)
		priv->width = val;
	if (dev_read_u32(dev, "height", (u32 *)&val) >= 0)
		priv->lines = val;
	if (dev_read_u32(dev, "font_size", (u32 *)&val) >= 0)
		priv->fonts = val;
	if (dev_read_u32(dev, "contrast", (u32 *)&val) >= 0)
		priv->contrast_std = val;
	if (dev_read_u32(dev, "regulation-ratio", (u32 *)&val) >= 0)
		priv->ratio_std = val;
	if (dev_read_u32(dev, "contrast-dimmer", (u32 *)&val) >= 0)
		priv->contrast_dmr = val;
	if (dev_read_u32(dev, "regulation-ratio-dimmer", (u32 *)&val) >= 0)
		priv->ratio_dmr = val;
	if (dev_read_u32(dev, "column-offset", (u32 *)&val) >= 0)
		priv->column_offset = val;
	if (dev_read_u32(dev, "frame-rate", (u32 *)&val) >= 0)
		priv->frame_rate = val;
	if (dev_read_u32(dev, "bias", (u32 *)&val) >= 0)
		priv->bias = val & 0x03;
	if (dev_read_u32(dev, "booster", (u32 *)&val) >= 0)
		priv->booster = val & 0x03;
	if (dev_read_bool(dev, "dimmer_on_boot"))
		priv->dimmer_flag = 1;
	priv->enable_dmr_gpio = dev_read_bool(dev, "enable_dimmer_gpio");
	ret = ist3602_read_board_id();
	if (ret < 0)
		return ret;
	else if (ret >= DIMMER_GPIO_BID)
		priv->enable_dmr_gpio = true;
	priv->boot_msg = dev_read_string(dev, "boot_message");
	printf("%s: columns=%d lines=%d font_size=%d\n",
		DEVICE_NAME, priv->width, priv->lines, priv->fonts);
	ist3602_initial(priv);

	if (priv->boot_msg)
		ret = ist3602_write(dev, 0, priv->boot_msg,
						strlen(priv->boot_msg));

	DEBUG("done\n");
	return 0;
}

static const struct misc_ops ist3602_ops = {
	.write = ist3602_write,
};

static const struct udevice_id ist3602_ids[] = {
	{ .compatible = "sony,ist3602" },
	{}
};

U_BOOT_DRIVER(ist3602) = {
	.name = DEVICE_NAME,
	.id = UCLASS_MISC,
	.of_match = ist3602_ids,
	.probe = ist3602_probe,
	.platdata_auto_alloc_size = sizeof(struct ist3602_priv),
	.ops = &ist3602_ops,
};
