/*
 * Copyright 2022 Sony Corporation
 */

#include <common.h>
#include <errno.h>
#include <command.h>
#include <dm.h>
#include <misc.h>
#include <power_led.h>
#include <asm/gpio.h>
#include <asm-generic/gpio.h>

/* Power LED driver name */
#define DEVICE_NAME	"power_led"

struct power_led_priv {
	struct gpio_desc pled_gpio;
};

void initr_power_led(void)
{
	int ret;
	struct udevice *dev = NULL;

	ret = uclass_get_device_by_name(UCLASS_MISC, DEVICE_NAME, &dev);
	if (ret != 0 || !dev) {
		printf("%s: Can not probe device: name=%s, ret=%d\n",
			__func__, DEVICE_NAME, ret);
	}
}

static int power_led_write(struct udevice *dev, int offset, const void *buf,
	int size)
{
	struct power_led_priv *priv = dev_get_platdata(dev);
	int ret = 0;
	int level = *(int *)buf;

	if (priv->pled_gpio.dev) {
		ret = dm_gpio_set_value(&priv->pled_gpio, level);
		if (ret)
			printf("%s: gpio_set_value error(%d)\n", DEVICE_NAME, ret);
	}

	return ret;
}


static int power_led_probe(struct udevice *dev)
{
	struct power_led_priv *priv = dev_get_platdata(dev);
	int ret;
	int level = 1;

	ret = gpio_request_by_name(dev, "gpios", 0, &priv->pled_gpio,
							GPIOD_IS_OUT);
	if (ret < 0) {
		printf("%s: gpios not detect(%d)\n", DEVICE_NAME, ret);
		priv->pled_gpio.dev = NULL;
		return ret;
	}

	ret = power_led_write(dev, 0, &level, sizeof(level));
	if (ret != 0) {
		printf("%s: power led initialize error: name=%s, ret=%d\n",
			__func__, DEVICE_NAME, ret);
	}

	return ret;
}

static const struct misc_ops power_led_ops = {
	.write = power_led_write,
};

static const struct udevice_id power_led_ids[] = {
	{ .compatible = "sony,power-led" },
	{}
};

U_BOOT_DRIVER(power_led) = {
	.name = DEVICE_NAME,
	.id = UCLASS_MISC,
	.of_match = power_led_ids,
	.probe = power_led_probe,
	.platdata_auto_alloc_size = sizeof(struct power_led_priv),
	.ops = &power_led_ops,
};
