// ------------------------------------------------------------------------
//
//                (C) COPYRIGHT 2011 - 2015 SYNOPSYS, INC.
//                          ALL RIGHTS RESERVED
//
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  version 2 as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, see <https://gnu.org/licenses/>.
//
// ------------------------------------------------------------------------

#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/io-mapping.h>
#include <linux/irqdomain.h>
#include <linux/of_irq.h>
#include <linux/err.h>
#include <linux/dma-mapping.h>
#include <linux/of_device.h>
#ifndef CONFIG_ARCH_CXD900XX_FPGA
#include <linux/delay.h>
#include <linux/clk.h>
#include <linux/reset.h>
static struct clk *clk;
static struct reset_control *rst;
#endif

#include <elppdu.h>

#define DRV_NAME        "dw-elpmem"

// max of 16 devices
#define MAX_DEV 16

static struct platform_device *devices[MAX_DEV];
static int dev_id;

static int spdu_init(unsigned long baseaddr, pdu_info *info)
{
	void *pdu_mem;

	pr_info("SPAcc-PDU base address: %.8lx\n", baseaddr);

	pdu_mem = ioremap_nocache(baseaddr, 0x1000);
	if (!pdu_mem)
		return -ENOMEM;

	/* Read the config to see what's up */
	pdu_get_version(pdu_mem, info);

	/* Enable SPAcc-PDU interrupts. */
	pdu_io_write32(pdu_mem, PDU_IRQ_EN_GLBL); // enable all ints!

	iounmap(pdu_mem);
	return 0;
}

static void register_device(const char *name, int id,
                            const struct resource *res, unsigned num,
                            pdu_info *info)
{
	char suffix[16] = "";
	struct platform_device_info pdevinfo = {
		.name = name,
		.id = id,
		.res = res,
		.num_res = num,
		.data = info,
		.size_data = sizeof *info,
		.dma_mask = 0xffffffff,
	};

	if (dev_id >= MAX_DEV) {
		pr_err("Too many devices; increase MAX_DEV.\n");
		return;
	}

	devices[dev_id] = platform_device_register_full(&pdevinfo);
	if (IS_ERR(devices[dev_id])) {
		if (id >= 0)
			snprintf(suffix, sizeof suffix, ".%d", id);
		pr_err("Failed to register %s%s\n", name, suffix);

		devices[dev_id] = NULL;
		return;
	}
	of_dma_configure(&devices[dev_id]->dev, NULL);

	dev_id++;
}

static int pdu_probe(struct platform_device *pdev)
{
	int irq;
	struct resource res[2];
	pdu_info info;
	int i, rc;
	void *pdu_mem;
	unsigned long vex_baseaddr;
	struct resource *pdu_res;
#ifndef CONFIG_ARCH_CXD900XX_FPGA
	int status;
#endif

	irq = platform_get_irq(pdev, 0);

	if (irq < 0) {
		res[1] = (struct resource) { 0 };
		pr_err("IRQ setup failed, not using IRQs\n");
	}
	else {
		res[1] = (struct resource) {
			.start = irq,
			.end   = irq,
			.flags = IORESOURCE_IRQ,
		};
	}

	pdu_res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!pdu_res)
		return -EINVAL;

	vex_baseaddr = pdu_res->start;

#ifndef CONFIG_ARCH_CXD900XX_FPGA
	//get gate clock from device node
	clk = devm_clk_get(&pdev->dev, NULL);
	if (IS_ERR(clk)) {
		dev_err(&pdev->dev, "could not retrieve SPAcc clock\n");
		return -ENODEV;
	}

	rst = devm_reset_control_get(&pdev->dev, NULL);
	if (IS_ERR(rst)) {
		dev_err(&pdev->dev, "can not get reset\n");
		return -ENODEV;
	}

	//enable the gate clock
	status = clk_prepare_enable(clk);
	if (status) {
		dev_err(&pdev->dev, "could not enable SPAcc clock\n");
		return -ENODEV;
	}

	/* Reset controller */
	reset_control_assert(rst);
	reset_control_deassert(rst);
	udelay(1);

#endif

	rc = spdu_init(vex_baseaddr, &info);
	if (rc < 0)
		return -1;

	pdu_mem = ioremap_nocache(vex_baseaddr, 0x1000);
	if (!pdu_mem)
		return -ENOMEM;
	for (i = 0; i < info.spacc_config.num_vspacc; i++) {
		unsigned long offset = i*0x40000;

		if (info.spacc_version.is_pdu)
			offset += 0x40000;

		res[0] = (struct resource) {
			.start = vex_baseaddr + offset,
			.end   = vex_baseaddr + offset + 0x40000-1,
			.flags = IORESOURCE_MEM,
		};

		if (info.spacc_version.is_pdu) {
			pdu_io_write32(pdu_mem, pdu_io_read32(pdu_mem) | PDU_IRQ_EN_VSPACC(i));
		}
		register_device("spacc", i | (info.spacc_version.project << 16), res, 2, &info);
	}

	iounmap(pdu_mem);
	return 0;
}

static int pdu_remove(struct platform_device *pdev)
{
	int i;

#ifndef CONFIG_ARCH_CXD900XX_FPGA
	/* Reset controller */
	reset_control_assert(rst);
	//disable the gate clcock
	clk_disable_unprepare(clk);
#endif

	for (i = 0; i < dev_id; i++) {
		platform_device_unregister(devices[i]);
	}

	return 0;
}

static const struct of_device_id pdu_of_match[] = {
	{ .compatible = "snps,designware-elpmem" },
        { /* end */ },
};
MODULE_DEVICE_TABLE(of, pdu_of_match);

static struct platform_driver pdu_driver = {
	.probe  = pdu_probe,
	.remove = pdu_remove,
	.driver = {
		.name = DRV_NAME,
		.of_match_table = pdu_of_match,
	},
};

module_platform_driver(pdu_driver);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Elliptic Technologies Inc.");
