/*
 * Copyright (c) 2015 Elliptic Technologies Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <elppdu.h>

// some defines
#define PDU_REG_SPACC_VERSION   0x00180UL
#define PDU_REG_SPACC_CONFIG    0x00184UL
#define PDU_REG_PDU_CONFIG      0x00188UL
#define PDU_REG_SPACC_CONFIG2   0x00190UL
#define PDU_REG_SECURE_LOCK     0x001C0UL
#define PDU_REG_SPACC_IV_OFFSET 0x00040UL

#ifndef SPACC_ID_MINOR
#define SPACC_ID_MINOR(x)   ((x)         & 0x0F)
#define SPACC_ID_MAJOR(x)   (((x) >>  4) & 0x0F)
#define SPACC_ID_QOS(x)     (((x) >>  8) & 0x01)
#define SPACC_ID_TYPE(x)    (((x) >>  9) & 0x03)
#define SPACC_TYPE_SPACCQOS 0
#define SPACC_TYPE_PDU 1
#define SPACC_TYPE_HSM 2
#define SPACC_ID_AUX(x)     (((x) >> 11) & 0x01)
#define SPACC_ID_VIDX(x)    (((x) >> 12) & 0x07)
#define SPACC_ID_PARTIAL(x) (((x) >> 15) & 0x01)
#define SPACC_ID_PROJECT(x) ((x)>>16)

// macros for v4.7 and below
#define SPACC_ID_AUX_V47(x)    (((x) >> 10) & 1)
#define SPACC_ID_QOS_V47(x)    (((x) >> 8) & 1)
#define SPACC_ID_PDU_V47(x)    (((x) >> 9) & 1)

#define SPACC_CFG_CTX_CNT(x)       ((x) & 0xFF)
#define SPACC_CFG_RC4_CTX_CNT(x)   (((x) >> 8) & 0xFF)
#define SPACC_CFG_VSPACC_CNT(x)    (((x) >> 16) & 0x0F)
#define SPACC_CFG_CIPH_CTX_SZ(x)   (((x) >> 20) & 0x07)
#define SPACC_CFG_HASH_CTX_SZ(x)   (((x) >> 24) & 0x0F)
#define SPACC_CFG_DMA_TYPE(x)      (((x) >> 28) & 0x03)

#define SPACC_CFG_CMD0_FIFO_QOS(x)   (((x)>>0)&0x7F)
#define SPACC_CFG_CMD0_FIFO(x)   (((x)>>0)&0x1FF)
#define SPACC_CFG_CMD1_FIFO(x)   (((x)>>8)&0x7F)
#define SPACC_CFG_CMD2_FIFO(x)   (((x)>>16)&0x7F)
#define SPACC_CFG_STAT_FIFO_QOS(x)   (((x)>>24)&0x7F)
#define SPACC_CFG_STAT_FIFO(x)       (((x)>>16)&0x1FF)


#define SPACC_PDU_CFG_MINOR(x)   ((x) & 0x0F)
#define SPACC_PDU_CFG_MAJOR(x)   (((x)>>4)  & 0x0F)
#define SPACC_PDU_CFG_RNG(x)     (((x)>>8)  & 0x01)
#define SPACC_PDU_CFG_PKA(x)     (((x)>>9)  & 0x01)
#define SPACC_PDU_CFG_RE(x)      (((x)>>10) & 0x01)
#define SPACC_PDU_CFG_KEP(x)     (((x)>>11) & 0x01)
#define SPACC_PDU_CFG_EA(x)      (((x)>>12) & 0x01)
#define SPACC_PDU_CFG_MPM(x)     (((x)>>13) & 0x01)

#define SPACC_HSM_CFG_MINOR(x)   ((x) & 0x0F)
#define SPACC_HSM_CFG_MAJOR(x)   (((x)>>4)  & 0x0F)
#define SPACC_HSM_CFG_PARADIGM(x)    (((x)>>8)  & 0x01)
#define SPACC_HSM_CFG_KEY_CNT(x)  (((x)>>16)&0xFF)
#define SPACC_HSM_CFG_KEY_SZ(x)   (((x)>>14)&0x03)

#define PDU_SECURE_LOCK_SPACC(x) (x)
#define PDU_SECURE_LOCK_RNG      (1UL<<16)
#define PDU_SECURE_LOCK_PKA      (1UL<<17)
#define PDU_SECURE_LOCK_RE       (1UL<<18)
#define PDU_SECURE_LOCK_KEP      (1UL<<19)
#define PDU_SECURE_LOCK_EA       (1UL<<20)
#define PDU_SECURE_LOCK_MPM      (1UL<<21)
#define PDU_SECURE_LOCK_CFG      (1UL<<30)
#define PDU_SECURE_LOCK_GLBL     (1UL<<31)

#endif

int pdu_get_version(void *dev, pdu_info *inf)
{
	unsigned long tmp;

	if (inf == NULL) {
		return -1;
	}

	memset(inf, 0, sizeof *inf);
	tmp = pdu_io_read32(dev + PDU_REG_SPACC_VERSION);

	if (SPACC_ID_MAJOR(tmp) < 0x05) {
		// we don't support SPAccs before v5.00
		printk("SPAcc MAJOR: %lu not supported\n", SPACC_ID_MAJOR(tmp));
		return -1;
	}

	// ***** Read the SPAcc version block this tells us the revision, project, and a few other feature bits *****
	inf->spacc_version = (spacc_version_block){SPACC_ID_MINOR(tmp),
		SPACC_ID_MAJOR(tmp),
		(SPACC_ID_MAJOR(tmp)<<4)|SPACC_ID_MINOR(tmp),
		SPACC_ID_QOS(tmp),
		(SPACC_ID_TYPE(tmp) == SPACC_TYPE_SPACCQOS) ? 1:0,
		(SPACC_ID_TYPE(tmp) == SPACC_TYPE_PDU) ? 1:0,
		(SPACC_ID_TYPE(tmp) == SPACC_TYPE_HSM) ? 1:0,
		SPACC_ID_AUX(tmp),
		SPACC_ID_VIDX(tmp),
		SPACC_ID_PARTIAL(tmp),
		SPACC_ID_PROJECT(tmp)};

	// try to autodetect
	pdu_io_write32(dev + PDU_REG_SPACC_IV_OFFSET, 0x80000000);
	if (pdu_io_read32(dev + PDU_REG_SPACC_IV_OFFSET) == 0x80000000) {
		inf->spacc_version.ivimport = 1;
	} else {
		inf->spacc_version.ivimport = 0;
	}


	// ***** Read the SPAcc config block which tells us how many contexts there are and context page sizes *****
	tmp = pdu_io_read32(dev + PDU_REG_SPACC_CONFIG);
	inf->spacc_config = (spacc_config_block){SPACC_CFG_CTX_CNT(tmp),
		SPACC_CFG_RC4_CTX_CNT(tmp),
		SPACC_CFG_VSPACC_CNT(tmp),
		SPACC_CFG_CIPH_CTX_SZ(tmp),
		SPACC_CFG_HASH_CTX_SZ(tmp),
		SPACC_CFG_DMA_TYPE(tmp),0,0,0,0};

	// ***** Read the SPAcc config2 block which tells us the FIFO depths of the core *****
	tmp = pdu_io_read32(dev + PDU_REG_SPACC_CONFIG2);
	inf->spacc_config.cmd0_fifo_depth = SPACC_CFG_CMD0_FIFO(tmp);
	inf->spacc_config.stat_fifo_depth = SPACC_CFG_STAT_FIFO(tmp);

	/* only read PDU config if it's actually a PDU engine */
	if (inf->spacc_version.is_pdu) {
		tmp = pdu_io_read32(dev + PDU_REG_PDU_CONFIG);
		inf->pdu_config = (pdu_config_block){SPACC_PDU_CFG_MINOR(tmp),
							SPACC_PDU_CFG_MAJOR(tmp),
							SPACC_PDU_CFG_RNG(tmp),
							SPACC_PDU_CFG_PKA(tmp),
							SPACC_PDU_CFG_RE(tmp),
							SPACC_PDU_CFG_KEP(tmp),
							SPACC_PDU_CFG_EA(tmp),
							SPACC_PDU_CFG_MPM(tmp)};

		pdu_io_write32(dev + PDU_REG_SECURE_LOCK, 0); // unlock all cores by default
	}

	return 0;
}
