/*
 * arch/arm/kernel/snsc_ns_state.c
 *
 * Copyright (C) 2018 Sony corp.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *
 * Show current state of NonSecuure/Secure of ARM TrustZone.
 *
 */


#include <linux/kernel.h>
#include <linux/stddef.h>
#include <linux/init.h>
#include <linux/cpu.h>
#include <linux/proc_fs.h>
#include <linux/hw_breakpoint.h>

typedef struct _ns_state_param {
	int cpu;
	int ns_state;
} ns_state_param_t;

static DEFINE_PER_CPU(ns_state_param_t, ns_state_param);

static int is_ns_state(void)
{
	int status = 0;
	int ns_state;

	// Note:
	// From Reference manual, instruction set is "APSR_nzcv, DBGDSCRint"
	// But build error of "asm volatile ("mrs APSR_nzcv, DBGDSCRint");"
	// on GNU compiler/assembler.
	// Therefore it was implementes vis "mrc p14" register as follows,
	// asm volatile ("mrc p14, 0, %0, c0, c1, 0" : "=r" (status));

	ARM_DBG_READ(c0, c1, 0, status);

	ns_state = (status & (1 << 18)) ? 1 : 0;
	return ns_state;

}

static void get_each_ns_state(void *cpu_id) {
	int cpu;

	if (cpu_id != NULL)
		cpu = *(int *)cpu_id;
	else
		cpu = smp_processor_id();

	per_cpu(ns_state_param, cpu).ns_state = is_ns_state();
}

static ssize_t show_ns_state(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	int this_cpu, ns_state;

	/* Update state for all CPUs */
	this_cpu = get_cpu();                 /* disable preemption */
	get_each_ns_state((void *)&this_cpu); /* For this_cpu */
	smp_call_function(get_each_ns_state, NULL, true);
	put_cpu();                            /* re-enable preemption */

	ns_state = per_cpu(ns_state_param, dev->id).ns_state;

	return snprintf(buf, PAGE_SIZE, "%d\n", ns_state);
}

static DEFINE_PER_CPU(struct device_attribute, ns_state) = { \
	.attr = { \
		.name  = "ns_state",  \
		.mode   = S_IRUGO | S_IWUGO \
	}, \
	.show   = show_ns_state, \
	.store  = NULL \
};

static int __init ns_state_init(void)
{
	int cpu, this_cpu, ns, ret;
	u8 debug_arch = arch_get_debug_arch();

	if (debug_arch < ARM_DEBUG_ARCH_V7_ECP14 ||
		ARM_DEBUG_ARCH_V7_1 < debug_arch) {
		printk(KERN_WARNING "ns_state: not supported arch(%d)\n",
			(int)debug_arch);
		return -ENODEV;
	}

	for_each_possible_cpu(cpu) {
		ret = sysfs_create_file(&(get_cpu_device(cpu)->kobj),
					&per_cpu(ns_state, cpu).attr);
		if (ret) {
			printk(KERN_ERR "ns_state failed on CPU%d\n", cpu);
			return -ENOMEM;
		}
	}

	/* Update state for all CPUs */
	this_cpu = get_cpu();                 /* disable preemption */
	get_each_ns_state((void *)&this_cpu); /* For this_cpu */
	smp_call_function(get_each_ns_state, NULL, true);
	put_cpu();                            /* re-enable preemption */

	for_each_possible_cpu(cpu) {
		ns = per_cpu(ns_state_param, cpu).ns_state;
		printk(KERN_INFO "CPU%d: ns_state=%d\n", cpu, ns);
	}

	return 0;
}

late_initcall(ns_state_init);

