/*
 *	wakeup_64_multiboot.S: Multiboot-compliant trampoline code
 *
 *	Copyright 2013,2015 Sony Corporation
 *
 *	This file is based on arch/x86/kernel/trampoline_64.S
*/
/*
 *
 *	Trampoline.S	Derived from Setup.S by Linus Torvalds
 *
 *	4 Jan 1997 Michael Chastain: changed to gnu as.
 *	15 Sept 2005 Eric Biederman: 64bit PIC support
 *
 *	Entry: CS:IP point to the start of our code, we are
 *	in real mode with no stack, but the rest of the
 *	trampoline page to make our stack and everything else
 *	is a mystery.
 *
 *	On entry to trampoline_data, the processor is in real mode
 *	with 16-bit addressing and 16-bit data.  CS has some value
 *	and IP is zero.  Thus, data addresses need to be absolute
 *	(no relocation) and are taken with regard to r_base.
 *
 *	With the addition of trampoline_level4_pgt this code can
 *	now enter a 64bit kernel that lives at arbitrary 64bit
 *	physical addresses.
 *
 *	If you work on this file, check the object module with objdump
 *	--full-contents --reloc to make sure there are no relocation
 *	entries.
 */

#include <linux/linkage.h>
#include <linux/init.h>
#include <asm/pgtable_types.h>
#include <asm/page_types.h>
#include <asm/msr.h>
#include <asm/segment.h>
#include <asm/processor-flags.h>

	.text
	.balign PAGE_SIZE
	.code32
	.globl wakeup_multiboot
wakeup_multiboot:
r_base = .
	cli			# We should be safe anyway
	wbinvd

	## wakeup routine for grub legacy
	movl    $r_base - __START_KERNEL_map, %esi

	lidtl   (tidt - r_base)(%esi)
	lgdtl   (tgdt - r_base)(%esi)

	movl    $__KERNEL_DS, %eax      # Initialize the %ds segment register
	movl    %eax, %ds

	movl    $X86_CR4_PAE, %eax
	movl    %eax, %cr4              # Enable PAE mode

                                        # Setup trampoline 4 level pagetables
	leal    (trampoline_level4_pgt_dup_grub - r_base)(%esi), %eax
	movl    %eax, %cr3

	movl    $MSR_EFER, %ecx
	movl    $(1 << _EFER_LME), %eax # Enable Long Mode
	xorl    %edx, %edx
	wrmsr

	# Enable paging and in turn activate Long Mode
	# Enable protected mode
	movl    $(X86_CR0_PG | X86_CR0_PE), %eax
	movl    %eax, %cr0

	/*
	 * At this point we're in long mode but in 32bit compatibility mode
	 * with EFER.LME = 1, CS.L = 0, CS.D = 1 (and in turn
	 * EFER.LMA = 1). Now we want to jump in 64bit mode, to do that we use
	 * the new gdt/idt that has __KERNEL_CS with CS.L = 1.
	*/
	ljmp    *(jmp_to_kernel_vector_grub - r_base)(%esi)

	## wakeup routine for efilinux
	.org 0x100
	.code64

	movq    $r_base - __START_KERNEL_map, %rsi
	# add offset
	movq    phys_base(%rip), %rbp
	addq    %rbp, %rsi

	addq    %rbp, trampoline_level4_pgt_dup_efi(%rip)
	addq    %rbp, trampoline_level4_pgt_dup_efi + (511*8)(%rip)
	addq    %rbp, trampoline_level3_pgt_dup_efi(%rip)

	# set physical address
	leaq    (trampoline_level4_pgt_dup_efi - r_base)(%rsi), %rax
	movq    %rax, %cr3

	# jump to virtual address
	movq    $(secondary_startup_64), %rax
	jmp     *%rax

	.code64
	.balign 4
jmp_to_kernel_grub:
        # Now jump into the kernel using virtual addresses
	movq    $secondary_startup_64, %rax
	jmp     *%rax

	.code32
/* temporary tgdt and tidt */
	.balign 4
tidt:
	.word   0                       # idt limit = 0
	.word   0, 0                    # idt base = 0L

	# Duplicate the global descriptor table
	# so the kernel can live anywhere
	.balign 4
tgdt:
	.short  tgdt_end - tgdt         # gdt limit
	.long   tgdt - __START_KERNEL_map
	.short  0
	.quad   0x00cf9b000000ffff      # __KERNEL32_CS
	.quad   0x00af9b000000ffff      # __KERNEL_CS
	.quad   0x00cf93000000ffff      # __KERNEL_DS
tgdt_end:

	.balign 4
jmp_to_kernel_vector_grub:
	.long   jmp_to_kernel_grub - __START_KERNEL_map
	.word   __KERNEL_CS, 0

	.org 0x1000
ENTRY(trampoline_level4_pgt_dup_grub)
	.quad   level3_ident_pgt - __START_KERNEL_map + _KERNPG_TABLE_NOENC
	.fill   510,8,0
	.quad   level3_kernel_pgt - __START_KERNEL_map + _KERNPG_TABLE_NOENC

	.org 0x2000
ENTRY(trampoline_level4_pgt_dup_efi)
	.quad   trampoline_level3_pgt_dup_efi - __START_KERNEL_map + _KERNPG_TABLE_NOENC
	.fill   510,8,0
	.quad   level3_kernel_pgt - __START_KERNEL_map + _KERNPG_TABLE_NOENC

ENTRY(trampoline_level3_pgt_dup_efi)
	.quad   trampoline_level2_pgt_dup_efi - __START_KERNEL_map + _KERNPG_TABLE_NOENC
	.fill   511,8,0

#define PMDS(START, PERM, COUNT) \
	i = 0 ; \
	.rept (COUNT) ; \
	.quad   (START) + (i << PMD_SHIFT) + (PERM) ; \
	i = i + 1; \
	.endr

ENTRY(trampoline_level2_pgt_dup_efi)
	PMDS(0, __PAGE_KERNEL_IDENT_LARGE_EXEC, PTRS_PER_PMD)

#undef PMDS
