// SPDX-License-Identifier: GPL-2.0
/*
 *  ssbootfs: inode handling
 *
 *  Copyright 2021 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 */

#include <linux/namei.h>
#include <linux/version.h>
#include <linux/list.h>
#include <linux/dcache.h>

#include "ssbootfs.h"

static int ssbootfs_inode_readlink(struct dentry *dentry, char __user *buf, int len)
{
	struct ssbootfs_priv *priv = ssbootfs_priv_from_dentry(dentry);
	int ret;
	struct dentry *mapped;

	ssbootfs_barrier(priv);

	SSBOOTFS_CHECK_ATTACHED(priv) {
		return -EIO;
	}

	ret = ssbootfs_mapping_map_dentry(dentry, &mapped);
	if (ret) {
		ssbootfs_debug("failed to lookup %s during readlink: %d\n", d_name(dentry), ret);
		return ret;
	}

	ret = vfs_readlink(mapped, buf, len);

	dput(mapped);

	return ret;
}

static struct inode *ssbootfs_inode_wire_up_inode(struct ssbootfs_priv *priv, struct dentry *fake_dentry,
		struct dentry *real_dentry, struct inode *real_inode)
{
	struct inode *i;

	/* Create a new shadow inode with the same properties as the real inode */
	i = ssbootfs_inode_new_fake_inode(priv->sb,
					  real_inode->i_ino,
					  real_inode->i_generation,
					  real_inode->i_mode);

	if (IS_ERR(i)) {
		goto out;
	}

	/* Copy the real inodes attributes into our inode */
	SSBOOTFS_SYNC_INODE(i, real_inode);
out:
	return i;
}

static int ssbootfs_inode_do_create(struct inode *dir, struct dentry *dentry, umode_t mode,
		bool want_excl, dev_t dev)
{
	struct ssbootfs_priv *priv = ssbootfs_priv_from_dentry(dentry);
	struct dentry *parent, *mappedparent, *realdentry;
	struct inode *inode;
	struct ssbootfs_creds creds;
	int ret = -EINVAL;

	ssbootfs_barrier(priv);

	SSBOOTFS_CHECK_ATTACHED(priv) {
		return -EIO;
	}

	/* first try to work out where the parent directory is */
	parent = dget_parent(dentry);
	if (!parent)
		goto out;

	ret = ssbootfs_switch_to_root(&creds);
	if (ret)
		goto out_parent;

	ret = ssbootfs_mapping_map_dentry(parent, &mappedparent);

	ssbootfs_switch_from_root(&creds);

	if (ret) {
		ssbootfs_debug("failed to lookup %s during create: %d\n", d_name(dentry), ret);
		goto out_parent;
	}

	lock_dentry(mappedparent);

	/*
	 * This looks like we're really trying to look up any existing dentry
	 * but we aren't. This essentially creates a negative dentry for
	 * the new file to jump into.
	 */
	realdentry = ssbootfs_map_target_dentry(mappedparent, dentry);
	if (IS_ERR(realdentry)) {
			ret = PTR_ERR(realdentry);
			goto out_locked;
	}

	 /* Now ask the backing fs to actually create the inode */
	ret = ssbootfs_backingfs_create(d_inode(mappedparent), realdentry, mode, want_excl, dev);
	if (ret)
		goto out_realdentry;

	inode = ssbootfs_inode_wire_up_inode(priv, dentry, realdentry, d_inode(realdentry));
	if (IS_ERR(inode)) {
		ret = PTR_ERR(inode);
		goto err_real_created;
	}

	/* Bind the inode to the dentry */
	inode->i_state = I_NEW;
	d_instantiate_new(dentry, inode);

#ifndef CONFIG_SNSC_SSBOOT_FS_DEBUG_DISABLE_DENTRY_LINKAGE
	ssbootfs_mapping_real_dentry_set(dentry, realdentry);
#endif

	goto out_success;

err_real_created:
	vfs_unlink(d_inode(mappedparent), realdentry, NULL);
out_success:
out_realdentry:
	dput(realdentry);
out_locked:
	unlock_dentry(mappedparent);
	dput(mappedparent);
out_parent:
	dput(parent);
out:
	return ret;
}

static int ssbootfs_inode_create(struct inode *dir, struct dentry *dentry, umode_t mode,
		bool want_excl)
{
	return ssbootfs_inode_do_create(dir, dentry, mode, want_excl, 0);
};

static struct dentry *ssbootfs_inode_lookup(struct inode *dir,
		struct dentry *dentry, unsigned int flags)
{
	int ret;
	struct ssbootfs_priv *priv = ssbootfs_priv_from_dentry(dentry);
	struct dentry *mapped, *res = NULL;
	struct inode *inode = NULL, *mapped_inode, *existing_inode;

	ssbootfs_barrier(priv);

	SSBOOTFS_CHECK_ATTACHED(priv) {
		return NULL;
	}

	/* Look up the real dentry */
	ret = ssbootfs_mapping_map_dentry(dentry, &mapped);
	if (ret)
		return NULL;

	mapped_inode = d_inode(mapped);

	/*
	 * For hardlinks we will have multiple dentry's for a
	 * single inode. Check if we have a shadow inode for the
	 * mapped real inode already.
	 */
	spin_lock(&priv->sb->s_inode_list_lock);
	list_for_each_entry(existing_inode, &priv->sb->s_inodes, i_sb_list) {
		if (ssbootfs_inode_issame(existing_inode, mapped_inode)) {
			inode = igrab(existing_inode);
			break;
		}
	}
	spin_unlock(&priv->sb->s_inode_list_lock);
	if (inode)
		goto out_splice;

	inode = ssbootfs_inode_wire_up_inode(priv, dentry, mapped, mapped_inode);
	if (IS_ERR(inode)) {
		goto out_mapped;
	}

out_splice:
	res = d_splice_alias(inode, dentry);
	ssbootfs_mapping_real_dentry_set(dentry, mapped);
out_mapped:
	dput(mapped);
	return res;
}


static int ssbootfs_inode_rename(struct inode *old_dir, struct dentry *old_dentry,
		struct inode *new_dir, struct dentry *new_dentry, unsigned int flags)
{
	struct ssbootfs_priv *priv = ssbootfs_priv_from_dentry(old_dentry);
	struct dentry *old_parent, *new_parent, *mapped_old_parent, *mapped_new_parent,
		      *mapped_old_dentry, *mapped_new_dentry;
	int ret;

	ssbootfs_barrier(priv);

	SSBOOTFS_CHECK_ATTACHED(priv){
		return -EIO;
	}

	old_parent = dget_parent(old_dentry);
	BUG_ON(!old_parent);

	new_parent = dget_parent(new_dentry);
	BUG_ON(!new_parent);

	ret = ssbootfs_mapping_map_dentry(old_parent, &mapped_old_parent);
	if (ret) {
		ssbootfs_err("failed to map old parent directory (%s) during rename: %d\n",
				d_name(old_parent), ret);
		goto out;
	}

	ret = ssbootfs_mapping_map_dentry(new_parent, &mapped_new_parent);
	if (ret) {
		ssbootfs_err("failed to map new parent directory (%s) during rename: %d\n",
				d_name(new_parent), ret);
		goto out_mapped_old_parent;
	}

	lock_rename(mapped_old_parent, mapped_new_parent);

	ret = ssbootfs_mapping_map_dentry(old_dentry, &mapped_old_dentry);
	if (ret) {
		ssbootfs_err("failed to map old dentry (%s) during rename: %d\n",
				d_name(old_dentry), ret);
		goto out_locked;
	}

	mapped_new_dentry = ssbootfs_map_target_dentry(mapped_new_parent, new_dentry);
	if (IS_ERR(mapped_new_dentry)) {
			ret = PTR_ERR(mapped_new_dentry);
			ssbootfs_err("failed to map new dentry (%s) during rename: %d\n",
					d_name(new_dentry), ret);
			goto out_mapped_old_dentry;
	}

	/*
	 * The real tree is going to change now.
	 * Block mapping until we are done.
	 */
	down_write(&priv->tree_rw_sem);

	ssbootfs_mapping_invalidate_inode(d_inode(old_dentry));

	ret = vfs_rename(d_inode(mapped_old_parent), mapped_old_dentry, d_inode(mapped_new_parent),
			mapped_new_dentry, NULL, flags);
	if (ret) {
		ssbootfs_debug("failed to do rename (%s -> %s) on backing fs during rename: %d\n",
				d_name(old_dentry), d_name(new_dentry), ret);
	}

	/*
	 * Invalidate any cache depths as the filesystem tree might have changed
	 */
	atomic_inc(&priv->token);

	up_write(&priv->tree_rw_sem);

	unlock_rename(mapped_old_parent, mapped_new_parent);

	dput(mapped_new_dentry);
out_mapped_old_dentry:
	dput(mapped_old_dentry);
out_locked:
	dput(mapped_new_parent);
out_mapped_old_parent:
	dput(mapped_old_parent);
out:
	dput(old_parent);
	dput(new_parent);
	return ret;
}

static int ssbootfs_inode_getattr(const struct path *path, struct kstat *stat,
		u32 request_mask, unsigned int query_flags)
{
	int ret;
	struct ssbootfs_priv *priv = ssbootfs_priv_from_dentry(path->dentry);
	struct dentry *mapped;
	struct path realpath;
	struct ssbootfs_creds creds;

	ssbootfs_barrier(priv);

	SSBOOTFS_CHECK_ATTACHED(priv){
		return -EIO;
	}

	ret = ssbootfs_mapping_map_dentry(path->dentry, &mapped);
	if (ret == -EACCES) {
		ret = ssbootfs_switch_to_root(&creds);
		if (ret)
			goto out;

		ret = ssbootfs_mapping_map_dentry(path->dentry, &mapped);

		ssbootfs_switch_from_root(&creds);

		if (ret)
			goto out;
	}
	else if (ret)
		goto out;

	realpath.mnt = priv->backingfs.mnt;
	realpath.dentry = mapped;

	/* go and ask the real fs */
	ret = vfs_getattr(&realpath, stat, request_mask, query_flags);

	dput(mapped);

out:
	return ret;
}

static int ssbootfs_inode_setattr(struct dentry *dentry, struct iattr *attr)
{
	struct ssbootfs_priv *priv = ssbootfs_priv_from_dentry(dentry);
	int ret;
	struct dentry *mapped;
	struct inode *realinode;
	struct ssbootfs_creds creds;

	ssbootfs_barrier(priv);

	SSBOOTFS_CHECK_ATTACHED(priv) {
		return -EIO;
	}

	ret = ssbootfs_mapping_map_dentry(dentry, &mapped);
	if (ret == -EACCES) {
		ret = ssbootfs_switch_to_root(&creds);
		if (ret)
			goto out;

		ret = ssbootfs_mapping_map_dentry(dentry, &mapped);

		ssbootfs_switch_from_root(&creds);

		if (ret)
			goto out;
	}
	else if (ret)
		goto out;

	realinode = d_inode(mapped);
	BUG_ON(!realinode);
	inode_lock(realinode);

	/* there is no vfs_setattr */
	if (realinode->i_op->setattr) {
		ret = realinode->i_op->setattr(mapped, attr);
	} else {
		ret = simple_setattr(mapped, attr);
	}

	/*
	 * Now apply the change on our side.
	 * We do this via simple_setattr instead of just
	 * copying the inode fields because there may
	 * need to be some adjustment to mmaps.
	 */
	simple_setattr(dentry, attr);

	inode_unlock(realinode);

	dput(mapped);
out:
	return ret;
}

static int ssbootfs_inode_link(struct dentry *from, struct inode *unused, struct dentry *to)
{
	struct ssbootfs_priv *priv = ssbootfs_priv_from_dentry(from);
	struct dentry *mapped_from, *new_parent, *mapped_new_parent, *mapped_to;
	struct inode *real, *proxy;
	int ret;

	ssbootfs_barrier(priv);

	SSBOOTFS_CHECK_ATTACHED(priv) {
		return -EIO;
	}

	ret = ssbootfs_mapping_map_dentry(from, &mapped_from);
	if (ret) {
		ssbootfs_err("failed to map from (%s) during link: %d\n", d_name(from), ret);
		goto out;
	}

	new_parent = dget_parent(to);
	if (!new_parent)
		goto out_mapped;

	ret = ssbootfs_mapping_map_dentry(new_parent, &mapped_new_parent);
	if (ret) {
		ssbootfs_err("failed to map new_parent (%s) during link: %d\n", d_name(new_parent), ret);
		goto out_new_parent;
	}

	lock_dentry(mapped_new_parent);
	mapped_to = ssbootfs_map_target_dentry(mapped_new_parent, to);
	/* we probably need to release the lock here for vfs_link */
	unlock_dentry(mapped_new_parent);
	if (!mapped_to)
		goto out_mapped_new_parent;

	ret = vfs_link(mapped_from, d_inode(mapped_new_parent), mapped_to, NULL);

	/* if creating the link worked we now need to create the proxy version */
	if (!ret) {
		real = d_inode(mapped_to);
		/* we must have a real inode here */
		BUG_ON(!real);

		/*
		 * This is a sanity check, we are creating the inode for this dentry,
		 * we shouldn't have a inode associated with it yet.
		 */
		BUG_ON(d_inode(to));

		proxy = ssbootfs_inode_wire_up_inode(priv, to, mapped_to, real);

		/* creating the proxy inode cannot fail. */
		BUG_ON(!proxy);

		/*
		 * Link up the proxy inode to the dentry and stash
		 * the real dentry in our cache
		 */
		d_instantiate(to, proxy);
		ssbootfs_mapping_real_dentry_set(to, mapped_to);
	}

	dput(mapped_to);
out_mapped_new_parent:
	dput(mapped_new_parent);
out_new_parent:
	dput(new_parent);
out_mapped:
	dput(mapped_from);
out:
	return ret;
}

static int ssbootfs_inode_unlink(struct inode *dir, struct dentry *dentry)
{
	struct ssbootfs_priv *priv = ssbootfs_priv_from_dentry(dentry);
	struct dentry *parent, *mapped_parent, *mapped_dentry, *open_parent;
	struct inode *inode;
	struct ssbootfs_inode *ssbfs_inode;
	struct ssbootfs_file_priv *file_priv, *tmp;
	int ret, opencount = 0;

	ssbootfs_barrier(priv);

	SSBOOTFS_CHECK_ATTACHED(priv) {
		return -EIO;
	}

	parent = dget_parent(dentry);

	/*
	 * Warn about processes that open a file and then unlink it with
	 * a reference held to implement "delete at exit" for temp files and
	 * such.
	 *
	 * The reason we need to handle this is that the first incarnation of
	 * a process could open and unlink a file and get saved in the snapshot,
	 * when the snapshot boots the process will revert to thinking it has
	 * an open file.
	 *
	 * When disengaging the real filesystem the reference count on the real
	 * files will drop to zero and the real filesystem will destroy the inode.
	 *
	 * If this becomes a big issue we will need implement a scheme for linking
	 * files like this somewhere else so the real filesystem doesn't destroy
	 * them.
	 */
	inode = d_inode(dentry);
	if (inode) {
		ssbfs_inode = SSBOOTFS_I_TO_SSBFSI(inode);
		if (!list_empty(&ssbfs_inode->files)) {
			list_for_each_entry_safe(file_priv, tmp, &ssbfs_inode->files, i_files) {
				open_parent = dget_parent(file_priv->fake_file->f_path.dentry);
				/* this should never happen! */
				if (!open_parent)
					continue;
				/*
				 * The parent d_entry pointer and the name should be enough to check that
				 * file being unlinked is the same as the item in our tracking this.
				 *
				 * Note that the dentry passed in here is not the same as the one
				 * pointed to in the file so we compare the name again here.
				 *
				 * To be honest this only really matter for hardlinks. The vast majority
				 * of unlinks while open will be for temp files that aren't hardlinked.
				 */
				if (open_parent == parent && (strcmp(d_name(dentry), f_name(file_priv->fake_file)) == 0)) {
					opencount++;
				}
				dput(open_parent);
			}
			ssbootfs_warn("unlinking %s/%s when inode %lu has open files, %d references\n",
					d_name(parent), d_name(dentry), inode->i_ino, opencount);
/*
 * You may want to ignore the unlink entirely to avoid issues with apps
 * coming back after snapshot and complaining their files are gone.
 *
 * This will leak the file as it will never get deleted.
 */
#if 0
			ret = 0;
			goto out_parent;
#endif
		}
	}

	ret = ssbootfs_mapping_map_dentry(parent, &mapped_parent);
	if (ret) {
		ssbootfs_err("failed to map parent(%s) during unlink: %d\n", d_name(parent), ret);
		goto out_parent;
	}

	ret = ssbootfs_mapping_map_dentry(dentry, &mapped_dentry);
	if (ret) {
		ssbootfs_err("failed to map dentry(%s) during unlink: %d\n", d_name(dentry), ret);
		goto out_mapped_parent;
	}

	ret = vfs_unlink(d_inode(mapped_parent), mapped_dentry, NULL);

	dput(mapped_dentry);
out_mapped_parent:
	dput(mapped_parent);
out_parent:
	dput(parent);
	return ret;
}

static int ssbootfs_inode_symlink(struct inode *dir, struct dentry *dentry, const char *content)
{
	struct ssbootfs_priv *priv = ssbootfs_priv_from_dentry(dentry);
	struct dentry *parent, *mapped_parent, *mapped_dentry;
	struct inode *proxy, *real;
	int ret;

	ssbootfs_barrier(priv);

	SSBOOTFS_CHECK_ATTACHED(priv) {
		return -EIO;
	}

	parent = dget_parent(dentry);

	ret = ssbootfs_mapping_map_dentry(parent, &mapped_parent);
	if (ret) {
		ssbootfs_err("failed to map parent (%s) during symlink: %d\n", d_name(parent), ret);
		goto out_parent;
	}

	lock_dentry(mapped_parent);

	mapped_dentry = ssbootfs_map_target_dentry(mapped_parent, dentry);
	if (IS_ERR(mapped_dentry)) {
		ret = PTR_ERR(mapped_dentry);
		goto out_parent_locked;
	}

	ret = vfs_symlink(d_inode(mapped_parent), mapped_dentry, content);

	if (!ret) {
		real = d_inode(mapped_dentry);
		/* we must have a real inode */
		BUG_ON(!real);
		/* there must not be an inode associated with the proxy dentry already */
		BUG_ON(d_inode(dentry));
		proxy = ssbootfs_inode_wire_up_inode(priv, dentry, mapped_dentry, real);
		/* failing to create the proxy inode is not an option */
		BUG_ON(!proxy);
		d_instantiate(dentry, proxy);
		ssbootfs_mapping_real_dentry_set(dentry, mapped_dentry);
	}

	dput(mapped_dentry);
out_parent_locked:
	unlock_dentry(mapped_parent);
	dput(mapped_parent);
out_parent:
	dput(parent);
	return ret;
}

static int ssbootfs_inode_mkdir(struct inode *dir, struct dentry *dentry, umode_t mode)
{
	return ssbootfs_inode_do_create(dir, dentry, mode | S_IFDIR, false, 0);
}

static int ssbootfs_inode_rmdir(struct inode *dir, struct dentry *dentry)
{
	struct ssbootfs_priv *priv = ssbootfs_priv_from_dentry(dentry);
	struct dentry *mapped, *mapped_parent;
	int ret;

	ssbootfs_barrier(priv);

	SSBOOTFS_CHECK_ATTACHED(priv) {
		return -EIO;
	}

	ret = ssbootfs_mapping_map_dentry(dentry, &mapped);
	if (ret)
		goto out;

	mapped_parent = dget_parent(mapped);
	if (!mapped_parent) {
		ret = -EINVAL;
		goto out_mapped;
	}

	ret = vfs_rmdir(d_inode(mapped_parent), mapped);

	dput(mapped_parent);
out_mapped:
	dput(mapped);
out:
	return ret;
}

static int ssbootfs_inode_mknod(struct inode *dir, struct dentry *dentry, umode_t mode, dev_t dev)
{
	return ssbootfs_inode_do_create(dir, dentry, mode, false, dev);
}

static const char *ssbootfs_inode_get_link(struct dentry *dentry, struct inode *inode,
		struct delayed_call *done)
{
	struct ssbootfs_priv *priv;
	struct dentry *mapped;
	const char *link = ERR_PTR(-EINVAL);
	int err;

	/*
	 * See Documentation/filesystems/vfs.txt
	 * A null dentry means we are being called in RCU.
	 * We can't do that so return -ECHILD to tell the caller
	 * to try again without RCU.
	 */
	if (!dentry)
		return ERR_PTR(-ECHILD);

	priv = ssbootfs_priv_from_dentry(dentry);

	ssbootfs_barrier(priv);

	SSBOOTFS_CHECK_ATTACHED(priv) {
		return ERR_PTR(-EIO);
	}

	err = ssbootfs_mapping_map_dentry(dentry, &mapped);
	if (err) {
		ssbootfs_debug("failed to map from %s during get_link\n", d_name(dentry));
		link = ERR_PTR(err);
		goto out;
	}

	/*
	 * It's possible we need to do something with the delayed call
	 * if there is one here.
	 */
	link = vfs_get_link(mapped, done);

	dput(mapped);
out:
	return link;
}

static const struct inode_operations ssbootfs_iops = {
	.readlink	= ssbootfs_inode_readlink,
	.lookup		= ssbootfs_inode_lookup,
	.create		= ssbootfs_inode_create,
	.mkdir		= ssbootfs_inode_mkdir,
	.rmdir		= ssbootfs_inode_rmdir,
	.link		= ssbootfs_inode_link,
	.unlink		= ssbootfs_inode_unlink,
	.symlink	= ssbootfs_inode_symlink,
	.rename		= ssbootfs_inode_rename,
	.getattr	= ssbootfs_inode_getattr,
	.setattr	= ssbootfs_inode_setattr,
	.mknod		= ssbootfs_inode_mknod,
};

static const struct inode_operations ssbootfs_symlink_iops = {
	.get_link	= ssbootfs_inode_get_link,
};

struct inode *ssbootfs_inode_new_fake_inode(struct super_block *sb, u64 ino, u32 generation, umode_t mode)
{
	struct inode *inode;
	struct ssbootfs_inode *si;
	unsigned int type = mode & S_IFMT;
	int ret = 0;

	inode = new_inode(sb);

	if (!inode)
		return ERR_PTR(-ENOMEM);

	si = SSBOOTFS_I_TO_SSBFSI(inode);

	inode->i_ino = ino;
	inode->i_generation = generation;
	inode->i_mode = mode;

	switch (type) {
	case S_IFDIR:
		inode->i_op = &ssbootfs_iops;
		inode->i_fop = &ssbootfs_dir_fops;
#ifndef CONFIG_SNSC_SSBOOT_FS_DEBUG_DISABLE_DENTRY_LINKAGE
		si->lookup_cache.real_dentry = NULL;
#endif
		ssbootfs_mapping_invalidate_inode(inode);
		break;
	case S_IFLNK:
		inode->i_op = &ssbootfs_symlink_iops;
		inode->i_fop = &ssbootfs_symlink_fops;
		break;
	case S_IFREG:
	case S_IFCHR:
	case S_IFBLK:
	case S_IFIFO:
	case S_IFSOCK:
		inode->i_op = &ssbootfs_iops;
		inode->i_fop = &ssbootfs_file_fops;
		inode->i_mapping->a_ops = &ssbootfs_file_aops;
		break;
	default:
		ssbootfs_err("unhandled inode type %x\n", type);
		ret = -EINVAL;
		goto err;
	}

	return inode;

err:
	iput(inode);

	return ERR_PTR(ret);
}

unsigned long ssbootfs_inode_disengage(struct ssbootfs_priv *priv, struct inode *inode)
{
	struct ssbootfs_file_priv *file_priv;
	struct ssbootfs_inode *ssbootfs_inode = SSBOOTFS_I_TO_SSBFSI(inode);
	struct file *file;
	/*
	 * Max possible files is probably the same as the maximum amount inodes
	 */
	unsigned long file_count = 0;

	ssbootfs_inode_lock(inode);

	/*
	 * We check if the list is empty here as there will be lots
	 * of inodes that are still allocated for the superblock
	 * but are on their way out and we don't care too much
	 * about.
	 */
	if (list_empty(&ssbootfs_inode->files))
		goto unlock;

	ssbootfs_debug_info("disengaging inode %lu\n", inode->i_ino);
	list_for_each_entry(file_priv, &ssbootfs_inode->files, i_files) {
		file = file_priv->fake_file;
		file_count += ssbootfs_file_disengage(priv, file);
	}

unlock:
	ssbootfs_mapping_invalidate_inode(inode);
	ssbootfs_inode_unlock(inode);

	return file_count;
}
