// SPDX-License-Identifier: GPL-2.0
/*
 *  ssbootfs: utility functions used by suspend/resume and sysfs interface
 *
 *  Copyright 2021 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 */

#include <linux/magic.h>
#include <linux/file.h>

#include "../mount.h"
#include "ssbootfs.h"

#ifdef CONFIG_SNSC_SSBOOT_FS_IN_KERNEL_MOUNT
int ssbootfs_util_mount(struct ssbootfs_priv *priv)
{
	int ret = 0;
	struct path mntpnt;

	ssbootfs_priv_lock(priv);

	/*
	 * dentry transit might call into here multiple times so this needs
	 * to be safe to enter multiple times.
	 */
	if (priv->backingdev && !priv->attached) {
		ssbootfs_warn("trying to automount %s...\n", priv->backingdev);
		ret = ssbootfs_backingfs_mntpath(priv, &mntpnt);
		if (ret) {
			ssbootfs_err("Failed to look up mount point path: %d\n", ret);
			goto out;
		}
		ret = do_new_mount(&priv->backingfs, priv->backingtype, 0, 0, priv->backingdev, priv->backingopts);
		if (ret) {
			ssbootfs_err("Failed to mount: %d\n", ret);
			goto out;
		}
		ret = ssbootfs_attach_unlocked(priv);
		if (ret) {
			ssbootfs_err("Failed to attach: %d\n", ret);
			/*
			 * TODO on failure the fs should be unmounted again!!
			 * But this causes a deadlock right now so there isn't much we
			 * can do here.
			 *
			 * Use deferred work?
			 */
			goto out;
		}

		priv->in_suspend = false;
	}

out:
	ssbootfs_priv_unlock(priv);
	return ret;
}

static void ssbootfs_util_disengage_all_sb(struct super_block *sb, void *arg)
{
	struct ssbootfs_priv *priv;
	int ret = *((int *)arg);

	/*
	 * A previous superblock might have thrown an
	 * error. We need to skip any superblocks after that
	 * point.
	 */
	if (ret)
		return;

	if (sb->s_magic == SSBOOTFS_MAGIC) {
		priv = sb->s_fs_info;

		ret = ssbootfs_disengage_sb(priv);
		if (ret)
			goto out;
	}

out:
	*((int *)arg) = ret;
}

int ssbootfs_util_disengage_all(void)
{
	int ret = 0;

	iterate_supers(ssbootfs_util_disengage_all_sb, &ret);

	return ret;
}

static void ssbootfs_util_reattach_all_sb(struct super_block *sb, void *arg)
{
	struct ssbootfs_priv *priv;
	int ret = *((int *)arg);

	/*
	 * A previous superblock might have thrown an
	 * error. We need to skip any superblocks after that
	 * point.
	 */
	if (ret)
		return;

	if (sb->s_magic == SSBOOTFS_MAGIC) {
		priv = sb->s_fs_info;
		ret = ssbootfs_util_mount(priv);
	}

	*((int *)arg) = ret;
}

int ssbootfs_util_reattach_all(void)
{
	int ret = 0;

	iterate_supers(ssbootfs_util_reattach_all_sb, &ret);

	return ret;
}

static void ssbootfs_util_detach_all_sb(struct super_block *sb, void *arg)
{
	struct ssbootfs_priv *priv;
	int ret = *((int *)arg);

	/*
	 * A previous superblock might have thrown an
	 * error. We need to skip any superblocks after that
	 * point.
	 */
	if (ret)
		return;

	if (sb->s_magic == SSBOOTFS_MAGIC) {
		priv = sb->s_fs_info;
		ret = ssbootfs_detach(priv, true);
	}

	*((int *)arg) = ret;
}

int ssbootfs_util_detach_all(void)
{
	int ret = 0;

	iterate_supers(ssbootfs_util_detach_all_sb, &ret);
	if (ret)
		iterate_supers(ssbootfs_util_reattach_all_sb, &ret);

	return ret;
}
#endif
