/*
 *  Open firmware pmmem feature
 *
 *  Copyright 2020 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 */

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/of_address.h>
#include <linux/of_platform.h>
#include <linux/memblock.h>
#include <linux/slab.h>

#include <linux/ssboot.h>
#include "ssboot/internal.h"

#define COMPATIBLE_SSBOOT_DISCARD       "sony,ssboot-discard"
#define RESERVED_MEMORY_NODE            "/reserved-memory"

#define PMMEM_DISCARDED                 (1 << 0)
#define PMMEM_RESERVED                  (1 << 1)

static struct device_node *reserved_memory_node;

static bool inline is_reserved_pmmem(struct platform_device *pdev)
{
	return pdev->dev.of_node->parent == reserved_memory_node;
}

static bool is_ssboot_image_create_done(void)
{
	return ssboot_is_resumed() && ssboot.resmode == SSBOOT_RESMODE_NORMAL;
}

static int pmmem_of_probe(struct platform_device *pdev)
{
	int i;
	int ret;
	struct resource *r;
	u32 *state;

	if (!is_reserved_pmmem(pdev))
		return -EINVAL;

	state = kzalloc(sizeof(*state), GFP_KERNEL);
	if (IS_ERR_OR_NULL(state)) {
		dev_err(&pdev->dev, "Failed to allocate state\n");
		return IS_ERR(state);
	}
	platform_set_drvdata(pdev, state);

	/* Already reserved by of_reserved_mem.c */
	*state |= PMMEM_RESERVED;

	for (i = 0; i < pdev->num_resources; i++) {
		r = &pdev->resource[i];

		if (IORESOURCE_MEM != resource_type(r))
			continue;

		dev_info(&pdev->dev, "Register%sPM memory: %px - %px\n",
			 is_reserved_pmmem(pdev) ? " reserved " : " ",
			 (void *)r->start, (void *)r->end);
		ret = ssboot_region_register_discard(r->start,
						     resource_size(r));
		if (ret < 0) {
			dev_err(&pdev->dev, "Failed to register PM memory\n");
			goto out;
		}
	}

	/* Mark all regions in this node had been discarded */
	*state |= PMMEM_DISCARDED;

	return 0;

out:
	for (i--; i > 0; i--) {
		r = &pdev->resource[i];

		if (IORESOURCE_MEM != resource_type(r))
			continue;

		if (ssboot_region_unregister(r->start, resource_size(r)) < 0)
			panic("Failed to unregister PM memory.\n");
	}

	return ret;
}

static int pmmem_of_remove(struct platform_device *pdev)
{
	int i;
	struct resource *r;
	u32 *state = platform_get_drvdata(pdev);
	resource_size_t size;

	/* Already removed */
	if (state == NULL)
		return 0;

	for (i = 0; i < pdev->num_resources; i++) {
		r = &pdev->resource[i];

		if (IORESOURCE_MEM != resource_type(r))
			continue;

		dev_info(&pdev->dev, "Unregister%sPM memory: %px - %px\n",
			 is_reserved_pmmem(pdev) ? " reserved " : " ",
			 (void *)r->start, (void *)r->end);

		size = resource_size(r);

		if (*state & PMMEM_DISCARDED) {
			if (ssboot_region_unregister(r->start, size) < 0)
				panic("Failed to unregister PM memory.\n");
		}

		if ((*state & PMMEM_RESERVED) && is_ssboot_image_create_done())
			memblock_free(r->start, size);
	}

	kfree(state);
	platform_set_drvdata(pdev, NULL);

	return 0;
}

static int pmmem_of_resume(struct platform_device *pdev)
{
	if (!is_ssboot_image_create_done())
		return 0;

	return pmmem_of_remove(pdev);
}

static const struct of_device_id pmmem_of_table[] = {
	{ .compatible = COMPATIBLE_SSBOOT_DISCARD, },
	{ }
};

static struct platform_driver pmmem_of_driver = {
	.driver		= {
		.name	= "PMMEM-OF",
		.of_match_table = pmmem_of_table,
	},
	.probe		= pmmem_of_probe,
	.remove		= pmmem_of_remove,
	.resume		= pmmem_of_resume,
};

static int __init pmmem_of_reserved_device_register(void)
{
	int ret;
	struct device_node *node;
	struct device_node *tmp;
	struct platform_device *pdev;

	for_each_available_child_of_node(reserved_memory_node, node) {
		if (!of_device_is_compatible(node,
					     COMPATIBLE_SSBOOT_DISCARD))
			continue;

		pdev = of_platform_device_create(node, NULL, NULL);
		if (IS_ERR_OR_NULL(pdev)) {
			ret = PTR_ERR(pdev);
			pr_err("of_platform_device_create() failed: %d\n", ret);
			goto device_destroy;
		}
	}

	return 0;

device_destroy:
	for_each_available_child_of_node(reserved_memory_node, tmp) {
		if (!of_device_is_compatible(tmp, COMPATIBLE_SSBOOT_DISCARD))
			continue;

		if (tmp == node)
			break;

		pdev = of_find_device_by_node(node);
		if (pdev)
			of_platform_device_destroy(&pdev->dev, NULL);
	}

	return ret;
}

static void __exit pmmem_of_reserved_device_unregister(void)
{
	struct device_node *node;
	struct platform_device *pdev;

	for_each_available_child_of_node(reserved_memory_node, node) {
		if (!of_device_is_compatible(node, COMPATIBLE_SSBOOT_DISCARD))
			continue;

		pdev = of_find_device_by_node(node);
		if (pdev)
			of_platform_device_destroy(&pdev->dev, NULL);
	}
}


/*
 * This is implemented with a platform driver to cleanly unregister the
 * reserved region when the kernel is resumes from the snapshot.
 */
static int __init pmmem_of_init(void)
{
	int ret;

	reserved_memory_node = of_find_node_by_path(RESERVED_MEMORY_NODE);
	if (IS_ERR(reserved_memory_node)) {
		pr_info("PMMEM-OF: Can not open /reserved-memory node\n");
		return PTR_ERR(reserved_memory_node);
	} else if (reserved_memory_node == NULL) {
		pr_info("PMMEM-OF: There is not /reserved-memory node\n");
		return -ENODEV;
	}

	ret = platform_driver_register(&pmmem_of_driver);
	if (ret < 0) {
		pr_err("Failed to register PM memory driver\n");
		return ret;
	}

	ret = pmmem_of_reserved_device_register();
	if (ret < 0) {
		pr_err("Failed to register reserved PM memory device\n");
		goto out;
	}

	return 0;

out:
	platform_driver_unregister(&pmmem_of_driver);

	return ret;
}

static void __exit pmmem_of_exit(void)
{
	pmmem_of_reserved_device_unregister();
	platform_driver_unregister(&pmmem_of_driver);
}

module_init(pmmem_of_init);
module_exit(pmmem_of_exit);

MODULE_LICENSE("GPL");
