#!/bin/bash
#
#  Script to make kernel header
#
#  Copyright 2006-2009 Sony Corporation.
#
set -e # stop on error

DEST=kernel_headers.tar.gz
HEADER=kernel_header.tar
KBUILD=kbuild_tree.tar

srctree=${srctree:=.}
objtree=${objtree:=.}

SCR_PATH=${srctree}/scripts
FIXUP=${objtree}/.fixup_kernel_headers
SYMVERS=${objtree}/Module.symvers
VERFILE=${objtree}/include/generated/uapi/linux/version.h
OBJTOOL_FILES="objtool fixdep sync-check.sh"
WORK=${objtree}/.work

#
# Files and directories required other than KConfig/Makefiles.
#
COPY_DIRS="scripts/"

# COPY_FILES should be synced with install_kernel_headers.sh.
COPY_FILES=".config .arch_name .cross_compile .target_name"

fatal () {
    echo "FATAL: $*" > /dev/tty
    rm -rf ${WORK}
    exit 1
}

usage () {
    echo "usage: $0 [ output_file_name.tar.gz ]" > /dev/tty
    exit 0
}

if [ "$#" -ge 2 ]; then
    usage
fi
if [ $# == 1 ]; then
    DEST=$1
fi

test -f ${VERFILE} || fatal "kernel is not configured"

for f in ${COPY_FILES}; do
    test -f ${objtree}/${f} || fatal "${f} is not found"
done
for d in ${COPY_DIRS} ; do
    test -d ${srctree}/${d} || fatal "${d} is not found"
done

ARCH=`${SCR_PATH}/install_headers_util.sh -a`
echo "Creating kernel headers (${DEST}) for ${ARCH}..."

#
# make kernel headers
#
rm -rf ${WORK}
mkdir -p ${WORK}/include

# copy all header files
for d in ${srctree} ${objtree} ; do
    tar cf - -C ${d}/include . | (cd ${WORK}/include ; tar xf -)
done
${SCR_PATH}/copy_headers.sh ${srctree}/arch/${ARCH} ${WORK}/arch/${ARCH}
if [ ${ARCH} = "arm64" ]; then
    ${SCR_PATH}/copy_headers.sh ${srctree}/arch/arm ${WORK}/arch/arm
fi

# Copy arch specific module.lds file
if [ -f ${srctree}/arch/${ARCH}/kernel/module.lds ]; then
	cp ${srctree}/arch/${ARCH}/kernel/module.lds \
		${WORK}/arch/${ARCH}/kernel/module.lds
fi

# fixup header files
if [ -e ${FIXUP} ]; then
    sh ${FIXUP} ${WORK}
fi

# Adding required cert files for module signing
. .config
if [ -n "${CONFIG_MODULE_SIG_KEY}" ]; then
     if [ -f "${CONFIG_MODULE_SIG_KEY}" ]; then
          mkdir -p ${WORK}/certs
          cp ${srctree}/${CONFIG_MODULE_SIG_KEY} ${WORK}/certs/
          cp ${srctree}/certs/signing_key.x509 ${WORK}/certs/
     fi
fi

# remove unnecessary directories/files
find ${WORK} -type d \( -name "CVS" -o -name ".svn" \) -print | xargs rm -rf
find ${WORK} -type f -name "*~" -print | xargs rm -rf

# change owner/permission
find ${WORK} -type f -print | xargs chmod 644
find ${WORK} -type d -print | xargs chmod 755

# create tarball
tar cf ${HEADER} -C ${WORK} .
rm -r ${WORK}
test -f ${HEADER}
if [ $? -ne 0 ]; then
    echo
    fatal "Cannot create kernel header"
fi

#
# make kbuild tree
#
mkdir -p ${WORK}
(cd ${srctree};
find . -type f \( -name "Makefile*" -o -name "Kconfig*" \) \
             ! \( -path "*.work*" -o -path "*.svn*" -o -path "*.pc*" \) | \
             cpio -o --format=newc --quiet | (cd ${WORK}; cpio -iumd --quiet))
for f in ${COPY_FILES} ; do
    cp ${objtree}/${f} ${WORK}
done
for d in ${COPY_DIRS} ; do
    cp -R  ${srctree}/${d} ${WORK}
    cp -fR ${objtree}/${d} ${WORK}
done
if [ -e ${SYMVERS} ]; then
    cp ${SYMVERS} ${WORK}
fi

# Copy objtool related files required for building out of tree modules
for i in $OBJTOOL_FILES
do
    if [ -f ${srctree}/tools/objtool/${i} ]; then
         cp ${srctree}/tools/objtool/${i} ${WORK}/tools/objtool/
    fi
done

# remove unnecessary directory
rm -rf ${WORK}/Documentation
rm -rf ${WORK}/include
rm -rf ${WORK}/.pc
rm -rf ${WORK}/patches

# remove unnecessary directories/files
find ${WORK} -type d \( -name "CVS" -o -name ".svn" \) -print | xargs rm -rf
find ${WORK} -type f -name "*~" -print | xargs rm -rf

# change permission
find ${WORK} -type d -print | xargs chmod 755

# create tarball
tar cf ${KBUILD} -C ${WORK} --exclude=Sony.INTERNAL .
rm -r ${WORK}
test -f ${KBUILD}
if [ $? -ne 0 ]; then
    echo
    fatal "Cannot create kbuild tree"
fi

#
# make archive file
#
tar cfz ${DEST} ${HEADER} ${KBUILD}
rm ${HEADER} ${KBUILD}

echo "done"
ls -l ${DEST}

exit 0
