/*
 * drivers/misc/nic.c
 *
 * Copyright 2022 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/arm-smccc.h>
#include <mach/regs-smc.h>
#include <linux/nic.h>

int nic_write(uint32_t reg, uint32_t val)
{
#ifdef CONFIG_EJ_USE_EL3
	struct arm_smccc_res res;

	arm_smccc_smc(CXD900XX_SMC_NIC_WR32, reg, val, 0, 0, 0, 0, 0, &res);
	return 0;
#else
	return -EPERM;
#endif /* CONFIG_EJ_USE_EL3 */
}
EXPORT_SYMBOL(nic_write);

int nic_read(uint32_t reg, uint32_t *val)
{
#ifdef CONFIG_EJ_USE_EL3
	struct arm_smccc_res res;

	arm_smccc_smc(CXD900XX_SMC_NIC_RD32, reg, 0, 0, 0, 0, 0, 0, &res);
	*val = (uint32_t)res.a0;
	return 0;
#else
	return -EPERM;
#endif /* CONFIG_EJ_USE_EL3 */
}
EXPORT_SYMBOL(nic_read);

static unsigned long reg2ul(const nic_reg *reg)
{
	return ((uint64_t)(reg->reg) << 0) | ((uint64_t)(reg->val) << 32);
}

int nic_write_multi(const nic_reg *reg, uint32_t num)
{
#ifdef CONFIG_EJ_USE_EL3
	struct arm_smccc_res res;

	if (reg == NULL)
		return -EINVAL;

	while (num > 0) {
		const uint32_t LENGTH = 7;
		unsigned long param[LENGTH]; /* NIC regsiter address and set value */
		uint32_t n = ((num <= LENGTH) ? num : LENGTH); /* number of valid parameters */
		uint32_t i;

		num -= n;
		for (i = 0; i < n; ++i) {
			param[i] = reg2ul(reg++);
		}
		for (; i < LENGTH; ++i) {
			param[i] = 0;
		}

		arm_smccc_smc((unsigned long)CXD900XX_SMC_NIC_WR32N | ((unsigned long)n << 32),
				param[0], param[1], param[2], param[3], param[4], param[5], param[6], &res);
	}

	return 0;
#else
	return -EPERM;
#endif /* CONFIG_EJ_USE_EL3 */
}
EXPORT_SYMBOL(nic_write_multi);
