/*
 * Copyright 2018 Sony Imaging Products & Solutions Inc.
 * Copyright 2018 Sony Corporation.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <linux/err.h>
#include <linux/io.h>
#include <linux/module.h>
#include <linux/mfd/syscon.h>
#include <linux/of.h>
#include <linux/platform_device.h>
#include <linux/reset-controller.h>
#include <linux/regmap.h>
#include <linux/types.h>
#include <linux/delay.h>
#include <dt-bindings/reset/cxd900xx_reset_regs.h>

struct cxd900xx_reset_data {
	struct regmap *scu;
	struct reset_controller_dev rcdev;
	u32 offset;
};

#define to_cxd900xx_reset_data(p)		\
	container_of((p), struct cxd900xx_reset_data, rcdev)

static int cxd900xx_reset_assert(struct reset_controller_dev *rcdev,
			     unsigned long id)
{
	struct cxd900xx_reset_data *priv = to_cxd900xx_reset_data(rcdev);

	int bank = id / CXD900XX_SCU_BITS;
	int offset = id % CXD900XX_SCU_BITS;

	pr_debug("%s: %s reset bank %u offset %u\n", KBUILD_MODNAME, __func__,
		 bank, offset);

	return regmap_write(priv->scu, priv->offset + (bank * 16 + 4), BIT(offset));
}

static int cxd900xx_reset_deassert(struct reset_controller_dev *rcdev,
			       unsigned long id)
{
	struct cxd900xx_reset_data *priv = to_cxd900xx_reset_data(rcdev);
	int ret;

	int bank = id / CXD900XX_SCU_BITS;
	int offset = id % CXD900XX_SCU_BITS;

	pr_debug("%s: %s reset bank %u offset %u\n", KBUILD_MODNAME, __func__,
		 bank, offset);

	ret = regmap_write(priv->scu, priv->offset + (bank * 16 + 8), BIT(offset));
	wmb();
	if (CXD900XX_RST_TIMER_0 <= id && id <= CXD900XX_RST_TIMER_15)
		udelay(8);
	else if (CXD900XX_RST_WDT_MAIN_CPU == id)
		udelay(4);
	else
		udelay(1);
	return ret;
}

static int cxd900xx_reset_status(struct reset_controller_dev *rcdev,
			     unsigned long id)
{
	struct cxd900xx_reset_data *priv = to_cxd900xx_reset_data(rcdev);

	int bank = id / CXD900XX_SCU_BITS;
	int offset = id % CXD900XX_SCU_BITS;
	int ret;
	u32 reg;

	pr_debug("%s: %s reset bank %u offset %u\n", KBUILD_MODNAME, __func__,
		 bank, offset);

	ret = regmap_read(priv->scu, priv->offset + (bank * 16), &reg);
	if (ret)
		return ret;

	return !!(reg & BIT(offset));
}

static const struct reset_control_ops cxd900xx_reset_ops = {
	.assert		= cxd900xx_reset_assert,
	.deassert	= cxd900xx_reset_deassert,
	.status		= cxd900xx_reset_status,
};

static int cxd900xx_reset_probe(struct platform_device *pdev)
{
	struct resource *res;
	struct cxd900xx_reset_data *priv;

	priv = devm_kzalloc(&pdev->dev, sizeof(*priv), GFP_KERNEL);
	if (!priv)
		return -ENOMEM;
	platform_set_drvdata(pdev, priv);

	priv->scu = syscon_regmap_lookup_by_phandle(pdev->dev.of_node,
						     "syscon");
	if (IS_ERR(priv->scu)) {
		dev_err(&pdev->dev, "unable to get cxd900xx-scu regmap");
		return PTR_ERR(priv->scu);
	}

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res) {
		dev_err(&pdev->dev, "missing IO resource\n");
		return -ENODEV;
	}

	priv->offset = res->start;

	priv->rcdev.owner = THIS_MODULE;
	priv->rcdev.nr_resets = resource_size(res) / 16 * CXD900XX_SCU_BITS;
	priv->rcdev.ops = &cxd900xx_reset_ops;
	priv->rcdev.of_node = pdev->dev.of_node;

	return devm_reset_controller_register(&pdev->dev, &priv->rcdev);
}

static const struct of_device_id cxd900xx_reset_dt_ids[] = {
	{ .compatible = "cxd900xx,reset", },
	{ /* sentinel */ },
};

static struct platform_driver cxd900xx_reset_driver = {
	.probe	= cxd900xx_reset_probe,
	.driver = {
		.name		= KBUILD_MODNAME,
		.of_match_table	= cxd900xx_reset_dt_ids,
	},
};

static int __init cxd900xx_reset_init(void)
{
	return platform_driver_register(&cxd900xx_reset_driver);
}
arch_initcall(cxd900xx_reset_init);

