/**
 *	@file	usbg_iap_cmn.c
 *	@brief	USB iAP(iAP Device Interface) - common
 *	
 *	Copyright 2024 Sony Group Corporation
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <asm/errno.h>
#include <asm/unaligned.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/version.h>

#include <linux/usb/ch9.h>
#include <linux/device.h>
#include <linux/usb/gadget.h>

#include <linux/usb/gcore/usb_event.h>
#include <linux/usb/gcore/usb_gadgetcore.h>
#include <linux/usb/specific_gadget/usbg_cmn.h>
#include <linux/usb/specific_gadget/usbg_type.h>
#include <linux/usb/specific_gadget/usbg_iap.h>
#include <linux/usb/specific_gadget/usb_kdbg.h>

#define FNAME "usbg_iap_cmn.c"
#include "usbg_iap_conf.h"
#include "usbg_iap_pvt.h"

#include <udif/malloc.h>

/**
 *	@brief		driver instance
 *	@note		usbg_dev == 0 means "driver is invalid"
 *				usbg_dev != 0 means "driver is valid"
 */
static struct func_data *iap_this = NULL;

/**
 *	@brief		Create driver instance
 *	@retval		0			success
 *	@retval		-ENOMEM		short of memory
 */
int iap_create_this(void)
{
    struct func_data*    p_this;
    struct func_data_ex* p_this_ex;
    
//    IAP_FNC();
    
    if(iap_this != NULL) {
		// instance is already created
		// nothint to do
		return 0;
	}
	
    // alloc driver instance area
	p_this = kmalloc(sizeof(struct func_data), GFP_ATOMIC);
	if(!p_this) {
		IAP_ERR("can not create instance\n");
		return -ENOMEM;
	}
	
    // alloc driver instance area (iAP extention)
	p_this_ex = kmalloc(sizeof(struct func_data_ex), GFP_ATOMIC);
	if(!p_this_ex) {
		IAP_ERR("can not create instance\n");
		kfree(p_this);
		return -ENOMEM;
	}
	
    // 0 fill 
	memset(p_this,    0, sizeof(struct func_data));
	memset(p_this_ex, 0, sizeof(struct func_data_ex));

	// link informations this->this_ex
	p_this->vp_user = p_this_ex;
	iap_this = p_this;
	
	return 0;
}


/**
 *	@brief		Free driver instance
 */
void iap_delete_this(void)
{
	struct func_data*    p_this;
	struct func_data_ex* p_this_ex;
	
//	IAP_FNC();
	
	p_this    = iap_get_this();
	p_this_ex = iap_get_this_ex();
	
	// free driver informations
	if(p_this_ex) {
		kfree(p_this_ex);
	}
	
	if(p_this) {
		kfree(p_this);
	}
	
	iap_this = NULL;
}

/**
 *	@brief		Get driver instance
 *	@retval		driver instance
 */
struct func_data *iap_get_this(void)
{
	return iap_this;
}

/**
 *	@brief		Get driver instance (iAP extention)
 *	@retval		driver instance (iAP extention)
 */
struct func_data_ex *iap_get_this_ex(void)
{
	if(!iap_this) {
		return NULL;
	}
	
	return (struct func_data_ex *)iap_this->vp_user;
}

/**
 *	@brief		Link driver instance to struct file
 *	@param		file	struct file
 */
void iap_link_this_to_file(struct file *file)
{
	if(file) {
		file->private_data = iap_this;
	}
}

/**
 *	@brief		Get ep struct
 *	@param		ep_id	Endpoint ID
 *	@retval		struct ep
 */
struct usb_ep *iap_get_ep(int ep_id)
{
	if(ep_id >= IAP_EP_IDX_MAX) {
		// invalid EP no.
		return NULL;
	}
	
	if(!iap_this ||
	   !iap_this->ptbl_ep) {
		IAP_INF("p_this->ptbl_ep==NULL\n");
		return NULL;
	}
	
	return iap_this->ptbl_ep[ep_id];
}

/**
 *	@brief		Get ep information
 *	@param		ep	struct ep
 *	@retval		ep information
 */
struct iap_ep_info *iap_get_ep_info(struct usb_ep *ep)
{
	if(!ep) return NULL;
	
	return (struct iap_ep_info*)ep->driver_data;
}

/**
 *	@brief		Start request
 *	@param		ep_id	Endpoint ID
 */
void iap_request_start(int ep_id)
{
    
    iap_set_requesting_flag( ep_id, true );
    
}

/**
 *	@brief		Stop request
 *	@param		ep_id	Endpoint ID
 */
void iap_request_stop(int ep_id)
{
    
    iap_set_requesting_flag( ep_id, false );
    
}

/**
 *	@brief		Requesting?
 *	@param		ep_id	Endpoint ID
 */
int  iap_is_requesting(int ep_id)
{
    
    struct func_data_ex *p_this_ex = iap_get_this_ex();
    
    if( !p_this_ex || !(p_this_ex->ep_info)) {
        
        IAP_INF( "p_this_ex->ep_info is NULL\n" );
        
    }else{
        
        IAP_REQUESTING_LOCK_ON();
        
        if( p_this_ex->ep_info[ep_id].data.requesting ){
            
            IAP_REQUESTING_LOCK_OFF();
            
            return true;
            
        }
        
        IAP_REQUESTING_LOCK_OFF();
        
    }
    
    return false;
    
}

/**
 *	@brief		Set requesting flag
 *	@param		ep_id	Endpoint ID
  *	@param		flag	requesting flag to be set. true or false.
 */
void iap_set_requesting_flag( int ep_id, int flag )
{
    
    struct func_data_ex *p_this_ex = iap_get_this_ex();
    
    if( !p_this_ex || !(p_this_ex->ep_info)) {
        
        IAP_INF( "p_this_ex->ep_info is NULL\n" );
        
    }else{
        
        if( ( flag == true ) && iap_is_requesting( ep_id ) ){
            
            IAP_INF("request already queued\n");
            
        }
        
        IAP_REQUESTING_LOCK_ON();
        
        p_this_ex->ep_info[ep_id].data.requesting = flag;
        
        IAP_REQUESTING_LOCK_OFF();
        
    }
    
}


/**
 *	@brief		Dequeue request
 *	@param		ep		struct ep
 */
void iap_ep_dequeue(struct usb_ep *ep)
{
    struct iap_ep_info *ep_inf = iap_get_ep_info(ep);
    
    IAP_FNC();
    
    if (!ep_inf) {
        IAP_INF("ep_inf==NULL\n");
        return;
    }
    
    switch (ep_inf->ep_no) {
    case 0x84: /* data - BULK_IN */
    case 0x05: /* data - BULK_OUT */
        IAP_REQUESTING_LOCK_ON();
        if(ep_inf->data.req && ep_inf->data.requesting) {
            IAP_REQUESTING_LOCK_OFF();
            usb_ep_dequeue(ep, ep_inf->data.req);
        }
        else {
            IAP_REQUESTING_LOCK_OFF();
            IAP_INF("request==NULL\n");
        }
        break;
    default:
        IAP_INF("unknown ep_no:%X\n", ep_inf->ep_no);
        break;
    }
}

/**
 *	@brief		Dequeue request from bulk ep
 */
void iap_ep_dequeue_bulk(void)
{
	struct usb_ep *ep;
	
	IAP_EP_LOCK_ON();

	ep = iap_get_ep(IAP_EP_IDX_BULK_IN);
	if(ep) {
		iap_ep_dequeue(ep);
	}

	ep = iap_get_ep(IAP_EP_IDX_BULK_OUT);
	if(ep) {
		iap_ep_dequeue(ep);
	}

	IAP_EP_LOCK_OFF();

}

/**
 *	@brief		Dequeue request from all ep
 */
void iap_ep_dequeue_all(void)
{
	int i;
	struct usb_ep **ep_list;
	
	if(!iap_this) {
		// no driver instance
		return;
	}
	
	ep_list = iap_this->ptbl_ep;
	if(!ep_list) {
		// no ep table
		return;
	}
	
	for(i = 0; i < IAP_EP_IDX_MAX; i++) {
		if(ep_list[i]) {
			iap_ep_dequeue(ep_list[i]);
		}
	}
}

/**
 *	@brief	first, check request complete status for completion
 */
bool iap_check_req_comp_waiting(void)
{
    struct func_data_ex *p_this_ex = iap_get_this_ex();

    IAP_REQUESTING_LOCK_ON();

    /* bulk in */
    if (p_this_ex->ep_info[IAP_EP_IDX_BULK_IN].data.requesting == true)
        goto found_waiting;

    /* bulk out */
    if (p_this_ex->ep_info[IAP_EP_IDX_BULK_OUT].data.requesting == true)
        goto found_waiting;

    IAP_REQUESTING_LOCK_OFF();

    return false;

found_waiting:
    p_this_ex->req_comp_waiting = true;

    IAP_REQUESTING_LOCK_OFF();

    return true;
}

/**
 *	@brief	recheck request complete status
 */
void iap_recheck_req_comp_waiting(void)
{
    struct func_data_ex *p_this_ex = iap_get_this_ex();

    IAP_REQUESTING_LOCK_ON();
    if (p_this_ex->req_comp_waiting) { // only when core_cb_stop() is running.
        /* bulk in */
        if (p_this_ex->ep_info[IAP_EP_IDX_BULK_IN].data.requesting == true)
            goto found_waiting;

        /* bulk out */
        if (p_this_ex->ep_info[IAP_EP_IDX_BULK_OUT].data.requesting == true)
            goto found_waiting;

        p_this_ex->req_comp_waiting = false;
        complete(&p_this_ex->req_wait_completion);
    }

found_waiting:
    IAP_REQUESTING_LOCK_OFF();

    return;
}

/**
 *	@brief		Get usb_request area
 *	@retval		usb_request address
 */
struct usb_request *iap_get_request(struct usb_ep *ep)
{
	struct iap_ep_info *ep_inf = iap_get_ep_info(ep);
	
	IAP_FNC();
	
	if(ep_inf) {
		return ep_inf->data.req;
	}
	else {
		IAP_INF("request==NULL\n");
		return NULL;
	}
}
