/*
 *  Provide IRQ stack usage for each online cpu.
 *  /proc/irq_stack
 */

#include <linux/fs.h>
#include <linux/init.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/nmi.h>
#include <asm/stacktrace.h>

static inline unsigned long irq_stack_used(int cpu)
{
	unsigned long *low = per_cpu(irq_stack_ptr, cpu);
	unsigned long high = (unsigned long)low + IRQ_STACK_SIZE;

	/* Address of the last usable long on the IRQ stack */
	do {
		low++;
	} while (!*low);

	return (high - (unsigned long)low);
}

static int irqstack_proc_show(struct seq_file *m, void *v)
{
	int cpu;

	seq_puts(m, "CPU  USED / TOTAL\n");

	for_each_online_cpu(cpu) {
		seq_printf(m, "%2d: %5ld / %5ld\n",
			   cpu,
			   irq_stack_used(cpu),
			   IRQ_STACK_SIZE);
	}

	return 0;
}

static int irqstack_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, irqstack_proc_show, NULL);
}

static const struct file_operations irqstack_proc_fops = {
	.open		= irqstack_proc_open,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release,
};

static int __init irqstack_init(void)
{
	proc_create("irq_stack", 0444, NULL, &irqstack_proc_fops);
	return 0;
}

late_initcall(irqstack_init);
