/*
 *  Copyright 2016 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include <common.h>
#include <bootm.h>
#include <command.h>
#include <linux/sizes.h>
#include <linux/ctype.h>
#include <elf.h>
#include <fs.h>
#include <malloc.h>
#include <boot_time.h>

#define FS_READ_SIZE SZ_16M

/*
 * Reserved for all functions of u-boot except SBI temporary buffer.
 * Such as filesystems，drivers，and other commands.
 */
#define MALLOC_RESRRVED_SIZE	SZ_4M

#if (CONFIG_SYS_MALLOC_LEN < FS_READ_SIZE + MALLOC_RESRRVED_SIZE)
#error "ssboot: CONFIG_SYS_MALLOC_LEN is too small, please increase it"
#endif

#ifdef CONFIG_CMD_SSBOOT_FS_TYPE
#define SSBOOT_FS_TYPE	CONFIG_CMD_SSBOOT_FS_TYPE
#else
#define SSBOOT_FS_TYPE	FS_TYPE_ANY
#endif

// #define CONFIG_CMD_SSBOOT_TIME


#ifdef CONFIG_CMD_SSBOOT_CKSUM

#define elf_cksum_algorithm(cksum_phdr)		((cksum_phdr)->p_offset)
#define elf_ehdr_cksum(cksum_phdr)			((cksum_phdr)->p_vaddr)
#define elf_phdr_cksum(cksum_phdr)			((cksum_phdr)->p_paddr)
#define elf_data_cksum(cksum_phdr)			((cksum_phdr)->p_filesz)

#define CKSUM_TYPE_CRC32	1

enum {
	SSBOOT_WITH_CRC32,
	SSBOOT_WITHOUT_CKSUM,
};
#endif

static const char *interface, *devpart, *filename;
static loff_t total_read_len = 0;
static loff_t act_read_len = 0;
static loff_t file_total_size = FS_READ_SIZE; /* temp init value */
static int fs_type = SSBOOT_FS_TYPE;

#ifdef CONFIG_CMD_SSBOOT_TIME
static void show_speed(unsigned long size, unsigned long time)
{
	unsigned long speed;
	unsigned long ms;
#define FLOAT(size, n) (((((size) >> (n)) & 0x3FF) * 1000) / 1024)

	if (size >> 30) {
		printf("Size : %lu.%03luGB\n", size >> 30, FLOAT(size, 20));
	} else if (size >> 20) {
		printf("Size : %lu.%03luMB\n", size >> 20, FLOAT(size, 10));
	} else if (size >> 10) {
		printf("Size : %lu.%03luKB\n", size >> 10, FLOAT(size, 0));
	} else {
		printf("Size : %luB\n", size);
	}
	ms = time / (CONFIG_SYS_HZ / 1000);
	if (ms > 1000) {
		printf("Time : %lu.%03lus\n", ms / 1000, ms % 1000);
	} else {
		printf("Time : %lums\n", ms);
	}

	speed = (size / ms) * 1000;
	if (speed >> 30) {
		printf("Speed: %lu.%03luGB/s\n", speed >> 30, FLOAT(speed, 20));
	} else if (speed >> 20) {
		printf("Speed: %lu.%03luMB/s\n", speed >> 20, FLOAT(speed, 10));
	} else if (speed >> 10) {
		printf("Speed: %lu.%03luKB/s\n", speed >> 10, FLOAT(speed, 0));
	} else {
		printf("Speed: %luB/s\n", speed);
	}
}
#endif

static loff_t exec_fs_size(const char *filename)
{
	loff_t filesize;

	if (fs_set_blk_dev(interface, devpart, fs_type)) {
		printf("Error: Init partition %s on %s failed\n", devpart, interface);
		return -1;
	}

	if (fs_size(filename, &filesize) < 0) {
		printf("Error: Can not get file size\n");
		return -1;
	}

	return filesize;
}

static loff_t get_elf_size(const char *filename, Elf_Phdr phdr)
{
	loff_t partition_size, filesize;
	struct blk_desc *fs_dev_desc;
	disk_partition_t fs_partition;

	if (blk_get_device_part_str(interface, devpart, &fs_dev_desc,
		&fs_partition, 1) < 0)
		return -1;

	partition_size = fs_partition.size * fs_partition.blksz;

	filesize = exec_fs_size(filename);
	if (filesize <= 0)
		return filesize;

	if (filesize < phdr.p_filesz + phdr.p_offset) {
		printf("Error: The ssboot image has been truncated.\n");
		return -1;
	}

	/* is rawfs */
	if (partition_size == filesize) {
		filesize = phdr.p_filesz + phdr.p_offset;

		if (filesize <= 0 || filesize > partition_size) {
			filesize = partition_size;
			printf("Warning: ELF size is invalid,"
					"try use partition size\n");
		}
	}

	return filesize;
}

static loff_t exec_fs_read(const void* addr)
{
	int ret;
	loff_t len;

	if (fs_set_blk_dev(interface, devpart, fs_type)) {
		printf("Error: Init partition %s on %s failed\n", devpart, interface);
		return -1;
	}

	if (file_total_size < total_read_len + FS_READ_SIZE)
		len = file_total_size - total_read_len;
	else
		len = FS_READ_SIZE;

	/* The entire ssboot file has been loaded, but more data is needed */
	if (len == 0)
		return -1;

	ret = fs_read(filename, (ulong)addr, total_read_len, len, &act_read_len);
	if (ret < 0 || act_read_len != len) {
		printf("Error: Read %lld bytes from %s:%lld failed\n",
				len, filename, total_read_len);
		return -1;
	}
	total_read_len += act_read_len;

	return act_read_len;
}

/*
 * Return
 * 0 : all data in buf is relocated
 * >0: some bytes data is not relocated
 * -1: error
 */
static loff_t elf_load_sect(const Elf_Phdr *phdr, void *buf, loff_t *rest_size)
{
	loff_t ret;
	void *src, *dst;
	loff_t filesz;

	filesz = phdr->p_filesz;
	dst = (void *)phdr->p_paddr;
	src = buf + phdr->p_offset - total_read_len + act_read_len;

	while (filesz > 0) {
		if (*rest_size == 0) {
			/* load new data */
			ret = exec_fs_read(buf);
			if (ret < 0)
				return -1;

			*rest_size = ret;
			src = buf;
		}

		if (*rest_size >= filesz) {
			debug("dst=%p src=%p, filesz=0x%llx\n", dst, src, filesz);
			memcpy(dst, src, filesz);
			*rest_size -= filesz;
			/* completed, turn to next */
			break;
		}
		/* not completed, copy all */
		memcpy(dst, src, *rest_size);
		debug("dst=%p src=%p, rest_size=0x%llx\n", dst, src, *rest_size);
		filesz -= *rest_size;
		dst += *rest_size;
		*rest_size = 0;
	}

	return 0;
}

#ifdef CONFIG_CMD_SSBOOT_CKSUM
static unsigned int ssboot_calc_data_checksum(Elf_Ehdr *ehdr, Elf_Phdr *phdr)
{
	int i;
	unsigned int data_cksum = 0;

	for (i = 0; i < ehdr->e_phnum; i++, phdr++) {
		if (phdr->p_type != PT_LOAD)
			continue;

		if (phdr->p_filesz == 0)
			continue;

		data_cksum = crc32(data_cksum, (void *)phdr->p_paddr,
				   phdr->p_filesz);
	}

	return data_cksum;
}

static int ssboot_verify_checksum(Elf_Ehdr *ehdr, Elf_Phdr *phdr)
{
	int i;
	Elf_Phdr *cksum_phdr = NULL;
	unsigned int ehdr_cksum, phdr_cksum, data_cksum;

	for (i = 0; i < ehdr->e_phnum; i++) {
		if (phdr[i].p_type != PT_LOAD) {
			if (phdr[i].p_offset == CKSUM_TYPE_CRC32)
				cksum_phdr = &phdr[i];
			continue;
		}
		break;
	}

	if (NULL == cksum_phdr) {
		printf("Warning: No checksum in image, skip verify\n");
		return 0;
	}

	ehdr_cksum = crc32(0, (void *)ehdr, sizeof(*ehdr));
	phdr_cksum = crc32(0, (void *)&cksum_phdr[1],
			ehdr->e_phnum * ehdr->e_phentsize
			- ((size_t)&cksum_phdr[1] - (size_t)phdr));
	data_cksum = ssboot_calc_data_checksum(ehdr, phdr);

	if (elf_ehdr_cksum(cksum_phdr) != ehdr_cksum ||
		elf_phdr_cksum(cksum_phdr) != phdr_cksum ||
		elf_data_cksum(cksum_phdr) != data_cksum) {
		debug("ehdr_cksum: %08x %08x\n",
			(unsigned int) elf_ehdr_cksum(cksum_phdr), ehdr_cksum);
		debug("phdr_cksum: %08x %08x\n",
			(unsigned int) elf_phdr_cksum(cksum_phdr), phdr_cksum);
		debug("data_cksum: %08x %08x\n",
			(unsigned int) elf_data_cksum(cksum_phdr), data_cksum);
		return -1;
	}

	return 0;
}
#endif

static int
do_ssbootelf(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
	int i;
	int ret;
	Elf_Ehdr ehdr;		/* Elf header structure pointer     */
	Elf_Phdr *phdr;		/* Program header structure pointer */
	char *tmpbuf;
	loff_t rest_size;
#ifdef CONFIG_CMD_SSBOOT_TIME
	unsigned long time;
#endif

	if (argc < 3)
		return CMD_RET_USAGE;

	interface = argv[1];
	devpart = argv[2];
	if (argc == 4) {
		fs_type = SSBOOT_FS_TYPE;
		filename = argv[3];
	}
	else {
		fs_type = FS_TYPE_RAW;
		filename = "raw data";
	}

#ifdef CONFIG_CMD_SSBOOT_TIME
	time = get_timer(0);
#endif

	BOOT_TIME_ADD("U-Boot: ssbootelf start");

	tmpbuf = memalign(ARCH_DMA_MINALIGN, FS_READ_SIZE);
	if (tmpbuf == NULL) {
		/* TODO: reduce the size and try again */
		printf("Error: alloc %dbytes memory for temp buffer failed\n",
				FS_READ_SIZE);
		goto out1;
	}

	ret = exec_fs_read(tmpbuf);
	if (ret < 0)
		goto out2;

	if (!valid_elf_image((unsigned long)tmpbuf)) {
		printf("Error: No ssboot ELF image at \"%s %s\"\n", interface, devpart);
		goto out2;
	}

	memcpy(&ehdr, tmpbuf, sizeof(ehdr));

	/* all of the header information should be read into temp buffer
	 * in the first time
	 */
	if (ehdr.e_phoff + ehdr.e_phnum * ehdr.e_phentsize > FS_READ_SIZE) {
		printf("Error: the temp buffer is too small\n");
		goto out2;
	}

	phdr = malloc(ehdr.e_phnum * ehdr.e_phentsize);
	if (phdr == NULL) {
		printf("Error: alloc memory for program header failed\n");
		goto out2;
	}

	memcpy(phdr, tmpbuf + ehdr.e_phoff, ehdr.e_phnum * ehdr.e_phentsize);

	/* "ehdr.e_phnum - 1" is the final section number */
	file_total_size = get_elf_size(filename, phdr[ehdr.e_phnum - 1]);
	if (file_total_size < 0)
		goto out3;

	for (i = 0; i < ehdr.e_phnum; i++) {
		if (phdr[i].p_type != PT_LOAD) {
			continue;
		}
		break;
	}

	rest_size = act_read_len - phdr[i].p_offset;

	for (; i < ehdr.e_phnum; i++) {
		if (phdr[i].p_type != PT_LOAD)
			continue;

		if (elf_load_sect(phdr + i, tmpbuf, &rest_size)) {
			printf("Error: load section %d failed\n", i);
			goto out3;
		}

		if (phdr[i].p_filesz != phdr[i].p_memsz)
			memset((void *)phdr[i].p_paddr + phdr[i].p_filesz, 0x00,
					phdr[i].p_memsz - phdr[i].p_filesz);
	}

	free(tmpbuf);

	BOOT_TIME_ADD("U-Boot: ssbootelf end");

#ifdef CONFIG_CMD_SSBOOT_CKSUM
	if (ssboot_verify_checksum(&ehdr, phdr) < 0) {
		printf("Error: Verify checksum failed\n");
		free(phdr);
		goto out1;
	}
#endif

	free(phdr);

#ifdef CONFIG_CMD_SSBOOT_TIME
	time = get_timer(time);
	show_speed(total_read_len, time);
#endif

	BOOT_TIME_ADD("U-Boot: boot_jump_ssboot");
	printf("## Booting snapshot image at 0x%p ...\n", (void *)ehdr.e_entry);

	cleanup_before_linux();
	/*
	 * Please don't do anything between cleanup_before_linux() and jump to boot
	 * entry. Because cleanup_before_linux() disabled all of the caches.
	 */
	((void (*)(void))ehdr.e_entry)();

	return CMD_RET_SUCCESS;

out3:
	free(phdr);

out2:
	free(tmpbuf);

out1:
	/* during error case, cmd may run again */
	total_read_len = 0;
	act_read_len = 0;
	file_total_size = FS_READ_SIZE;

	return CMD_RET_FAILURE;
}

U_BOOT_CMD(
	ssbootelf,      4,      0,      do_ssbootelf,
	"Load ssboot ELF image and boot",
	"<interface> <dev:part> [filename]\n"
	"\t- Load ssboot ELF image 'filename' from 'dev:part' on 'interface'"
);
