/*
 *  Copyright 2020 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include <common.h>
#include <linux/compat.h>
#include <fdtdec.h>
#include <pmmem.h>

#define PMMEM_COMPATIBLE "sony,ssboot-discard"

int pmmem_reserve(void *fdt, phys_addr_t base, phys_size_t size,
		  const char *basename)
{
	int ret, off;
	uint32_t phandle;
	struct fdt_memory mem;

	if ((base | size) & (PAGE_SIZE - 1)) {
		printf("Error: PM memory should alignment with page size\n");
		return -EINVAL;
	}

	mem.start = base;
	mem.end = base + size - 1;

	ret = fdtdec_add_reserved_memory(fdt, basename, &mem, &phandle);
	if (ret < 0) {
		printf("Can't create node in /reserved-memory.\n");
		return ret;
	}

	off = fdt_node_offset_by_phandle(fdt, phandle);
	if (off < 0) {
		printf("Can't open the reserved node\n");
		return off;
	}

	return fdt_setprop_string(fdt, off, "compatible", PMMEM_COMPATIBLE);
}

#if CONFIG_IS_ENABLED(PMMEM_UBOOT)
static int pmmem_reserve_uboot(void *fdt)
{
	phys_addr_t base;
	phys_size_t size;

	/*
	 * To ensure that the memory region is same for PM memory and LMB
	 * reserved, the memory region is exported to "uboot_pmmem_base"
	 * and "uboot_pmmem_size" environment variables in arch_lmb_reserve().
	 */

	base = env_get_hex("uboot_pmmem_base", ~0);
	if (base == ~0) {
		printf("Error: env uboot_pmmem_base is not set\n");
		return -EINVAL;
	}

	size = env_get_hex("uboot_pmmem_size", ~0);
	if (size == ~0) {
		printf("Error: env uboot_pmmem_size is not set\n");
		return -EINVAL;
	}

	return pmmem_reserve(fdt, base, size, "uboot");
}
#endif

int fdt_fixup_pmmem(void *fdt)
{
	int ret = 0;

#if CONFIG_IS_ENABLED(PMMEM_UBOOT)
	ret = pmmem_reserve_uboot(fdt);
	if (ret < 0) {
		printf("Failed to add U-Boot memory into reserved PM memory\b");
		return ret;
	}
#endif

	return ret;
}
