/*
 *  Copyright 2016 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include <common.h>
#include <config.h>
#include <exports.h>
#include <rawfs.h>
#include <part.h>
#include <malloc.h>
#include <linux/compiler.h>
#include <linux/ctype.h>
#include <div64.h>

#if !CONFIG_IS_ENABLED(BLK)
#define  blk_dread(dev, start, nblk, buf) \
	dev->block_read ? dev->block_read((dev), (start), (nblk), (buf)) : (-1)
#endif

static struct blk_desc *cur_dev;
static disk_partition_t cur_part_info;

static inline loff_t to_blocks(loff_t n)
{
	uint32_t blksz = cur_dev->blksz;

	if (n < blksz)
		return 0;

	for (; blksz > 1; blksz >>= 1)
		n >>= 1;

	return n;
}

int rawfs_set_blk_dev(struct blk_desc *dev_desc, disk_partition_t *info)
{
	cur_dev = dev_desc;
	cur_part_info = *info;

	return 0;
}

int rawfs_file_ls(const char *dir)
{
	loff_t size;

	rawfs_size(NULL, &size);
	printf("%8lld   (raw data)\n\n", size);

	return 1;
}

int rawfs_exists(const char *filename)
{
	return 1;
}

int rawfs_size(const char *filename, loff_t *size)
{
	*size = cur_part_info.size * cur_dev->blksz;

	return 0;
}

int rawfs_read(const char *filename, void *buf, loff_t offset, loff_t len,
		  loff_t *actread)
{
	loff_t size, vsize, head, blksz;
	lbaint_t start, nblk;
	char *tmpbuf;
	unsigned long res;

	*actread = 0;

	if (!cur_dev)
		return -1;

	blksz = cur_dev->blksz;

	tmpbuf = memalign(ARCH_DMA_MINALIGN, blksz);
	if (tmpbuf == NULL) {
		printf("malloc failed\n");
		return -1;
	}

	rawfs_size(filename, &size);
	if (len == 0)
		len = size;
	if (len + offset > size)
		len = size - offset;

	head = offset & (blksz - 1); /* bytes */
	start = to_blocks(offset) + cur_part_info.start; /* blocks */

	while (len > 0) {
		nblk = ((head || len < blksz) ? 1 : to_blocks(len));

		if (nblk == 1) {
			res = blk_dread(cur_dev, start, nblk, tmpbuf);
			if (res != nblk)
				goto err;
			vsize = (head + len) > blksz ? blksz - head : len;
			memcpy(buf, tmpbuf + head, vsize);
			head = 0;
		} else {
			res = blk_dread(cur_dev, start, nblk, buf);
			if (res != nblk)
				goto err;
			vsize = len & ~(blksz - 1);
		}

		start += nblk;
		buf += vsize;
		*actread += vsize;
		len -= vsize;
	}

	free(tmpbuf);
	return 0;

err:
	free(tmpbuf);
	printf("rawfs read failed\n");
	return -1;
}

void rawfs_close(void)
{
}
