/*
 * (C) Copyright 2016 Sony Corporation
 */

#include <boot_time.h>
#include <common.h>
#include <div64.h>

DECLARE_GLOBAL_DATA_PTR;

static struct boot_time_t *pBootTime;
static int boot_time_max_size;
#define BOOT_TIME_ENTRY_SIZE	(sizeof(struct boot_time_entry_t) + boot_time_max_size)
#define PENTRY(off)	((struct boot_time_entry_t *)((unsigned long)pBootTime + off))
#ifndef CONFIG_BOOT_TIME_RINGBUFFER
#define INC_OFFSET(head, off, max)  do { (off) += BOOT_TIME_ENTRY_SIZE; } while (0)
#else
#define INC_OFFSET(head, off, max)				\
	do {							\
		if (((off)+BOOT_TIME_ENTRY_SIZE) >= (max)) {	\
			(off) = (head);				\
		} else {					\
			(off) += BOOT_TIME_ENTRY_SIZE;		\
		}						\
	} while (0)
#endif

#define WORD_ALIGNED(addr)	(((unsigned long)(addr) & (4 - 1)) == 0)

static void
boot_time_count_set(struct boot_time_entry_t *pEntry)
{
	unsigned long long t;

	t = (unsigned long long)timer_get_us() * 1000;

	pEntry->count_lo = (u_int32_t)t;
	pEntry->count_hi = (u_int32_t)(t >> 32);
}

void
boot_time_add(char *comment)
{
	struct boot_time_entry_t  *p;
	int nr_entry;

	if (pBootTime == NULL)
		return;

#ifndef CONFIG_BOOT_TIME_RINGBUFFER
	if (pBootTime->offNext >= pBootTime->offMax)
		return;
#endif
	p = PENTRY(pBootTime->offNext);
	boot_time_count_set(p);
	if (comment)
		strncpy(p->comment, comment, MAX_COMMENT);
	else
		p->comment[0] = '\0';
	INC_OFFSET(pBootTime->offHead, pBootTime->offNext, pBootTime->offMax);
	nr_entry = (pBootTime->offMax - pBootTime->offHead) /
			BOOT_TIME_ENTRY_SIZE;

	if (pBootTime->numWritten < nr_entry)
		pBootTime->numWritten++;
	return;
}

void
boot_time_init(void)
{
	u_int32_t header_size;
	int nr_entry;

	pBootTime = (struct boot_time_t *)gd->boottime_base;

	header_size = (unsigned long)&(pBootTime->first_entry) -
			(unsigned long)pBootTime;

	boot_time_max_size = MAX_COMMENT;

	/* force offMax align to entry start address */
	nr_entry = (gd->boottime_size - header_size) /
		   BOOT_TIME_ENTRY_SIZE;
	pBootTime->offMax = header_size + nr_entry * BOOT_TIME_ENTRY_SIZE;

	if ((pBootTime->magic != BOOT_TIME_MAGIC)
	   || ((u_int32_t)pBootTime->offHead > (u_int32_t)pBootTime->offMax)
	   || ((u_int32_t)pBootTime->offNext > (u_int32_t)pBootTime->offMax)
	   || (!WORD_ALIGNED(pBootTime->offHead))
	   || (!WORD_ALIGNED(pBootTime->offNext))
	   || (pBootTime->max_comment != MAX_COMMENT)) {
		/* initialize boot-time data region */
		pBootTime->magic = BOOT_TIME_MAGIC;
		pBootTime->offHead = header_size;
		pBootTime->offNext = pBootTime->offHead;
		pBootTime->numWritten = 0;
		pBootTime->max_comment = MAX_COMMENT;
	}
}

int
shell_boottime(int argc, char *const argv[])
{
	struct boot_time_entry_t *p;
	u_int32_t off;
	int loop, num;
#ifdef CONFIG_BOOT_TIME_RINGBUFFER
	int nr_entry;
#endif
	if (argc > 1) {
		if (strcmp(argv[1], "-c") == 0) {
			/* clear boot-time region */
			pBootTime->offNext = pBootTime->offHead;
			pBootTime->numWritten = 0;
			flush_dcache_all();
			return 0;
		}
	}

#ifndef CONFIG_BOOT_TIME_RINGBUFFER
	num = (pBootTime->offNext - pBootTime->offHead) / BOOT_TIME_ENTRY_SIZE;
	off = pBootTime->offHead;
#else
	nr_entry = (pBootTime->offMax - pBootTime->offHead) /
			BOOT_TIME_ENTRY_SIZE;
	num = pBootTime->numWritten;
	if (pBootTime->numWritten >= nr_entry)
		off = pBootTime->offNext;
	else
		off = pBootTime->offHead;
#endif
	for (loop = 0; loop < num; loop++) {
		int i;
		int space = 0;
		int usec;

		p = PENTRY(off);
		usec = lldiv(((unsigned long long)p->count_hi << 32) |
				p->count_lo, 1000);
		printf("%7u.%03u ms : ", usec / 1000, usec % 1000);
		for (i = 0; i < MAX_COMMENT; i++) {
			if (space || p->comment[i] == '\0' || p->comment[i] == '\n')
				space = 1;
			else
				putc(p->comment[i]);
		}
		printf("\n");
		INC_OFFSET(pBootTime->offHead, off, pBootTime->offMax);
	}
	printf("\n");
	return 0;
}
