/*
 *  Copyright (C) 2011 Sony Corporation.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA
 */

#include "core/state.h"
#include "gfx/convert.h"

#include "ev1_gfxdriver.h"
#include "ev1_2d.h"

// hardware
#include <fcntl.h>
#include <linux/fb.h>
#include <linux/fs.h>
#include <sys/mman.h>
#include <sys/ioctl.h>
#include <mach/fbcommon.h>

#include <string.h>

D_DEBUG_DOMAIN(EV1_2d, "EV1/2D", "EV1 2D graphics");

#define DFB_TRANSFORM(x, y, m, affine) \
do { \
     s32 _x, _y, _w; \
     if (affine) { \
          _x = ((x) * (m)[0] + (y) * (m)[1] + (m)[2] + 0x8000) >> 16; \
          _y = ((x) * (m)[3] + (y) * (m)[4] + (m)[5] + 0x8000) >> 16; \
     } \
     else { \
          _x = ((x) * (m)[0] + (y) * (m)[1] + (m)[2]); \
          _y = ((x) * (m)[3] + (y) * (m)[4] + (m)[5]); \
          _w = ((x) * (m)[6] + (y) * (m)[7] + (m)[8]); \
          if (!_w) { \
               _x = (_x < 0) ? -0x7fffffff : 0x7fffffff; \
               _y = (_y < 0) ? -0x7fffffff : 0x7fffffff; \
          } \
          else { \
               _x /= _w; \
               _y /= _w; \
          } \
     } \
     (x) = _x; \
     (y) = _y; \
} while (0)

void ev1_set_src(EV1DeviceData *dev, CardState *state);
void ev1_set_dst(EV1DeviceData *dev, CardState *state);
void ev1_set_color(EV1DeviceData *dev, CardState *state);
void ev1_set_matrix(EV1DeviceData *dev, CardState *state);
void ev1_set_clip(EV1DeviceData *dev, CardState *state);
void ev1_set_src_colorkey(EV1DeviceData *dev, CardState *state);
unsigned int convert_format(DFBSurfacePixelFormat format);

inline bool is_ev1_fill_rectangle_valid(EV1DeviceData* dev_data);
inline bool is_ev1_blit_valid(EV1DeviceData* dev_data);
inline bool is_ev1_stretch_blit_valid(EV1DeviceData* dev_data);
inline bool is_ev1_fb_blit_siz_valid(DFBRectangle* srect, DFBRectangle* drect);
inline bool ev1_fb_blit(EV1DriverData* drv_data, EV1DeviceData* dev_data, DFBRectangle* srect, DFBRectangle* drect, unsigned int alpha);

void ev1_clip_rectangle(const DFBRegion *clip, DFBRectangle *rect);
void ev1_clip_blit(const DFBRegion *clip, DFBRectangle *srect, int *dx, int *dy);
void ev1_clip_stretchblit(const DFBRegion *clip, DFBRectangle *srect, DFBRectangle *drect);

inline bool is_ev1_fill_rectangle_valid(EV1DeviceData* dev_data)
{
    if(dev_data->dst_format == DSPF_RGB24){
            return true;
    }
    return false;
}

inline bool is_ev1_stretch_blit_valid(EV1DeviceData* dev_data)
{
    if(dev_data->src_format == DSPF_RGB24 && dev_data->dst_format == DSPF_RGB24){
            return true;
    }
    return false;
}

inline bool is_ev1_blit_valid(EV1DeviceData* dev_data)
{
    if((dev_data->src_format == DSPF_RGB24 || dev_data->src_format == DSPF_ARGB)
        && (dev_data->dst_format == DSPF_RGB24 || dev_data->dst_format == DSPF_ARGB)){
        return true;
    }
    return false;
}

inline bool is_ev1_blit_rgb16_valid(EV1DeviceData* dev_data)
{
    if(dev_data->src_format == DSPF_RGB16 && (dev_data->dst_format == DSPF_RGB24 || dev_data->dst_format == DSPF_ARGB)){
        return true;
    }
    return false;
}

inline bool is_ev1_fb_blit_siz_valid(DFBRectangle* srect, DFBRectangle* drect)
{
    // HW仕様による制限
    if(srect->x == 0 && srect->y == 0 && srect->w%4==0 && srect->w>1 && srect->h>1 && drect->w>1 && drect->h>1 && (drect->w <= (256 * srect->w)) && (drect->h <=(256 * srect->h)) && (srect->w <= (64 * drect->w)) && (srect->h <= (64 * srect->h))) {
        return true;
    }
    return false;
}

inline bool ev1_fb_blit(EV1DriverData* drv_data, EV1DeviceData* dev_data, DFBRectangle* srect, DFBRectangle* drect, unsigned int alpha)
{
    if(srect->w < 1 || srect->h < 1 || drect->h < 1 || drect->h < 1) {
        DEBUG_PRINT("warning: EMXX_FB_BLIT invalid size. \n");
        return false;
    }

    struct emxx_fb_rect src_rect;
    struct emxx_fb_rect dst_rect;
    struct emxx_fb_img src_img;
    struct emxx_fb_img dst_img;
    src_rect.x = (unsigned)srect->x; src_rect.y = (unsigned)srect->y; src_rect.w = (unsigned)srect->w; src_rect.h = (unsigned)srect->h;
    dst_rect.x = (unsigned)drect->x; dst_rect.y = (unsigned)drect->y; dst_rect.w = (unsigned)drect->w; dst_rect.h = (unsigned)drect->h;
    src_img.width = (unsigned)dev_data->src_width; src_img.height = (unsigned)dev_data->src_height; src_img.format = convert_format(dev_data->src_format); src_img.offset = dev_data->src_addr - drv_data->base_addr; src_img.memory_id = drv_data->fd;
    dst_img.width = (unsigned)dev_data->dst_width; dst_img.height = (unsigned)dev_data->dst_height; dst_img.format = convert_format(dev_data->dst_format); dst_img.offset = dev_data->dst_addr - drv_data->base_addr; dst_img.memory_id = drv_data->fd;

    DEBUG_PRINT("fb_blit:src rect: %d, %d, %d, %d\n", src_rect.x, src_rect.y, src_rect.w, src_rect.h);
    DEBUG_PRINT("fb_blit:dst rect: %d, %d, %d, %d\n", dst_rect.x, dst_rect.y, dst_rect.w, dst_rect.h);
    DEBUG_PRINT("fb_blit:src img(w,h): %d, %d\n", src_img.width, src_img.height);
    DEBUG_PRINT("fb_blit:dst img(w,h): %d, %d\n", dst_img.width, dst_img.height);

    struct emxx_fb_blit_req blit_req;
    blit_req.src_rect = src_rect;
    blit_req.dst_rect = dst_rect;
    blit_req.src = src_img;
    blit_req.dst = dst_img;
    blit_req.alpha = alpha;
    blit_req.transp_mask = EMXX_FB_TRANSP_NOP;
    blit_req.flags = EMXX_FB_ROT_NOP;

    struct {
        uint32_t count;
        struct emxx_fb_blit_req req[1];
    } req_list;
    memset(&req_list, 0, sizeof(req_list));
    req_list.count = 1;
    req_list.req[0] = blit_req;

    if(ioctl(drv_data->fd, EMXX_FB_BLIT, &req_list)) {
        DEBUG_PRINT("warning: EMXX_FB_BLIT return false.\n");
        return false;
    }

    //printf("***** EMXX_FB_BLIT success *****\n");
    return true;
}

void ev1_clip_rectangle( const DFBRegion *clip, DFBRectangle *rect )
{
     if ((clip->x1 >= rect->x + rect->w) ||
         (clip->x2 < rect->x) ||
         (clip->y1 >= rect->y + rect->h) ||
         (clip->y2 < rect->y))
          return;

     if (clip->x1 > rect->x) {
          rect->w += rect->x - clip->x1;
          rect->x = clip->x1;
     }

     if (clip->y1 > rect->y) {
          rect->h += rect->y - clip->y1;
          rect->y = clip->y1;
     }

     if (clip->x2 < rect->x + rect->w - 1)
          rect->w = clip->x2 - rect->x + 1;

     if (clip->y2 < rect->y + rect->h - 1)
          rect->h = clip->y2 - rect->y + 1;
}

void ev1_clip_blit( const DFBRegion *clip, DFBRectangle *srect, int *dx, int *dy )
{
     if (clip->x1 > *dx ) {
          srect->w = MIN( (clip->x2 - clip->x1) + 1,
                    (*dx + srect->w) - clip->x1);

          srect->x+= clip->x1 - *dx;
          *dx = clip->x1;
     }
     else if (clip->x2 < *dx + srect->w - 1) {
          srect->w = clip->x2 - *dx + 1;
     }

     if (clip->y1 > *dy ) {
          srect->h = MIN( (clip->y2 - clip->y1) + 1,
                          (*dy + srect->h) - clip->y1);
          srect->y+= clip->y1 - *dy;
          *dy = clip->y1;
     }
     else if (clip->y2 < *dy + srect->h - 1) {
          srect->h = clip->y2 - *dy + 1;
     }
}

void ev1_clip_stretchblit( const DFBRegion *clip, DFBRectangle *srect, DFBRectangle *drect )
{
     DFBRectangle orig_dst = *drect;

     ev1_clip_rectangle( clip, drect );

     if (drect->x != orig_dst.x)
          srect->x += (int)( (drect->x - orig_dst.x) *
                             (srect->w / (float)orig_dst.w) );

     if (drect->y != orig_dst.y)
          srect->y += (int)( (drect->y - orig_dst.y) *
                             (srect->h / (float)orig_dst.h) );

     if (drect->w != orig_dst.w)
          srect->w = (int)( srect->w * (drect->w / (float)orig_dst.w) );

     if (drect->h != orig_dst.h)
          srect->h = (int)( srect->h * (drect->h / (float)orig_dst.h) );
}

void ev1CheckState(void *drv, void *dev, CardState *state, DFBAccelerationMask accel)
{
    FUNCTION_ENTER();

    if(accel & ~(EV1_SUPPORTED_DRAWINGFUNCTIONS | EV1_SUPPORTED_BLITTINGFUNCTIONS)) {
        return;
    }

#ifdef EV1_DEBUG_STATE
    debugPrintState(state);
#endif // DEBUG_STATE

    switch(state->destination->config.format) {
    case DSPF_RGB24:
    case DSPF_ARGB:
        break;
    default:
        DEBUG_PRINT("unsupported format(dst) 0x%08x\n", state->destination->config.format);
        return;
    }

#ifdef EV1_ENABLE_FILLRECTANGLE
    if (!(accel & ~EV1_SUPPORTED_DRAWINGFUNCTIONS)) { 
        if(state->drawingflags & ~EV1_SUPPORTED_DRAWINGFLAGS) {
            return;
        }
        state->accel |= EV1_SUPPORTED_DRAWINGFUNCTIONS;
    }
#endif // EV1_ENABLE_FILLRECTANGLE

    if (!(accel & ~EV1_SUPPORTED_BLITTINGFUNCTIONS)) { 
        switch(state->source->config.format) {
            case DSPF_RGB24:
                break;
            case DSPF_ARGB:
                if(state->blittingflags != DSBLIT_NOFX) {
                    // EMXX_FB_BLITで意図不明の仕様により失敗するケースは除外
                    return;
                }
                break;
            case DSPF_RGB16:
                break;
            default:
                return;
        }

        if(state->blittingflags & ~EV1_SUPPORTED_BLITTINGFLAGS) {
            return;
        }

        state->accel |= EV1_SUPPORTED_BLITTINGFUNCTIONS;
    }
    //DEBUG_PRINT("state->blittingflags = 0x%08x\n", state->blittingflags);

    FUNCTION_EXIT();
}

void ev1SetState(void *drv, void *dev, GraphicsDeviceFuncs *funcs, CardState *state, DFBAccelerationMask accel)
{
    FUNCTION_ENTER();

    EV1DeviceData* dev_data = (EV1DeviceData*)dev;

#ifdef EV1_DEBUG_STATE
    debugPrintState(state);
#endif // EV1_DEBUG_STATE

    if (state->mod_hw) {
        if ((state->mod_hw & SMF_SOURCE) && state->source) {
            dev_data->v_flags &= ~SMF_SOURCE;
        }
        if (state->mod_hw & SMF_DESTINATION) {
            dev_data->v_flags &= ~SMF_DESTINATION;
        }
        if (state->mod_hw & SMF_COLOR) {
            dev_data->v_flags &= ~SMF_COLOR;
        }
        if (state->mod_hw & SMF_CLIP) {
            dev_data->v_flags &= ~SMF_CLIP;
        }
        if (state->mod_hw & SMF_SRC_COLORKEY) {
            dev_data->v_flags &= ~SMF_SRC_COLORKEY;
        }
        if (state->mod_hw & SMF_MATRIX) {
            dev_data->v_flags &= ~SMF_MATRIX;
        }
        if (state->mod_hw & SMF_RENDER_OPTIONS) {
            dev_data->v_flags &= ~(SMF_MATRIX | SMF_RENDER_OPTIONS);
        }
    }

    switch (accel) {
    case DFXL_FILLRECTANGLE:
        ev1_set_dst(dev_data, state);
        ev1_set_color(dev_data, state);
        ev1_set_matrix(dev_data, state);
        ev1_set_clip(dev_data, state);
        state->set |= EV1_SUPPORTED_DRAWINGFUNCTIONS;
        break;
    case DFXL_BLIT:
    case DFXL_STRETCHBLIT:
        ev1_set_src(dev_data, state);
        ev1_set_dst(dev_data, state);
        ev1_set_color(dev_data, state);
        ev1_set_matrix(dev_data, state);
        ev1_set_clip(dev_data, state);
        if (state->blittingflags & DSBLIT_SRC_COLORKEY)
            ev1_set_src_colorkey(dev_data, state);
        state->set |= EV1_SUPPORTED_BLITTINGFUNCTIONS;
        break;
    default:
        DEBUG_PRINT("unexpected drawing/blitting function");
        break;
    }

    dev_data->drawingflags = state->drawingflags;
    dev_data->blittingflags = state->blittingflags;

    state->mod_hw = 0;

    FUNCTION_EXIT();
}

#ifdef EV1_ENABLE_FILLRECTANGLE
// FillRectangleはEMXX_FB_BLITの拡大矩形転送で実現
// SWで塗りつぶした領域(SRC_RECT_W x SRC_RECT_H)を拡大
#define SRC_RECT_W 12
#define SRC_RECT_H 4
bool ev1FillRectangle(void *drv, void *dev, DFBRectangle *rect)
{
    FUNCTION_ENTER();

    EV1DriverData* drv_data = (EV1DriverData*)drv;
    EV1DeviceData* dev_data = (EV1DeviceData*)dev;

    if(is_ev1_fill_rectangle_valid(dev_data) == false) {
        return false;
    }

    DFBRectangle drect = (DFBRectangle) {rect->x, rect->y, rect->w, rect->h};

    // matrix
    if(dev_data->render_options & DSRO_MATRIX) {
        if (dev_data->matrix[1] != 0 || dev_data->matrix[3] != 0) {
             DEBUG_PRINT("warning: rotation or triangle not yet implemented" );
             return false;
        }
        
        int x1, y1, x2, y2;
        x1 = rect->x; y1 = rect->y; x2 = x1 + rect->w; y2 = y1 + rect->h;
        DFB_TRANSFORM(x1, y1, dev_data->matrix, dev_data->affine_matrix);
        DFB_TRANSFORM(x2, y2, dev_data->matrix, dev_data->affine_matrix);
        drect.x = x1; drect.y = y1; drect.w = x2-x1; drect.h = y2-y1;
    }

    // clip
    DFBRegion clip = (DFBRegion) {dev_data->clip_x1, dev_data->clip_y1, dev_data->clip_x2, dev_data->clip_y2};
    ev1_clip_rectangle(&clip, &drect);

    if(drect.w < SRC_RECT_W || drect.h < SRC_RECT_H) {
        DEBUG_PRINT("warning: ev1FillRectangle invalid size. \n");
        return false;
    }
    
    // EMXX_FB_BLITの拡大転送を利用するのでstretchblitの制限対応
    DFBRectangle srect;
    srect.x = drect.x; srect.y = drect.y; srect.w = SRC_RECT_W; srect.h = SRC_RECT_H;
    if(is_ev1_fb_blit_siz_valid(&srect, &drect) == false) {
        return false;
    }

    // source prepare
    char red = 0xff & (dev_data->color_pixel >> 16);
    char green = 0xff & (dev_data->color_pixel >> 8);
    char blue = 0xff & (dev_data->color_pixel);
    char* fb_ptr = (char *)dev_data->dst_map_addr;
    int x = 0;
    int y = 0;
    for(y = 0; y < srect.h; y++) {
        fb_ptr = dev_data->dst_map_addr + dev_data->dst_pitch * y;
        for(x = 0; x < srect.w; x++) {
            *fb_ptr = blue; fb_ptr++;
            *fb_ptr = green; fb_ptr++;
            *fb_ptr = red; fb_ptr++;
        }
    }

    // emmx_fb_blit
    struct emxx_fb_rect src_rect;
    struct emxx_fb_rect dst_rect;
    struct emxx_fb_img src_img;
    struct emxx_fb_img dst_img;

    src_rect.x = (unsigned)srect.x; src_rect.y = (unsigned)srect.y; src_rect.w = (unsigned)srect.w; src_rect.h = (unsigned)srect.h;
    DEBUG_PRINT("src rect: %d, %d, %d, %d\n", src_rect.x, src_rect.y, src_rect.w, src_rect.h);
    dst_rect.x = (unsigned)drect.x; dst_rect.y = (unsigned)drect.y; dst_rect.w = (unsigned)drect.w; dst_rect.h = (unsigned)drect.h;
    DEBUG_PRINT("dst rect: %d, %d, %d, %d\n", dst_rect.x, dst_rect.y, dst_rect.w, dst_rect.h);
    src_img.width = (unsigned)dev_data->dst_width; src_img.height = (unsigned)dev_data->dst_height; src_img.format = convert_format(dev_data->dst_format); src_img.offset = dev_data->dst_addr - drv_data->base_addr; src_img.memory_id = drv_data->fd;
    DEBUG_PRINT("src img: width=%d, height=%d, offset=%d\n", src_img.width, src_img.height, src_img.offset);
    dst_img.width = (unsigned)dev_data->dst_width; dst_img.height = (unsigned)dev_data->dst_height; dst_img.format = convert_format(dev_data->dst_format); dst_img.offset = dev_data->dst_addr - drv_data->base_addr; dst_img.memory_id = drv_data->fd;
    DEBUG_PRINT("dst img: width=%d, height=%d, offset=%d\n", dst_img.width, dst_img.height, dst_img.offset);

    struct emxx_fb_blit_req blit_req;
    blit_req.src_rect = src_rect;
    blit_req.dst_rect = dst_rect;
    blit_req.src = src_img;
    blit_req.dst = dst_img;
    blit_req.alpha = EMXX_FB_ALPHA_NOP;
    blit_req.transp_mask = EMXX_FB_TRANSP_NOP;
    blit_req.flags = EMXX_FB_ROT_NOP;

    struct {
        uint32_t count;
        struct emxx_fb_blit_req req[1];
    } req_list;
    memset(&req_list, 0, sizeof(req_list));
    req_list.count = 1;
    req_list.req[0] = blit_req;

    if(ioctl(drv_data->fd, EMXX_FB_BLIT, &req_list)) {
        DEBUG_PRINT("EMXX_FB_BLIT false !\n");
        return false;
    }
    //printf("***** EMXX_FB_BLIT success *****\n");
    
    FUNCTION_EXIT();
    return true;
}
#endif // EV1_ENABLE_FILLRECTANGLE

bool ev1Blit(void *drv, void *dev, DFBRectangle *srect, int dx, int dy)
{
    FUNCTION_ENTER();

    EV1DriverData* drv_data = (EV1DriverData*)drv;
    EV1DeviceData* dev_data = (EV1DeviceData*)dev;

    DFBRectangle src_rect = *srect;
    DFBRectangle dst_rect;

//    if (!dfb_config->hw_blit) {
//        return false;
//    }

    if(is_ev1_blit_valid(dev_data) == false && is_ev1_blit_rgb16_valid(dev_data) == false) {
        return false;
    }

    // matrix
    if(dev_data->render_options & DSRO_MATRIX) {
        if (dev_data->matrix[0] < 0  || dev_data->matrix[1] != 0 ||
            dev_data->matrix[3] != 0 || dev_data->matrix[4] < 0  ||
            dev_data->matrix[6] != 0 || dev_data->matrix[7] != 0) {
             DEBUG_PRINT("warning: rotation not yet implemented" );
             return false;
        }
        
        int x1, y1, x2, y2;
        x1 = dx; y1 = dy; x2 = dx + src_rect.w; y2 = dy + src_rect.h;
        DFB_TRANSFORM(x1, y1, dev_data->matrix, dev_data->affine_matrix);
        DFB_TRANSFORM(x2, y2, dev_data->matrix, dev_data->affine_matrix);
        dst_rect = (DFBRectangle) {x1, y1, x2-x1, y2-y1};
        dx = dst_rect.x;
        dy = dst_rect.y;

    }

    // clip
    DEBUG_PRINT("src: 0x%08x, %d, %d, %d, %d\n", dev_data->src_addr, src_rect.x, src_rect.y, src_rect.w, src_rect.h);
    DEBUG_PRINT("dst: 0x%08x, %d, %d\n", dev_data->dst_addr, dx, dy);
    DEBUG_PRINT("clip: (%d, %d), (%d, %d)\n", dev_data->clip_x1, dev_data->clip_y1, dev_data->clip_x2, dev_data->clip_y2);

    DFBRegion clip = (DFBRegion) {dev_data->clip_x1, dev_data->clip_y1, dev_data->clip_x2, dev_data->clip_y2};

    ev1_clip_blit(&clip, &src_rect, &dx, &dy);

    dst_rect.x = dx;
    dst_rect.y = dy;
    dst_rect.w = src_rect.w;
    dst_rect.h = src_rect.h;

    DEBUG_PRINT("src after: 0x%08x, %d, %d, %d, %d\n", dev_data->src_addr, src_rect.x, src_rect.y, src_rect.w, src_rect.h);
    DEBUG_PRINT("dst after: 0x%08x, %d, %d, %d, %d\n", dev_data->dst_addr, dst_rect.x, dst_rect.y, dst_rect.w, dst_rect.h);

    // color alpha
    unsigned int color_alpha = 0xff & (dev_data->color_pixel >> 24);
    if(!(dev_data->blittingflags & DSBLIT_BLEND_COLORALPHA) || color_alpha == 0xff) {
         color_alpha = EMXX_FB_ALPHA_NOP;
    }
    DEBUG_PRINT("color_alpha = %d\n", color_alpha);

    // emxx_fb_blit
    if(ev1_fb_blit(drv_data, dev_data, &src_rect, &dst_rect, color_alpha) == false) {
        return false;
    }

    FUNCTION_EXIT();
    return true;
}

bool ev1StretchBlit(void *drv, void *dev, DFBRectangle *srect, DFBRectangle *drect)
{
    FUNCTION_ENTER();

    EV1DriverData* drv_data = (EV1DriverData*)drv;
    EV1DeviceData* dev_data = (EV1DeviceData*)dev;

    if(is_ev1_stretch_blit_valid(dev_data) == false) {
        return false;
    }

    DFBRectangle dst_rect = (DFBRectangle) {drect->x, drect->y, drect->w, drect->h};
    DFBRectangle src_rect = (DFBRectangle) {srect->x, srect->y, srect->w, srect->h};

    // matrix
    if(dev_data->render_options & DSRO_MATRIX) {
        if (dev_data->matrix[0] < 0  || dev_data->matrix[1] != 0 ||
            dev_data->matrix[3] != 0 || dev_data->matrix[4] < 0  ||
            dev_data->matrix[6] != 0 || dev_data->matrix[7] != 0) {
             DEBUG_PRINT("warning: rotation not yet implemented" );
             return false;
        }

        int x1, y1, x2, y2;
        x1 = drect->x; y1 = drect->y; x2 = x1 + drect->w; y2 = y1 + drect->h;
        DFB_TRANSFORM(x1, y1, dev_data->matrix, dev_data->affine_matrix);
        DFB_TRANSFORM(x2, y2, dev_data->matrix, dev_data->affine_matrix);
        dst_rect.x = x1; dst_rect.y = y1; dst_rect.w = x2 - x1; dst_rect.h = y2 - y1;
    }

    // clip
    DEBUG_PRINT("src: 0x%08x, %d, %d, %d, %d\n", dev_data->src_addr, src_rect.x, src_rect.y, src_rect.w, src_rect.h);
    DEBUG_PRINT("dst: 0x%08x, %d, %d, %d, %d\n", dev_data->dst_addr, dst_rect.x, dst_rect.y, dst_rect.w, dst_rect.h);
    DEBUG_PRINT("clip: (%d, %d), (%d, %d)\n", dev_data->clip_x1, dev_data->clip_y1, dev_data->clip_x2, dev_data->clip_y2);

    DFBRegion clip = (DFBRegion) {dev_data->clip_x1, dev_data->clip_y1, dev_data->clip_x2, dev_data->clip_y2};

    ev1_clip_stretchblit(&clip, &src_rect, &dst_rect);

    DEBUG_PRINT("src after: 0x%08x, %d, %d, %d, %d\n", dev_data->src_addr, src_rect.x, src_rect.y, src_rect.w, src_rect.h);
    DEBUG_PRINT("dst after: 0x%08x, %d, %d, %d, %d\n", dev_data->dst_addr, dst_rect.x, dst_rect.y, dst_rect.w, dst_rect.h);

    // HW仕様
    if(is_ev1_fb_blit_siz_valid(&src_rect, &dst_rect) == false) {
        return false;
    }

    // color alpha
    unsigned int color_alpha = 0xff & (dev_data->color_pixel >> 24);
    if(!(dev_data->blittingflags & DSBLIT_BLEND_COLORALPHA) || color_alpha == 0xff) {
         color_alpha = EMXX_FB_ALPHA_NOP;
    }
    DEBUG_PRINT("color_alpha = %d\n", color_alpha);

    // emxx_fb_blit
    if(ev1_fb_blit(drv_data, dev_data, &src_rect, &dst_rect, color_alpha) == false) {
        return false;
    }
    
    FUNCTION_EXIT();
    return true;
}

void ev1_set_src(EV1DeviceData *dev, CardState *state)
{
    if (dev->v_flags & SMF_SOURCE)
        return;

    dev->src_addr = state->src.phys;
    DEBUG_PRINT("src_addr=%p\n", dev->src_addr);
    dev->src_pitch = state->src.pitch;
    dev->src_format = state->src.buffer->format;
    dev->src_depth = DFB_BYTES_PER_PIXEL(dev->src_format);
    dev->src_width = state->source->config.size.w;
    dev->src_height = state->source->config.size.h;
    DEBUG_PRINT("src_width=%d height=%d\n", dev->src_width, dev->src_height);
    dev->v_flags |= SMF_SOURCE;
    dev->src_map_addr = state->src.addr;
}

void ev1_set_dst(EV1DeviceData *dev, CardState *state)
{
    if (dev->v_flags & SMF_DESTINATION)
        return;

    dev->dst_addr = state->dst.phys;
    DEBUG_PRINT("dst_addr=%p\n", dev->dst_addr);
    dev->dst_pitch = state->dst.pitch;
    dev->dst_format = state->dst.buffer->format;
    dev->dst_depth = DFB_BYTES_PER_PIXEL(dev->dst_format);
    dev->dst_width = state->destination->config.size.w;
    dev->dst_height = state->destination->config.size.h;
    DEBUG_PRINT("dst_width=%d height=%d\n", dev->dst_width, dev->dst_height);
    dev->v_flags |= SMF_DESTINATION;
    dev->dst_map_addr = state->dst.addr;
}

void ev1_set_color(EV1DeviceData *dev, CardState *state)
{
    if (dev->v_flags & SMF_COLOR)
        return;

    switch(dev->dst_format) {
    case DSPF_RGB24:
        DEBUG_PRINT("dst surface set color (A, R, G, B) = (0x%02x, 0x%02x, 0x%02x, 0x%02x)\n", state->color.a, state->color.r, state->color.g, state->color.b);
        dev->color_pixel = PIXEL_ARGB(state->color.a, state->color.r, state->color.g, state->color.b);
    }
    dev->v_flags |= SMF_COLOR;
}

void ev1_set_matrix(EV1DeviceData *dev, CardState *state)
{
    if (dev->v_flags & SMF_MATRIX)
        return;

    int i = 0;
    for(i = 0; i < 9; i++) {
        dev->matrix[i] = state->matrix[i];
    }
    dev->affine_matrix = state->affine_matrix;
    dev->render_options = state->render_options;
    dev->v_flags |= SMF_MATRIX;
}

void ev1_set_clip(EV1DeviceData *dev, CardState *state)
{
    if (dev->v_flags & SMF_CLIP)
        return;

    dev->clip_x1 = state->clip.x1;
    dev->clip_x2 = state->clip.x2;
    dev->clip_y1 = state->clip.y1;
    dev->clip_y2 = state->clip.y2;
    dev->v_flags |= SMF_CLIP;

    DEBUG_PRINT("state->clip (%d, %d), (%d, %d)\n", state->clip.x1, state->clip.y1, state->clip.x2, state->clip.y2);
}

void ev1_set_src_colorkey(EV1DeviceData *dev, CardState *state)
{
    if (dev->v_flags & SMF_SRC_COLORKEY)
        return;

    dev->src_colorkey = state->src_colorkey;
    dev->v_flags |= SMF_SRC_COLORKEY;
}

inline unsigned int convert_format(DFBSurfacePixelFormat format)
{
    switch(format) {
    case DSPF_ARGB:
        return EMXX_FB_ABGR_8888;
    case DSPF_RGB24:
        return EMXX_FB_BGR_888;
    case DSPF_RGB16:
        return EMXX_FB_RGB_565;
    default:
        return EMXX_FB_IMGTYPE_LIMIT;
    }
}

/* debug function */
#if defined(EV1_DEBUG)
inline void debugPrintFormat(DFBSurfacePixelFormat format)
{
        switch(format) {
        case DSPF_UNKNOWN:
                fprintf(stderr, "unknown\n");
                break;
        case DSPF_ARGB1555:
                fprintf(stderr, "argb1555\n");
                break;
        case DSPF_RGB16:
                fprintf(stderr, "rgb16\n");
                break;
        case DSPF_RGB24:
                fprintf(stderr, "rgb24\n");
                break;
        case DSPF_RGB32:
                fprintf(stderr, "rgb32\n");
                break;
        case DSPF_ARGB:
                fprintf(stderr, "argb\n");
                break;
        case DSPF_A8:
                fprintf(stderr, "a8\n");
                break;
        case DSPF_YUY2:
                fprintf(stderr, "yuy2\n");
                break;
        case DSPF_RGB332:
                fprintf(stderr, "rgb332\n");
                break;
        default:
                fprintf(stderr, "else\n");
                break;
        }
}

inline void debugPrintState(CardState* state)
{
        if (state->source) {
                fprintf(stderr, "source format: ");
                debugPrintFormat(state->source->config.format);

        }

        if (state->destination) {
                fprintf(stderr, "destination format: ");
                debugPrintFormat(state->destination->config.format);
        }

        fprintf(stderr, "drawingflags: %08x, blittingflags: %08x\n", state->drawingflags, state->blittingflags);

}
#endif // EV1_DEBUG

