/*
 *  Copyright (C) 2011 Sony Corporation.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA
 */

#include <stdio.h>

#include <directfb.h>

#include <direct/debug.h>
#include <direct/messages.h>

#include <core/gfxcard.h>
#include <core/system.h>
#include <core/coretypes.h>
#include <core/graphics_driver.h>

#include <misc/conf.h>

#include <fbdev/fbdev.h>
#include "ev1_gfxdriver.h"
#include "ev1_2d.h"

D_DEBUG_DOMAIN( EV1_Driver, "EV1/Driver", "EV1 graphics driver" );
DFB_GRAPHICS_DRIVER( ev1 );

static int
driver_probe( CoreGraphicsDevice *device )
{
    FUNCTION_ENTER(); 

    FUNCTION_EXIT(); 
    return 1;
}

static void
driver_get_info( CoreGraphicsDevice *device,
                 GraphicsDriverInfo *info )
{
    FUNCTION_ENTER(); 

    /* fill driver info structure */
    snprintf( info->name,
        DFB_GRAPHICS_DRIVER_INFO_NAME_LENGTH,
        "EV1 Driver" );

    snprintf( info->vendor,
        DFB_GRAPHICS_DRIVER_INFO_VENDOR_LENGTH,
        "SONY Corporation" );
    
    info->version.major = 1;
    info->version.minor = 3;

    info->driver_data_size = sizeof(EV1DriverData);
    info->device_data_size = sizeof(EV1DeviceData);

    FUNCTION_EXIT(); 
}

static DFBResult
driver_init_driver( CoreGraphicsDevice  *device,
                    GraphicsDeviceFuncs *funcs,
                    void                *driver_data,
                    void                *device_data,
                    CoreDFB             *core )
{
    FUNCTION_ENTER(); 

    EV1DriverData* drv = (EV1DriverData*)driver_data;

    // system=fbdev前提
    FBDev *dfb_fbdev = dfb_system_data();
    drv->fd = dfb_fbdev->fd;
    drv->base_addr = dfb_fbdev->shared->fix.smem_start;
    DEBUG_PRINT("frame buffer fd = %d\n", drv->fd);
    DEBUG_PRINT("frame buffer size = %x\n", dfb_fbdev->shared->fix.smem_start);
    DEBUG_PRINT("frame buffer len = %d\n", dfb_fbdev->shared->fix.smem_len);

    /* initialize function pointers */
    funcs->CheckState    = ev1CheckState;
    funcs->SetState      = ev1SetState;
#ifdef EV1_ENABLE_FILLRECTANGLE
    funcs->FillRectangle = ev1FillRectangle;
#endif // EV1_ENABLE_FILLRECTANGLE
    funcs->Blit          = ev1Blit;
    funcs->StretchBlit   = ev1StretchBlit;

    FUNCTION_EXIT(); 
    return DFB_OK;
}

static DFBResult
driver_init_device( CoreGraphicsDevice *device,
                    GraphicsDeviceInfo *device_info,
                    void               *driver_data,
                    void               *device_data )
{
    FUNCTION_ENTER(); 

    /* fill device info */
    snprintf( device_info->vendor, DFB_GRAPHICS_DEVICE_INFO_VENDOR_LENGTH, "Renesas Electronics Corporation" );
    snprintf( device_info->name,   DFB_GRAPHICS_DEVICE_INFO_NAME_LENGTH,   "EMMA EV1 ES3.0" );

    device_info->caps.flags    = CCF_CLIPPING | CCF_RENDEROPTS;
    device_info->caps.accel    = EV1_SUPPORTED_DRAWINGFUNCTIONS |
                                 EV1_SUPPORTED_BLITTINGFUNCTIONS;
    device_info->caps.drawing  = EV1_SUPPORTED_DRAWINGFLAGS;
    device_info->caps.blitting = EV1_SUPPORTED_BLITTINGFLAGS;

    FUNCTION_EXIT();
    return DFB_OK;
}

static void
driver_close_device( CoreGraphicsDevice *device,
                     void               *driver_data,
                     void               *device_data )
{
    FUNCTION_ENTER(); 
    FUNCTION_EXIT(); 
}

static void
driver_close_driver( CoreGraphicsDevice *device,
                     void               *driver_data )
{
    FUNCTION_ENTER(); 
    FUNCTION_EXIT(); 
}

