#ifndef __ASM_SPINLOCK_H
#define __ASM_SPINLOCK_H

#if __LINUX_ARM_ARCH__ < 6
#error SMP not supported on pre-ARMv6 CPUs
#endif

static inline void dsb_sev(void)
{
#if __LINUX_ARM_ARCH__ >= 7
	__asm__ __volatile__ (
		"dsb\n"
		"sev"
	);
#elif defined(CONFIG_CPU_32v6K)
	__asm__ __volatile__ (
		"mcr p15, 0, %0, c7, c10, 4\n"
		"sev"
		: : "r" (0)
	);
#endif
}

/*
 * ARMv6 Spin-locking.
 *
 * We exclusively read the old value.  If it is zero, we may have
 * won the lock, so we try exclusively storing it.  A memory barrier
 * is required after we get a lock, and before we release it, because
 * V6 CPUs are assumed to have weakly ordered memory.
 *
 * Unlocked value: 0
 * Locked value: 1
 */

#define __raw_spin_is_locked(x)		((x)->lock != 0)
#define __raw_spin_unlock_wait(lock) \
	do { while (__raw_spin_is_locked(lock)) cpu_relax(); } while (0)

#define __raw_spin_lock_flags(lock, flags) __raw_spin_lock(lock)

#ifdef CONFIG_ARM_ERRATA_351422
#define SPINLOCK_BACKOFF_DELAY				\
	"	mrc     p15, 0, %0, c0, c0, 5\n"	\
	"	and	%0, %0, #0xf\n"			\
	"2:	subs	%0, %0, #1\n"			\
	"	bpl	2b\n"
#endif

static inline void __raw_spin_lock(__raw_spinlock_t *lock)
{
	unsigned long tmp;

	__asm__ __volatile__(
"1:\n"
"	ldrex	%0, [%1]\n"
"	teq	%0, #0\n"
#ifdef CONFIG_CPU_32v6K
"	wfene\n"
#endif
"	strexeq	%0, %2, [%1]\n"
"	teqeq	%0, #0\n"
#ifndef CONFIG_ARM_ERRATA_351422
"	bne	1b"
#else
"	beq	3f\n"
	SPINLOCK_BACKOFF_DELAY
"	b	1b\n"
"3:"
#endif
	: "=&r" (tmp)
	: "r" (&lock->lock), "r" (1)
	: "cc");

	smp_mb();
}

static inline int __raw_spin_trylock(__raw_spinlock_t *lock)
{
	unsigned long tmp;

	__asm__ __volatile__(
"	ldrex	%0, [%1]\n"
"	teq	%0, #0\n"
"	strexeq	%0, %2, [%1]"
	: "=&r" (tmp)
	: "r" (&lock->lock), "r" (1)
	: "cc");

	if (tmp == 0) {
		smp_mb();
		return 1;
	} else {
#ifdef CONFIG_ARM_ERRATA_351422
		__asm__ __volatile__(
		SPINLOCK_BACKOFF_DELAY
		: "=&r" (tmp)
		:
		: "cc");
#endif
		return 0;
	}
}

static inline void __raw_spin_unlock(__raw_spinlock_t *lock)
{
	smp_mb();

	__asm__ __volatile__(
"	str	%1, [%0]\n"
	:
	: "r" (&lock->lock), "r" (0)
	: "cc");

	dsb_sev();
}

/*
 * RWLOCKS
 *
 *
 * Write locks are easy - we just set bit 31.  When unlocking, we can
 * just write zero since the lock is exclusively held.
 */

static inline void __raw_write_lock(__raw_rwlock_t *rw)
{
	unsigned long tmp;

	__asm__ __volatile__(
"1:\n"
"	ldrex	%0, [%1]\n"
"	teq	%0, #0\n"
#ifdef CONFIG_CPU_32v6K
"	wfene\n"
#endif
"	strexeq	%0, %2, [%1]\n"
"	teq	%0, #0\n"
#ifndef CONFIG_ARM_ERRATA_351422
"	bne	1b"
#else
"	beq	3f\n"
	SPINLOCK_BACKOFF_DELAY
"	b	1b\n"
"3:"
#endif
	: "=&r" (tmp)
	: "r" (&rw->lock), "r" (0x80000000)
	: "cc");

	smp_mb();
}

static inline int __raw_write_trylock(__raw_rwlock_t *rw)
{
	unsigned long tmp;

	__asm__ __volatile__(
"1:	ldrex	%0, [%1]\n"
"	teq	%0, #0\n"
"	strexeq	%0, %2, [%1]"
	: "=&r" (tmp)
	: "r" (&rw->lock), "r" (0x80000000)
	: "cc");

	if (tmp == 0) {
		smp_mb();
		return 1;
	} else {
#ifdef CONFIG_ARM_ERRATA_351422
		__asm__ __volatile__(
		SPINLOCK_BACKOFF_DELAY
		: "=&r" (tmp)
		:
		: "cc");
#endif
		return 0;
	}
}

static inline void __raw_write_unlock(__raw_rwlock_t *rw)
{
	smp_mb();

	__asm__ __volatile__(
	"str	%1, [%0]\n"
	:
	: "r" (&rw->lock), "r" (0)
	: "cc");

	dsb_sev();
}

/* write_can_lock - would write_trylock() succeed? */
#define __raw_write_can_lock(x)		((x)->lock == 0)

/*
 * Read locks are a bit more hairy:
 *  - Exclusively load the lock value.
 *  - Increment it.
 *  - Store new lock value if positive, and we still own this location.
 *    If the value is negative, we've already failed.
 *  - If we failed to store the value, we want a negative result.
 *  - If we failed, try again.
 * Unlocking is similarly hairy.  We may have multiple read locks
 * currently active.  However, we know we won't have any write
 * locks.
 */
static inline void __raw_read_lock(__raw_rwlock_t *rw)
{
	unsigned long tmp, tmp2;

	__asm__ __volatile__(
"1:\n"
"	ldrex	%0, [%2]\n"
"	adds	%0, %0, #1\n"
"	strexpl	%1, %0, [%2]\n"
#ifdef CONFIG_CPU_32v6K
"	wfemi\n"
#endif
"	rsbpls	%0, %1, #0\n"
#ifndef CONFIG_ARM_ERRATA_351422
"	bmi	1b"
#else
"	bpl	3f\n"
	SPINLOCK_BACKOFF_DELAY
"	b	1b\n"
"3:"
#endif
	: "=&r" (tmp), "=&r" (tmp2)
	: "r" (&rw->lock)
	: "cc");

	smp_mb();
}

static inline void __raw_read_unlock(__raw_rwlock_t *rw)
{
	unsigned long tmp, tmp2;

	smp_mb();

	__asm__ __volatile__(
"1:\n"
"	ldrex	%0, [%2]\n"
"	sub	%0, %0, #1\n"
"	strex	%1, %0, [%2]\n"
"	teq	%1, #0\n"
#ifndef CONFIG_ARM_ERRATA_351422
"	bne	1b"
#else
"	beq	3f\n"
	SPINLOCK_BACKOFF_DELAY
"	b	1b\n"
"3:"
#endif
	: "=&r" (tmp), "=&r" (tmp2)
	: "r" (&rw->lock)
	: "cc");

	if (tmp == 0)
		dsb_sev();
}

static inline int __raw_read_trylock(__raw_rwlock_t *rw)
{
	unsigned long tmp, tmp2 = 1;

	__asm__ __volatile__(
"1:	ldrex	%0, [%2]\n"
"	adds	%0, %0, #1\n"
"	strexpl	%1, %0, [%2]\n"
	: "=&r" (tmp), "+r" (tmp2)
	: "r" (&rw->lock)
	: "cc");

	if (tmp2 == 0) {
		smp_mb();
		return 1;
	} else {
#ifdef CONFIG_ARM_ERRATA_351422
		__asm__ __volatile__(
		SPINLOCK_BACKOFF_DELAY
		: "=&r" (tmp2)
		:
		: "cc");
#endif
		return 0;
	}
}

/* read_can_lock - would read_trylock() succeed? */
#define __raw_read_can_lock(x)		((x)->lock < 0x80000000)

#define __raw_spin_relax(lock)	cpu_relax()
#define __raw_read_relax(lock)	cpu_relax()
#define __raw_write_relax(lock)	cpu_relax()

#endif /* __ASM_SPINLOCK_H */
