/*
 *  linux/arch/arm/kernel/irq.c
 *
 *  Copyright (C) 1992 Linus Torvalds
 *  Modifications for ARM processor Copyright (C) 1995-2000 Russell King.
 *
 *  Support for Dynamic Tick Timer Copyright (C) 2004-2005 Nokia Corporation.
 *  Dynamic Tick Timer written by Tony Lindgren <tony@atomide.com> and
 *  Tuukka Tikkanen <tuukka.tikkanen@elektrobit.com>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 *  This file contains the code used by various IRQ handling routines:
 *  asking for different IRQ's should be done through these routines
 *  instead of just grabbing them. Thus setups with different IRQ numbers
 *  shouldn't result in any weird surprises, and installing new handlers
 *  should be easier.
 *
 *  IRQ's are in fact implemented a bit like signal handlers for the kernel.
 *  Naturally it's not a 1:1 relation, but there are similarities.
 */
#include <linux/kernel_stat.h>
#include <linux/module.h>
#include <linux/signal.h>
#include <linux/ioport.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/slab.h>
#include <linux/random.h>
#include <linux/smp.h>
#include <linux/init.h>
#include <linux/seq_file.h>
#include <linux/errno.h>
#include <linux/list.h>
#include <linux/kallsyms.h>
#include <linux/proc_fs.h>
#include <linux/percpu.h>
#include <linux/rt_trace_lite.h>
#include <linux/rt_trace_lite_irq.h>

#include <asm/system.h>
#include <asm/mach/irq.h>
#include <asm/mach/time.h>

/*
 * No architecture-specific irq_finish function defined in arm/arch/irqs.h.
 */
#ifndef irq_finish
#define irq_finish(irq) do { } while (0)
#endif

void (*init_arch_irq)(void) __initdata = NULL;
unsigned long irq_err_count;

static void __handle_irq(unsigned int irq)
{
#ifdef CONFIG_THREAD_MONITOR
	s64 t1 = sched_clock();
#endif

	generic_handle_irq(irq);

#ifdef CONFIG_THREAD_MONITOR
	{
		int cpu = raw_smp_processor_id();
		extern int trace_cpu;
		extern void add_trace_entry(int cpu, struct task_struct *prev, struct task_struct *next, int data);

		if (trace_cpu & (1 << cpu)) {
			s64 t2 = sched_clock();
			u32 delta = t2 - t1;
			delta /= 1000;
			add_trace_entry(cpu, current, current, irq | (delta << 8));
		}
	}
#endif

}

#ifdef CONFIG_SNSC_DEBUG_IRQ_OVERHEAD_ONCE

static DEFINE_PER_CPU(unsigned long, count_irq);
static DEFINE_PER_CPU(unsigned long, count_irq_enter);
#ifdef CONFIG_SMP
static DEFINE_PER_CPU(unsigned long, ipi_lt_without_irq);
static DEFINE_PER_CPU(unsigned long, in_ipi_count);
unsigned long get_ipi_count(int);
#endif

static void show_irq_ipi_lt(struct seq_file *p)
{
	int cpu;
	unsigned long count_irq;
	unsigned long count_irq_enter;
#ifdef CONFIG_SMP
	unsigned long ipi_lt;
	unsigned long ipi_lt_no_irq;
#endif

	seq_putc(p, '\n');

#ifdef CONFIG_SMP
	seq_printf(p, "ipi/lt                         ");
	for_each_present_cpu(cpu) {
		ipi_lt = get_ipi_count(cpu) + irq_stat[cpu].local_timer_irqs;
		seq_printf(p, " %10lu", ipi_lt);
	}
	seq_putc(p, '\n');

	seq_printf(p, "ipi/lt with no external irq    ");
	for_each_present_cpu(cpu)
		seq_printf(p, " %10lu", per_cpu(ipi_lt_without_irq, cpu));
	seq_putc(p, '\n');

	seq_printf(p, "                               ");
	for_each_present_cpu(cpu) {
		ipi_lt = get_ipi_count(cpu) + irq_stat[cpu].local_timer_irqs;
		ipi_lt_no_irq = per_cpu(ipi_lt_without_irq, cpu);
		seq_printf(p, " %9lu%%",
			(ipi_lt_no_irq * 100) / ((ipi_lt) ? ipi_lt : 1));
	}
	seq_putc(p, '\n');
	seq_printf(p, "(these ipi/lt have more overhead with CONFIG_SNSC_IRQ_OVERHEAD_ONCE)\n\n");

	seq_printf(p, "IPI pending on do_IPI() return ");
	for_each_present_cpu(cpu)
		seq_printf(p, " %10lu", per_cpu(in_ipi_count, cpu));
	seq_putc(p, '\n');
	seq_printf(p, "(these would have less overhead if ack_IPI_irq() moved to do_IPI())\n\n");
#endif
	seq_printf(p, "external irq                   ");
	for_each_present_cpu(cpu)
		seq_printf(p, " %10lu", per_cpu(count_irq, cpu));
	seq_putc(p, '\n');


	seq_printf(p, "irq_enter() calls              ");
	for_each_present_cpu(cpu)
		seq_printf(p, " %10lu", per_cpu(count_irq_enter, cpu));
	seq_putc(p, '\n');


	seq_printf(p, "irq_enter() calls avoided:     ");
	for_each_present_cpu(cpu)
		seq_printf(p, " %10lu",
			per_cpu(count_irq, cpu) -
			per_cpu(count_irq_enter, cpu));
	seq_putc(p, '\n');


	seq_printf(p, "                               ");
	for_each_present_cpu(cpu) {
		count_irq       = per_cpu(count_irq,       cpu);
		count_irq_enter = per_cpu(count_irq_enter, cpu);
		seq_printf(p, " %9lu%%",
			((count_irq - count_irq_enter) * 100) /
			((count_irq) ? count_irq : 1));
	}
	seq_putc(p, '\n');
}
#endif


int show_interrupts(struct seq_file *p, void *v)
{
	int i = *(loff_t *) v, cpu;
	struct irqaction * action;
	unsigned long flags;

	if (i == 0) {
		char cpuname[12];

		seq_printf(p, "    ");
		for_each_present_cpu(cpu) {
			sprintf(cpuname, "CPU%d", cpu);
			seq_printf(p, " %10s", cpuname);
		}
		seq_putc(p, '\n');
	}

	if (i < NR_IRQS) {
		spin_lock_irqsave(&irq_desc[i].lock, flags);
		action = irq_desc[i].action;
		if (!action)
			goto unlock;

		seq_printf(p, "%3d: ", i);
		for_each_present_cpu(cpu)
			seq_printf(p, "%10u ", kstat_irqs_cpu(i, cpu));
		seq_printf(p, " %10s", irq_desc[i].chip->name ? : "-");
		seq_printf(p, "  %s", action->name);
		for (action = action->next; action; action = action->next)
			seq_printf(p, ", %s", action->name);

		seq_putc(p, '\n');
unlock:
		spin_unlock_irqrestore(&irq_desc[i].lock, flags);
	} else if (i == NR_IRQS) {
#ifdef CONFIG_ARCH_ACORN
		show_fiq_list(p, v);
#endif
#ifdef CONFIG_SMP
		show_ipi_list(p);
		show_local_irqs(p);
#endif
		seq_printf(p, "Err: %10lu\n", irq_err_count);

#ifdef CONFIG_SNSC_DEBUG_IRQ_OVERHEAD_ONCE
 		show_irq_ipi_lt(p);
#endif
	}
	return 0;
}


#ifdef CONFIG_SNSC_DEBUG_IRQ_DURATION
int show_irq_stat(struct seq_file *p, void *v)
{
	int i = *(loff_t *) v, cpu;
	struct irqaction * action;
	unsigned long flags;

	if (i == 0) {
		char cpuname[12];

		seq_printf(p, "  ");
		for_each_present_cpu(cpu) {
			sprintf(cpuname, "CPU%d", cpu);
			seq_printf(p, " %10s               ", cpuname);
		}

		seq_putc(p, '\n');
		seq_printf(p, "     ");
		for_each_present_cpu(cpu) {
			seq_printf(p, "      count  min  avg  max");
		}
		seq_putc(p, '\n');
	}

	if (i < NR_IRQS) {

		spin_lock_irqsave(&irq_desc[i].lock, flags);
		action = irq_desc[i].action;
		if (!action)
			goto unlock;

		seq_printf(p, "%3d: ", i);

		show_rt_trace_irq_stat(p, i);

		seq_printf(p, "  %s", action->name);
		for (action = action->next; action; action = action->next)
			seq_printf(p, ", %s", action->name);

		seq_putc(p, '\n');
unlock:
		spin_unlock_irqrestore(&irq_desc[i].lock, flags);

	} else if (i == NR_IRQ_IPI) {

		seq_printf(p, "IPI: ");
		show_rt_trace_irq_stat(p, i);
		seq_printf(p, "  do_IPI");
		seq_putc(p, '\n');

	} else if (i == NR_IRQ_LOC) {

		seq_printf(p, "LOC: ");
		show_rt_trace_irq_stat(p, i);
		seq_printf(p, "  do_local_timer");
		seq_putc(p, '\n');

	} else if (i == NR_IRQ_INV) {

		seq_printf(p, "INV: ");
		show_rt_trace_irq_stat(p, i);
		seq_printf(p, "  invalid irq");
		seq_putc(p, '\n');

	}
	return 0;
}
#endif


#ifdef CONFIG_SNSC_IRQ_OVERHEAD_ONCE

#include <mach/entry-macro.h>

asmlinkage void __exception notrace asm_do_IRQ_IPI_LT(struct pt_regs *regs)
{
	struct pt_regs *old_regs = set_irq_regs(regs);
	unsigned int irq;
	int in_irq = 0;
#if defined(CONFIG_SNSC_DEBUG_IRQ_OVERHEAD_ONCE) && defined(CONFIG_SMP)
	int had_external_interrupt = 0;
	int in_ipi = 0;
#endif

	/*
	 * Why call irq_exit() in cases IRQ_IS_IP and IRQ_IS_LOCALTIMER?
	 * waiting until falling out of the while loop would allow a scenario:
	 *
	 *   irq A:
	 *     irq_enter()
	 *     handle irq A
	 *   IPI:
	 *     handle IPI
	 *   irq B:
	 *     handle irq B
	 *   irq_exit()
	 *
	 * instead of:
	 *
	 *   irq A:
	 *     irq_enter()
	 *     handle irq A
	 *   IPI:
	 *     irq_exit()
	 *     handle IPI
	 *   irq B:
	 *     irq_enter()
	 *     handle irq B
	 *   irq_exit()
	 *
	 * which would save an irq_exit() and an irq_enter().  But that would
	 * put the IPI handling between the irq_enter() and the irq_exit().
	 * That might work with the current code, but any changes to do_IPI()
	 * and do_localtimer() that does not expect a subsequent call to
	 * irq_exit() could cause problems.  I am being exceedingly cautious
	 * to minimize the amount of code flow change in
	 * CONFIG_SNSC_IRQ_OVERHEAD_ONCE.
	 */

	get_irq_preamble();
	do {
		irq = get_irq();

		if (IRQ_IS_DONE(irq)) {

			/* done, no more irqs */
			irq = 0;

		} else if (IRQ_IS_EXTERNAL_INTERRUPT(irq)) {
#ifdef CONFIG_SNSC_DEBUG_IRQ_OVERHEAD_ONCE
			per_cpu(count_irq, smp_processor_id())++;
#ifdef CONFIG_SMP
			had_external_interrupt = 1;
#endif
#endif
			if (!in_irq) {
				in_irq = 1;
#ifdef CONFIG_SNSC_DEBUG_IRQ_OVERHEAD_ONCE
				per_cpu(count_irq_enter, smp_processor_id())++;
#endif
				irq_enter();
			}

			/*
			 * Some hardware gives randomly wrong interrupts.
			 * Rather than crashing, do something sensible.
			 */
			if (unlikely(irq >= NR_IRQS)) {
				if (printk_ratelimit())
					printk(KERN_WARNING "Bad IRQ%u\n", irq);
				ack_bad_irq(irq);
			} else {
#ifdef CONFIG_SNSC_LITE_IRQSOFF_EXT_IRQ_NO_TRACE
				trace_hardirqs_on();
#else
				trace_off_mark_irq(0x510, irq);
#endif
				irq_handler_rt_trace_enter(irq);
				__handle_irq(irq);
				irq_handler_rt_trace_exit(irq);
#ifndef CONFIG_SNSC_LITE_IRQSOFF_EXT_IRQ_NO_TRACE
				trace_off_mark_irq(0x520, irq);
#else
				trace_hardirqs_off();
#endif
			}

			/* AT91 specific workaround */
			irq_finish(irq);

#ifdef CONFIG_SMP
		} else if (IRQ_IS_IPI(irq)) {

#ifdef CONFIG_SNSC_DEBUG_IRQ_OVERHEAD_ONCE
			if (in_ipi)
				per_cpu(in_ipi_count, smp_processor_id())++;
			else
				in_ipi = 1;
#endif

			if (in_irq) {
				in_irq = 0;
				irq_exit();
			}
			ack_IPI_irq(irq);
			do_IPI(regs);
#endif

#ifdef CONFIG_LOCAL_TIMERS
		} else if (IRQ_IS_LOCALTIMER(irq)) {
			if (in_irq) {
				in_irq = 0;
				irq_exit();
			}
			ack_local_timer_irq(irq);
			do_local_timer(regs);
#endif
		}
	} while (irq);

	if (in_irq)
		irq_exit();

#if defined(CONFIG_SNSC_DEBUG_IRQ_OVERHEAD_ONCE) && defined(CONFIG_SMP)
	if (!had_external_interrupt)
		per_cpu(ipi_lt_without_irq, smp_processor_id())++;
#endif

	set_irq_regs(old_regs);
}

#else /* !CONFIG_SNSC_IRQ_OVERHEAD_ONCE */

/*
 * do_IRQ handles all hardware IRQ's.  Decoded IRQs should not
 * come via this function.  Instead, they should provide their
 * own 'handler'
 */
asmlinkage void __exception asm_do_IRQ(unsigned int irq, struct pt_regs *regs)
{
	struct pt_regs *old_regs = set_irq_regs(regs);

	irq_enter();

	/*
	 * Some hardware gives randomly wrong interrupts.  Rather
	 * than crashing, do something sensible.
	 */
	if (unlikely(irq >= NR_IRQS)) {
		if (printk_ratelimit())
			printk(KERN_WARNING "Bad IRQ%u\n", irq);
		ack_bad_irq(irq);
	} else {
#ifdef CONFIG_SNSC_LITE_IRQSOFF_EXT_IRQ_NO_TRACE
		trace_hardirqs_on();
#else
		trace_off_mark_irq(0x510, irq);
#endif
		irq_handler_rt_trace_enter(irq);
		__handle_irq(irq);
		irq_handler_rt_trace_exit(irq);
#ifndef CONFIG_SNSC_LITE_IRQSOFF_EXT_IRQ_NO_TRACE
		trace_off_mark_irq(0x520, irq);
#else
		trace_hardirqs_off();
#endif
	}

	/* AT91 specific workaround */
	irq_finish(irq);

	irq_exit();
	set_irq_regs(old_regs);
}

#endif

void set_irq_flags(unsigned int irq, unsigned int iflags)
{
	struct irq_desc *desc;
	unsigned long flags;

	if (irq >= NR_IRQS) {
		printk(KERN_ERR "Trying to set irq flags for IRQ%d\n", irq);
		return;
	}

	desc = irq_desc + irq;
	spin_lock_irqsave(&desc->lock, flags);
	desc->status |= IRQ_NOREQUEST | IRQ_NOPROBE | IRQ_NOAUTOEN;
	if (iflags & IRQF_VALID)
		desc->status &= ~IRQ_NOREQUEST;
	if (iflags & IRQF_PROBE)
		desc->status &= ~IRQ_NOPROBE;
	if (!(iflags & IRQF_NOAUTOEN))
		desc->status &= ~IRQ_NOAUTOEN;
	spin_unlock_irqrestore(&desc->lock, flags);
}

void __init init_IRQ(void)
{
	int irq;

	for (irq = 0; irq < NR_IRQS; irq++)
		irq_desc[irq].status |= IRQ_NOREQUEST | IRQ_NOPROBE;

	init_arch_irq();
}

#ifdef CONFIG_HOTPLUG_CPU

static void route_irq(struct irq_desc *desc, unsigned int irq, unsigned int cpu)
{
	pr_debug("IRQ%u: moving from cpu%u to cpu%u\n", irq, desc->cpu, cpu);

	spin_lock_irq(&desc->lock);
	desc->chip->set_affinity(irq, cpumask_of(cpu));
	spin_unlock_irq(&desc->lock);
}

/*
 * The CPU has been marked offline.  Migrate IRQs off this CPU.  If
 * the affinity settings do not allow other CPUs, force them onto any
 * available CPU.
 */
void migrate_irqs(void)
{
	unsigned int i, cpu = smp_processor_id();

	for (i = 0; i < NR_IRQS; i++) {
		struct irq_desc *desc = irq_desc + i;

		if (desc->cpu == cpu) {
			unsigned int newcpu = cpumask_any_and(desc->affinity,
							      cpu_online_mask);
			if (newcpu >= nr_cpu_ids) {
				if (printk_ratelimit())
					printk(KERN_INFO "IRQ%u no longer affine to CPU%u\n",
					       i, cpu);

				cpumask_setall(desc->affinity);
				newcpu = cpumask_any_and(desc->affinity,
							 cpu_online_mask);
			}

			route_irq(desc, i, newcpu);
		}
	}
}
#endif /* CONFIG_HOTPLUG_CPU */
