/*
 *  linux/arch/arm/kernel/smp.c
 *
 *  Copyright (C) 2002 ARM Limited, All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */
#include <linux/module.h>
#include <linux/delay.h>
#include <linux/init.h>
#include <linux/spinlock.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/cache.h>
#include <linux/profile.h>
#include <linux/errno.h>
#include <linux/mm.h>
#include <linux/err.h>
#include <linux/cpu.h>
#include <linux/smp.h>
#include <linux/seq_file.h>
#include <linux/irq.h>
#include <linux/rt_trace_lite.h>
#include <linux/rt_trace_lite_irq.h>

#include <asm/atomic.h>
#include <asm/cacheflush.h>
#include <asm/cpu.h>
#include <asm/mmu_context.h>
#include <asm/pgtable.h>
#include <asm/pgalloc.h>
#include <asm/processor.h>
#include <asm/tlbflush.h>
#include <asm/ptrace.h>
#include <asm/cputype.h>

/*
 * as from 2.5, kernels no longer have an init_tasks structure
 * so we need some other way of telling a new secondary core
 * where to place its SVC stack
 */
struct secondary_data secondary_data;

/*
 * structures for inter-processor calls
 * - A collection of single bit ipi messages.
 */
struct ipi_data {
	raw_spinlock_t lock;
	unsigned long ipi_count;
	unsigned long bits;
};

static DEFINE_PER_CPU(struct ipi_data, ipi_data) = {
	.lock	= RAW_SPIN_LOCK_UNLOCKED(ipi_data.lock),
};

enum ipi_msg_type {
	IPI_TIMER,
	IPI_RESCHEDULE,
	IPI_CALL_FUNC,
	IPI_CALL_FUNC_SINGLE,
	IPI_CPU_STOP,
};

int __cpuinit __cpu_up(unsigned int cpu)
{
	struct cpuinfo_arm *ci = &per_cpu(cpu_data, cpu);
	struct task_struct *idle = ci->idle;
	pgd_t *pgd;
	pmd_t *pmd;
	int ret;

	/*
	 * Spawn a new process manually, if not already done.
	 * Grab a pointer to its task struct so we can mess with it
	 */
	if (!idle) {
		idle = fork_idle(cpu);
		if (IS_ERR(idle)) {
			printk(KERN_ERR "CPU%u: fork() failed\n", cpu);
			return PTR_ERR(idle);
		}
		ci->idle = idle;
	}

	/*
	 * Allocate initial page tables to allow the new CPU to
	 * enable the MMU safely.  This essentially means a set
	 * of our "standard" page tables, with the addition of
	 * a 1:1 mapping for the physical address of the kernel.
	 */
#ifndef CONFIG_EJ_SPLIT_PAGETABLE
	pgd = pgd_alloc(&init_mm);
#else
	pgd = init_mm.pgd;
#endif
	pmd = pmd_offset(pgd + pgd_index(PHYS_OFFSET), PHYS_OFFSET);
	*pmd = __pmd((PHYS_OFFSET & PGDIR_MASK) |
		     PMD_TYPE_SECT | PMD_SECT_AP_WRITE);
	flush_pmd_entry(pmd);
	outer_clean_range(__pa(pmd), __pa(pmd + 1));

	/*
	 * We need to tell the secondary core where to find
	 * its stack and the page tables.
	 */
	secondary_data.stack = task_stack_page(idle) + THREAD_START_SP;
	secondary_data.pgdir = virt_to_phys(pgd);
	__cpuc_flush_dcache_page(&secondary_data);
	outer_clean_range(__pa(&secondary_data), __pa(&secondary_data + 1));

	/*
	 * Now bring the CPU into our world.
	 */
	ret = boot_secondary(cpu, idle);
	if (ret == 0) {
		unsigned long timeout;

		/*
		 * CPU was successfully started, wait for it
		 * to come online or time out.
		 */
		timeout = jiffies + HZ;
		while (time_before(jiffies, timeout)) {
			if (cpu_online(cpu))
				break;

			udelay(10);
			barrier();
		}

		if (!cpu_online(cpu))
			ret = -EIO;
	}

	secondary_data.stack = NULL;
	secondary_data.pgdir = 0;

	*pmd = __pmd(0);
	clean_pmd_entry(pmd);
#ifndef CONFIG_EJ_SPLIT_PAGETABLE
	pgd_free(&init_mm, pgd);
#endif

	if (ret) {
		printk(KERN_CRIT "CPU%u: processor failed to boot\n", cpu);

		/*
		 * FIXME: We need to clean up the new idle thread. --rmk
		 */
	}

	return ret;
}

#ifdef CONFIG_HOTPLUG_CPU
/*
 * __cpu_disable runs on the processor to be shutdown.
 */
int __cpuexit __cpu_disable(void)
{
	unsigned int cpu = smp_processor_id();
	struct task_struct *p;
	int ret;

	ret = mach_cpu_disable(cpu);
	if (ret)
		return ret;

	/*
	 * Take this CPU offline.  Once we clear this, we can't return,
	 * and we must not schedule until we're ready to give up the cpu.
	 */
	cpu_clear(cpu, cpu_online_map);

	/*
	 * OK - migrate IRQs away from this CPU
	 */
	migrate_irqs();

	/*
	 * Stop the local timer for this CPU.
	 */
	local_timer_stop();

	/*
	 * Flush user cache and TLB mappings, and then remove this CPU
	 * from the vm mask set of all processes.
	 */
	flush_cache_all();
	local_flush_tlb_all();

	read_lock(&tasklist_lock);
	for_each_process(p) {
		if (p->mm)
			cpu_clear(cpu, p->mm->cpu_vm_mask);
	}
	read_unlock(&tasklist_lock);

	return 0;
}

/*
 * called on the thread which is asking for a CPU to be shutdown -
 * waits until shutdown has completed, or it is timed out.
 */
void __cpuexit __cpu_die(unsigned int cpu)
{
	if (!platform_cpu_kill(cpu))
		printk("CPU%u: unable to kill\n", cpu);
}

/*
 * Called from the idle thread for the CPU which has been shutdown.
 *
 * Note that we disable IRQs here, but do not re-enable them
 * before returning to the caller. This is also the behaviour
 * of the other hotplug-cpu capable cores, so presumably coming
 * out of idle fixes this.
 */
void __cpuexit cpu_die(void)
{
	unsigned int cpu = smp_processor_id();

	local_irq_disable();
	idle_task_exit();

	/*
	 * actual CPU shutdown procedure is at least platform (if not
	 * CPU) specific
	 */
	platform_cpu_die(cpu);

	/*
	 * Do not return to the idle loop - jump back to the secondary
	 * cpu initialisation.  There's some initialisation which needs
	 * to be repeated to undo the effects of taking the CPU offline.
	 */
	__asm__("mov	sp, %0\n"
	"	b	secondary_start_kernel"
		:
		: "r" (task_stack_page(current) + THREAD_START_SP));
}
#endif /* CONFIG_HOTPLUG_CPU */

/*
 * This is the secondary CPU boot entry.  We're using this CPUs
 * idle thread stack, but a set of temporary page tables.
 */
asmlinkage void __cpuinit secondary_start_kernel(void)
{
	struct mm_struct *mm = &init_mm;
	unsigned int cpu = smp_processor_id();

	printk("CPU%u: Booted secondary processor\n", cpu);

	/*
	 * All kernel threads share the same mm context; grab a
	 * reference and switch to it.
	 */
	atomic_inc(&mm->mm_users);
	atomic_inc(&mm->mm_count);
	current->active_mm = mm;
	cpu_set(cpu, mm->cpu_vm_mask);
	cpu_switch_mm(mm->pgd, mm);
	enter_lazy_tlb(mm, current);
	local_flush_tlb_all();

	cpu_init();
	preempt_disable();

	/*
	 * Give the platform a chance to do its own initialisation.
	 */
	platform_secondary_init(cpu);

	/*
	 * Enable local interrupts.
	 */
	notify_cpu_starting(cpu);
	local_irq_enable();
	local_fiq_enable();

	/*
	 * Setup local timer for this CPU.
	 */
	local_timer_setup();

	calibrate_delay();

	smp_store_cpu_info(cpu);

	/*
	 * OK, now it's safe to let the boot CPU continue
	 */
	cpu_set(cpu, cpu_online_map);

	/*
	 * OK, it's off to the idle thread for us
	 */
	cpu_idle();
}

/*
 * Called by both boot and secondaries to move global data into
 * per-processor storage.
 */
void __cpuinit smp_store_cpu_info(unsigned int cpuid)
{
	struct cpuinfo_arm *cpu_info = &per_cpu(cpu_data, cpuid);

	cpu_info->loops_per_jiffy = loops_per_jiffy;
}

void __init smp_cpus_done(unsigned int max_cpus)
{
	int cpu;
	unsigned long bogosum = 0;

	for_each_online_cpu(cpu)
		bogosum += per_cpu(cpu_data, cpu).loops_per_jiffy;

	printk(KERN_INFO "SMP: Total of %d processors activated "
	       "(%lu.%02lu BogoMIPS).\n",
	       num_online_cpus(),
	       bogosum / (500000/HZ),
	       (bogosum / (5000/HZ)) % 100);
}

void __init smp_prepare_boot_cpu(void)
{
	unsigned int cpu = smp_processor_id();

	per_cpu(cpu_data, cpu).idle = current;
}

static void send_ipi_message(cpumask_t callmap, enum ipi_msg_type msg)
{
	unsigned long flags;
	unsigned int cpu;

	trace_off_mark_irq(0x100, msg);

	local_irq_save(flags);

	for_each_cpu_mask(cpu, callmap) {
		struct ipi_data *ipi = &per_cpu(ipi_data, cpu);

		spin_lock(&ipi->lock);
		ipi->bits |= 1 << msg;
		spin_unlock(&ipi->lock);
	}

	/*
	 * Call the platform specific cross-CPU call function.
	 */
	smp_cross_call(callmap);

	local_irq_restore(flags);
}

void arch_send_call_function_ipi(cpumask_t mask)
{
	send_ipi_message(mask, IPI_CALL_FUNC);
}

void arch_send_call_function_single_ipi(int cpu)
{
	send_ipi_message(cpumask_of_cpu(cpu), IPI_CALL_FUNC_SINGLE);
}

#ifdef CONFIG_EJ_ARCH_HAVE_SMP_CALL_FUNCTION
/*
 * Bring smp_call_function() from 2.6.23
 */
struct smp_call_struct {
	void (*func)(void *info);
	void *info;
	int wait;
	cpumask_t pending;
	cpumask_t unfinished;
};

static struct smp_call_struct * volatile smp_call_function_data;
static DEFINE_RAW_SPINLOCK(smp_call_function_lock);

/*
 * You must not call this function with disabled interrupts, from a
 * hardware interrupt handler, nor from a bottom half handler.
 */
void smp_call_function_many(const struct cpumask *mask,
			    void (*func)(void *), void *info, bool wait)
{
	struct smp_call_struct data;
	unsigned long timeout;
	int ret = 0;
	cpumask_t callmap;

	cpumask_copy(&callmap, mask);

	data.func = func;
	data.info = info;
	data.wait = wait;

	cpu_clear(smp_processor_id(), callmap);
	if (cpus_empty(callmap))
		goto out;

	data.pending = callmap;
	if (wait)
		data.unfinished = callmap;

	BUG_ON(irqs_disabled());

	/*
	 * disable softirq to avoid deadlock
	 */
	local_bh_disable();

	/*
	 * try to get the mutex on smp_call_function_data
	 */
	spin_lock(&smp_call_function_lock);
	smp_call_function_data = &data;

	send_ipi_message(callmap, IPI_CALL_FUNC);

	timeout = jiffies + HZ;
	while (!cpus_empty(data.pending) && time_before(jiffies, timeout))
		barrier();

	/*
	 * did we time out?
	 */
	if (!cpus_empty(data.pending)) {
		/*
		 * this may be causing our panic - report it
		 */
		printk(KERN_CRIT
		       "CPU%u: smp_call_function timeout for %p(%p)\n"
		       "      callmap %lx pending %lx, %swait\n",
		       smp_processor_id(), func, info, *cpus_addr(callmap),
		       *cpus_addr(data.pending), wait ? "" : "no ");

		/*
		 * TRACE
		 */
	retry:
		timeout = jiffies + (5 * HZ);
		while (!cpus_empty(data.pending) && time_before(jiffies, timeout))
			barrier();

		if (cpus_empty(data.pending))
			printk(KERN_CRIT "     RESOLVED\n");
		else {
			printk(KERN_CRIT "     STILL STUCK\n");
			printk(KERN_CRIT "     Try Again\n");
			goto retry;
		}
	}

	/*
	 * whatever happened, we're done with the data, so release it
	 */
	smp_call_function_data = NULL;
	spin_unlock(&smp_call_function_lock);

	local_bh_enable();

	if (!cpus_empty(data.pending)) {
		ret = -ETIMEDOUT;
		goto out;
	}

	if (wait)
		while (!cpus_empty(data.unfinished))
			barrier();
 out:

	return;
}
EXPORT_SYMBOL(smp_call_function_many);

int smp_call_function(void (*func)(void *info), void *info, int wait)
{
	smp_call_function_many(cpu_online_mask, func, info, wait);

	return 0;
}
EXPORT_SYMBOL_GPL(smp_call_function);

int smp_call_function_single(int cpu, void (*func)(void *info), void *info,
			     int wait)
{
	/* prevent preemption and reschedule on another processor */
	int current_cpu = get_cpu();
	int ret = 0;

	if (cpu == current_cpu) {
		local_irq_disable();
		func(info);
		local_irq_enable();
	} else
		smp_call_function_many(get_cpu_mask(cpu),
				       func, info, wait);

	put_cpu();

	return ret;
}
EXPORT_SYMBOL_GPL(smp_call_function_single);

/*
 * for caller who has a pre-allocated data structure
 */
void __smp_call_function_single(int cpu, struct call_single_data *data,
				int wait)
{
	/* Can deadlock when called with interrupts disabled */
	WARN_ON_ONCE(wait && irqs_disabled() && !oops_in_progress);

	smp_call_function_single(cpu, data->func, data->info, wait);
}

/*
 * ipi_call_function - handle IPI from smp_call_function()
 *
 * Note that we copy data out of the cross-call structure and then
 * let the caller know that we're here and have done with their data
 */
static void ipi_call_function(unsigned int cpu)
{
	struct smp_call_struct *data = smp_call_function_data;
	void (*func)(void *info) = data->func;
	void *info = data->info;
	int wait = data->wait;

	cpu_clear(cpu, data->pending);

	trace_off_mark_irq(0x300, (unsigned long)func);
	func(info);
	trace_off_mark_irq(0x3ff, 0);

	if (wait)
		cpu_clear(cpu, data->unfinished);
}
#endif /* CONFIG_EJ_ARCH_HAVE_SMP_CALL_FUNCTION */

unsigned long get_ipi_count(int cpu)
{
	return per_cpu(ipi_data, cpu).ipi_count;
}

void show_ipi_list(struct seq_file *p)
{
	unsigned int cpu;

	seq_puts(p, "IPI:");

	for_each_present_cpu(cpu)
		seq_printf(p, " %10lu", per_cpu(ipi_data, cpu).ipi_count);

	seq_putc(p, '\n');
}

void show_local_irqs(struct seq_file *p)
{
	unsigned int cpu;

	seq_printf(p, "LOC: ");

	for_each_present_cpu(cpu)
		seq_printf(p, "%10u ", irq_stat[cpu].local_timer_irqs);

	seq_putc(p, '\n');
}

static void ipi_timer(void)
{
	irq_enter();
	local_timer_interrupt();
	irq_exit();
}

#ifdef CONFIG_LOCAL_TIMERS
#ifdef CONFIG_SNSC_IRQ_OVERHEAD_ONCE
           void __exception do_local_timer(struct pt_regs *regs)
#else
asmlinkage void __exception do_local_timer(struct pt_regs *regs)
#endif
{
	struct pt_regs *old_regs = set_irq_regs(regs);
	int cpu = smp_processor_id();

	trace_off_mark_irq(0x200, 0);

	irq_handler_rt_trace_enter(NR_IRQ_LOC);
	if (local_timer_ack()) {
		irq_stat[cpu].local_timer_irqs++;
#ifdef CONFIG_SNSC_LOCAL_TIMER_AVOID_WAKEUP_SOFTIRQ
		/*
		 * wakeup_softirqd() will be called in this sequence:
		 *    local_timer_interrupt()
		 *       run_local_timers()
		 *          raise_softirq()
		 *             wakeup_softirqd()
		 *
		 * Avoid a second call to wakeup_softirq() from irq_exit().
		 */
		irq_enter();
		local_timer_interrupt();
		irq_exit_no_invoke_softirq();
#else
		ipi_timer();
#endif
	}
	irq_handler_rt_trace_exit(NR_IRQ_LOC);

	trace_off_mark_irq(0x2ff, 0);

	set_irq_regs(old_regs);
}
#endif

static DEFINE_RAW_SPINLOCK(stop_lock);

/*
 * ipi_cpu_stop - handle IPI from smp_send_stop()
 */
static void ipi_cpu_stop(unsigned int cpu)
{
	spin_lock(&stop_lock);
	printk(KERN_CRIT "CPU%u: stopping\n", cpu);
	dump_stack();
	spin_unlock(&stop_lock);

	cpu_clear(cpu, cpu_online_map);

	local_fiq_disable();
	local_irq_disable();

	while (1)
		cpu_relax();
}

/*
 * Main handler for inter-processor interrupts
 *
 * For ARM, the ipimask now only identifies a single
 * category of IPI (Bit 1 IPIs have been replaced by a
 * different mechanism):
 *
 *  Bit 0 - Inter-processor function call
 */
#ifdef CONFIG_SNSC_IRQ_OVERHEAD_ONCE
           void __exception do_IPI(struct pt_regs *regs)
#else
asmlinkage void __exception do_IPI(struct pt_regs *regs)
#endif
{
	unsigned int cpu = smp_processor_id();
	struct ipi_data *ipi = &per_cpu(ipi_data, cpu);
	struct pt_regs *old_regs = set_irq_regs(regs);

	irq_handler_rt_trace_enter(NR_IRQ_IPI);
	ipi->ipi_count++;

	for (;;) {
		unsigned long msgs;

		spin_lock(&ipi->lock);
		msgs = ipi->bits;
		ipi->bits = 0;
		spin_unlock(&ipi->lock);

		if (!msgs)
			break;

		do {
			unsigned nextmsg;

			nextmsg = msgs & -msgs;
			msgs &= ~nextmsg;
			nextmsg = ffz(~nextmsg);

			trace_off_mark_irq(0x400, nextmsg);

			switch (nextmsg) {
			case IPI_TIMER:
				trace_off_mark_irq_dtl(0x410, nextmsg);
				ipi_timer();
				trace_off_mark_irq_dtl(0x41f, 0);
				break;

			case IPI_RESCHEDULE:
				/*
				 * nothing more to do - eveything is
				 * done on the interrupt return path
				 */
				break;

#ifndef CONFIG_EJ_ARCH_HAVE_SMP_CALL_FUNCTION
			case IPI_CALL_FUNC:
				trace_off_mark_irq_dtl(0x420, nextmsg);
				generic_smp_call_function_interrupt();
				trace_off_mark_irq_dtl(0x42f, 0);
				break;

			case IPI_CALL_FUNC_SINGLE:
				trace_off_mark_irq_dtl(0x430, nextmsg);
				generic_smp_call_function_single_interrupt();
				trace_off_mark_irq_dtl(0x43f, 0);
				break;
#else
			case IPI_CALL_FUNC:
				trace_off_mark_irq_dtl(0x420, nextmsg);
				ipi_call_function(cpu);
				trace_off_mark_irq_dtl(0x42f, 0);
				break;

			case IPI_CALL_FUNC_SINGLE:
				trace_off_mark_irq_dtl(0x430, nextmsg);
				ipi_call_function(cpu);
				trace_off_mark_irq_dtl(0x43f, 0);
				break;
#endif /* CONFIG_EJ_ARCH_HAVE_SMP_CALL_FUNCTION */

			case IPI_CPU_STOP:
				trace_off_mark_irq_dtl(0x440, nextmsg);
				ipi_cpu_stop(cpu);
				trace_off_mark_irq_dtl(0x44f, 0);
				break;

			default:
				printk(KERN_CRIT "CPU%u: Unknown IPI message 0x%x\n",
				       cpu, nextmsg);
				break;
			}
		} while (msgs);
#ifdef CONFIG_EJ_DO_IPI_EARLY_BREAK
		/* Skip pulling next message */
		break;
#endif
	}

	irq_handler_rt_trace_exit(NR_IRQ_IPI);
	trace_off_mark_irq(0x4ff, 0);

	set_irq_regs(old_regs);
}

void smp_send_reschedule(int cpu)
{
#ifdef CONFIG_EJ_DO_NOT_USE_IPI_RESCHEDULE
	/* a interrupt is enough for reschedule */
	smp_cross_call(cpumask_of_cpu(cpu));
#else /* CONFIG_EJ_DO_NOT_USE_IPI_RESCHEDULE */
	send_ipi_message(cpumask_of_cpu(cpu), IPI_RESCHEDULE);
#endif /* CONFIG_EJ_DO_NOT_USE_IPI_RESCHEDULE */
}

/*
 * this function sends a 'reschedule' IPI to all other CPUs.
 * This is used when RT tasks are starving and other CPUs
 * might be able to run them:
 */
void smp_send_reschedule_allbutself(void)
{
#ifdef CONFIG_EJ_DO_NOT_USE_IPI_RESCHEDULE
	cpumask_t mask = cpu_online_map;
	preempt_disable();
	cpu_clear(smp_processor_id(), mask);
	preempt_enable();
	smp_cross_call(mask);
#else /* CONFIG_EJ_DO_NOT_USE_IPI_RESCHEDULE */
	unsigned int cpu;

	for_each_online_cpu(cpu) {
		preempt_disable();
		if (cpu != smp_processor_id())
			smp_send_reschedule(cpu);
		preempt_enable();
	}
#endif /* CONFIG_EJ_DO_NOT_USE_IPI_RESCHEDULE */
}

void smp_send_timer(void)
{
	cpumask_t mask = cpu_online_map;
	cpu_clear(smp_processor_id(), mask);
	send_ipi_message(mask, IPI_TIMER);
}

void smp_timer_broadcast(cpumask_t mask)
{
	send_ipi_message(mask, IPI_TIMER);
}

void smp_send_stop(void)
{
	cpumask_t mask = cpu_online_map;
	cpu_clear(smp_processor_id(), mask);
	send_ipi_message(mask, IPI_CPU_STOP);
}

/*
 * not supported here
 */
int setup_profiling_timer(unsigned int multiplier)
{
	return -EINVAL;
}

#define dprintk pr_debug

#ifdef CONFIG_EJ_ON_EACH_CPU_THREADED
struct cpu_work {
	struct work_struct work;
	void (*func)(void *);
	void *info;
	struct completion done;
};

static void cpu_work_thread(struct work_struct *work)
{
	struct cpu_work *cpu_work = container_of(work, struct cpu_work, work);
	dprintk("%s:cpu=%d\n", __func__, raw_smp_processor_id());
	cpu_work->func(cpu_work->info);
	complete(&cpu_work->done);
}

static DECLARE_MUTEX(mtx);

static int
__on_each_cpu_mask_thread(void (*func)(void *), void *info, int wait,
		 cpumask_t cpumask)
{
	int cpu;
	int this_cpu;
	int this_cpu_on = 0;
	struct cpu_work works[NR_CPUS];

	dprintk("%s:cpu=%d cpus=%d start\n", __func__, raw_smp_processor_id(), cpus_weight(cpumask));
	this_cpu = raw_smp_processor_id();
	if (cpu_isset(this_cpu, cpumask)) {
		this_cpu_on = 1;
		cpu_clear(this_cpu, cpumask);
	}

	for_each_cpu_mask(cpu, cpumask) {
		INIT_WORK(&works[cpu].work, cpu_work_thread);
		init_completion(&works[cpu].done);
		works[cpu].func = func;
		works[cpu].info = info;
		schedule_work_cpu(&works[cpu].work, cpu);
	}

	if (this_cpu_on) {
		func(info);
	}

	for_each_cpu_mask(cpu, cpumask) {
		wait_for_completion(&works[cpu].done);
	}

	dprintk("%s:cpu=%d end\n", __func__, raw_smp_processor_id());

	return 0;
}

static int
__on_each_cpu_mask_atomic(void (*func)(void *), void *info, int wait,
		 cpumask_t cpumask)
{
	unsigned long flags;

	smp_call_function_many(&cpumask, func, info, wait);
	local_irq_save(flags);
	if (cpu_isset(smp_processor_id(), cpumask))
		func(info);
	local_irq_restore(flags);

	return 0;
}

static int __sched
__on_each_cpu_mask(void (*func)(void *), void *info, int wait,
		 cpumask_t cpumask)
{
	int ret;
	migration_t flg;

	migration_disable(&flg);

#ifdef CONFIG_EJ_THREADED_CPU_BROADCAST
	if (in_atomic() || irqs_disabled() || system_state < SYSTEM_RUNNING) {
		if (system_state == SYSTEM_RUNNING)
			WARN_ON_ONCE("from atomic\n");
#else
	if (1) {
#endif
		ret = __on_each_cpu_mask_atomic(func, info, wait, cpumask);

	}
	else
		ret = __on_each_cpu_mask_thread(func, info, wait, cpumask);

	migration_enable(flg);

	return ret;
}

static int __sched
on_each_cpu_mm(void (*func)(void *), void *info, int wait,
		 struct mm_struct *mm)
{
	cpumask_t cpumask = mm->cpu_vm_mask;

	if (unlikely(mm != current->active_mm))
		cpumask = cpu_online_map;

	return __on_each_cpu_mask(func, info, wait, cpumask);
}

#ifndef CONFIG_EJ_THREADED_CPU_BROADCAST_ALL
static int
__on_each_cpu_mask_fast(void (*func)(void *), void *info, int wait,
			cpumask_t cpumask)
{
	int ret;
	migration_t flg;

	migration_disable(&flg);

	ret = __on_each_cpu_mask_atomic(func, info, wait, cpumask);

	migration_enable(flg);

	return ret;
}

static int
on_each_cpu_mm_fast(void (*func)(void *), void *info, int wait,
		    struct mm_struct *mm)
{
	cpumask_t cpumask = mm->cpu_vm_mask;

	if (unlikely(mm != current->active_mm))
		cpumask = cpu_online_map;

	return __on_each_cpu_mask_fast(func, info, wait, cpumask);
}
#endif

/*
 * Call a function on all processors
 */
int __sched on_each_cpu(void (*func) (void *info), void *info, int wait)
{
	return __on_each_cpu_mask(func, info, wait, cpu_online_map);
}
EXPORT_SYMBOL(on_each_cpu);

#else /* CONFIG_EJ_ON_EACH_CPU_THREADED */
static int __sched
on_each_cpu_mm(void (*func)(void *), void *info, int wait,
		 struct mm_struct *mm)
{
	int ret = 0;

	if (unlikely(mm != current->active_mm))
		return on_each_cpu(func, info, wait);

	preempt_disable();

	ret = smp_call_function_mask(mm->cpu_vm_mask, func, info, wait);
	if (cpu_isset(smp_processor_id(), mm->cpu_vm_mask))
		func(info);

	preempt_enable();

	return ret;
}
#endif /* CONFIG_EJ_ON_EACH_CPU_THREADED */

/**********************************************************************/

/*
 * TLB operations
 */
struct tlb_args {
	struct vm_area_struct *ta_vma;
	unsigned long ta_start;
	unsigned long ta_end;
};

/* all SMP configurations have the extended CPUID registers */
static inline int tlb_ops_need_broadcast(void)
{
	return ((read_cpuid_ext(CPUID_EXT_MMFR3) >> 12) & 0xf) < 2;
}

static inline void ipi_flush_tlb_all(void *ignored)
{
	local_flush_tlb_all();
}

static inline void ipi_flush_tlb_mm(void *arg)
{
	struct mm_struct *mm = (struct mm_struct *)arg;

	local_flush_tlb_mm(mm);
}

static inline void ipi_flush_tlb_page(void *arg)
{
	struct tlb_args *ta = (struct tlb_args *)arg;

	local_flush_tlb_page(ta->ta_vma, ta->ta_start);
}

static inline void ipi_flush_tlb_kernel_page(void *arg)
{
	struct tlb_args *ta = (struct tlb_args *)arg;

	local_flush_tlb_kernel_page(ta->ta_start);
}

static inline void ipi_flush_tlb_range(void *arg)
{
	struct tlb_args *ta = (struct tlb_args *)arg;

	local_flush_tlb_range(ta->ta_vma, ta->ta_start, ta->ta_end);
}

static inline void ipi_flush_tlb_kernel_range(void *arg)
{
	struct tlb_args *ta = (struct tlb_args *)arg;

	local_flush_tlb_kernel_range(ta->ta_start, ta->ta_end);
}

void __sched flush_tlb_all(void)
{
	if (tlb_ops_need_broadcast())
		on_each_cpu(ipi_flush_tlb_all, NULL, 1);
	else
		local_flush_tlb_all();
}

void __sched flush_tlb_mm(struct mm_struct *mm)
{
	if (tlb_ops_need_broadcast()) {
#if defined(CONFIG_EJ_THREADED_CPU_BROADCAST_ALL) || !defined(CONFIG_EJ_ON_EACH_CPU_THREADED)
		on_each_cpu_mm(ipi_flush_tlb_mm, mm, 1, mm);
#else
		on_each_cpu_mm_fast(ipi_flush_tlb_mm, mm, 1, mm);
#endif
	} else
		local_flush_tlb_mm(mm);
}

void __sched flush_tlb_page(struct vm_area_struct *vma, unsigned long uaddr)
{
	if (tlb_ops_need_broadcast()) {
		struct tlb_args ta;
		ta.ta_vma = vma;
		ta.ta_start = uaddr;
#if defined(CONFIG_EJ_THREADED_CPU_BROADCAST_ALL) || !defined(CONFIG_EJ_ON_EACH_CPU_THREADED)
		on_each_cpu_mm(ipi_flush_tlb_page, &ta, 1, vma->vm_mm);
#else
		on_each_cpu_mm_fast(ipi_flush_tlb_page, &ta, 1, vma->vm_mm);
#endif
	} else
		local_flush_tlb_page(vma, uaddr);
}

void __sched flush_tlb_kernel_page(unsigned long kaddr)
{
	if (tlb_ops_need_broadcast()) {
		struct tlb_args ta;
		ta.ta_start = kaddr;
		on_each_cpu(ipi_flush_tlb_kernel_page, &ta, 1);
	} else
		local_flush_tlb_kernel_page(kaddr);
}

void __sched flush_tlb_range(struct vm_area_struct *vma,
                     unsigned long start, unsigned long end)
{
	if (tlb_ops_need_broadcast()) {
		struct tlb_args ta;
		ta.ta_vma = vma;
		ta.ta_start = start;
		ta.ta_end = end;
		on_each_cpu_mm(ipi_flush_tlb_range, &ta, 1, vma->vm_mm);
	} else
		local_flush_tlb_range(vma, start, end);
}

void __sched flush_tlb_kernel_range(unsigned long start, unsigned long end)
{
	if (tlb_ops_need_broadcast()) {
		struct tlb_args ta;
		ta.ta_start = start;
		ta.ta_end = end;
		on_each_cpu(ipi_flush_tlb_kernel_range, &ta, 1);
	} else
		local_flush_tlb_kernel_range(start, end);
}
