/*
 *  arch/arm/mach-emxx/cpld/jtgvup_gpio_drv.c
 *  for version up
 *
 *  Copyright 2011 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/io.h>
#include <linux/gpio.h>
#include <linux/init.h>
#include <linux/device.h>
#include <linux/cdev.h>
#include <linux/types.h>
#include <linux/fs.h>
#include <asm/uaccess.h>

#ifdef CONFIG_SNSC_HSS
#include <linux/snsc_hss.h>
#endif  /* #ifdef CONFIG_SNSC_HSS */

/* typedef */
/*
 * @brief gpio pin and pin's value
 */
typedef struct __gpio_pin_value {
    unsigned int pin;
    unsigned int value;
} gpio_pin_value;

struct gpio_chr_dev {
    struct cdev cdev;
};

/* define */
#define GPIO_DRV_NAME                   "gpio_jtgvup_drv"

#define EMXX_CHG_BASE       0xe0140000
#define CHG_PINSEL_G000     (IO_ADDRESS(EMXX_CHG_BASE) + 0x0200)
#define CHG_PINSEL_G032     (IO_ADDRESS(EMXX_CHG_BASE) + 0x0204)
#define CHG_PINSEL_G064     (IO_ADDRESS(EMXX_CHG_BASE) + 0x0208)
#define CHG_PINSEL_G096     (IO_ADDRESS(EMXX_CHG_BASE) + 0x020C)
#define CHG_PINSEL_G128     (IO_ADDRESS(EMXX_CHG_BASE) + 0x0210)

#define CHG_PULL14      (IO_ADDRESS(EMXX_CHG_BASE) + 0x0338)            /* gpio_11:7-4 gpio_12:11-8 gpio_15:23-20 */
#define CHG_PULL10      (IO_ADDRESS(EMXX_CHG_BASE) + 0x0328)            /* gpio_99:31-28 gpio_97:23-20 */
#define CHG_PULL15      (IO_ADDRESS(EMXX_CHG_BASE) + 0x033C)            /* gpio_109 ~ gpio_114 */
#define CHG_PULL16      (IO_ADDRESS(EMXX_CHG_BASE) + 0x0340)            /* gpio_115 ~ gpio_121 */

#define JTGVUP_MELON_PORT_TMS           (99)                                 // VUP_TMS の Melon のポートのビット位置
#define JTGVUP_MELON_PORT_TCK           (12)                                 // TCK     の Melon のポートのビット位置
#define JTGVUP_MELON_PORT_TDI           (97)                                 // TDI     の Melon のポートのビット位置
#define JTGVUP_MELON_PORT_TDO           (15)                                 // TDO     の Melon のポートのビット位置
#define JTGVUP_MELON_PORT_VUP_EN        (11)                                 // VUP JTAG ON の Melon のポートのビット位置
#define JTGVUP_MELON_PORT_KY_EN         (119)                                // JKLD JTAG ON の Melon のポートのビット位置

#define gpio_warning(fmt,args...) \
    printk(KERN_WARNING fmt,##args)

#ifdef CONFIG_SNSC_HSS
#define JTGVUP_GPIO_HSS_NODE_NAME   "jtgvup_gpio_hss_node_name"
#define JTGVUP_GPIO_HSS_NODE_PARENT NULL    /* parent is gpio, registered in LDM */
#endif  /* #ifdef CONFIG_SNSC_HSS */

/* globals */
static dev_t gpio_chr_devno = {0};
static struct gpio_chr_dev* gpio_cdevp = NULL;

#ifdef CONFIG_SNSC_HSS
struct hss_node* jtgvup_gpio_hss_node;
#endif  /* #ifdef CONFIG_SNSC_HSS */

/* functions */
/*
 * @brief       driver open
 * @param[OUT]  なし
 * @param[IN]   inode: standard linux device driver parameter
 * @param[IN]   file: standard linux device driver parameter
 * @return      status
 * @pre         なし
 * @post        なし
 * @attention   なし
 */
static int  jtgvup_gpio_open(struct inode* inode, struct file* filp)
{
    return 0;
}

/*
 * @brief       driver close
 * @param[OUT]  なし
 * @param[IN]   inode: standard linux device driver parameter
 * @param[IN]   file: standard linux device driver parameter
 * @return      status
 * @pre         なし
 * @post        なし
 * @attention   なし
 */
static int  jtgvup_gpio_release(struct inode* inode, struct file* filp)
{

    return 0;
}

/*
 * @brief       driver ioctl
 * @param[OUT]  なし
 * @param[IN]   inodep: standard linux device driver parameter
 * @param[IN]   filp: standard linux device driver parameter
 * @param[IN]   command: command for ioctl request
 * @param[IN]   arg:  content which is transferred from user to kernel
 * @return      status
 * @pre         なし
 * @post        なし
 * @attention   なし
 */
static int  jtgvup_gpio_ctl_ioctl(struct inode* inode, struct file* filp, unsigned int command, unsigned long arg)
{
    int err = 0;

    return err;
}

/*
 * @brief       read gpio's value
 * @param[OUT]  buf: user buffer address for writing gpio's value
 * @param[IN]   file: standard linux device driver parameter
 * @param[IN]   buf: user buffer address for reading gpio number
 * @param[IN]   count: data size to be read.
 * @param[IN]   f_pos: standard linux device driver parameter
 * @return      size
 * @pre         なし
 * @post        なし
 * @attention   なし
 */
static ssize_t  jtgvup_gpio_read(struct file* filp, char __user* buf, size_t count, loff_t* f_pos)
{
    int ret = 0;
    gpio_pin_value gpv = {0};
    ret = copy_from_user(&gpv, buf, sizeof(gpio_pin_value));
    gpv.value = gpio_get_value(gpv.pin);
    ret = copy_to_user(buf, &gpv, sizeof(gpio_pin_value));
    return ret;
}

/*
 * @brief       write gpio's value
 * @param[IN]   buf: gpio's value and number
 * @param[IN]   count: data size to be written.
 * @param[IN]   ppos: standard linux device driver parameter
 * @return      size
 * @pre         なし
 * @post        なし
 * @attention   なし
 */
static ssize_t  jtgvup_gpio_write(struct file* filp, const char __user* buf, size_t count, loff_t* f_pos)
{
    int ret = 0;
    gpio_pin_value gpv = {0};
    ret = copy_from_user(&gpv, buf, sizeof(gpio_pin_value));
    gpio_set_value(gpv.pin, gpv.value);
    return ret;
}

/*
 * @brief       initialize gpio value
 * @param[IN]   なし
 * @return      なし
 * @pre         なし
 * @post        なし
 * @attention   なし
 */
static void jtgvup_gpio_pin_init(void)
{
    /* set gpio's CHG_PULLxx register,set gpio_15 enable input */
    writel( (readl(CHG_PULL14)) & (0xFFDFF00F), CHG_PULL14);
    writel( (readl(CHG_PULL14)) | (0x00500000), CHG_PULL14);
    writel( (readl(CHG_PULL10)) & (0x0F0FFFFF), CHG_PULL10);
    writel( (readl(CHG_PULL16)) & (0xFFF0FFFF), CHG_PULL16);

    writel(readl(CHG_PINSEL_G000) | 0x00009800, CHG_PINSEL_G000);
    writel(readl(CHG_PINSEL_G096) | 0x0080000A, CHG_PINSEL_G096);

    /* set gpio_15 input,TDO */
    gpio_direction_input(JTGVUP_MELON_PORT_TDO);

    /* set gpio_119 output,JKLD_EN */
    gpio_direction_output(JTGVUP_MELON_PORT_KY_EN, 1);

    /* set gpio_99 output,TMS */
    gpio_direction_output(JTGVUP_MELON_PORT_TMS, 0);

    /* set gpio_97 output,TDI */
    gpio_direction_output(JTGVUP_MELON_PORT_TDI, 0);

    /* set gpio_11 output,VUP_EN */
    gpio_direction_output(JTGVUP_MELON_PORT_VUP_EN, 1);

    /* set gpio_12 output,TCK */
    gpio_direction_output(JTGVUP_MELON_PORT_TCK, 0);
}

#ifdef CONFIG_SNSC_HSS
/*
 * @brief       suspend jtgvup gpio
 * @param[IN]   なし
 * @return      なし
 * @pre         なし
 * @post        なし
 * @attention   なし
 */
static int jtgvup_gpio_suspend(struct hss_node* node)
{
    return 0;
}

/*
 * @brief       resume jtgvup gpio
 * @param[IN]   なし
 * @return      なし
 * @pre         なし
 * @post        なし
 * @attention   なし
 */
static int jtgvup_gpio_resume(struct hss_node* node)
{
    jtgvup_gpio_pin_init();
    return 0;
}
#endif  /* #ifdef CONFIG_SNSC_HSS */

/*
 * Char driver ops
 */
static struct file_operations gpio_ctl_fops = {
    .owner = THIS_MODULE,
    .ioctl = jtgvup_gpio_ctl_ioctl,
    .open = jtgvup_gpio_open,
    .release = jtgvup_gpio_release,
    .read = jtgvup_gpio_read,
    .write = jtgvup_gpio_write,
};

#ifdef CONFIG_SNSC_HSS
/*
 * HSS framework ops
 */
static struct hss_node_ops hss_jtgvup_gpio_node_ops = {
    .suspend = jtgvup_gpio_suspend,
    .resume = jtgvup_gpio_resume,
};
#endif  /* #ifdef CONFIG_SNSC_HSS */

/*
 * @brief   {driver初期化}　gpio初期化を行う。
 * @param   なし
 * @return[状態]　状態を返す
 * @pre     なし
 * @post    なし
 * @attention  なし
 */
static int __init jtgvup_gpio_chr_init(void)
{
    int err = 0;

#ifdef CONFIG_SNSC_HSS
    /*
     * alloc the HSS node
     */
    jtgvup_gpio_hss_node = hss_alloc_node(JTGVUP_GPIO_HSS_NODE_NAME);
    if (!jtgvup_gpio_hss_node) {
        printk(KERN_INFO "%s(): alloc_node(jtgvup_gpio) failed\n", __func__);
        return -ENOMEM;
    }

    /*
     * Register the node.  If you want to get the callbacks to be called
     * before the specified node's one, specify the third parameter with
     * the name of the parent node. NULL means for no parent needed.
     */
    err = hss_register_node(jtgvup_gpio_hss_node, &hss_jtgvup_gpio_node_ops, JTGVUP_GPIO_HSS_NODE_PARENT);
    if (err) {
        printk(KERN_INFO "%s: register_node(jtgvup_gpio) failed\n", __func__);
        hss_free_node(jtgvup_gpio_hss_node);
        return err;
    }
#endif  /* #ifdef CONFIG_SNSC_HSS */

    /* alloc_chrdev_region */
    err = alloc_chrdev_region(&gpio_chr_devno, 0, 1, GPIO_DRV_NAME);
    if( err < 0 ) {
        return err;
    }
    gpio_cdevp = kmalloc(sizeof(struct gpio_chr_dev), GFP_KERNEL);
    if( !gpio_cdevp ) {
        err = -ENOMEM;
        goto fail_malloc;
    }
    memset(gpio_cdevp, 0, sizeof(struct gpio_chr_dev));
    cdev_init(&gpio_cdevp->cdev, &gpio_ctl_fops);
    gpio_cdevp->cdev.owner = THIS_MODULE;

    err = cdev_add(&gpio_cdevp->cdev, gpio_chr_devno, 1);
    if( err ) {
        gpio_warning("Error adding gpio_jtgvup_drv!\n");
        goto init_fail;
    }

    jtgvup_gpio_pin_init();

    return err;
init_fail:
    kfree(gpio_cdevp);
fail_malloc:
    unregister_chrdev_region(gpio_chr_devno, 1);
    return err;
}

/*
 * @brief   remove gpio_jtgvup_drv module
 * @param   なし
 * @return  なし
 * @pre     なし
 * @post    なし
 * @attention  なし
 */
static void __exit jtgvup_gpio_chr_exit(void)
{
#ifdef CONFIG_SNSC_HSS
    hss_unregister_node(jtgvup_gpio_hss_node);
    hss_free_node(jtgvup_gpio_hss_node);
#endif  /* #ifdef CONFIG_SNSC_HSS */
    cdev_del(&gpio_cdevp->cdev);
    kfree(gpio_cdevp);
    unregister_chrdev_region(gpio_chr_devno, 1);
    gpio_warning("\n gpio_jtgvup_drv module remove success\n");
    return ;
}

module_init(jtgvup_gpio_chr_init);
module_exit(jtgvup_gpio_chr_exit);
MODULE_LICENSE("GPL");
