/*
 *  arch/arm/mach-emxx/csi/csi_Ctrl.c
 *  CSI_CONCTRL interface
 *
 *  Copyright 2011 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>
#include <linux/unistd.h>
#include <linux/ioctl.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/uaccess.h>
#include <linux/mm.h>
#include <linux/semaphore.h>
#include <linux/sched.h>
#include <linux/dma-mapping.h>
#include <linux/gpio.h>
#include "../kmsg_que.h"
#include "csi_datalink.h"
#include "csiCtrl.h"

#define USE_DMA_ALLOC_COHERENT

dma_addr_t csi_ctrl_mmap_sbuff_bus_base;
dma_addr_t csi_ctrl_mmap_rbuff_bus_base;
#ifdef USE_DMA_ALLOC_COHERENT
static unsigned long csi_virt_to_phys(unsigned long addr);
#define __PA_CSI(addr) csi_virt_to_phys((unsigned long)(addr))
#endif

/* the count of interrupt */
extern unsigned int csi_lowlevel_count;

/*
** @brief status of the csi_drv
*/
typedef enum {
    CSI_DMA_STATUS_NONE = 0U,  /*idle*/
    CSI_DMA_STATUS_REQRECEIVE, /*receive*/
    CSI_DMA_STATUS_REQSEND,    /*send*/
    CSI_DMA_STATUS_RCV_FULL,   /*receive buffer is full*/
    CSI_DMA_STATUS_ERROR,      /*error*/
    CSI_DMA_STATUS_MAX         /*the max value of the enem*/
} CSI_DMA_STATUS;

static struct cdev csi_ctrl_cdev;
static kmsg_que_id_t csi_ctrl_kmsg_que;
static unsigned long csi_ctrl_mmap_sbuff_base = 0UL;
static unsigned long csi_ctrl_mmap_rbuff_base = 0UL;
static CSI_DMA_STATUS csi_ctrl_dma_status = CSI_DMA_STATUS_NONE;
static CSI_CIRC_BUFF* pScb_mgr = NULL;  /* send circle buffer manager */
static CSI_CIRC_BUFF* pRcb_mgr = NULL;  /* receive circle buffer manager */
static CSI_MSG_DATA notifyRcvDataInfo;

/* dummydata */
static dma_addr_t dummy_phys;
void* dummy_virt;


#define CSI_CTRL_SEND_BUFF_MGR          (csi_ctrl_mmap_sbuff_base + CSI_CTRL_SEND_BUFF_MGR_OFFSET)
#define CSI_CTRL_SEND_BUFF_BASE         ((CSI_PACKET *)(csi_ctrl_mmap_sbuff_base + CSI_CTRL_SEND_BUFF_OFFSET))
#define CSI_CTRL_RECEIVE_BUFF_MGR     (csi_ctrl_mmap_rbuff_base + CSI_CTRL_RECEIVE_BUFF_MGR_OFFSET)
#define CSI_CTRL_RECEIVE_BUFF_BASE    ((CSI_PACKET *)(csi_ctrl_mmap_rbuff_base + CSI_CTRL_RECEIVE_BUFF_OFFSET))

#define CSI_CTRL_GetPkgCnt(size)            ((unsigned long)(((size+CSILIB_SEND_DATA_SIZE)+(CSI_CTRL_PACKET_DATA_SIZE-1))/CSI_CTRL_PACKET_DATA_SIZE))

static void csi_ctrl_buf_init(CSI_CIRC_BUFF* cb);
static bool csi_ctrl_isRcvedDataValid(void);
static void csi_ctrl_rec_first_pkt(CSI_PACKET* rPkt);
static void csi_ctrl_clear_rcved_pkts(void);
static void csi_ctrl_update_write_point(void);
static void csi_ctrl_last_pkt_handler(void);
static void csi_ctrl_dma_completed(unsigned int* pReceive);
static int csi_ctrl_dma_send_receive(CSI_PACKET* pSend);
static int csi_ctrl_send_data(void);
static int csi_ctrl_receive_data(unsigned long arg);
static int csi_ctrl_set_Send_Pointer(unsigned int* pSend);

static int csi_ctrl_drv_open(struct inode* inode, struct file* file);
static int csi_ctrl_drv_mmap(struct file* file, struct vm_area_struct* vma);
static int csi_ctrl_drv_release(struct inode* inode, struct file* file);
static int csi_ctrl_drv_ioctl(struct inode* inode, struct file* file, unsigned int cmd, unsigned long arg);
static void __exit csi_ctrl_drv_exit(void);
static int __init csi_ctrl_drv_init(void);


#define CSI_CTRL_ERROR(fmt, args...)      \
    printk(KERN_ERR "%s: " fmt, __func__, ## args)


/**
* @brief  Cancel receive data for receiver
* @param[OUT]  None
* @param[IN]  None
* @return None
* @pre None
* @post None
* @attention None
*/
static void csi_ctrl_cancel_receive_data(void)
{
    CSI_MSG_DATA dummydata;
    int sys_ret = 0;

    dummydata.data_addr = 0x00;
    dummydata.packet_count = 0x00;

    sys_ret = kmsg_que_send(&csi_ctrl_kmsg_que, &dummydata, sizeof(CSI_MSG_DATA), NO_WAIT);
    if (sys_ret != 0) {
        CSI_CTRL_ERROR("csi ctrl kmsg_que_send message send error!\n");
    }
    return;
}

/**
* @brief  Initialize all data in the circular buffer
* @param[IN]  None
* @param[OUT]  <<cb>> the conctrol buffer
* @return None
* @pre None
* @post None
* @attention cb can not bu NULL
*/
static void csi_ctrl_buf_init(CSI_CIRC_BUFF* cb)
{
    cb->write = cb->read = 0;
    return ;
}


/**
* @brief  Check the received data valid or not.
* @param[IN]   None
* @param[OUT]  None
* @return the validity of the received data
* @pre None
* @post None
* @attention None
*/
static bool csi_ctrl_isRcvedDataValid(void)
{
    CSI_PACKET* pkt = notifyRcvDataInfo.data_addr + CSI_CTRL_RECEIVE_BUFF_BASE;
    unsigned long rcvSize = *((unsigned long*)(pkt->Data));
    unsigned long sndPktCnt = 0;

    sndPktCnt = CSI_CTRL_GetPkgCnt(rcvSize);

    return (sndPktCnt == notifyRcvDataInfo.packet_count) ? TRUE : FALSE;
}

/**
* @brief  Record the first packet for sending to user.
* @param[IN]   <<rPkt>> the pointer of the packet
* @param[OUT]  None
* @return  None
* @pre None
* @post None
* @attention the pointer of the packet can not be NULL
*/
static void csi_ctrl_rec_first_pkt(CSI_PACKET* rPkt)
{
    if (!notifyRcvDataInfo.packet_count) {
        notifyRcvDataInfo.data_addr = (unsigned long)(pRcb_mgr->write);
        notifyRcvDataInfo.packet_count = rPkt->seqNo;
    }
    return;
}

/**
* @brief  When dma transfer error, it is used for clear received packed.
* @param[IN]   None
* @param[OUT]  None
* @return  None
* @pre None
* @post None
* @attention None
*/
static void csi_ctrl_clear_rcved_pkts(void)
{
    if (notifyRcvDataInfo.packet_count) {
        pRcb_mgr->write = (int)(notifyRcvDataInfo.data_addr);  /* delete the received packet */
        notifyRcvDataInfo.data_addr = 0x00;
        notifyRcvDataInfo.packet_count = 0x00;
    }
    return;
}

/**
* @brief  Update the receive buffer write pointer.
* @param[IN]   None
* @param[OUT]  None
* @return  None
* @pre None
* @post None
* @attention None
*/
static void csi_ctrl_update_write_point(void)
{
    pRcb_mgr->write = CSI_MOD((pRcb_mgr->write + 1), (CSI_CTRL_CIRC_BUFF_SIZE - 1));
    /* update rbuff write pointer to RAM */
#ifndef USE_DMA_ALLOC_COHERENT
    dma_sync_single_for_device(NULL, (__pa(csi_ctrl_mmap_rbuff_base) + 32), 32, DMA_TO_DEVICE);
#endif
    /* update rbuff read pointer from RAM */

    /* Check receive buffer full or not */
    if ( CSI_CIRC_SPACE_TO_END(pRcb_mgr->write, pRcb_mgr->read, CSI_CTRL_CIRC_BUFF_SIZE) <= 0) {
        csi_ctrl_dma_status = CSI_DMA_STATUS_RCV_FULL;
    } else if (CSI_DMA_STATUS_RCV_FULL == csi_ctrl_dma_status) {
        csi_ctrl_dma_status = CSI_DMA_STATUS_NONE;
    } else {
        /* do nothing; */
    }
    return;
}

/**
* @brief  Handle the last packet.
* @param[IN]   None
* @param[OUT]  None
* @return  None
* @pre None
* @post None
* @attention None
*/
static void csi_ctrl_last_pkt_handler(void)
{
    int sys_ret = 0;

    /* Check the received data size */
    if (csi_ctrl_isRcvedDataValid()) {
        /* Notify Receiver to get data */
        sys_ret = kmsg_que_send(&csi_ctrl_kmsg_que, &notifyRcvDataInfo, sizeof(CSI_MSG_DATA), NO_WAIT);
        if (sys_ret != 0) {
            CSI_CTRL_ERROR("csi ctrl receive datainfo message send error!\n");
        }

        /* clear the receive data info for next */
        notifyRcvDataInfo.data_addr = 0x00;
        notifyRcvDataInfo.packet_count = 0x00;

        /* update receive buffer write */
        csi_ctrl_update_write_point();
    } else {
        /* The received data is invalid, need clear receive buffer */
        csi_ctrl_clear_rcved_pkts();
    }

}

/**
 * @brief  the handler of csi_dma
 * @param[IN]   <<pReceive>> the pointer of the received packet
 * @param[OUT]  None
 * @return  None
 * @pre None
 * @post None
 * @attention the pointer of the received packet can't be NULL
 */
static void csi_ctrl_dma_completed(unsigned int* pReceive)
{
    int sys_ret = 0;
    CSI_PACKET* rPkt = NULL;

    if (csi_lowlevel_count > 0) {
        csi_lowlevel_count--;
    }

    if (CSI_DMA_STATUS_NONE == csi_ctrl_dma_status) {
        csi_ctrl_dma_status = CSI_DMA_STATUS_REQRECEIVE;
    }

    /* update read buff  read pointer only */
#ifndef USE_DMA_ALLOC_COHERENT
    dma_sync_single_for_device(NULL, (__pa(csi_ctrl_mmap_rbuff_base) + 64), 32, DMA_FROM_DEVICE);
#endif

    rPkt = pRcb_mgr->buf + pRcb_mgr->write; /* Point to the receive packet */

    /* update send buffer read */
    if (CSI_DMA_STATUS_REQSEND == csi_ctrl_dma_status) {
        pScb_mgr->read = CSI_MOD((pScb_mgr->read + 1), (CSI_CTRL_CIRC_BUFF_SIZE - 1));
#ifndef USE_DMA_ALLOC_COHERENT
        /* update send buffer read pointer to ram */
        dma_sync_single_for_device(NULL, (__pa(csi_ctrl_mmap_sbuff_base) + 64), 32, DMA_TO_DEVICE);
        /* update send buffer write pointer from ram */
        dma_sync_single_for_device(NULL, (__pa(csi_ctrl_mmap_sbuff_base) + 32), 32, DMA_FROM_DEVICE);

#endif
    }

    /* update receive buffer write */
    /* Check receive packet is dummy or not */
    switch (rPkt->seqNo) {
    case 0x0000:  /* dummy packet */
        /* Do nothing for dummy packet */
        break;

    case 0x0001:  /* the last packet */
        /* The first packet received, record the start address and packet count */
        csi_ctrl_rec_first_pkt(rPkt);

        /* Handle the last packet */
        csi_ctrl_last_pkt_handler();

        break;

    default:  /* middle packet */
        /* The first packet received, record the start address and packet count */
        csi_ctrl_rec_first_pkt(rPkt);

        /* update receive buffer write */
        csi_ctrl_update_write_point();

        break;
    }

#ifdef USE_DMA_ALLOC_COHERENT
    *pReceive = __PA_CSI(pRcb_mgr->buf + pRcb_mgr->write);
#else
    *pReceive = __pa(pRcb_mgr->buf + pRcb_mgr->write);
#endif
    /* Check recevier buffer full or not */
    if (CSI_DMA_STATUS_RCV_FULL == csi_ctrl_dma_status) {
        /* Don't send data, because of receive buffer full */
        printk("**Receive Memory is full,no space!!***");
        return ;
    }

    /* Check send buffer empty or not,and receive the last packet */
    if ((CSI_CIRC_CNT(pScb_mgr->write, pScb_mgr->read, CSI_CTRL_CIRC_BUFF_SIZE) <= 0)
            && (rPkt->seqNo <= 0x0001)) {
        csi_ctrl_dma_status = CSI_DMA_STATUS_NONE;
        gpio_set_value(GPIO_P3, 1);
        if(csi_lowlevel_count > 1) {
            csi_lowlevel_callback();
        }

        return ;
    } else if ((CSI_CIRC_CNT(pScb_mgr->write, pScb_mgr->read, CSI_CTRL_CIRC_BUFF_SIZE) <= 0)
               && (rPkt->seqNo > 0x0001)) {
        csi_ctrl_dma_status = CSI_DMA_STATUS_REQRECEIVE;
        gpio_set_value(GPIO_P3, 1);
        if(csi_lowlevel_count > 1) {
            csi_lowlevel_callback();
        }
        return ;
    } else {
        csi_ctrl_dma_status = CSI_DMA_STATUS_REQSEND;
        /* do nothing */
    }

    gpio_set_value(GPIO_P3, 1);

    /* send packet by csi ctrl dma */
    sys_ret = csi_ctrl_dma_send_receive(pScb_mgr->buf + pScb_mgr->read);

    return;
}

/**
 * @brief  set value of the send pointer
 * @param[IN]   <<pSend>> the pointer of the send pointer
 * @param[OUT]  None
 * @return  None
 * @pre None
 * @post None
 * @attention None
 */
static int csi_ctrl_set_Send_Pointer(unsigned int* pSend)
{
    if (CSI_CIRC_CNT(pScb_mgr->write, pScb_mgr->read, CSI_CTRL_CIRC_BUFF_SIZE) > 0) {
#ifdef USE_DMA_ALLOC_COHERENT
        *pSend = __PA_CSI(pScb_mgr->buf + pScb_mgr->read);
#else
        *pSend = __pa(pScb_mgr->buf + pScb_mgr->read);
#endif
        csi_ctrl_dma_status = CSI_DMA_STATUS_REQSEND;
    } else {
        *pSend = dummy_phys;
        csi_ctrl_dma_status = CSI_DMA_STATUS_REQRECEIVE;
    }
    return 0;
}

/**
 * @brief  send and receive the data
 * @param[IN]   <<pSend>> the pointer of the send pointer
 * @param[OUT]  None
 * @return  None
 * @pre None
 * @post None
 * @attention None
 */
static int csi_ctrl_dma_send_receive(CSI_PACKET* pSend)
{
    int sys_ret = 0;

    /* dummy here */
    /* copy send packet to receive buffer */
#ifdef USE_DMA_ALLOC_COHERENT
    sys_ret = csi_dl_send_packet(__PA_CSI(pSend), 1);
#else
    sys_ret = csi_dl_send_packet(__pa(pSend), 1);
#endif

    return sys_ret;
}

/**
 * @brief  send data
 * @param[IN]   None
 * @param[OUT]  None
 * @return  None
 * @pre None
 * @post None
 * @attention None
 */
static int csi_ctrl_send_data(void)
{
    int sys_ret = 0;

    /* update write only */
#ifndef USE_DMA_ALLOC_COHERENT
    dma_sync_single_for_device(NULL, (__pa(csi_ctrl_mmap_sbuff_base) + 32), 32, DMA_FROM_DEVICE);
#endif

    if (CSI_DMA_STATUS_NONE == csi_ctrl_dma_status) {
        if (CSI_DMA_STATUS_NONE == csi_ctrl_dma_status) {
            if (0 == CSI_CIRC_CNT(pScb_mgr->write, pScb_mgr->read, CSI_CTRL_CIRC_BUFF_SIZE)) {
                return sys_ret;
            }
            csi_ctrl_dma_status = CSI_DMA_STATUS_REQSEND;
#ifdef USE_DMA_ALLOC_COHERENT
            sys_ret = csi_dl_send_packet(__PA_CSI(pScb_mgr->buf + pScb_mgr->read), 0);
#else
            sys_ret = csi_dl_send_packet(__pa(pScb_mgr->buf + pScb_mgr->read), 0);
#endif

        } else {
            return sys_ret;
        }
    }
    return sys_ret;
}

/**
 * @brief  receive data
 * @param[IN]   <<arg>> the pointer of userspace
 * @param[OUT]  None
 * @return  None
 * @pre None
 * @post None
 * @attention None
 */
static int csi_ctrl_receive_data(unsigned long arg)
{
    CSI_MSG_DATA rcv_data;
    int sys_ret = 0;

    /* update  read only */
#ifndef USE_DMA_ALLOC_COHERENT
    dma_sync_single_for_device(NULL, (__pa(csi_ctrl_mmap_rbuff_base) + 64), 32, DMA_FROM_DEVICE);
#endif
    sys_ret = kmsg_que_receive(&csi_ctrl_kmsg_que, &rcv_data, sizeof(CSI_MSG_DATA), WAIT_FOREVER);
    if (sys_ret) {
        CSI_CTRL_ERROR("receive msg from queue failed: sys_ret=%d\n", sys_ret);
        return sys_ret;
    }

    sys_ret = copy_to_user( (void __user*)arg, &rcv_data, sizeof(CSI_MSG_DATA) );
    if (0 != sys_ret) {
        CSI_CTRL_ERROR(" copy to user failed: sys_ret=%d\n", sys_ret );
        return -ENXIO;
    }

    return sys_ret;
}

/* file operations */
static struct file_operations csi_ctrl_fops = {
    .owner      = THIS_MODULE,
    .open       = csi_ctrl_drv_open,
    .mmap     = csi_ctrl_drv_mmap,
    .ioctl        = csi_ctrl_drv_ioctl,
    .release    = csi_ctrl_drv_release,
};

static int csi_ctrl_drv_open(struct inode* inode, struct file* file)
{
    /* do nothing */
    return 0;
}

static int csi_ctrl_drv_mmap(struct file* file, struct vm_area_struct* vma)
{
    unsigned long offset = vma->vm_pgoff << PAGE_SHIFT;
    unsigned long size = vma->vm_end - vma->vm_start;

    /*support reserved memory*/
    if (0 == offset) {
        /* check the request-map-size. */
        if (size > CSI_CTRL_MMAP_SBUFF_SIZE) {
            CSI_CTRL_ERROR("size[0x%08x] > CSI_CTRL_MMAP_SBUFF_SIZE. \n", (unsigned int)size);
            return -ENXIO;
        }

        vma->vm_flags |= VM_LOCKED;

        /* make it uncacheable... */
        vma->vm_page_prot = pgprot_noncached(vma->vm_page_prot);
#ifdef USE_DMA_ALLOC_COHERENT
        /* map physical memory to user space */
        if (remap_pfn_range(vma, vma->vm_start, (csi_ctrl_mmap_sbuff_bus_base) >> PAGE_SHIFT, size, vma->vm_page_prot)) {
            CSI_CTRL_ERROR("remap_pfn_range failed. \n");
            return -ENXIO;
        }
#else
        /* map physical memory to user space */
        if (remap_pfn_range(vma, vma->vm_start, __pa(csi_ctrl_mmap_sbuff_base) >> PAGE_SHIFT, size, vma->vm_page_prot)) {
            CSI_CTRL_ERROR("remap_pfn_range failed. \n");
            return -ENXIO;
        }
#endif
    } else if ((unsigned long)CSI_CTRL_MMAP_SBUFF_SIZE == offset) {
        /* check the request-map-size. */
        if (size > CSI_CTRL_MMAP_RBUFF_SIZE) {
            CSI_CTRL_ERROR("size[0x%08x] > CSI_CTRL_MMAP_RBUFF_SIZE. \n", (unsigned int)size);
            return -ENXIO;
        }

        vma->vm_flags |= VM_LOCKED;

        /* make it uncacheable... */
        vma->vm_page_prot = pgprot_noncached(vma->vm_page_prot);
#ifdef USE_DMA_ALLOC_COHERENT
        /* map physical memory to user space */
        if (remap_pfn_range(vma, vma->vm_start, (csi_ctrl_mmap_rbuff_bus_base) >> PAGE_SHIFT, size, vma->vm_page_prot)) {
            CSI_CTRL_ERROR("remap_pfn_range failed. \n");
            return -ENXIO;
        }
#else
        /* map physical memory to user space */
        if (remap_pfn_range(vma, vma->vm_start, __pa(csi_ctrl_mmap_rbuff_base) >> PAGE_SHIFT, size, vma->vm_page_prot)) {
            CSI_CTRL_ERROR("remap_pfn_range failed. \n");
            return -ENXIO;
        }
#endif
    } else {
        CSI_CTRL_ERROR("offset != 0, so need another remap. \n");
        return -EPERM;
    }

    return 0;
}

static int csi_ctrl_drv_release(struct inode* inode, struct file* file)
{
    return 0;
}

static int csi_ctrl_drv_ioctl(struct inode* inode, struct file* file, unsigned int cmd, unsigned long arg)
{
    void __user* usermem = (void __user*) arg;
    unsigned long size;
    int  ret = 0;

    /* Check the validation of input argument. */
    if ((_IOC_TYPE(cmd) != CSI_CTRL_IO_MAGIC) ||
            (_IOC_NR(cmd) > CSI_CTRL_MAX_NR)) {
        CSI_CTRL_ERROR("Invalid cmd: [0x%08x]\n", cmd );
        return -ENOTTY;
    }

    switch (cmd) {
        /* use for cancel receiver */
    case CSI_CTRL_RECEIVER_CANCEL:
        csi_ctrl_cancel_receive_data();
        break;
        /* maybe no use for user space */
    case CSI_CTRL_GET_SBUFF_PHY_ADDR:
        size = csi_ctrl_mmap_sbuff_base;
        if (copy_to_user(usermem, &size, sizeof(unsigned long)))
        {
            CSI_CTRL_ERROR("CSI_CTRL_GET_SBUFF_PHY_ADDR: copy_to_user failed. \n");
            ret = -ENXIO;
        }
        break;
    case CSI_CTRL_GET_RBUFF_PHY_ADDR:
        size = csi_ctrl_mmap_rbuff_base;
        if (copy_to_user(usermem, &size, sizeof(unsigned long)))
        {
            CSI_CTRL_ERROR("CSI_CTRL_GET_RBUFF_PHY_ADDR: copy_to_user failed. \n");
            ret = -ENXIO;
        }
        break;
    case CSI_CTRL_GET_BUFF_SIZE:

        size = CSI_CTRL_MMAP_SBUFF_SIZE;
        if (copy_to_user(usermem, &size, sizeof(unsigned long)))
        {
            CSI_CTRL_ERROR("CSI_CTRL_MMAP_SBUFF_SIZE: copy_to_user failed. \n");
            ret = -ENXIO;
        }
        break;
    case CSI_CTRL_SEND_DATA:
        /* Send data to DMA or msgQ */
        csi_ctrl_send_data();
        break;
    case CSI_CTRL_RECEIVE_DATA:
        /* Receive data for user */
        ret = csi_ctrl_receive_data(arg);
        break;
    case CSI_CTRL_DRV_INIT:
        /* init kernel flag */
        csi_drv_rdyflag_init();
        break;
    default :
        CSI_CTRL_ERROR( "Csi ctrl IOCtrl: no type run \n");
        ret = -EPERM;
    }

    return ret;
}

#ifdef USE_DMA_ALLOC_COHERENT

unsigned long csi_virt_to_phys(unsigned long addr)
{
    unsigned long ret;
    if (addr > csi_ctrl_mmap_sbuff_base && addr < (csi_ctrl_mmap_sbuff_base + CSI_CTRL_MMAP_SBUFF_SIZE)) {
        ret = addr -   csi_ctrl_mmap_sbuff_base + csi_ctrl_mmap_sbuff_bus_base;
    } else if (addr > csi_ctrl_mmap_rbuff_base && addr < (csi_ctrl_mmap_rbuff_base + CSI_CTRL_MMAP_RBUFF_SIZE)) {
        ret = addr -   csi_ctrl_mmap_rbuff_base + csi_ctrl_mmap_rbuff_bus_base;
    } else {
        printk(KERN_WARNING "[MEM]%s error!!\n", __FUNCTION__);
        return 0xFFFFFFFF;
    }
    return ret;
}
#endif

static void __exit csi_ctrl_drv_exit(void)
{
    kmsg_que_destroy(&csi_ctrl_kmsg_que);

    cdev_del(&csi_ctrl_cdev);
    unregister_chrdev_region(csi_ctrl_cdev.dev, 1);

    /* Free mmap buffer */
#ifdef USE_DMA_ALLOC_COHERENT
    dma_free_coherent(NULL, CSI_CTRL_MMAP_SBUFF_SIZE, csi_ctrl_mmap_sbuff_base, csi_ctrl_mmap_sbuff_bus_base);
#else
    free_pages(csi_ctrl_mmap_sbuff_base, get_order(CSI_CTRL_MMAP_SBUFF_SIZE));
#endif
#ifdef USE_DMA_ALLOC_COHERENT
    dma_free_coherent(NULL, CSI_CTRL_MMAP_RBUFF_SIZE, csi_ctrl_mmap_rbuff_base, csi_ctrl_mmap_rbuff_bus_base);
#else
    free_pages(csi_ctrl_mmap_rbuff_base, get_order(CSI_CTRL_MMAP_RBUFF_SIZE));
#endif
    csi_ctrl_mmap_sbuff_base = 0;
    csi_ctrl_mmap_rbuff_base = 0;
    pScb_mgr = NULL;
    pRcb_mgr = NULL;


    dma_free_coherent(NULL, (sizeof(CSI_PACKET)), dummy_virt , dummy_phys);


}

static int __init csi_ctrl_drv_init(void)
{
    int sys_ret = 0;

#ifdef USE_DMA_ALLOC_COHERENT
    csi_ctrl_mmap_sbuff_base = dma_alloc_coherent(NULL, CSI_CTRL_MMAP_SBUFF_SIZE, &csi_ctrl_mmap_sbuff_bus_base, GFP_KERNEL);
    printk(KERN_WARNING "%s base = 0x%x, bus = 0x%x\n", __FUNCTION__, csi_ctrl_mmap_sbuff_base, csi_ctrl_mmap_sbuff_bus_base);
#else
    /* Malloc mmap sbuffer  */
    csi_ctrl_mmap_sbuff_base = __get_free_pages(GFP_KERNEL, get_order(CSI_CTRL_MMAP_SBUFF_SIZE));
#endif
    if (0 == csi_ctrl_mmap_sbuff_base) {
        CSI_CTRL_ERROR("__get_free_pages for csi ctrl failed.\n");
        return -ENOMEM;
    }
#ifdef USE_DMA_ALLOC_COHERENT
    csi_ctrl_mmap_rbuff_base = dma_alloc_coherent(NULL, CSI_CTRL_MMAP_RBUFF_SIZE, &csi_ctrl_mmap_rbuff_bus_base, GFP_KERNEL);
    printk(KERN_WARNING "%s base = 0x%x, bus = 0x%x\n", __FUNCTION__, csi_ctrl_mmap_rbuff_base, csi_ctrl_mmap_rbuff_bus_base);
#else
    /* Malloc mmap sbuffer  */
    csi_ctrl_mmap_rbuff_base = __get_free_pages(GFP_KERNEL, get_order(CSI_CTRL_MMAP_RBUFF_SIZE));
#endif
    if (0 == csi_ctrl_mmap_rbuff_base) {
        free_pages(csi_ctrl_mmap_sbuff_base, get_order(CSI_CTRL_MMAP_SBUFF_SIZE));
        csi_ctrl_mmap_sbuff_base = 0;
        CSI_CTRL_ERROR("__get_free_pages for csi ctrl failed.\n");
        return -ENOMEM;
    }

    cdev_init(&csi_ctrl_cdev, &csi_ctrl_fops);

    csi_ctrl_cdev.owner = csi_ctrl_fops.owner;

    sys_ret = alloc_chrdev_region(&csi_ctrl_cdev.dev, 0, 1, CSI_CTRL_NAME);
    if (0 != sys_ret) {
        CSI_CTRL_ERROR("csi_ctrl register err!\n");
        free_pages(csi_ctrl_mmap_sbuff_base, get_order(CSI_CTRL_MMAP_SBUFF_SIZE));
        free_pages(csi_ctrl_mmap_rbuff_base, get_order(CSI_CTRL_MMAP_RBUFF_SIZE));
        csi_ctrl_mmap_sbuff_base = 0;
        csi_ctrl_mmap_rbuff_base = 0;
    }

    sys_ret = cdev_add(&csi_ctrl_cdev, csi_ctrl_cdev.dev, 1);
    if (0 != sys_ret) {
        CSI_CTRL_ERROR("cdev_add csi_ctrl to devices failed!\n");
        free_pages(csi_ctrl_mmap_sbuff_base, get_order(CSI_CTRL_MMAP_SBUFF_SIZE));
        free_pages(csi_ctrl_mmap_rbuff_base, get_order(CSI_CTRL_MMAP_RBUFF_SIZE));
        csi_ctrl_mmap_sbuff_base = 0;
        csi_ctrl_mmap_rbuff_base = 0;
        unregister_chrdev_region(csi_ctrl_cdev.dev, 1);
    }

    /*create the messaged queue.*/
    sys_ret = kmsg_que_create(&csi_ctrl_kmsg_que, 256, sizeof(CSI_MSG_DATA));
    if (0 != sys_ret) {
        CSI_CTRL_ERROR("msgQCreate csi_kmsg_que err!\n");
        free_pages(csi_ctrl_mmap_sbuff_base, get_order(CSI_CTRL_MMAP_SBUFF_SIZE));
        free_pages(csi_ctrl_mmap_rbuff_base, get_order(CSI_CTRL_MMAP_RBUFF_SIZE));
        csi_ctrl_mmap_sbuff_base = 0;
        csi_ctrl_mmap_rbuff_base = 0;
        cdev_del(&csi_ctrl_cdev);
        unregister_chrdev_region(csi_ctrl_cdev.dev, 1);
    }

    pScb_mgr = (CSI_CIRC_BUFF*)CSI_CTRL_SEND_BUFF_MGR;
    pScb_mgr->buf = (CSI_PACKET*)CSI_CTRL_SEND_BUFF_BASE;

    csi_ctrl_buf_init(pScb_mgr);
    pRcb_mgr = (CSI_CIRC_BUFF*)CSI_CTRL_RECEIVE_BUFF_MGR;
    pRcb_mgr->buf = (CSI_PACKET*)CSI_CTRL_RECEIVE_BUFF_BASE;
    csi_ctrl_buf_init(pRcb_mgr);
    /* update buff/write/read */
#ifndef USE_DMA_ALLOC_COHERENT
    dma_sync_single_for_device(NULL, __pa(csi_ctrl_mmap_sbuff_base), 96, DMA_TO_DEVICE);
#endif

#ifndef USE_DMA_ALLOC_COHERENT
    dma_sync_single_for_device(NULL, __pa(csi_ctrl_mmap_rbuff_base), 96, DMA_TO_DEVICE);
#endif
    /* initialize for receive data */
    notifyRcvDataInfo.data_addr = 0x00;
    notifyRcvDataInfo.packet_count = 0x00;

    dummy_virt = dma_alloc_coherent(NULL, (sizeof(CSI_PACKET)), &dummy_phys, 0);
#ifdef USE_DMA_ALLOC_COHERENT
    csi_dl_init(csi_ctrl_dma_completed, csi_ctrl_set_Send_Pointer, dummy_phys, __PA_CSI(pRcb_mgr->buf + pRcb_mgr->write));
#else
    csi_dl_init(csi_ctrl_dma_completed, csi_ctrl_set_Send_Pointer, dummy_phys, __pa(pRcb_mgr->buf + pRcb_mgr->write));
#endif

    return sys_ret;
}

module_init(csi_ctrl_drv_init);
module_exit(csi_ctrl_drv_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("SONY");
MODULE_ALIAS("csi:char");
MODULE_DESCRIPTION("csi.");


