/* 2014-08-28: File added and changed by Sony Corporation */
/*
 *  arch/arm/mach-emxx/emif/emif_ctrl.c
 *  EMIF_CTRL control datalink and some interfaces for syscall
 *
 *  Copyright 2011 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>
#include <linux/unistd.h>
#include <linux/ioctl.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/uaccess.h>
#include <linux/mm.h>
#include <linux/semaphore.h>
#include <linux/sched.h>
#include <linux/dma-mapping.h>
#include <linux/spinlock.h>
#include <linux/timer.h>
#include <linux/delay.h>

#include "../kmsg_que.h"
#include "emif_datalink.h"

#include "emif_ctrl.h"

#define USE_DMA_ALLOC_COHERENT 1
dma_addr_t emif_ctrl_mmap_sbuff_bus_base;
dma_addr_t emif_ctrl_mmap_rbuff_bus_base;
unsigned long emif_virt_to_phys(unsigned long vaddr);
#define __PA_EMIF(addr) emif_virt_to_phys((unsigned long)(addr))

/*
 * @brief Emif dma status
 */
typedef enum {
    EMIF_DMA_STATUS_NONE = 0,
    EMIF_DMA_STATUS_IDLE,
    EMIF_DMA_STATUS_SEND,
    EMIF_DMA_STATUS_RECV,
    EMIF_DMA_STATUS_BUSY,
    EMIF_DMA_STATUS_RCV_FULL,
    EMIF_DMA_STATUS_ERROR,
    EMIF_DMA_STATUS_MAX,
} EMIF_CTRL_DMA_STATUS;


static struct cdev emif_ctrl_cdev = {0};
static kmsg_que_id_t emif_ctrl_kmsg_que;
static unsigned long emif_ctrl_mmap_sbuff_base = 0;
static unsigned long emif_ctrl_mmap_rbuff_base = 0;
static EMIF_CTRL_DMA_STATUS emif_ctrl_dma_status = EMIF_DMA_STATUS_NONE;
static EMIF_CIRC_BUFF* pScb_mgr = NULL;  /* send circle buffer manager */
static EMIF_CIRC_BUFF* pRcb_mgr = NULL;  /* receive circle buffer manager */
static EMIF_MSG_DATA notifyRcvDataInfo;
spinlock_t send_lock; /* global ioctl send lock */


/*add for 10s*/
spinlock_t ready_lock;  /* global ioctl ready lock*/
static int emif_ctrl_emma_status = 0;
static int emif_ctrl_emma_first_send = 0;

#define EMIF_CTRL_CXD4239_READY_YES      (0x00000001)
#define EMIF_FIRST_SEND_YES 0
#define EMIF_FIRST_SEND_NO  1
#define EMIF_CTRL_EMMA_READY_OK                 1
#define EMIF_CTRL_EMMA_READY_ON                 0
#define EMIF_CTRL_DELAY_TIME                    10

#define EMIF_CTRL_SEND_BUFF_MGR          (emif_ctrl_mmap_sbuff_base + EMIF_CTRL_SEND_BUFF_MGR_OFFSET)
#define EMIF_CTRL_SEND_BUFF_BASE         ((EMIF_PACKET *)(emif_ctrl_mmap_sbuff_base + EMIF_CTRL_SEND_BUFF_OFFSET))
#define EMIF_CTRL_RECEIVE_BUFF_MGR     (emif_ctrl_mmap_rbuff_base + EMIF_CTRL_RECEIVE_BUFF_MGR_OFFSET)
#define EMIF_CTRL_RECEIVE_BUFF_BASE    ((EMIF_PACKET *)(emif_ctrl_mmap_rbuff_base + EMIF_CTRL_RECEIVE_BUFF_OFFSET))

#define EMIF_CTRL_GetPkgCnt(size)            ((unsigned long)(((size+EMIFLIB_SEND_DATA_SIZE)+(EMIF_CTRL_PACKET_DATA_SIZE-1))/EMIF_CTRL_PACKET_DATA_SIZE))

static void emif_ctrl_buf_init(EMIF_CIRC_BUFF* cb);
static bool emif_ctrl_isRcvedDataValid(void);
static void emif_ctrl_rec_first_pkt(EMIF_PACKET* rPkt);
static void emif_ctrl_clear_rcved_pkts(void);
static void emif_ctrl_update_write_point(void);
static void emif_ctrl_last_pkt_handler(void);
static int emif_ctrl_dma_send_packet(EMIF_PACKET* pSend);
static int emif_ctrl_send_data(void);
static int emif_ctrl_receive_data(unsigned long arg);
static int emif_ctrl_spi_write(unsigned long arg);
static int emif_ctrl_spi_read(unsigned long arg);
static int emif_ctrl_send_or_recv_packet(void);
static int emif_ctrl_dma_transfer_complete(EMIF_DL_DMA_STATUS emif_dma_last_status);

static int emif_ctrl_drv_open(struct inode* inode, struct file* file);
static int emif_ctrl_drv_mmap(struct file* file, struct vm_area_struct* vma);
static int emif_ctrl_drv_release(struct inode* inode, struct file* file);
static int emif_ctrl_drv_ioctl(struct inode* inode, struct file* file, unsigned int cmd, unsigned long arg);
static void __exit emif_ctrl_drv_exit(void);
static int __init emif_ctrl_drv_init(void);




#define EMIF_CTRL_ERROR(fmt, args...)      \
    printk(KERN_ERR "%s: " fmt, __func__, ## args)




/*
* @brief       Cancel receive data for receiver.
* @param[OUT]  none
* @param[IN]   none
* @return      none
* @pre         none
* @post        none
* @attention   none
*/
static void emif_ctrl_cancel_receive_data(void)
{
    EMIF_MSG_DATA dummydata = {0};
    int sys_ret = 0;

    dummydata.data_addr = 0x00;
    dummydata.packet_count = 0x00;

    sys_ret = kmsg_que_send(&emif_ctrl_kmsg_que, &dummydata, sizeof(EMIF_MSG_DATA), NO_WAIT);
    if(sys_ret != 0) {
        EMIF_CTRL_ERROR("emif ctrl kmsg_que_send message send error!\n");
    }

    return;
}

/*
 * @brief       Initialize all data in the circular buffer.
 * @param[OUT]  none
 * @param[IN]   cb : write and read lock
 * @return      none
 * @pre         none
 * @post        none
 * @attention   none
 */
static void emif_ctrl_buf_init(EMIF_CIRC_BUFF* cb)
{
    cb->write = cb->read = 0;
    return;
}

/*
 * @brief       Check the received data valid or not.
 * @param[OUT]  none
 * @param[IN]   none
 * @return      none
 * @pre         none
 * @post        none
 * @attention   none
 */
static bool emif_ctrl_isRcvedDataValid(void)
{
    EMIF_PACKET* pkt = notifyRcvDataInfo.data_addr + EMIF_CTRL_RECEIVE_BUFF_BASE;
    unsigned long rcvSize = *((unsigned long*)(pkt->Data));
    unsigned long sndPktCnt = 0;

    sndPktCnt = EMIF_CTRL_GetPkgCnt(rcvSize);
    return (sndPktCnt == notifyRcvDataInfo.packet_count) ? TRUE : FALSE;
}

/*
 * @brief       Record the first packet for sending to user.
 * @param[OUT]  none
 * @param[IN]   rPkt : receive
 * @return      none
 * @pre         none
 * @post        none
 * @attention   none
 */
static void emif_ctrl_rec_first_pkt(EMIF_PACKET* rPkt)
{
    if(!notifyRcvDataInfo.packet_count ) {
        notifyRcvDataInfo.data_addr = (unsigned long)(pRcb_mgr->write);
        notifyRcvDataInfo.packet_count = rPkt->seqNo;
    }
    return;
}

/*
 * @brief       When dma transfer error, it is used for clear received packed.
 * @param[OUT]  none
 * @param[IN]   none
 * @return      none
 * @pre         none
 * @post        none
 * @attention   none
 */
static void emif_ctrl_clear_rcved_pkts(void)
{
    if(notifyRcvDataInfo.packet_count) {
        pRcb_mgr->write = (int)(notifyRcvDataInfo.data_addr);  /* delete the received packet */
        notifyRcvDataInfo.data_addr = 0x00;
        notifyRcvDataInfo.packet_count = 0x00;
    }

    return;
}

/*
 * @brief       Update the receive buffer write pointer.
 * @param[OUT]  none
 * @param[IN]   none
 * @return      none
 * @pre         none
 * @post        none
 * @attention   none
 */
static void emif_ctrl_update_write_point(void)
{
    pRcb_mgr->write = EMIF_MOD((pRcb_mgr->write + 1), (EMIF_CTRL_CIRC_BUFF_SIZE - 1) );
#ifndef USE_DMA_ALLOC_COHERENT


    /* update read buffer write pointer */
    dma_sync_single_for_device(NULL, (__pa(emif_ctrl_mmap_rbuff_base) + 32), 32, DMA_TO_DEVICE);
    /*  update write buff read poiner from ram */
    dma_sync_single_for_device(NULL, (__pa(emif_ctrl_mmap_sbuff_base) + 64), 32, DMA_FROM_DEVICE);
#endif

    return;
}

/*
 * @brief       Handle the last packet.
 * @param[OUT]  none
 * @param[IN]   none
 * @return      none
 * @pre         none
 * @post        none
 * @attention   none
 */
static void emif_ctrl_last_pkt_handler(void)
{
    int sys_ret = 0;

    /* Check the received data size */
    if(emif_ctrl_isRcvedDataValid()) {
        /* Notify Receiver to get data */
        sys_ret = kmsg_que_send(&emif_ctrl_kmsg_que, &notifyRcvDataInfo, sizeof(EMIF_MSG_DATA), NO_WAIT);
        if(sys_ret != 0) {
            EMIF_CTRL_ERROR("emif ctrl receive datainfo message send error!\n");
        }

        /* clear the receive data info for next */
        notifyRcvDataInfo.data_addr = 0x00;
        notifyRcvDataInfo.packet_count = 0x00;

        /* update receive buffer write */
        emif_ctrl_update_write_point();
    } else {
        /* The received data is invalid, need clear receive buffer */
        emif_ctrl_clear_rcved_pkts();
    }

    return;
}

/*
* @brief       emif_ctrl_dma_transfer_complete
* @param[OUT]  none
* @param[IN]   emif_dma_last_status:last dma status
* @return      none
* @pre         none
* @post        none
* @attention   none
*/
static int emif_ctrl_dma_transfer_complete(EMIF_DL_DMA_STATUS emif_dma_last_status)
{
    int ret = 0;

    /* the last dma transfer is write data to write buffer */
    if(EMIF_DL_STATUS_WRITE == emif_dma_last_status) {
        /* the first time ,update send buffer read pointer*/
        pScb_mgr->read = EMIF_MOD((pScb_mgr->read + 1), (EMIF_CTRL_CIRC_BUFF_SIZE - 1));
        /* update write buff read poiner to ram */
#ifndef USE_DMA_ALLOC_COHERENT


        dma_sync_single_for_device(NULL, (__pa(emif_ctrl_mmap_sbuff_base) + 64), 32, DMA_TO_DEVICE);
#endif
        /*this request is send data */
    }
    /* the last dma transfer is read data form write buffer */
    else if(EMIF_DL_STATUS_READ == emif_dma_last_status) {
        /* Check the packet is the last packet or not and update receive buffer write pointer */
        EMIF_PACKET* rPkt = NULL;
        /* Point to the receive packet */
        rPkt = pRcb_mgr->buf + pRcb_mgr->write;

        switch (rPkt->seqNo) {
        case 0x0000:  /* dummy packet */
            /* Do nothing for dummy packet */
            break;
        case 0x0001:  /* the last packet */
            /* The first packet received, record the start address and packet count */
            emif_ctrl_rec_first_pkt(rPkt);
            /* Handle the last packet and update write point */
            emif_ctrl_last_pkt_handler();
            break;
        default:  /* middle packet */
            /* The first packet received, record the start address and packet count */
            emif_ctrl_rec_first_pkt(rPkt);
            /* update receive buffer write */
            emif_ctrl_update_write_point();
            break;
        }
    } else {
        EMIF_CTRL_ERROR(" emif_dma_last_status is error \n");
        return -1;
    }
    ret = emif_ctrl_send_or_recv_packet();
    return  ret;
}

/*
* @brief       emif_ctrl_send_or_recv_packet
* @param[OUT]  none
* @param[IN]   none
* @return      none
* @pre         none
* @post        none
* @attention   none
*/
static int emif_ctrl_send_or_recv_packet(void)
{
    int ret = DMA_REQUEST;
    unsigned int pSend;
    unsigned int pReceive;
    short int request_mode = 0;

    /*add for 10s*/
    if(EMIF_CTRL_EMMA_READY_ON == emif_ctrl_emma_status) {
        emif_dl_enable_gpio();
        return 0;
    }

    request_mode = emif_dl_get_request_mode();

    if(EMIF_WRITE_REQUEST == request_mode || EMIF_BOTH_READ_WRITE_REQUEST == request_mode) {
#ifndef USE_DMA_ALLOC_COHERENT


        /*update write buff write poiner from ram*/
        dma_sync_single_for_device(NULL, (__pa(emif_ctrl_mmap_sbuff_base) + 32), 32, DMA_FROM_DEVICE);
#endif
        /* Check there are sent packets to send in send buffer */
        if(EMIF_CIRC_CNT(pScb_mgr->write, pScb_mgr->read, EMIF_CTRL_CIRC_BUFF_SIZE) > 0) {
#ifdef USE_DMA_ALLOC_COHERENT
            pSend = __PA_EMIF(pScb_mgr->buf + pScb_mgr->read);
#else
            pSend = __pa(pScb_mgr->buf + pScb_mgr->read);
#endif
            emif_ctrl_dma_status = EMIF_DMA_STATUS_SEND;
            emif_dl_send_packet(pSend);
        } else if(EMIF_BOTH_READ_WRITE_REQUEST == request_mode) { /* thera are not sent packets to send but recive packets to recive */
            if(EMIF_CIRC_SPACE(pRcb_mgr->write, pRcb_mgr->read, EMIF_CTRL_CIRC_BUFF_SIZE) <=  0) {
                /* recive buffer is full already*/
                emif_ctrl_dma_status = EMIF_DMA_STATUS_RCV_FULL;
                ret = NO_REQUEST;
                /*get memory ring buffer status*/
                printk("recv memory pool is already full\n");
            } else {
#ifdef USE_DMA_ALLOC_COHERENT
                pReceive = __PA_EMIF(pRcb_mgr->buf + pRcb_mgr->write);
#else
                pReceive = __pa(pRcb_mgr->buf + pRcb_mgr->write);
#endif
                emif_ctrl_dma_status = EMIF_DMA_STATUS_RECV;
                emif_dl_recv_packet(pReceive);
            }
        } else {
            /* the last packet is sended finish and no recive packets*/
            emif_ctrl_dma_status = EMIF_DMA_STATUS_IDLE;
            ret = NO_REQUEST;
            /*get memory ring buffer status*/
        }
    } else if(EMIF_READ_REQUEST == request_mode) {
        /* Check recevier buffer full or not ???*/
        if(EMIF_CIRC_SPACE(pRcb_mgr->write, pRcb_mgr->read, EMIF_CTRL_CIRC_BUFF_SIZE) <=  0) {
            EMIF_CTRL_ERROR("recv memory pool is already full\n");
            /* recive buffer is full already*/
            emif_ctrl_dma_status = EMIF_DMA_STATUS_RCV_FULL;

        } else {
#ifdef USE_DMA_ALLOC_COHERENT
            pReceive = __PA_EMIF(pRcb_mgr->buf + pRcb_mgr->write);
#else
            pReceive = __pa(pRcb_mgr->buf + pRcb_mgr->write);
#endif
            emif_ctrl_dma_status = EMIF_DMA_STATUS_RECV;
            emif_dl_recv_packet(pReceive);
        }
    } else {
        /*the write buffer is full and read buffer is empty */
        emif_ctrl_dma_status = EMIF_DMA_STATUS_IDLE;
        ret = NO_REQUEST;
        /*get memory ring buffer status*/

    }
    return ret;
}

/*
* @brief       emif_ctrl_dma_send_packet
* @param[OUT]  none
* @param[IN]   pSend:send data address
* @return      status
* @pre         none
* @post        none
* @attention   none
*/
static int emif_ctrl_dma_send_packet(EMIF_PACKET* pSend)
{
    int sys_ret = 0;
    short int request_mode = 0;
    int timer_num = 0;
    /*update write buffer write pointer form RAM*/
#ifndef USE_DMA_ALLOC_COHERENT
    dma_sync_single_for_device(NULL, (__pa(emif_ctrl_mmap_sbuff_base) + 32), 32, DMA_FROM_DEVICE);
#endif

    if(pSend != pScb_mgr->buf + pScb_mgr->read ) {
        emif_dl_enable_gpio();
        return sys_ret;
    }

    /*add for 10s*/
    if(EMIF_FIRST_SEND_YES == emif_ctrl_emma_first_send) {
        emif_ctrl_emma_first_send = EMIF_FIRST_SEND_NO;
        for(timer_num = 0; timer_num < EMIF_CTRL_DELAY_TIME * HZ; timer_num++) {
            if(EMIF_CTRL_CXD4239_READY_YES == emif_dl_get_communication_status()) {
                break;
            }
            msleep(EMIF_CTRL_DELAY_TIME);
        }
        if(HZ == timer_num) {
            printk("cxd4239 has not ready !\n");
        }
        /* 10s time out */
        if(EMIF_CTRL_DELAY_TIME* HZ == timer_num) {
            printk("----------emma emif timeout----------------\n");
            return sys_ret;
        }
    }

    request_mode = emif_dl_get_request_mode();
    if(-1 == request_mode) {
        printk("emif_ctrl_dma_send_packet  emif_ctrl_dma_status = %d \n", emif_ctrl_dma_status);
    }

    if(request_mode != EMIF_WRITE_REQUEST && request_mode != EMIF_BOTH_READ_WRITE_REQUEST) {
        /*enable mask*/
        emif_dl_enable_gpio();
        return sys_ret;
    }
    if (pScb_mgr->write == pScb_mgr->read) {
        emif_dl_enable_gpio();
        return sys_ret;
    }

    emif_ctrl_dma_status = EMIF_DMA_STATUS_SEND;
#ifdef USE_DMA_ALLOC_COHERENT
    sys_ret = emif_dl_send_packet(__PA_EMIF(pSend));
#else
    sys_ret = emif_dl_send_packet(__pa(pSend));
#endif

    return sys_ret;
}

/*
* @brief       emif_ctrl_send_data
* @param[OUT]  none
* @param[IN]   none
* @return      status
* @pre         none
* @post        none
* @attention   none
*/
static int emif_ctrl_send_data(void)
{
    int sys_ret = 0;
    if ((EMIF_DMA_STATUS_RECV == emif_ctrl_dma_status) || (EMIF_DMA_STATUS_SEND == emif_ctrl_dma_status)) {
        return sys_ret;
    }
    else {
        /* Require to send packet by emif ctrl dma */
        sys_ret = emif_ctrl_dma_send_packet(pScb_mgr->buf + pScb_mgr->read);
    }

    return sys_ret;
}

/*
* @brief       emif_ctrl_receive_data
* @param[OUT]  arg:user buffer address for reading data
* @param[IN]   none
* @return      status
* @pre         none
* @post        none
* @attention   none
*/
static int emif_ctrl_receive_data(unsigned long arg)
{
    EMIF_MSG_DATA rcv_data;
    int sys_ret = 0;
    sys_ret = kmsg_que_receive(&emif_ctrl_kmsg_que, &rcv_data, sizeof(EMIF_MSG_DATA), WAIT_FOREVER);
    if (sys_ret) {
        EMIF_CTRL_ERROR("receive msg from queue failed: sys_ret=%d\n", sys_ret);
		return sys_ret;
    }

    sys_ret = copy_to_user( (void __user*)arg, &rcv_data, sizeof(EMIF_MSG_DATA) );
    if( 0 != sys_ret ) {
        EMIF_CTRL_ERROR(" copy to user failed: sys_ret=%d\n", sys_ret );
        return -ENXIO;
    }

    return sys_ret;
}

/*
* @brief       emif_ctrl_spi_write
* @param[OUT]  arg:user buffer address for write spi register data
* @param[IN]   none
* @return      status
* @pre         none
* @post        none
* @attention   none
*/
static int emif_ctrl_spi_write(unsigned long arg)
{
    EMIF_ACCESS_SPI spi_data;
    int sys_ret = 0;
    sys_ret = copy_from_user(&spi_data,(void __user*)arg, sizeof(EMIF_ACCESS_SPI));
    if( 0 != sys_ret ) {
        EMIF_CTRL_ERROR(" copy from user failed: sys_ret=%d\n", sys_ret );
        return -1;
    }
    emif_dl_ab0_set_async_mode();
    emif_dl_async_write(spi_data.spi_addr,spi_data.spi_value);
    return 0;
}

/*
* @brief       emif_ctrl_spi_read
* @param[OUT]  arg:user buffer address for read and write spi register data
* @param[IN]   none
* @return      status
* @pre         none
* @post        none
* @attention   none
*/
static int emif_ctrl_spi_read(unsigned long arg)
{
    EMIF_ACCESS_SPI spi_data;
    int sys_ret = 0;
    sys_ret = copy_from_user(&spi_data,(void __user*)arg, sizeof(EMIF_ACCESS_SPI));
    if( 0 != sys_ret ) {
        EMIF_CTRL_ERROR(" copy from user failed: sys_ret=%d\n", sys_ret );
        return -1;
    }
    emif_dl_ab0_set_async_mode();
    spi_data.spi_value = emif_dl_async_read(spi_data.spi_addr);
    sys_ret = copy_to_user((void __user*)arg, &spi_data, sizeof(EMIF_ACCESS_SPI));
    if( 0 != sys_ret ) {
        EMIF_CTRL_ERROR(" copy to user failed: sys_ret=%d\n", sys_ret );
        return -1;
    }
    return 0;
}

/* file operations */
static struct file_operations emif_ctrl_fops = {
    .owner      = THIS_MODULE,
    .open       = emif_ctrl_drv_open,
    .mmap     = emif_ctrl_drv_mmap,
    .ioctl        = emif_ctrl_drv_ioctl,
    .release    = emif_ctrl_drv_release,
};

/*
 * @brief       driver open
 * @param[OUT]  none
 * @param[IN]   inode: standard linux device driver parameter
 * @param[IN]   file: standard linux device driver parameter
 * @return      status
 * @pre         none
 * @post        none
 * @attention   none
 */
static int emif_ctrl_drv_open(struct inode* inode, struct file* file)
{
    /* do nothing; */
    return 0;
}

/*
 * @brief       emif mmap
 * @param[OUT]  none
 * @param[IN]   file: standard linux device driver parameter
 * @param[IN]   vma: userland vm_area
 * @return      status
 * @pre         none
 * @post        none
 * @attention   none
 */
static int emif_ctrl_drv_mmap(struct file* file, struct vm_area_struct* vma)
{
    unsigned long offset = vma->vm_pgoff << PAGE_SHIFT;
    unsigned long size = vma->vm_end - vma->vm_start;

    /*support reserved memory*/
    if(0 == offset) {
        /* check the request-map-size. */
        if(size > EMIF_CTRL_MMAP_SBUFF_SIZE) {
            EMIF_CTRL_ERROR("size[0x%08x] > EMIF_CTRL_MMAP_SBUFF_SIZE. \n", (unsigned int)size);
            return -ENXIO;
        }

        vma->vm_flags |= VM_LOCKED;

        /* make it uncacheable... */
        vma->vm_page_prot = pgprot_noncached(vma->vm_page_prot);
#ifdef USE_DMA_ALLOC_COHERENT
        /* map physical memory to user space */
        if(remap_pfn_range(vma, vma->vm_start, (emif_ctrl_mmap_sbuff_bus_base) >> PAGE_SHIFT, size, vma->vm_page_prot)) {
            EMIF_CTRL_ERROR("remap_pfn_range failed. \n");
            return -ENXIO;
        }
#else
        /* map physical memory to user space */
        if(remap_pfn_range(vma, vma->vm_start, __pa(emif_ctrl_mmap_sbuff_base) >> PAGE_SHIFT, size, vma->vm_page_prot)) {
            EMIF_CTRL_ERROR("remap_pfn_range failed. \n");
            return -ENXIO;
        }
#endif
    } else if ((unsigned long)EMIF_CTRL_MMAP_SBUFF_SIZE == offset) {
        /* check the request-map-size. */
        if(size > EMIF_CTRL_MMAP_RBUFF_SIZE) {
            EMIF_CTRL_ERROR("size[0x%08x] > EMIF_CTRL_MMAP_RBUFF_SIZE. \n", (unsigned int)size);
            return -ENXIO;
        }

        vma->vm_flags |= VM_LOCKED;

        /* make it uncacheable... */
        vma->vm_page_prot = pgprot_noncached(vma->vm_page_prot);
#ifdef USE_DMA_ALLOC_COHERENT
        /* map physical memory to user space */
        if(remap_pfn_range(vma, vma->vm_start, (emif_ctrl_mmap_rbuff_bus_base) >> PAGE_SHIFT, size, vma->vm_page_prot)) {
            EMIF_CTRL_ERROR("remap_pfn_range failed. \n");
            return -ENXIO;
        }
#else
        /* map physical memory to user space */
        if(remap_pfn_range(vma, vma->vm_start, __pa(emif_ctrl_mmap_rbuff_base) >> PAGE_SHIFT, size, vma->vm_page_prot)) {
            EMIF_CTRL_ERROR("remap_pfn_range failed. \n");
            return -ENXIO;
        }
#endif
    } else {
        EMIF_CTRL_ERROR("offset != 0, so need another remap. \n");
        return -EPERM;
    }

    return 0;
}

/*
 * @brief       driver close
 * @param[OUT]  none
 * @param[IN]   inode: standard linux device driver parameter
 * @param[IN]   file: standard linux device driver parameter
 * @return      status
 * @pre         none
 * @post        none
 * @attention   none
 */
static int emif_ctrl_drv_release(struct inode* inode, struct file* file)
{
    return 0;
}

/*
 * @brief       emif get ready
 * @param[OUT]  none
 * @param[IN]   none
 * @return      status
 * @pre         none
 * @post        none
 * @attention   add for 10s wait
 */
static int emif_ctrl_get_ready(void)
{
    unsigned long flags;
    spin_lock_irqsave(&ready_lock, flags);
    /*mean that this ioctl is not first*/
    if(EMIF_CTRL_EMMA_READY_OK == emif_ctrl_emma_status) {
        spin_unlock_irqrestore(&ready_lock, flags);
        return 0;
    } else {
        emif_ctrl_emma_status = EMIF_CTRL_EMMA_READY_OK;
        if(EMIF_CTRL_CXD4239_READY_YES == emif_dl_get_communication_status()) {
            emif_dl_disable_gpio();
            spin_unlock_irqrestore(&ready_lock, flags);
            emif_ctrl_send_or_recv_packet();
            return 0;
        } else {
            spin_unlock_irqrestore(&ready_lock, flags);
            return 0;
        }
    }
}

/*
 * @brief       emif driver ioctl
 * @param[OUT]  none
 * @param[IN]   inodep: standard linux device driver parameter
 * @param[IN]   filp: standard linux device driver parameter
 * @param[IN]   command: command for ioctl request
 * @param[IN]   arg:  content which is transferred from kernel to user
 * @return      status
 * @pre         none
 * @post        none
 * @attention   none
 */
static int emif_ctrl_drv_ioctl(struct inode* inode, struct file* file, unsigned int cmd, unsigned long arg)
{
    int  ret = 0;
    unsigned long flags;
    /* Check the validation of input argument. */
    if( ((_IOC_TYPE(cmd)) != EMIF_CTRL_IO_MAGIC) ||
            ((_IOC_NR(cmd)) > EMIF_CTRL_MAX_NR) ) {
        EMIF_CTRL_ERROR("Invalid cmd: [0x%08x]\n", cmd );
        return -ENOTTY;
    }

    switch (cmd) {

        /* use for cancel receiver */
    case EMIF_CTRL_RECEIVER_CANCEL:
        emif_ctrl_cancel_receive_data();
        break;

    case EMIF_CTRL_SEND_DATA:
        /* Send data to DMA or msgQ */
        spin_lock_irqsave(&send_lock, flags);
        if ((EMIF_DMA_STATUS_RECV == emif_ctrl_dma_status) || (EMIF_DMA_STATUS_SEND == emif_ctrl_dma_status)) {
            spin_unlock_irqrestore(&send_lock, flags);
            return ret;
        }
        emif_dl_disable_gpio();
        emif_dl_start_workqueue();
        spin_unlock_irqrestore(&send_lock, flags);
        break;

    case EMIF_CTRL_RECEIVE_DATA:
        /* Receive data for user */
        ret = emif_ctrl_receive_data(arg);
        break;

    case EMIF_CTRL_DRV_INIT:
        emif_ctrl_get_ready();
        break;

    case EMIF_CTRL_SPI_WRITE:
        emif_ctrl_spi_write(arg);
        break;

    case EMIF_CTRL_SPI_READ:
        emif_ctrl_spi_read(arg);
        break;

    default :
        EMIF_CTRL_ERROR( " emif ctrl IOCtrl: no type run \n");
        ret = -EPERM;
    }
    return ret;
}

/*
 * @brief   emif va to pa
 * @param   vaddr: va
 * @return  pa
 * @pre     none
 * @post    none
 * @attention  none
 */
unsigned long emif_virt_to_phys(unsigned long vaddr)
{
    unsigned long ret;
    if(vaddr > emif_ctrl_mmap_sbuff_base && vaddr < (emif_ctrl_mmap_sbuff_base + EMIF_CTRL_MMAP_SBUFF_SIZE)) {
        ret = vaddr - emif_ctrl_mmap_sbuff_base + emif_ctrl_mmap_sbuff_bus_base;
    } else if(vaddr > emif_ctrl_mmap_rbuff_base && vaddr < (emif_ctrl_mmap_rbuff_base + EMIF_CTRL_MMAP_RBUFF_SIZE)) {
        ret = vaddr -   emif_ctrl_mmap_rbuff_base + emif_ctrl_mmap_rbuff_bus_base;
    } else {
        printk(KERN_WARNING "%s error!!\n", __FUNCTION__);
        return 0xFFFFFFFF;
    }
    return ret;
}

/*
 * @brief   emif module exit
 * @param   none
 * @return  none
 * @pre     none
 * @post    none
 * @attention  none
 */
static void __exit emif_ctrl_drv_exit(void)
{
    kmsg_que_destroy(&emif_ctrl_kmsg_que);
    cdev_del(&emif_ctrl_cdev);
    unregister_chrdev_region(emif_ctrl_cdev.dev, 1);
#ifdef USE_DMA_ALLOC_COHERENT
    dma_free_coherent(NULL, EMIF_CTRL_MMAP_SBUFF_SIZE, (void*)(emif_ctrl_mmap_sbuff_base), emif_ctrl_mmap_sbuff_bus_base);
    dma_free_coherent(NULL, EMIF_CTRL_MMAP_RBUFF_SIZE, (void*)(emif_ctrl_mmap_rbuff_base), emif_ctrl_mmap_rbuff_bus_base);
#else
    /* Free mmap buffer */
    free_pages(emif_ctrl_mmap_sbuff_base, get_order(EMIF_CTRL_MMAP_SBUFF_SIZE));
    free_pages(emif_ctrl_mmap_rbuff_base, get_order(EMIF_CTRL_MMAP_RBUFF_SIZE));
#endif
    emif_ctrl_mmap_sbuff_base = 0;
    emif_ctrl_mmap_rbuff_base = 0;
    pScb_mgr = NULL;
    pRcb_mgr = NULL;
}

/*
 * @brief       emif driver init
 * @param   none
 * @return      status
 * @pre     none
 * @post    none
 * @attention   none
 */
static int __init emif_ctrl_drv_init(void)
{
    int sys_ret = 0;
    /* Malloc mmap sbuffer  */
#ifdef USE_DMA_ALLOC_COHERENT
    emif_ctrl_mmap_sbuff_base = (unsigned long)dma_alloc_coherent(NULL, EMIF_CTRL_MMAP_SBUFF_SIZE, &emif_ctrl_mmap_sbuff_bus_base, GFP_KERNEL);
#else
    emif_ctrl_mmap_sbuff_base = __get_free_pages(GFP_KERNEL, get_order(EMIF_CTRL_MMAP_SBUFF_SIZE));
#endif
    if (0 == emif_ctrl_mmap_sbuff_base) {
        EMIF_CTRL_ERROR("__get_free_pages for emif ctrl failed.\n");
        return -ENOMEM;
    }
    /* Malloc mmap sbuffer  */
#ifdef USE_DMA_ALLOC_COHERENT
    emif_ctrl_mmap_rbuff_base = (unsigned long)dma_alloc_coherent(NULL, EMIF_CTRL_MMAP_RBUFF_SIZE, &emif_ctrl_mmap_rbuff_bus_base, GFP_KERNEL);
#else
    emif_ctrl_mmap_rbuff_base = __get_free_pages(GFP_KERNEL, get_order(EMIF_CTRL_MMAP_RBUFF_SIZE));
#endif
    if (0 == emif_ctrl_mmap_rbuff_base) {
        free_pages(emif_ctrl_mmap_sbuff_base, get_order(EMIF_CTRL_MMAP_SBUFF_SIZE));
        emif_ctrl_mmap_sbuff_base = 0;
        EMIF_CTRL_ERROR("__get_free_pages for emif ctrl failed.\n");
        return -ENOMEM;
    }

    cdev_init(&emif_ctrl_cdev, &emif_ctrl_fops);

    emif_ctrl_cdev.owner = emif_ctrl_fops.owner;

    sys_ret = alloc_chrdev_region(&emif_ctrl_cdev.dev, 0, 1, EMIF_CTRL_NAME);
    if(0  != sys_ret) {
        EMIF_CTRL_ERROR("emif_ctrl register err!\n");
        free_pages(emif_ctrl_mmap_sbuff_base, get_order(EMIF_CTRL_MMAP_SBUFF_SIZE));
        free_pages(emif_ctrl_mmap_rbuff_base, get_order(EMIF_CTRL_MMAP_RBUFF_SIZE));
        emif_ctrl_mmap_sbuff_base = 0;
        emif_ctrl_mmap_rbuff_base = 0;
    }

    sys_ret = cdev_add(&emif_ctrl_cdev, emif_ctrl_cdev.dev, 1);
    if(0  != sys_ret) {
        EMIF_CTRL_ERROR("cdev_add emif_ctrl to devices failed!\n");
        free_pages(emif_ctrl_mmap_sbuff_base, get_order(EMIF_CTRL_MMAP_SBUFF_SIZE));
        free_pages(emif_ctrl_mmap_rbuff_base, get_order(EMIF_CTRL_MMAP_RBUFF_SIZE));
        emif_ctrl_mmap_sbuff_base = 0;
        emif_ctrl_mmap_rbuff_base = 0;
        unregister_chrdev_region(emif_ctrl_cdev.dev, 1);
    }

    /*create the messaged queue.*/
    sys_ret = kmsg_que_create(&emif_ctrl_kmsg_que, 256, sizeof(EMIF_MSG_DATA));
    if(0  != sys_ret) {
        EMIF_CTRL_ERROR("msgQCreate emif_kmsg_que err!\n");
        free_pages(emif_ctrl_mmap_sbuff_base, get_order(EMIF_CTRL_MMAP_SBUFF_SIZE));
        free_pages(emif_ctrl_mmap_rbuff_base, get_order(EMIF_CTRL_MMAP_RBUFF_SIZE));
        emif_ctrl_mmap_sbuff_base = 0;
        emif_ctrl_mmap_rbuff_base = 0;
        cdev_del(&emif_ctrl_cdev);
        unregister_chrdev_region(emif_ctrl_cdev.dev, 1);
    }


    pScb_mgr = (EMIF_CIRC_BUFF*)EMIF_CTRL_SEND_BUFF_MGR;
    pScb_mgr->buf = (EMIF_PACKET*)EMIF_CTRL_SEND_BUFF_BASE;
    emif_ctrl_buf_init(pScb_mgr);
    pRcb_mgr = (EMIF_CIRC_BUFF*)EMIF_CTRL_RECEIVE_BUFF_MGR;
    pRcb_mgr->buf = (EMIF_PACKET*)EMIF_CTRL_RECEIVE_BUFF_BASE;
    emif_ctrl_buf_init(pRcb_mgr);

#ifndef USE_DMA_ALLOC_COHERENT
    dma_sync_single_for_device(NULL, __pa(emif_ctrl_mmap_sbuff_base), 96, DMA_TO_DEVICE);
    dma_sync_single_for_device(NULL, __pa(emif_ctrl_mmap_rbuff_base), 96, DMA_TO_DEVICE);
#endif
    /*init spin lock*/
    spin_lock_init(&send_lock);

    /* initialize for receive data */
    notifyRcvDataInfo.data_addr = 0x00;
    notifyRcvDataInfo.packet_count = 0x00;
    emif_dl_init(emif_ctrl_dma_transfer_complete, emif_ctrl_send_or_recv_packet,emif_ctrl_send_data);

    return sys_ret;
}

module_init(emif_ctrl_drv_init);
module_exit(emif_ctrl_drv_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("SONY");
MODULE_ALIAS("emif:char");
MODULE_DESCRIPTION("emif.");


