/* 2014-08-28: File added and changed by Sony Corporation */
/*
 *  arch/arm/mach-emxx/emif/emif_datalink.c
 *  EMIF_DATALINK interface and function for datalink
 *
 *  Copyright 2011 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/poll.h>
#include <linux/delay.h>
#include <linux/dma-mapping.h>
#include <linux/interrupt.h>
#include <linux/vmalloc.h>
#include <linux/io.h>
#include <linux/gpio.h>
#include <linux/time.h>
#include <linux/workqueue.h>
#include <linux/semaphore.h>
#include <mach/dma.h>
#include <asm/mach/irq.h>
#include <stdbool.h>
#include "emif_datalink.h"
#include "ab0.h"

#ifdef CONFIG_SNSC_HSS
#include <linux/snsc_hss.h>
#endif /* #ifdef CONFIG_SNSC_HSS */

#define USE_DMA_ALLOC_COHERENT 1

/* about gpio */
#define EMIF_DL_GPIO GPIO_P101
#define CLR_GPIO_101_INTERRUPT outl(0x00000020,0xe00501a8)
/* enable p101 gpio function and enable p101 input */
#define ENABLE_GPIO_101_PINS(n) (n) = inl(0xe014020c);  \
    (n) |= 0x00000020; \
    outl((n),0xe014020c);   \
    (n) = inl(0xe014032c);  \
    (n) |= 0x00000050; \
    outl((n),0xe014032c)
/* async falling dectection */
#define SET_GPIO_101_INTERRUPT_ASYNCHRONOUS(n)  (n) = inl(0xe00501c0); \
    (n) &= 0xff0fffff;\
    (n) |= 0x00900000;\
    outl((n),0xe00501c0)

/* async low level */
/*
#define SET_GPIO_101_INTERRUPT_ASYNCHRONOUS(n)  (n) = inl(0xe00501c0); \
                                                (n) &= 0xff0fffff;\
                                                (n) |= 0x00b00000;\
                                                outl((n),0xe00501c0)
*/

/* about ab0 for emif */
#define EMIF_CLK         (0x01)
#define EMIF_SET_MODE    (0x00)
#define EMIF_CS2         (0x01)
#define EMIF_CS3         (0x01)
#define EMIF_CS2_ASYNC_ENBALE       (0x00000004)
#define EMIF_CS3_BURST_ENBALE       (0x00000008)
#define EMIF_CS2_BASE_ADDR   (0x00000000)
#define EMIF_CS3_BASE_ADDR   (0x00020000)
#define EMIF_CS2_ADDR_MASK   (0xFFFE0000)
#define EMIF_CS3_ADDR_MASK   (0xFFFC0000)
#define EMIF_BURST_WAIT_MASK    (0x00000001)
#define EMIF_BURST_WAITCTRL    (0x01000101)
#define EMIF_BURST_WAITCTRL_W  (0x00000101)
#define EMIF_BURST_READCTRL    (0x00000003)
#define EMIF_BURST_CONTROL     (0x00000000)
#define EMIF_BURST_FLASHRCR    (0x00001503)
#define EMIF_BURST_FLASHWCR    (0x00000000)

#define EMIF_ASYNC_WAIT_MASK    (0x00000001)
#define EMIF_ASYNC_WAITCTRL    (0x01000102)
#define EMIF_ASYNC_WAITCTRL_W  (0x00000102)
#define EMIF_ASYNC_READCTRL    (0x00000000)
#define EMIF_ASYNC_CONTROL     (0x00000004)
#define EMIF_ASYNC_FLASHRCR    (0x00009503)
#define EMIF_ASYNC_FLASHWCR    (0x00000001)
#define EMIF_ASYNC_1_FIX        (0x00000000)
#define EMIF_ASYNC_2_FIX        (0x00008000)
#define EMIF_ASYNC_4_FIX        (0x00010000)
/* 32 increase */
#define EMIF_BURST_32_INC       (0x00016000)
/* 8 bytes incress */
#define EMIF_4_BURST_SIZE         (32)
#define EMIF_AB0_BURST_MODE        1
#define EMIF_AB0_ASYNC_MODE        2
#define EMIF_AB0_DEST_REGISTERS    0x0001fe00
#define REMAP(x)        ((x) - (0x2FFF0000) + (unsigned long)ab0_base_addr)

/* about smu */
#define SMU_BASE_ADDR           (0xe0110000)
#define SMU_SIZE                (0x2000)
#define SMU_NOVRMALA_DIV        (0xe0110320)
#define SMU_FLASHCLK_CTRL       (0xe0110364)
#define SMU_CLKCTRL             (0xe011039C)
#define SMU_ABGCLKCTRL          (0xe0110440)
#define SMU_FLAGCLKCTRL         (0xe0110444)
#define SMU_CKRQMODE_MASK1      (0xe0110704)
#define SMU_CKRQ_MODE           (0xe0110708)
#define EMIF_NOVRMALA_DIV       (0x01517100)
#define EMIF_FLASHCLK_CTRL      (0x00000001)
#define EMIF_CLKCTRL            (0x0021f321)
#define EMIF_ABGCLKCTRL         (0x00000001)
#define EMIF_FLAGCLKCTRL        (0x00000001)
#define EMIF_CKRQMODE_MASK1     (0x77ec4fff)
#define EMIF_CKRQ_MODE          (0xf0000001)
#define REMAP_SMU(x)    ((x) - (0xe0110000) + (unsigned long)emif_smu_base_addr)

/*about emma dma */
#define CHANNEL_WIDTH  EMXX_DMAC_DEFMODE_32BIT
#define DMA_BLOCK_SIZE     32
#define DMA_CHANNAL_NUMBER 8


/*about ring buffer*/
#define EMIF_DL_AXI_HEAD_ADDR       (0xFFFE0000)
#define EMIF_DL_AXI_OFFSET          (0x00000000)
#define EMIF_DL_STATUS_LENGTH       (512)
#define EMIF_DL_PACKET_LENGTH       (512)
#define EMIF_DL_WRITE_BUFF_SIZE     (5)
#define EMIF_DL_READ_BUFF_SIZE      (2)
/* Return a mod b  */
#define EMIF_DL_MOD(a, b)   (((a)+(b))%(b))
/* Return count in buffer. */
#define EMIF_DL_CIRC_CNT(write,read,size) (EMIF_DL_MOD(((write) - (read)), (size)))
/* Return space available, 0..size-1.  We always leave one free cell^M
   as a completely full buffer has write == read, which is the same as^M
   empty.  */
#define EMIF_DL_CIRC_SPACE(write,read,size) EMIF_DL_CIRC_CNT((read),((write)+1),(size))

#define EMIF_DL_GET_DMA_ADDR(x,y)   (EMIF_DL_AXI_HEAD_ADDR) + (EMIF_DL_STATUS_LENGTH) + (EMIF_DL_PACKET_LENGTH)*(y) + ((x)*(EMIF_DL_PACKET_LENGTH))

/* status for write buffer and read buffer */
#define WRITE_BUFFER_WP  (0xFFFE0000)           /* writer point of write buffer */
#define WRITE_BUFFER_RP  (0xFFFE0004)           /* writer point of read buffer */
#define WRITE_BUFFER_FULL_FLAG  (0xFFFE0008)    /* write buffer is full or not  */
#define READ_BUFFER_WP  (0xFFFE000C)            /* read point of write buffer */
#define READ_BUFFER_RP  (0xFFFE0010)            /* read point of read buffer */
#define READ_BUFFER_FULL_FLAG   (0xFFFE0014)    /* read buffer is full or not */
#define INTERRUPT_TO_CXD4239    (0xF2048004)    /* set interrupt to cxd4239  */
#define INTERRUPT_FROM_CXD4239  (0xF3000614)    /* clear interrupt from cxd4239 */
#define EMIF_COMMUNICATION_STATUS   (0xFFFE0018)        /* communication enbale  */

#define SET_INTERRUPT           (1)
#define CLR_INTERRUPT           (1<<16)

/* defined Global Variables */
static EMIF_DL_DMA_STATUS emif_dl_dma_status = EMIF_DL_STATUS_NONE;
static unsigned int emif_ab0_mode = 0;
static unsigned int emif_dl_dma_write_send_p;
static unsigned int emif_dl_dma_write_recv_p;
static unsigned int emif_dl_dma_read_send_p;
static unsigned int emif_dl_dma_read_recv_p;
static int (* irq_callback_func_for_dma)(EMIF_DL_DMA_STATUS emif_dma_last_status);
static int (* irq_callback_func_for_gpio)(void);
static int (* send_callback_func_for_ioctl)(void);
static unsigned int emif_channel_id = EMXX_DMAC_M2M_ACPU_LCH0;  /* < default channel id selected */
static dma_regs_t* emif_dma_args;   /* dma args pointer*/
static dma_addr_t emif_set_burst_phys;
void* emif_set_burst_virt;
void __iomem* ab0_base_addr;    /* ab0 */
void __iomem* emif_smu_base_addr;   /* smu */
void __iomem* emif_register_addr_base;   /* 1fe00 */
void __iomem* emif_4_fix_addr_base;      /* 0x10000 */
static int irq  = 0;
static int interrupt_type  = 0;
static struct work_struct   emif_dl_work;
static unsigned long emif_dl_baseaddr = 0;
#ifdef CONFIG_SNSC_HSS
struct hss_emif_dl_info {
    struct hss_node *node;
};
struct hss_emif_dl_info *p_hss_emif_dl_info;
#endif  /* #ifdef CONFIG_SNSC_HSS */


/* Internal functon declaration */
static int emif_dl_ab0_init(void);
static int emif_dl_ab0_set_burst_mode(void);
static void emif_dl_dma_irq_callback(void* data, int intsts, int intrawsts);
static irqreturn_t emif_dl_gpio_irq_handler(int irq, void* id);
static int emif_dl_start_dma_transfer(unsigned int dma_src, unsigned int dma_des, unsigned int dma_leng);
static int emif_dl_request_dma_channel(void);
static int emif_dl_gpio_init(void);
static int emif_dl_gpio_exit(void);
static void emif_dl_workqueue(struct work_struct* data);


/**
 * @brief  handle the bottom of interrupt
 * @param[out] none
 * @param[in] <<data>>
 * @return  None
 * @post None
 * @attention None
 */
static void emif_dl_workqueue(struct work_struct* data)
{
    int ret = DMA_REQUEST;

    if(interrupt_type == DMA_INTERRUPT) {
        unsigned long status_value;
        if(EMIF_DL_STATUS_SET_WRITE == emif_dl_dma_status) {
            emif_dl_dma_status = EMIF_DL_STATUS_WRITE;
            emif_dl_start_dma_transfer(emif_dl_dma_write_send_p, emif_dl_dma_write_recv_p, (unsigned int)EMIF_DL_PACKET_LENGTH);
            return ;
        } else if(EMIF_DL_STATUS_SET_READ == emif_dl_dma_status) {
            emif_dl_dma_status = EMIF_DL_STATUS_READ;
            emif_dl_start_dma_transfer(emif_dl_dma_read_send_p, emif_dl_dma_read_recv_p, (unsigned int)EMIF_DL_PACKET_LENGTH);
            return ;
        }

        if(emif_ab0_mode != EMIF_AB0_ASYNC_MODE) {
            emif_dl_ab0_set_async_mode();
        }
        if(EMIF_DL_STATUS_WRITE == emif_dl_dma_status) {
            /*modify write point of write buffer and add 1*/
            status_value = emif_dl_async_read(WRITE_BUFFER_WP);
            status_value = EMIF_DL_MOD((status_value + 1), EMIF_DL_WRITE_BUFF_SIZE);
            emif_dl_async_write(WRITE_BUFFER_WP, status_value);

            /*set interrupt to cxd4239*/
            emif_dl_async_write(INTERRUPT_TO_CXD4239, SET_INTERRUPT) ;
            emif_dl_async_write(0xF2048008, 1);
        } else if(EMIF_DL_STATUS_READ == emif_dl_dma_status) {
            /*first modify write point of write buffer and add 1*/
            status_value = emif_dl_async_read(READ_BUFFER_RP);
            status_value = EMIF_DL_MOD((status_value + 1), EMIF_DL_READ_BUFF_SIZE);
            emif_dl_async_write(READ_BUFFER_RP, status_value);
            /*last set interrupt to cxd4239*/
            emif_dl_async_write(INTERRUPT_TO_CXD4239, SET_INTERRUPT);
            emif_dl_async_write(0xF2048008, 1);
        } else {
            printk(KERN_INFO"%s():%d the status is error %d!\n", __func__, __LINE__, emif_dl_dma_status);
        }
        if(irq_callback_func_for_dma) {
            ret = irq_callback_func_for_dma(emif_dl_dma_status);
        }
    } else if(GPIO_INTERRUPT == interrupt_type) {
        if(irq_callback_func_for_gpio) {
            ret = irq_callback_func_for_gpio();
        }
    } else if(IOCTL_SEND == interrupt_type) {
        if(send_callback_func_for_ioctl) {
            send_callback_func_for_ioctl();
        }
    } else {
        /* do nothing */
    }

    if(ret == NO_REQUEST) {
        interrupt_type = NO_INTERRUPT;
        emif_dl_enable_gpio();
    }
}


/**
 * @brief ab0 common init
 * @param[in] none
 * @return init sucess or failed
 * @pre this module has been installed
 * @post      none
 * @attention
 */
static int emif_dl_ab0_init(void)
{
    /* ioremap for setting ab0 status */
    ab0_base_addr = ioremap_nocache(AB0_BASE_ADDR, AB0_SIZE);
    if(NULL == ab0_base_addr) {
        printk("Error occurred at ioremap AB0!");
        return -1;
    }
    /* smu init clock 33mhz */
    emif_smu_base_addr = ioremap_nocache(SMU_BASE_ADDR, SMU_SIZE);
    if(NULL == emif_smu_base_addr) {
        printk("Error occurred at ioremap smu!");
        return -1;
    }
    emif_dl_baseaddr = *(volatile unsigned long*)REMAP(AB0_CS2BASEADD);


    emif_dl_ab0_set_async_mode();

    /* ioremap 0x1fe00 */
    emif_register_addr_base = ioremap_nocache(EMIF_AB0_DEST_REGISTERS, sizeof(unsigned long));
    if(NULL == emif_register_addr_base) {
        printk("Error occurred at ioremap command addr 0x1fe00!");
        return -1;
    }

    /* ioremap 0x10000   modify 09281 */
    emif_4_fix_addr_base = ioremap_nocache(EMIF_ASYNC_4_FIX, sizeof(unsigned long));
    if(NULL == emif_4_fix_addr_base) {
        printk("Error occurred at ioremap command addr 0x10000!");
        return -1;
    }

    return 0;
}

/**
 * @brief set ab0 burst mode
 * @param[in] none
 * @return set burst sucess or failed
 * @pre ab0 has been inited
 * @post None
 * @attention
 */
static int emif_dl_ab0_set_burst_mode(void)
{
    int  ret = 0;
    /*cs3 burst init*/
    *(volatile unsigned long*)REMAP(AB0_CS2BASEADD) = emif_dl_baseaddr;

    *(volatile unsigned long*)REMAP(AB0_CS3BASEADD) = EMIF_CS3_BASE_ADDR;
    *(volatile unsigned long*)REMAP(AB0_CS3BITCOMP) = EMIF_CS3_ADDR_MASK;
    *(volatile unsigned long*)REMAP(AB0_CS3WAIT_MASK) = EMIF_BURST_WAIT_MASK;
    *(volatile unsigned long*)REMAP(AB0_CS3WAITCTRL) = EMIF_BURST_WAITCTRL;
    *(volatile unsigned long*)REMAP(AB0_CS3WAITCTRL_W) = EMIF_BURST_WAITCTRL_W;
    *(volatile unsigned long*)REMAP(AB0_CS3READCTRL) = EMIF_BURST_READCTRL;
    *(volatile unsigned long*)REMAP(AB0_CS3CONTROL) = EMIF_BURST_CONTROL;
    *(volatile unsigned long*)REMAP(AB0_CS3FLASHRCR) = EMIF_BURST_FLASHRCR;
    *(volatile unsigned long*)REMAP(AB0_CS3FLASHWCR) = EMIF_BURST_FLASHWCR;
    *(volatile unsigned long*)REMAP_SMU(SMU_NOVRMALA_DIV) = EMIF_NOVRMALA_DIV;
    *(volatile unsigned long*)REMAP_SMU(SMU_FLASHCLK_CTRL) = EMIF_FLASHCLK_CTRL;
    *(volatile unsigned long*)REMAP_SMU(SMU_CLKCTRL) = EMIF_CLKCTRL;
    *(volatile unsigned long*)REMAP_SMU(SMU_ABGCLKCTRL) = EMIF_ABGCLKCTRL;
    *(volatile unsigned long*)REMAP_SMU(SMU_FLAGCLKCTRL) = EMIF_FLAGCLKCTRL;
    *(volatile unsigned long*)REMAP_SMU(SMU_CKRQMODE_MASK1) = EMIF_CKRQMODE_MASK1;
    *(volatile unsigned long*)REMAP_SMU(SMU_CKRQ_MODE) = EMIF_CKRQ_MODE;
    *(volatile unsigned long*)REMAP(AB0_FLASHCOMSET) = EMIF_CS3_BURST_ENBALE;
    /*clock driver capi 12ma*/
    outl(0xaaaaaaab,0xe0140408);

    emif_ab0_mode = EMIF_AB0_BURST_MODE;

    return ret;
}

/**
 * @brief  set ab0 async mode
 * @param[in] none
 * @return set async sucess or failed
 * @pre ab0 has been inited
 * @post    none
 * @attention
 */
int emif_dl_ab0_set_async_mode(void)
{
    int  ret = 0;
    /*cs2 async init*/
    *(volatile unsigned long*)REMAP(AB0_CS2BASEADD) = EMIF_CS2_BASE_ADDR;
    *(volatile unsigned long*)REMAP(AB0_CS2BITCOMP) = EMIF_CS2_ADDR_MASK;
    *(volatile unsigned long*)REMAP(AB0_CS2WAITCTRL) = EMIF_ASYNC_WAITCTRL;
    *(volatile unsigned long*)REMAP(AB0_CS2WAITCTRL_W) = EMIF_ASYNC_WAITCTRL_W;
    *(volatile unsigned long*)REMAP(AB0_CS2READCTRL) = EMIF_ASYNC_READCTRL;
    *(volatile unsigned long*)REMAP(AB0_CS2WAIT_MASK) = EMIF_ASYNC_WAIT_MASK;
    *(volatile unsigned long*)REMAP(AB0_CS2CONTROL) = EMIF_ASYNC_CONTROL;
    *(volatile unsigned long*)REMAP(AB0_CS2FLASHRCR) = EMIF_ASYNC_FLASHRCR;
    *(volatile unsigned long*)REMAP(AB0_CS2FLASHWCR) = EMIF_ASYNC_FLASHWCR;
    *(volatile unsigned long*)REMAP_SMU(SMU_NOVRMALA_DIV) = EMIF_NOVRMALA_DIV;
    *(volatile unsigned long*)REMAP_SMU(SMU_FLASHCLK_CTRL) = EMIF_FLASHCLK_CTRL;
    *(volatile unsigned long*)REMAP_SMU(SMU_CLKCTRL) = EMIF_CLKCTRL;
    *(volatile unsigned long*)REMAP_SMU(SMU_ABGCLKCTRL) = EMIF_ABGCLKCTRL;
    *(volatile unsigned long*)REMAP_SMU(SMU_FLAGCLKCTRL) = EMIF_FLAGCLKCTRL;
    *(volatile unsigned long*)REMAP_SMU(SMU_CKRQMODE_MASK1) = EMIF_CKRQMODE_MASK1;
    *(volatile unsigned long*)REMAP_SMU(SMU_CKRQ_MODE) = EMIF_CKRQ_MODE;
    *(volatile unsigned long*)REMAP(AB0_FLASHCOMSET) = EMIF_CS2_ASYNC_ENBALE;

    emif_ab0_mode = EMIF_AB0_ASYNC_MODE;

    return ret;
}
EXPORT_SYMBOL(emif_dl_ab0_set_async_mode);

/**
 * @brief  handle after dma transfer is complete  as callback func
 * @param[out] none
 * @param[in] <<data>>
 * @param[in] <<intsts>>
 * @param[in] <<intrawsts>>
 * @return normal
 * @pre registers in emif_dl_request_dma_channel
 * @post None
 * @attention None
 */
static void emif_dl_dma_irq_callback(void* data, int intsts, int intrawsts)
{
    interrupt_type = DMA_INTERRUPT;
    schedule_work(&emif_dl_work);


#ifndef USE_DMA_ALLOC_COHERENT

    dma_sync_single_for_device(NULL, emif_dl_dma_read_recv_p, EMIF_DL_PACKET_LENGTH, DMA_FROM_DEVICE);
#endif
}

/**
 * @brief  handle GPIO irq interrupt
 * @param[in]  <<irq>> Num of interrupt
 * @param[in]  <<id>>  interrupt info
 * @return status of this process
 * @pre none
 * @post none
 * @attention none
 */
static irqreturn_t emif_dl_gpio_irq_handler(int irq, void* id)
{
    emif_dl_disable_gpio();/*the first time mask interrupt  */
    interrupt_type = GPIO_INTERRUPT;
    schedule_work(&emif_dl_work);

    return IRQ_HANDLED;
}

/**
 *  @brief  init dma and start dma。
 *  @param[in]  <<dma_src>>  dma src addr
 *  @param[in]  <<dma_des>>  dma des addr
 *  @param[in]  <<dma_leng>>  dma the number of transfer byte
 *  @return start dma success or failed
 *  @pre    none
 *  @post   none
 *  @attention none
 */
static int emif_dl_start_dma_transfer(unsigned int dma_src, unsigned int dma_des, unsigned int dma_leng)
{
    int ret = 0;
    emif_dma_args->aadd = dma_src;
    emif_dma_args->badd = dma_des;
    emif_dma_args->aoff = 0;
    emif_dma_args->boff = 0;
    emif_dma_args->size = DMA_BLOCK_SIZE;
    emif_dma_args->leng = dma_leng;
    emif_dma_args->mode = CHANNEL_WIDTH;

    /* start DMA transfer   */
    ret = emxx_start_m2m_dma(EMXX_M2M_DMA_LCH(emif_channel_id), EMXX_DMAC_INT_LENG_EN);
    if (ret) {
        emxx_free_dma(emif_channel_id);
        printk(KERN_INFO "dma start error\n");
    }
    return ret;
}

/**
 * @brief       request a idle dma channel。
 * @param[OUT]  none
 * @param[IN]   none
 * @return      sucess or failed
 * @pre         none
 * @post        none
 * @attention   none
 */
static int emif_dl_request_dma_channel(void)
{
    int ret;
    int channel_num = 0;

    for(channel_num = 0; channel_num < DMA_CHANNAL_NUMBER; channel_num ++) {
        switch (channel_num) {
        case 0:
            emif_channel_id = EMXX_DMAC_M2M_ACPU_LCH0;
            break;
        case 1:
            emif_channel_id = EMXX_DMAC_M2M_ACPU_LCH1;
            break;
        case 2:
            emif_channel_id = EMXX_DMAC_M2M_ACPU_LCH2;
            break;
        case 3:
            emif_channel_id = EMXX_DMAC_M2M_ACPU_LCH3;
            break;
        case 4:
            emif_channel_id = EMXX_DMAC_M2M_ACPU_LCH4;
            break;
        case 5:
            emif_channel_id = EMXX_DMAC_M2M_ACPU_LCH5;
            break;
        case 6:
            emif_channel_id = EMXX_DMAC_M2M_ACPU_LCH6;
            break;
        case 7:
            emif_channel_id = EMXX_DMAC_M2M_ACPU_LCH7;
            break;
        }
        /* request DMA channel */
        ret = emxx_request_dma(emif_channel_id, "emif_dl_dma", emif_dl_dma_irq_callback, NULL, &emif_dma_args);
        if (0 == ret) {
            break;
        }
    }
    return ret;
}

/**
 * @brief initiate conditions of gpio install handler of gpio interrupt
 * @param[OUT] None
 * @param[IN] None
 * @return status of this process
 * @pre
 * @post None
 * @attention None
 */
static int emif_dl_gpio_init(void)
{

    int ret = 0;
    unsigned int val = 0;
    ENABLE_GPIO_101_PINS(val);
    irq = gpio_to_irq(EMIF_DL_GPIO);
    set_irq_type(irq, IRQ_TYPE_EDGE_FALLING); /* interrupt detection mode setup IRQ_TYPE_EDGE_FALLING */
    SET_GPIO_101_INTERRUPT_ASYNCHRONOUS(val);

    /* enable irq  */
    enable_irq(irq);
    gpio_direction_input(EMIF_DL_GPIO);/* input */
    ret = request_irq(INT_GPIO_BASE + EMIF_DL_GPIO, emif_dl_gpio_irq_handler, IRQF_DISABLED, "EMIFDLGPIO", NULL);
    if (ret) {
        printk(KERN_INFO"%s():%d irq request failed \n", __func__, __LINE__);
    }
    return ret;
}

/**
 * @brief  clear conditions of gpio free interrupt handler
 * @param[OUT] None
 * @param[IN] None
 * @return normal
 * @pre
 * @post None
 * @attention None
 */
static int emif_dl_gpio_exit(void)
{
    disable_irq(irq);
    free_irq(INT_GPIO_BASE + EMIF_DL_GPIO, NULL);
    return 0;
}


/**
 * @brief get read or write request
 * @param[IN] none
 * @return request mode
 * @pre none
 * @post none
 * @attention  none
 */
short int emif_dl_get_request_mode(void)
{
    short int irq_mode = 0;
    if(emif_ab0_mode != EMIF_AB0_ASYNC_MODE) {
        emif_dl_ab0_set_async_mode();
    }

    if(emif_dl_async_read(WRITE_BUFFER_RP) > EMIF_DL_WRITE_BUFF_SIZE ) {
        printk("Error:the value of WRITE_BUFFER_RP cross the border!\n");
        return -1;
    }

    if(emif_dl_async_read(WRITE_BUFFER_WP) > EMIF_DL_WRITE_BUFF_SIZE ) {
        printk("Error:the value of WRITE_BUFFER_WP cross the border!\n");
        return -1;
    }

    if(emif_dl_async_read(READ_BUFFER_RP) > EMIF_DL_READ_BUFF_SIZE ) {
        printk("Error:the value of READ_BUFFER_RP cross the border!\n");
        return -1;
    }

    if(emif_dl_async_read(READ_BUFFER_WP) > EMIF_DL_READ_BUFF_SIZE ) {
        printk("Error:the value of READ_BUFFER_WP cross the border!\n");
        return -1;
    }

    if(EMIF_DL_CIRC_SPACE((emif_dl_async_read(WRITE_BUFFER_WP)), (emif_dl_async_read(WRITE_BUFFER_RP)), EMIF_DL_WRITE_BUFF_SIZE) > 0) {
        irq_mode = EMIF_WRITE_REQUEST;
        if(EMIF_DL_CIRC_CNT((emif_dl_async_read(READ_BUFFER_WP)), (emif_dl_async_read(READ_BUFFER_RP)), EMIF_DL_READ_BUFF_SIZE) > 0) {
            irq_mode = EMIF_BOTH_READ_WRITE_REQUEST;
        }
    } else if(EMIF_DL_CIRC_CNT((emif_dl_async_read(READ_BUFFER_WP)), (emif_dl_async_read(READ_BUFFER_RP)), EMIF_DL_READ_BUFF_SIZE) > 0) {
        irq_mode = EMIF_READ_REQUEST;
    } else {
        /* do nothing */
    }

    return irq_mode;
}
EXPORT_SYMBOL(emif_dl_get_request_mode);



/**
 * @brief set call back func pointer
 * @param[OUT] None
 * @param[IN]  <<pfunc_dma_complete_callback>> the func point of dma callback
 * @param[IN]  <<pfunc_gpio_callback>> the func point of gpio callback
 * @return status of the emif_dl_init
 * @pre this module has been installed
 * @post None
 * @attention pFunc pointer can not be NULL
 */
int emif_dl_init(void* pfunc_dma_complete_callback, void* pfunc_gpio_callback, void* pfunc_ioctl_send)
{
    int ret = 0;
    irq_callback_func_for_dma = pfunc_dma_complete_callback;
    irq_callback_func_for_gpio = pfunc_gpio_callback;
    send_callback_func_for_ioctl = pfunc_ioctl_send;

    /* before enable irq clear interrupt */
    if(emif_ab0_mode != EMIF_AB0_ASYNC_MODE) {
        emif_dl_ab0_set_async_mode();
    }
    emif_dl_async_write(INTERRUPT_FROM_CXD4239, CLR_INTERRUPT);

    /* last init gpio interrupt */
    ret = emif_dl_gpio_init();
    if(ret != 0) {
        printk(KERN_INFO"gpio init failed\n");
        return ret;
    }

    return ret;
}
EXPORT_SYMBOL(emif_dl_init);

/**
 * @brief set DMA pointers and trans one packet
 * @param[OUT] None
 * @param[IN] <<dma_send_p>> the physical address of send packet
 * @return status of this process
 * @pre the datalink module is installed
 * @post None
 * @attention
 */
int emif_dl_send_packet(unsigned long dma_send_p)
{
    int ret = 0;
    unsigned long status_value;

    if(emif_ab0_mode != EMIF_AB0_ASYNC_MODE) {
        emif_dl_ab0_set_async_mode();
    }

    /* as content write to addr---emif_set_burst_virt */
    status_value = emif_dl_async_read(WRITE_BUFFER_WP);
    if(status_value > 4)
    {
        printk("emif_dl_send_packet Error:the value of WRITE_BUFFER_RP cross the border!\n");
        return -1;
    }

    *(unsigned long*)emif_set_burst_virt = (unsigned long)EMIF_DL_GET_DMA_ADDR(status_value, 0);

    emif_dl_dma_write_recv_p = EMIF_BURST_32_INC;
    emif_dl_dma_write_send_p = (unsigned long)dma_send_p;

    /* addr send to EMIF_AB0_DEST_REGISTERS */
    if(emif_ab0_mode != EMIF_AB0_BURST_MODE) {
        emif_dl_ab0_set_burst_mode();
    }
    emif_dl_dma_status = EMIF_DL_STATUS_SET_WRITE;
    ret = emif_dl_start_dma_transfer((unsigned int)emif_set_burst_phys, (unsigned int)EMIF_AB0_DEST_REGISTERS, DMA_BLOCK_SIZE);

    return ret;
}
EXPORT_SYMBOL(emif_dl_send_packet);

/**
 * @brief set DMA pointers and trans one packet
 * @param[OUT] None
 * @param[IN] <<dma_recv_p>> the physical address of send packet
 * @return status of this process
 * @pre
 * @post None
 * @attention none
 */
int emif_dl_recv_packet(unsigned long dma_recv_p)
{
    int ret = 0;
    unsigned long status_value;

    if(emif_ab0_mode != EMIF_AB0_ASYNC_MODE) {
        emif_dl_ab0_set_async_mode();
    }
    /* as content write to addr---emif_set_burst_virt */
    status_value = emif_dl_async_read(READ_BUFFER_RP);
    if(status_value > 2)
    {
        printk("emif_dl_recv_packet Error:the value of READ_BUFFER_RP cross the border!\n");
        return -1;

    }

    *(unsigned long*)emif_set_burst_virt = (unsigned long)EMIF_DL_GET_DMA_ADDR(status_value, EMIF_DL_WRITE_BUFF_SIZE);
    emif_dl_dma_read_recv_p  = (unsigned int)dma_recv_p;
    emif_dl_dma_read_send_p = EMIF_BURST_32_INC;
    if(emif_ab0_mode != EMIF_AB0_BURST_MODE) {
        emif_dl_ab0_set_burst_mode();
    }
    emif_dl_dma_status = EMIF_DL_STATUS_SET_READ;
    ret = emif_dl_start_dma_transfer((unsigned int)emif_set_burst_phys, (unsigned int)EMIF_AB0_DEST_REGISTERS, DMA_BLOCK_SIZE);

    return ret;
}
EXPORT_SYMBOL(emif_dl_recv_packet);

/**
 * @brief unmask GPIO interrupt
 * @param[OUT] None
 * @param[IN] None
 * @return normal
 * @pre
 * @post None
 * @attention None
 */
void emif_dl_enable_gpio(void)
{
    struct irq_desc* desc = NULL;
    desc = irq_to_desc(irq);
    if (desc != NULL)
        desc->chip->unmask(irq);

    return ;
}
EXPORT_SYMBOL(emif_dl_enable_gpio);

/**
 * @brief mask GPIO interrupt
 * @param[OUT] None
 * @param[IN] None
 * @return normal
 * @pre
 * @post None
 * @attention None
 */
void emif_dl_disable_gpio(void)
{
    struct irq_desc* desc = NULL;
    desc = irq_to_desc(irq);
    if (desc != NULL)
        desc->chip->mask(irq);
    CLR_GPIO_101_INTERRUPT;

    return ;
}
EXPORT_SYMBOL(emif_dl_disable_gpio);

/**
 * @brief get the interrupt tpye
 * @param[OUT] interrupt type
 * @param[IN] None
 * @return normal
 * @pre
 * @post None
 * @attention None
 */
int emif_dl_start_workqueue(void)
{
    interrupt_type = IOCTL_SEND;
    schedule_work(&emif_dl_work);
    return 0;
}
EXPORT_SYMBOL(emif_dl_start_workqueue);

/**
 * @brief mask GPIO interrupt
 * @param[OUT] None
 * @param[IN] None
 * @return normal
 * @pre
 * @post None
 * @attention None
 */
int emif_dl_get_communication_status(void)
{
    int ret = 0;
    if(emif_ab0_mode != EMIF_AB0_ASYNC_MODE) {
        emif_dl_ab0_set_async_mode();
    }
    ret = emif_dl_async_read(EMIF_COMMUNICATION_STATUS);

    return ret;
}
EXPORT_SYMBOL(emif_dl_get_communication_status);

/**
 * @brief cs2 async write
 * @param[in] <<async_addr>>   write addr
 * @param[in] <<async_value>>  write value
 * @return   sucess or failed
 * @pre
 * @post None
 * @attention None
 */
int emif_dl_async_write(unsigned long async_addr, unsigned long async_value)
{
    iowrite32(async_addr, emif_register_addr_base);

    iowrite32(async_value, emif_4_fix_addr_base);
    return 0;
}
EXPORT_SYMBOL(emif_dl_async_write);

/**
 * @brief cs2 async read
 * @param[in] <<async_addr>>   write addr
 * @return   the value of the input address
 * @pre
 * @post None
 * @attention None
 */
 unsigned long emif_dl_async_read(unsigned long async_addr)
{
    unsigned long async_value;
    iowrite32(async_addr, emif_register_addr_base);

    async_value = (unsigned long)ioread32(emif_4_fix_addr_base);

    return async_value;
}
EXPORT_SYMBOL(emif_dl_async_read);

#ifdef CONFIG_SNSC_HSS
static int emif_dl_hss_node_suspend(struct hss_node *node)
{
    emxx_free_dma(emif_channel_id);
    emif_dl_gpio_exit();
    return 0;
}

static int emif_dl_hss_node_resume(struct hss_node *node)
{
    int ret = 0;

    /*request dma channel*/
    ret = emif_dl_request_dma_channel();
    if(ret != 0) {
        printk(KERN_INFO "all DMA channels are busy in emif_dl_hss_node_resume\n");
        return ret;
    }

    /* set ab0 async mode */
    emif_dl_ab0_set_async_mode();
    emif_dl_async_write(WRITE_BUFFER_WP, 0);
    emif_dl_async_write(WRITE_BUFFER_RP, 0);
    emif_dl_async_write(INTERRUPT_FROM_CXD4239, CLR_INTERRUPT);

    /* init gpio interrupt */
    ret = emif_dl_gpio_init();
    if(ret != 0) {
        printk(KERN_INFO"gpio init failed in emif_dl_hss_node_resume\n");
        return ret;
    }

    return ret;
}

static struct hss_node_ops hss_emif_dl_node_ops = {
    .suspend = emif_dl_hss_node_suspend,
    .resume = emif_dl_hss_node_resume,
};

static int emif_dl_hss_node_init(void)
{
    struct hss_node *node;
    int error = 0;

    /* alloc for emif_dl info */
    p_hss_emif_dl_info = kzalloc(sizeof(*p_hss_emif_dl_info), GFP_KERNEL);
    if (!p_hss_emif_dl_info) {
        return -ENOMEM;
    }

    /*
     * alloc the HSS node
     */
    node = hss_alloc_node(EMIF_DL_E_HSS_NODE_NAME);
    if (!node) {
        kfree(p_hss_emif_dl_info);
        return -ENOMEM;
    }

    /*
     * save struct to HSS
     */
    hss_set_private(node, p_hss_emif_dl_info);
    p_hss_emif_dl_info->node = node;

    /*
     * Register the node.  If you want to get the callbacks to be called
     * before the specified node's one, specify the third parameter with
     * the name of the parent node. NULL means for no parent needed.
     */

    /* register HSS Platform for suspend/resume */
    error = hss_register_node(node, &hss_emif_dl_node_ops, EMIF_DL_E_HSS_NODE_PARENT);

    if (error) {
        hss_free_node(node);
        kfree(p_hss_emif_dl_info);
    }

    return error;
}

static void emif_dl_hss_node_exit(void)
{
    hss_unregister_node(p_hss_emif_dl_info->node);
    hss_free_node(p_hss_emif_dl_info->node);
    kfree(p_hss_emif_dl_info);
}
#endif /* #ifdef CONFIG_SNSC_HSS */

/** initialize
 * @brief  datalink driver install
 * @param[OUT] None
 * @param[IN] None
 * @return status of install
 * @pre dma module has been complied into kernel
 * @post None
 * @attention None
 */
static int __init emif_dl_drv_init(void)
{
    int ret = 0;
    printk(KERN_INFO"%s():%d emif dl init ioctl 0713-4 33m! \n", __func__, __LINE__);

#ifdef CONFIG_SNSC_HSS
    /* register driver to hss */
    ret = emif_dl_hss_node_init();
    if(ret != 0) {
        return ret;
    }
#endif /* #ifdef CONFIG_SNSC_HSS */

    /*request dma channel*/
    ret = emif_dl_request_dma_channel();
    if(ret != 0) {
        printk(KERN_INFO "all DMA channels are busy\n");
        return ret;
    }

    emif_set_burst_virt = dma_alloc_coherent(NULL, DMA_BLOCK_SIZE, &emif_set_burst_phys, 0);

    /* init ab0 common port */
    ret = emif_dl_ab0_init();
    if(ret != 0) {
        printk(KERN_INFO"ab0 init failed\n");
        emxx_free_dma(emif_channel_id);
        dma_free_coherent(NULL, sizeof(unsigned long), emif_set_burst_virt , emif_set_burst_phys);
    }

    INIT_WORK(&emif_dl_work, emif_dl_workqueue);

    /*init read ,write pointer*/
    if(emif_ab0_mode != EMIF_AB0_ASYNC_MODE) {
        emif_dl_ab0_set_async_mode();
    }
    emif_dl_async_write(WRITE_BUFFER_WP, 0);
    emif_dl_async_write(WRITE_BUFFER_RP, 0);

        return ret;
}

/** exit
 * @brief  datalink driver uninstall
 * @param[OUT] None
 * @param[IN] None
 * @return None
 * @pre spi1 working in FDX way as part of kernel
 * @post None
 * @attention None
 */
static void __exit emif_dl_exit(void)
{
    emxx_free_dma(emif_channel_id);
    emif_dl_gpio_exit();
    if(ab0_base_addr) {
        iounmap(ab0_base_addr);
    }

    if(emif_smu_base_addr) {
        iounmap(emif_smu_base_addr);
    }

    if(emif_register_addr_base) {
        iounmap(emif_register_addr_base);
    }

    if(emif_4_fix_addr_base) {
        iounmap(emif_4_fix_addr_base);
    }

    dma_free_coherent(NULL, DMA_BLOCK_SIZE, emif_set_burst_virt , emif_set_burst_phys);

#ifdef CONFIG_SNSC_HSS
    /* unregister driver from hss */
    emif_dl_hss_node_exit();
#endif /* #ifdef CONFIG_SNSC_HSS */

    return ;
}

module_init(emif_dl_drv_init);
module_exit(emif_dl_exit);
MODULE_LICENSE("GPL");

