﻿/*
 *  Sony EV/SY board specific code
 *
 *  Copyright 2010,2011,2012 Sony Corporation
 *  Based on arch/arm/mach-emxx/emev_board.c
 *
 *  Copyright (C) Renesas Electronics Corporation 2010
 *
 *  This program is free software;you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by Free
 *  Softwere Foundation; either version 2 of License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warrnty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 *  more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; If not, write to the Free Software Foundation, Inc., 59
 *  Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */
#include <linux/platform_device.h>
#include <linux/gpio.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/io.h>
#include <linux/types.h>
#include <linux/i2c.h>
#include <linux/irq.h>
#include <linux/smsc911x.h>
#ifdef CONFIG_ANDROID_PMEM
#include <linux/android_pmem.h>
#endif
#include <mach/emxx_mem.h>

#include <mach/smu.h>
#include <asm/mach-types.h>
#include <asm/mach/arch.h>
#include <asm/hardware/gic.h>
#include <asm/hardware/cache-l2x0.h>

#include <mach/hardware.h>
#include <mach/sdio.h>
#include <mach/evsy.h>

#include "../generic.h"
#include "../timer.h"

#if defined(CONFIG_MMC_EMXX_SDIO) || defined(CONFIG_MMC_EMXX_SDIO_MODULE)
#define __CONFIG_MMC_EMXX_SDIO_EVSY
#endif

static DEFINE_SPINLOCK(mod_reg_lock);

static void _mod_reg(unsigned int addr, u32 mask, u32 val)
{
	u32 org;
	unsigned long flag;

	spin_lock_irqsave(&mod_reg_lock, flag);
	org = readl(addr);
	org &= ~mask;
	org |= val & mask;
	writel(org, addr);
	spin_unlock_irqrestore(&mod_reg_lock, flag);
}

/*
 * for shared SDC/MSHC pin control
 */
void evsy_alter_sdc_pins(enum evsy_sdcmshc pin_type)
{
	switch (pin_type) {
	case EVSY_SDCMSHC_SDC:
	case EVSY_SDCMSHC_NOCARD:
		/* PINSEL to SD */
		_mod_reg(CHG_PINSEL_SD, 0x00000003, 0x00000000);
		/* Set GPIO_48 to SD_CKI */
		_mod_reg(CHG_PINSEL_G032, 0x00010000, 0x00000000);
		/* Drive capability */
		_mod_reg(CHG_DRIVE1, 0xF0000000, 0x50000000);
		/* Pull Up/Down */
		if (pin_type == EVSY_SDCMSHC_SDC) {
			/*
			 *   common: schmit, input enable
			 *   Data: pull UP
			 *   CMD: pull UP
			 *   CKI: Open
			 */
			_mod_reg(CHG_PULL4, 0x00000FFF, 0x00000DDC);
		} else {
			/*
			 *   common: normal input, mask input
			 *   Data: pull Down
			 *   CMD: pull Down
			 */
			_mod_reg(CHG_PULL4, 0x00000FF0, 0x00000330);
		}
		break;
	case EVSY_SDCMSHC_MSHC:
		/* PINSEL to MS */
		_mod_reg(CHG_PINSEL_SD, 0x00000003, 0x00000001);
		/* Drive capability */
		_mod_reg(CHG_DRIVE1, 0xF0000000, 0x50000000);
		/*
		 * Pull Up/Down
		 *   common: schmit, input enable
		 *   DATA: pull DOWN
		 *   BS(=CMD): Open
		 */
		_mod_reg(CHG_PULL4, 0x00000FF0, 0x00000FC0);
		break;
	default:
		pr_err("%s: Unknown type %d\n", __func__, pin_type);
	}
}

EXPORT_SYMBOL(evsy_alter_sdc_pins);

static int _evsy_gpio_on(unsigned port, int on_val, int on)
{
	int v;

	if (gpio_get_value(port) == on_val)
		v = 1;
	else
		v = 0;

	if (on)
		gpio_set_value(port, on_val);
	else
		gpio_set_value(port, !on_val);

	return v;
}

int evsy_media_power(int on)
{
	return _evsy_gpio_on(EVSY_GPIO_MC_POWER_OFF_L, 0, on);
}
EXPORT_SYMBOL(evsy_media_power);

int evsy_media_ins_led(int on)
{
	return _evsy_gpio_on(EVSY_GPIO_MS_INS_LED, 1, on);
}
EXPORT_SYMBOL(evsy_media_ins_led);

int evsy_media_acc_led(int on)
{
	return _evsy_gpio_on(EVSY_GPIO_MS_ACS_LED, 1, on);
}
EXPORT_SYMBOL(evsy_media_acc_led);


static int evsy_serial_ports[] = { 1, 1, 0, 0 };
int *emxx_resume_serials = evsy_serial_ports;

/*
 * for AB0_CSnBITCOMP registers
 * lower 16bits are reserved (read as zero)
 */
static u32 _ab0_cscomp_mask(u32 size)
{
	return ~(size - 1) & 0xFFFF0000;
}
/*
 * for AB0_CSnBASEADD registers
 * top most 2 and lower 16bits are reserved (read as zero)
 */
static u32 _ab0_csbaseadd(u32 addr)
{
	return addr & 0x3FFF0000;
}


/* SMSC 9221 (911x driver) */
#ifdef CONFIG_SMSC911X
static struct smsc911x_platform_config smsc911x_platdata = {
	.flags		= SMSC911X_USE_32BIT,
	.irq_type	= SMSC911X_IRQ_TYPE_PUSH_PULL,
	.irq_polarity	= SMSC911X_IRQ_POLARITY_ACTIVE_HIGH,
};

static struct resource smsc911x_resources[] = {
	[0] = {
		.start	= EVSY_ETHER_BASE,
		.end	= EVSY_ETHER_BASE + EVSY_ETHER_SIZE - 1,
		.flags	= IORESOURCE_MEM,
	},
	[1] = {
		.start	= EVSY_INT_LAN_INT,
		.end	= EVSY_INT_LAN_INT,
		.flags	= IORESOURCE_IRQ | IRQF_TRIGGER_HIGH,
	},
};

static struct platform_device smc911x_device = {
	.name	= "smsc911x",
	.id	= 0,
	.dev	= {
		  .platform_data = &smsc911x_platdata,
		},
	.num_resources	= ARRAY_SIZE(smsc911x_resources),
	.resource	= smsc911x_resources,
};
#endif /* CONFIG_SMSC911X */

#ifdef __CONFIG_MMC_EMXX_SDIO_EVSY
/* SDIO0: eMMC */
static void evsy_sdio_nop_release(struct device *dev)
{
	/* nothing to be released */
}
static struct platform_device evsy_sdio0_device = {
	.name = "emxx_sdio", /* Looks wrong? See emxx_sdio.c */
	.id = 0,
	.dev = {
		.release = evsy_sdio_nop_release,
	}
};
#endif /* __CONFIG_MMC_EMXX_SDIO_EVSY */

#ifdef CONFIG_ANDROID_PMEM
/* PMEM */
static struct android_pmem_platform_data android_pmem_pdata = {
	.name	= "pmem",
	.start	= EMXX_PMEM_BASE,
	.size	= EMXX_PMEM_SIZE,
	.no_allocator = 1,
	.cached	= 1,
};
static struct platform_device android_pmem_device = {
	.name	= "android_pmem",
	.id	= 0,
	.dev	= {
		  .platform_data = &android_pmem_pdata
		},
};
#endif

static struct platform_device *evsy_devs[] __initdata = {
#ifdef CONFIG_SMSC911X
	&smc911x_device,
#endif
#ifdef __CONFIG_MMC_EMXX_SDIO_EVSY
	&evsy_sdio0_device,
#endif
#ifdef CONFIG_ANDROID_PMEM
	&android_pmem_device,
#endif
};

/* I2C devices */
#if defined (CONFIG_RTC_RX8025) || defined (CONFIG_RTC_RX8803)
# define RTC_GPIO	GPIO_P25
# define RTC_GPIO_PULL	(0x7 << 28) /* Enable input, Enable pull down */
# define RTC_GPIO_INTR	(0x9 << 4)  /* Asynchronous falling edge detection */

static void evsy_rtc_init_port(void)
{
	unsigned long tmp;

	tmp = readl(CHG_PINSEL_G000);
	tmp |= 1 << RTC_GPIO;
	writel(tmp, CHG_PINSEL_G000);

	tmp = readl(CHG_PULL0);
	tmp |= RTC_GPIO_PULL;
	writel(tmp, CHG_PULL0);

	tmp = readl(VA_GIO + GIO_IDT3);
	tmp |= RTC_GPIO_INTR;
	writel(tmp, VA_GIO + GIO_IDT3);

	set_irq_type(gpio_to_irq(GPIO_P25), IRQ_TYPE_EDGE_FALLING);
}

struct i2c_rtc_device {
	int ch;
	struct i2c_board_info info;
};

#define RTC_I2C_SLAVE_ADDR	0x32
#define RTC_IRQ			(RTC_GPIO + INT_GPIO_BASE)

static struct i2c_rtc_device evsy_rtc_devices[] = {
#ifdef CONFIG_RTC_RX8025
	{
		.ch = 0,
		.info = {
			.type		= "rx8025",
			.addr		= RTC_I2C_SLAVE_ADDR,
			.irq		= RTC_IRQ,
			.platform_data	= evsy_rtc_init_port,
		},
	},
#endif
#ifdef CONFIG_RTC_RX8803
	{
		.ch = 1,
		.info = {
			.type		= "rx8803",
			.addr		= RTC_I2C_SLAVE_ADDR,
			.irq		= RTC_IRQ,
			.platform_data	= evsy_rtc_init_port,
		},
	},
#endif
};

static char *rtc_name = 0;
module_param_named(rtc, rtc_name, charp, S_IRUSR|S_IWUSR);

static int __init evsy_rtc_init(void)
{
	int i, err;
	struct i2c_rtc_device *dev = 0;

	if (!rtc_name) {
		pr_warning("EMEV: must specify board.rtc = [devname]\n");
		return -EINVAL;
	}

	for (i = 0; i < ARRAY_SIZE(evsy_rtc_devices); i++) {
		dev = &evsy_rtc_devices[i];
		if (strncmp(dev->info.type, rtc_name,
		    sizeof(dev->info.type)) == 0) {
			break;
		}
	}

	if (i >= ARRAY_SIZE(evsy_rtc_devices)) {
		pr_warning("EMEV: Not found rtc device %s\n", rtc_name);
		return -ENODEV;
	}

	pr_info("EMEV: i2c%d Found rtc device %s \n", dev->ch, rtc_name);

	err = i2c_register_board_info(dev->ch, &dev->info, 1);
	if (err) {
		pr_warning("EMEV: Failed rtc device register err:%d, %d, %s\n",
			   err, dev->ch, rtc_name);
	}

	return err;
}
#else
# define evsy_rtc_init() do {} while (0)
#endif /* (CONFIG_RTC_RX8025) || defined (CONFIG_RTC_RX8803) */

static void __init evsy_board_map_io(void)
{
	/* EMEV internal devices */
	emxx_map_io();

	pr_info("EMEV system_rev=%#x\n", system_rev);
}

static void __init evsy_init_irq(void)
{
	gic_dist_init(0, __io_address(EMXX_INTA_DIST_BASE), INT_CPU_TIM);

	/*
	 * Set GPIO interrupt setting IDT[0..3] here if need be.
	 * emxx_gio_init() will configure GPIO interrupts
	 * according to IDT[0..3]
	 */

	gic_cpu_init(0, __io_address(EMXX_INTA_CPU_BASE));
}

static void __init evsy_board_init(void)
{
	emxx_serial_init(evsy_serial_ports);

#ifdef CONFIG_EMXX_L310
	{
		void __iomem *l2cc_base = (void *)EMXX_L2CC_VIRT;
#ifdef CONFIG_EMXX_L310_WT
		/* Force L2 write through */
		writel(0x002, l2cc_base + L2X0_DEBUG_CTRL);
#endif
		writel(0x111, l2cc_base + L2X0_TAG_LATENCY_CTRL);
		writel(0x111, l2cc_base + L2X0_DATA_LATENCY_CTRL);
#ifdef CONFIG_EMXX_L310_8WAY
		/* 8-way 32KB cache, Early BRESP */
		writel(0, SMU_CPU_ASSOCIATIVITY);	/* 0:8-way 1:16-way */
		writel(2, SMU_CPU_WAYSIZE);		/* 0,1:16KB 2:32KB */
		l2x0_init(l2cc_base, 0x40040000, 0x00000fff);
#else
		/* 16-way 16KB cache, Early BRESP */
		writel(1, SMU_CPU_ASSOCIATIVITY);	/* 0:8-way 1:16-way */
		writel(1, SMU_CPU_WAYSIZE);		/* 0,1:16KB 2:32KB */
		l2x0_init(l2cc_base, 0x40030000, 0x00000fff);
#endif
	}
#endif	/* CONFIG_EMXX_L310 */

	/* Configure AB0 */
	/* CS0: FRAM */
	writel(_ab0_csbaseadd(EVSY_FRAM_BASE), AB_CS0BASEADD);
	writel(_ab0_cscomp_mask(EVSY_FRAM_SIZE), AB_CS0BITCOMP);
	/* CS1: Ether */
	writel(_ab0_csbaseadd(EVSY_ETHER_BASE), AB_CS1BASEADD);
	writel(_ab0_cscomp_mask(EVSY_ETHER_SIZE), AB_CS1BITCOMP);
	/* CS2: Unused */
	writel(_ab0_csbaseadd(EVSY_AB0_CS2_BASE), AB_CS2BASEADD);
	writel(_ab0_cscomp_mask(EVSY_AB0_CS2_SIZE), AB_CS2BITCOMP);
	/* CS3: PLD */
	writel(_ab0_csbaseadd(EVSY_PLD_BASE), AB_CS3BASEADD);
	writel(_ab0_cscomp_mask(EVSY_PLD_SIZE), AB_CS3BITCOMP);

	/* Configure SDC/MSHC pins and clock */
	/* SDC Clk */
	_mod_reg(SMU_SDCGCLKCTRL, 0x0000007, 0x00000005);
	/* MSHC Clk */
	_mod_reg(SMU_MSPGCLKCTRL, 0x0000007, 0x00000007);
	/* default to SD */
//	evsy_alter_sdc_pins(EVSY_SDCMSHC_SDC);

	/* board specific devices */
	platform_add_devices(evsy_devs, ARRAY_SIZE(evsy_devs));
	evsy_rtc_init();

#ifdef CONFIG_EMXX_QR
	writel(0x00444444, SMU_QR_WAITCNT);
	writel(0x00000003, SMU_QR_WFI);
#endif

}

void evsy_prepare_l2x0_resume(void)
{
#ifdef CONFIG_EMXX_L310
	void __iomem *l2cc_base = (void *)EMXX_L2CC_VIRT;
#ifdef CONFIG_EMXX_L310_WT
	/* Force L2 write through */
	writel(0x002, l2cc_base + L2X0_DEBUG_CTRL);
#endif
	writel(0x111, l2cc_base + L2X0_TAG_LATENCY_CTRL);
	writel(0x111, l2cc_base + L2X0_DATA_LATENCY_CTRL);
#ifdef CONFIG_EMXX_L310_8WAY
	/* 8-way 32KB cache, Early BRESP */
	writel(0, SMU_CPU_ASSOCIATIVITY);	/* 0:8-way 1:16-way */
	writel(2, SMU_CPU_WAYSIZE);		/* 0,1:16KB 2:32KB */
#else
	/* 16-way 16KB cache, Early BRESP */
	writel(1, SMU_CPU_ASSOCIATIVITY);	/* 0:8-way 1:16-way */
	writel(1, SMU_CPU_WAYSIZE);		/* 0,1:16KB 2:32KB */
#endif
#endif	/* CONFIG_EMXX_L310 */
}

void evsy_board_resume(void)
{
	/* Configure AB0 */
	/* CS0: FRAM */
	writel(_ab0_csbaseadd(EVSY_FRAM_BASE), AB_CS0BASEADD);
	writel(_ab0_cscomp_mask(EVSY_FRAM_SIZE), AB_CS0BITCOMP);
	/* CS1: Ether */
	writel(_ab0_csbaseadd(EVSY_ETHER_BASE), AB_CS1BASEADD);
	writel(_ab0_cscomp_mask(EVSY_ETHER_SIZE), AB_CS1BITCOMP);
	/* CS2: Unused */
	writel(_ab0_csbaseadd(EVSY_AB0_CS2_BASE), AB_CS2BASEADD);
	writel(_ab0_cscomp_mask(EVSY_AB0_CS2_SIZE), AB_CS2BITCOMP);
	/* CS3: PLD */
	writel(_ab0_csbaseadd(EVSY_PLD_BASE), AB_CS3BASEADD);
	writel(_ab0_cscomp_mask(EVSY_PLD_SIZE), AB_CS3BITCOMP);

	/* Configure SDC/MSHC pins and clock */
	/* SDC Clk */
	_mod_reg(SMU_SDCGCLKCTRL, 0x0000007, 0x00000005);
	/* MSHC Clk */
	_mod_reg(SMU_MSPGCLKCTRL, 0x0000007, 0x00000007);

#ifdef CONFIG_EMXX_QR
	writel(0x00444444, SMU_QR_WAITCNT);
	writel(0x00000003, SMU_QR_WFI);
#endif
}

MACHINE_START(EVSY, "EVSY")
	.phys_io      = EMXX_UART0_BASE,
	.io_pg_offst  = (IO_ADDRESS(EMXX_UART0_BASE) >> 18) & 0xfffc,
	.boot_params  = PHYS_OFFSET + 0x100,
	.soft_reboot  = 1,
	.map_io       = evsy_board_map_io,
	.init_irq     = evsy_init_irq,
	.init_machine = evsy_board_init,
	.timer        = &emxx_timer,
MACHINE_END
