/*
 *  Copyright 2011,2012 Sony Corporation.
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#include <linux/platform_device.h>
#include <linux/gpio.h>
#include <linux/ioport.h>
#include <linux/irq.h>
#include <linux/mmc/mmc.h>
#include <linux/mmc/host.h>
#include <linux/mmc/sd.h>
#include <linux/mmc/core.h>
#include <linux/mmc/card.h>
#include <linux/leds.h>

#include <mach/hardware.h>
#include <mach/smu.h>
#include <mach/sdc.h>
#include <mach/evsy.h>

#include "../../../drivers/mmc/host/emxx_sdc.h"

static int evsy_sdc_is_media_present(struct mmc_host *mmc)
{
	if (gpio_get_value(EVSY_GPIO_SD_INS))
		return 0;
	else
		return 1; /* media present */
}

static int evsy_sdc_get_ro(struct mmc_host *mmc)
{
	if (gpio_get_value(EVSY_GPIO_SDWPZ))
		return 1; /* Write protected or uninserted */
	else
		return 0;
}

static irqreturn_t evsy_sdc_detect_irq(int irq, void *dev_id)
{
	struct emxx_sdc_host *host;
	struct mmc_host *mmc;

	host = (struct emxx_sdc_host *)dev_id;
	mmc = host->mmc;

	if (evsy_sdc_is_media_present(mmc)) {
		host->connect = 1;
		evsy_alter_sdc_pins(EVSY_SDCMSHC_SDC);
		evsy_media_power(1);
	} else {
		host->connect = 0;
#ifdef CONFIG_EVSY_SDC_POWER_OFF_WHEN_REMOVING_CARD
		evsy_media_power(0);
		evsy_alter_sdc_pins(EVSY_SDCMSHC_NOCARD);
#endif
	}

	mmc_detect_change(host->mmc, msecs_to_jiffies(100));

	return IRQ_HANDLED;
}

#ifdef CONFIG_LEDS_CLASS
#define MSEC_PER_JIFFY          (1000/HZ)
#define LED_BLINK_INTERVAL      70 /* ms */
#define LED_BLINK_INTERVAL_TICK (LED_BLINK_INTERVAL / MSEC_PER_JIFFY)
#define LED_BLINK_COUNT_MIN     1

struct evsy_sdc_led_st {
    int on;
    int blink_count;
    struct timer_list blinker;
};
static struct evsy_sdc_led_st evsy_sdc_led;

/*
 * access LED support
 */
static void evsy_sdc_led_control(int on)
{

    evsy_sdc_led.on = on;
    evsy_media_acc_led(evsy_sdc_led.on);
}

static void evsy_sdc_led_invert(void)
{

    evsy_sdc_led_control(~evsy_sdc_led.on & 0x1);
}

static void led_blinker(unsigned long data)
{

    if (evsy_sdc_led.blink_count == 0) {
            evsy_sdc_led_control(0);
    } else {
            evsy_sdc_led_invert();
            evsy_sdc_led.blink_count--;
            mod_timer(&evsy_sdc_led.blinker,
                      jiffies + LED_BLINK_INTERVAL_TICK);
    }
}

static void evsy_sdc_led_blinker_start(void)
{

    evsy_sdc_led.blink_count = LED_BLINK_COUNT_MIN;
    if (!timer_pending(&evsy_sdc_led.blinker)) {
            /* If led is not blinking */
            mod_timer(&evsy_sdc_led.blinker,
                      jiffies + LED_BLINK_INTERVAL_TICK);
    }
}

static void evsy_sdc_led_init(void)
{

    evsy_sdc_led.on = 0;
    evsy_sdc_led.blink_count = 0;
    evsy_sdc_led.blinker.function = led_blinker;
    init_timer(&evsy_sdc_led.blinker);
}

static void evsy_sdc_led_fin(void)
{

    del_timer_sync(&evsy_sdc_led.blinker);
}

static void evsy_acc_ledclass_brightness_set(struct led_classdev *led_cdev,
					     enum led_brightness brightness)
{

	if (brightness == LED_FULL)
		evsy_sdc_led_blinker_start();
}

static enum led_brightness
evsy_acc_ledclass_brightness_get(struct led_classdev *led_cdev)
{
	if (gpio_get_value(EVSY_GPIO_MS_ACS_LED))
		return LED_FULL;
	else
		return LED_OFF;
}

static struct led_classdev evsy_accled_classdev = {
	.brightness = LED_OFF,
	.brightness_set = evsy_acc_ledclass_brightness_set,
	.brightness_get = evsy_acc_ledclass_brightness_get,
};
#endif /* CONFIG_LEDS_CLASS */

static int evsy_sdc_probe(struct platform_device *pdev, struct mmc_host *mmc)
{

	struct emxx_sdc_host *host;
	int ret;

	host = mmc_priv(mmc);

	/* SD card media detection */
	host->detect_irq = EVSY_INT_SD_INS;
	set_irq_type(host->detect_irq, IRQ_TYPE_EDGE_BOTH);

	if (evsy_sdc_is_media_present(mmc)) {
		host->connect = 1;
		evsy_alter_sdc_pins(EVSY_SDCMSHC_SDC);
		evsy_media_power(1);
	} else {
		host->connect = 0;
#ifdef CONFIG_EVSY_SDC_POWER_OFF_WHEN_REMOVING_CARD
		evsy_media_power(0);
		evsy_alter_sdc_pins(EVSY_SDCMSHC_NOCARD);
#endif
	}

	ret = request_irq(host->detect_irq, evsy_sdc_detect_irq, IRQF_DISABLED,
			  pdev->name, host);
	if (ret) {
		pr_err("%s: failed to register card detect irq\n", __func__);
		host->connect = 0;
#ifdef CONFIG_EVSY_SDC_POWER_OFF_WHEN_REMOVING_CARD
		evsy_media_power(0);
		evsy_alter_sdc_pins(EVSY_SDCMSHC_NOCARD);
#endif
		host->detect_irq = 0;
		goto done;
	}

#ifdef CONFIG_LEDS_CLASS
	evsy_accled_classdev.name = mmc_hostname(mmc);
	evsy_accled_classdev.default_trigger = mmc_hostname(mmc);
	ret = led_classdev_register(mmc_dev(mmc), &evsy_accled_classdev);
	if (ret) {
		pr_err("%s: failed to register led (%d)\n", __func__, ret);
		free_irq(host->detect_irq, host);
		host->connect = 0;
#ifdef CONFIG_EVSY_SDC_POWER_OFF_WHEN_REMOVING_CARD
		evsy_media_power(0);
		evsy_alter_sdc_pins(EVSY_SDCMSHC_NOCARD);
#endif
		host->detect_irq = 0;
	}
#endif
done:
	return ret;
}

static void evsy_sdc_remove(struct platform_device *pdev)
{
	struct emxx_sdc_host *host;

	host = platform_get_drvdata(pdev);

	if (host)
		free_irq(host->detect_irq, host);
	else
		pr_err("%s:emxx_sdc_host already freed\n", __func__);

#ifdef CONFIG_LEDS_CLASS
	led_classdev_unregister(&evsy_accled_classdev);
#endif
	return;
}

static void evsy_sdc_nop_release(struct device *dev)
{
	/* nothing to be released */
}

#ifdef CONFIG_PM
int evsy_sdc_suspend(struct platform_device *pdev, pm_message_t state)
{
	return 0;
}

int evsy_sdc_resume(struct platform_device *pdev)
{
	struct emxx_sdc_host *host;
	struct mmc_host *mmc;

	host = platform_get_drvdata(pdev);
	mmc = host->mmc;

	if (evsy_sdc_is_media_present(mmc)) {
		host->connect = 1;
		evsy_alter_sdc_pins(EVSY_SDCMSHC_SDC);
		evsy_media_power(1);
	} else {
		host->connect = 0;
#ifdef CONFIG_EVSY_SDC_POWER_OFF_WHEN_REMOVING_CARD
		evsy_media_power(0);
		evsy_alter_sdc_pins(EVSY_SDCMSHC_NOCARD);
 #endif
	}

	return 0;
}
#endif /* CONFIG_PM */

static struct emxx_sdc_platform_data evsy_sdc_platform_data = {
	.sdc_get_ro = evsy_sdc_get_ro,
	.sdc_probe = evsy_sdc_probe,
	.sdc_remove = evsy_sdc_remove,
#ifdef CONFIG_PM
	.sdc_suspend = evsy_sdc_suspend,
	.sdc_resume = evsy_sdc_resume,
#endif
};

static struct platform_device evsy_sdc_device = {
	.name = "emxx_sdc",
	.id = -1,
	.dev = {
		.platform_data = &evsy_sdc_platform_data,
		.release       = evsy_sdc_nop_release,
	},
};

static int __init evsy_sdc_init(void)
{
	evsy_sdc_led_init();
	return platform_device_register(&evsy_sdc_device);
}

static void __exit evsy_sdc_exit(void)
{

	evsy_sdc_led_fin();
	platform_device_unregister(&evsy_sdc_device);
}

arch_initcall(evsy_sdc_init);
module_exit(evsy_sdc_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Sony Corporation");
