/*
 *  Copyright 2011 Sony Corporation.
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <linux/platform_device.h>
#include <linux/gpio.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/delay.h>
#include <linux/usb/ch9.h>
#include <linux/usb/gadget.h>

#include <mach/hardware.h>
#include <mach/udc.h>

#include "../../../../drivers/usb/gadget/emxx_udc.h"

#define EVSY_VBUS_CHATTERING_MDELAY	1
#define EVSY_DMA_DISABLE_TIME		10

static int evsy_udc_check_vbus(void)
{
	return gpio_get_value(EVSY_GPIO_VBUS);
}

static void evsy_udc_wait_vbus_chattering(void)
{
	mdelay(EVSY_VBUS_CHATTERING_MDELAY);
}

static void evsy_udc_wait_dma_abort(void)
{
	mdelay(EVSY_DMA_DISABLE_TIME);
}

static void evsy_udc_nop_release(struct device *dev)
{
	/* nothing to do */
}

static irqreturn_t evsy_vbus_irq(int irq, void *_udc)
{
	struct nbu2ss_udc *udc = (struct nbu2ss_udc *)_udc;

	spin_lock(&udc->lock);
	if (udc->config->udc_vbus_work) {
		udc->config->udc_vbus_work(udc);
	}
	spin_unlock(&udc->lock);

	return IRQ_HANDLED;
}

static int evsy_udc_probe(struct nbu2ss_udc *udc)
{
	int status;

	set_irq_type(EVSY_INT_VBUS, IRQ_TYPE_EDGE_BOTH);
	status = request_irq(EVSY_INT_VBUS,
			     evsy_vbus_irq,
			     IRQF_SHARED,
			     "emxx_udc",
			     udc);
	return status;
}

static void evsy_udc_remove(struct nbu2ss_udc *udc)
{
	free_irq(EVSY_INT_VBUS, udc);
}

struct emxx_udc_platform_data evsy_udc_config = {
	.udc_pre_clr_dpd		= NULL,
	.udc_post_set_dpd		= NULL,
	.udc_check_vbus			= evsy_udc_check_vbus,
	.udc_wait_vbus_chattering	= evsy_udc_wait_vbus_chattering,
	.udc_wait_dma_abort		= evsy_udc_wait_dma_abort,
	.udc_vbus_work			= NULL,
	.udc_probe			= evsy_udc_probe,
	.udc_remove			= evsy_udc_remove,
};

/* USB device */
static struct platform_device evsy_udc_device = {
	.name = "emxx_udc",
	.id = -1,
	.dev = {
		.release = evsy_udc_nop_release,
		.platform_data = &evsy_udc_config,
	}
};

static int __init evsy_udc_init(void)
{
	return platform_device_register(&evsy_udc_device);
}

static void __exit evsy_udc_exit(void)
{
	platform_device_unregister(&evsy_udc_device);
}

arch_initcall(evsy_udc_init);
module_exit(evsy_udc_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Sony Corporation");
