/*
 *  File Name       : arch/arm/mach-emxx/generic.c
 *  Function        : generic
 *  Release Version : Ver 1.00
 *  Release Date    : 2010/02/05
 *
 *  Copyright (C) NEC Electronics Corporation 2010
 *
 *  This program is free software;you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by Free
 *  Softwere Foundation; either version 2 of License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warrnty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 *  more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; If not, write to the Free Software Foundation, Inc., 59
 *  Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include <linux/init.h>
#include <linux/platform_device.h>
#include <linux/io.h>
#include <linux/serial_8250.h>
#include <linux/serial_reg.h>
#include <linux/mman.h>
#include <linux/wait.h>

#include <asm/mach/map.h>

#include <mach/hardware.h>
#include <mach/pmu.h>
#include <mach/smu.h>

#include <mach/dma.h>
#include "spi.h"
#include "generic.h"

/*
 * Common EMXX I/O Mapping
 *
 * Logical  Physical
 * e0000000 e0000000-e2ffffff
 * e3000000 1e000000-1e0fffff
 */

static struct map_desc standard_io_desc[] __initdata = {
	{
		.virtual	= IO_ADDRESS(EMXX_INTERNAL_IO_BASE1),
		.pfn		= __phys_to_pfn(EMXX_INTERNAL_IO_BASE1),
		.length		= EMXX_INTERNAL_IO_SIZE1,
		.type		= MT_DEVICE,
	},
	{
		.virtual	= IO_ADDRESS(EMXX_INTERNAL_IO_BASE2),
		.pfn		= __phys_to_pfn(EMXX_INTERNAL_IO_BASE2),
		.length		= EMXX_INTERNAL_IO_SIZE2,
		.type		= MT_DEVICE_SO,
	},
	{
		.virtual	= IO_ADDRESS(EMXX_INTERNAL_IO_BASE3),
		.pfn		= __phys_to_pfn(EMXX_INTERNAL_IO_BASE3),
		.length		= EMXX_INTERNAL_IO_SIZE3,
		.type		= MT_DEVICE,
	},
	{
		.virtual	= EMXX_CPU_LOCAL_VIRT,
		.pfn		= __phys_to_pfn(EMXX_CPU_LOCAL_BASE),
		.length		= EMXX_CPU_LOCAL_SIZE,
		.type		= MT_DEVICE,
	},
	{
		.virtual	= EMXX_SRAM_VIRT,
		.pfn		= __phys_to_pfn(EMXX_SRAM_BASE),
		.length		= EMXX_SRAM_SIZE,
		.type		= MT_DEVICE,
	},
	{
		/* AB0 control registers */
		.virtual	= IO_ADDRESS(EMXX_AB_BASE),
		.pfn		= __phys_to_pfn(EMXX_AB_BASE),
		.length		= SZ_4K,
		.type		= MT_DEVICE,
	}
};

static struct plat_serial8250_port serial_platform_data[] = {
	{
	 .membase = (char *)IO_ADDRESS(EMXX_UART0_BASE),
	 .mapbase = (unsigned long)EMXX_UART0_BASE,
	 .irq = INT_UART0,
	 .flags = UPF_BOOT_AUTOCONF | UPF_NO_TXEN_TEST,
	 .iotype = UPIO_MEM32,
	 .regshift = 2,
	 .uartclk = EMXX_BASE_BAUD,
	 },
	{
	 .membase = (char *)IO_ADDRESS(EMXX_UART1_BASE),
	 .mapbase = (unsigned long)EMXX_UART1_BASE,
	 .irq = INT_UART1,
	 .flags = UPF_BOOT_AUTOCONF | UPF_NO_TXEN_TEST,
	 .iotype = UPIO_MEM32,
	 .regshift = 2,
	 .uartclk = EMXX_BASE_BAUD,
	 },
	{
	 .membase = (char *)IO_ADDRESS(EMXX_UART2_BASE),
	 .mapbase = (unsigned long)EMXX_UART2_BASE,
	 .irq = INT_UART2,
	 .flags = UPF_BOOT_AUTOCONF | UPF_NO_TXEN_TEST,
	 .iotype = UPIO_MEM32,
	 .regshift = 2,
	 .uartclk = EMXX_BASE_BAUD,
	 },
	{
	 .membase = (char *)IO_ADDRESS(EMXX_UART3_BASE),
	 .mapbase = (unsigned long)EMXX_UART3_BASE,
	 .irq = INT_UART3,
	 .flags = UPF_BOOT_AUTOCONF | UPF_NO_TXEN_TEST,
	 .iotype = UPIO_MEM32,
	 .regshift = 2,
	 .uartclk = EMXX_BASE_BAUD,
	 },
	{
	 /* terminate */
	 },
};

static struct platform_device dma_device = {
	.name = "dma",
	.id = -1,
};

/* spi transfer info */
#define DECLARE_DMA_RX(ch)			\
	[SPI_DEV_SP##ch] = {			\
		.lch = EMXX_DMAC_P2M_SIO##ch,	\
		.data = SPx_RX_DATA_PHYS(ch),	\
		.name = SPI_NAME #ch "_rx_dma",	\
	}
#define DECLARE_DMA_TX(ch)			\
	[SPI_DEV_SP##ch] = {			\
		.lch = EMXX_DMAC_M2P_SIO##ch,	\
		.data = SPx_TX_DATA_PHYS(ch),	\
		.name = SPI_NAME #ch "_tx_dma",	\
	}

static struct spi_dma dma_sp_rx[] = {
	DECLARE_DMA_RX(0),
	DECLARE_DMA_RX(1),
	DECLARE_DMA_RX(2),
	DECLARE_DMA_RX(3),
	DECLARE_DMA_RX(4),
	DECLARE_DMA_RX(5),
};

static struct spi_dma dma_sp_tx[] = {
	DECLARE_DMA_TX(0),
	DECLARE_DMA_TX(1),
	DECLARE_DMA_TX(2),
	DECLARE_DMA_TX(3),
	DECLARE_DMA_TX(4),
	DECLARE_DMA_TX(5),
};

/* spi config info */
#define DECLARE_CONFIG(ch, bit, clk)		\
	[SPI_DEV_SP##ch] = {			\
		.dev = SPI_DEV_SP##ch,		\
		.nbr = SPI_NB_##bit,		\
		.nbw = SPI_NB_##bit,		\
		.cs_sel = SPI_CS_SEL_CS0,	\
		.m_s = SPI_M_S_MASTER,		\
		.dma = SPI_DMA_OFF,		\
		.pol = SPI_POL_SP##ch##_CS,	\
		.sclk = SPI_SCLK_##clk,		\
		.tiecs = SPI_TIECS_NORMAL,	\
	}
static SPI_CONFIG config_sp[] = {
	DECLARE_CONFIG(0, 8BIT, 3MHZ),
	DECLARE_CONFIG(1, 16BIT, 3MHZ),
	DECLARE_CONFIG(2, 8BIT, 930KHZ),
	DECLARE_CONFIG(3, 8BIT, 930KHZ),
	DECLARE_CONFIG(4, 8BIT, 930KHZ),
	DECLARE_CONFIG(5, 8BIT, 930KHZ),
};

/* smu info */
#define DECLARE_SMU(ch, type, div, sft)								\
	[SPI_DEV_SP##ch] = {									\
		.pclk = (EMXX_CLK_USI##type##_S##ch##_P | EMXX_CLK_USI##type##_S##ch##_H),	\
		.sclk = EMXX_CLK_USI##type##_S##ch##_S,						\
		.pclk_ctrl = EMXX_CLKCTRL_USI##type##S##ch##PCLK,				\
		.sclk_ctrl = EMXX_CLKCTRL_USI##type##S##ch,					\
		.reset = EMXX_RST_USI##type##_S##ch##_S | EMXX_RST_USI##type##_S##ch##_A,	\
		.sclk_div = SMU_USI##div##SCLKDIV,						\
    		.sclk_div_shift = sft,								\
	}
static struct spi_smu smu_sp[] = {
	DECLARE_SMU(0, A, A, 0),
	DECLARE_SMU(1, A, A, 16),
	DECLARE_SMU(2, B, B0, 0),
	DECLARE_SMU(3, B, B0, 16),
	DECLARE_SMU(4, B, B1, 0),
	DECLARE_SMU(5, B, B1, 16),
};

/* spi private data info */
#define DECLARE_SPI(_name, ch, _rx, _tx, _config, _smu)			\
	[SPI_DEV_SP##ch] = {						\
		.regs = (struct spi_regs *)(SPx_ADDR(SPI_DEV_SP##ch)),	\
		.dma_rx = &_rx[SPI_DEV_SP##ch],				\
		.dma_tx = &_tx[SPI_DEV_SP##ch],				\
		.config = &_config[SPI_DEV_SP##ch],			\
		.smu = &_smu[SPI_DEV_SP##ch],				\
		.int_spi = INT_SIO##ch,					\
		.pol = SPI_POL_SP##ch,					\
		.name = SPI_NAME #ch,					\
		.ready = 0,						\
		.busy = 0,						\
		.done = 0,						\
		.priv = 0,						\
		.lockflags = 0,						\
		.wq = __WAIT_QUEUE_HEAD_INITIALIZER(_name[SPI_DEV_SP##ch].wq),		\
		.spinlock =  __SPIN_LOCK_UNLOCKED(_name[SPI_DEV_SP##ch].spinlock)	\
	}
static struct spi_data spi_drv_data[] = {
	DECLARE_SPI(spi_drv_data, 0, dma_sp_rx, dma_sp_tx, config_sp, smu_sp),
	DECLARE_SPI(spi_drv_data, 1, dma_sp_rx, dma_sp_tx, config_sp, smu_sp),
	DECLARE_SPI(spi_drv_data, 2, dma_sp_rx, dma_sp_tx, config_sp, smu_sp),
	DECLARE_SPI(spi_drv_data, 3, dma_sp_rx, dma_sp_tx, config_sp, smu_sp),
	DECLARE_SPI(spi_drv_data, 4, dma_sp_rx, dma_sp_tx, config_sp, smu_sp),
	DECLARE_SPI(spi_drv_data, 5, dma_sp_rx, dma_sp_tx, config_sp, smu_sp),
};

static struct platform_device spi_device = {
	.name = SPI_NAME,
	.id = -1,
	.dev = {
		.platform_data = spi_drv_data,
	},
};

static struct platform_device i2c0_device = {
	.name = "i2c",
	.id = 0,
};

#ifdef CONFIG_I2C_EMXX_ENABLE_CH2
static struct platform_device i2c1_device = {
	.name = "i2c",
	.id = 1,
};
#endif

static struct platform_device serial8250_device = {
	.name = "serial8250",
	.id = 0,
	.dev = {
		.platform_data = serial_platform_data,
		},
};

static struct platform_device pcm_device = {
	.name = "pcm",
	.id = -1,
};

static struct platform_device pcm1_device = {
	.name = "pcm1",
	.id = -1,
};

static struct platform_device emxx_fb_device = {
	.name = "emxx_fb",
	.id = -1,
};

static struct platform_device emxx_image_device = {
	.name = "emxx_image",
	.id = -1,
};

static struct platform_device emxx_dsp_device = {
	.name = "emxx_dsp",
	.id = -1,
};

#ifndef CONFIG_SNSC_EMXX_PLATFORM
static struct platform_device emxx_udc_device = {
	.name = "emxx_udc",
	.id = -1,
};

static struct platform_device emxx_ehci_device = {
	.name = "emxx-ehci-driver",
	.id = -1,
	.dev			= {
		.dma_mask	= (void *)0xffffffff,
		.coherent_dma_mask = 0xffffffff,
	},
};
static struct platform_device emxx_ohci_device = {
	.name = "emxx-ohci-driver",
	.id = -1,
	.dev			= {
		.dma_mask	= (void *)0xffffffff,
		.coherent_dma_mask = 0xffffffff,
	},
};

static struct platform_device emxx_sdio0_device = {
	.name = "emxx_sdio",
	.id = 0,
};
static struct platform_device emxx_sdio1_device = {
	.name = "emxx_sdio1",
	.id = 1,
};
#endif /* !CONFIG_SNSC_EMXX_PLATFORM */

#ifndef CONFIG_SNSC_EMXX_PLATFORM
static struct platform_device emxx_sdc_device = {
	.name = "emxx_sdc",
	.id = -1,
};
#endif /* !CONFIG_SNSC_EMXX_PLATFORM */

static struct resource emxx_cfi_resources[] = {
	[0] = {
		.start	= EMXX_CFI_BASE,
		.end	= EMXX_CFI_BASE + SZ_1M- 1,
		.flags	= IORESOURCE_MEM,
	},
	[1] = {
		.start	= INT_CFI,
		.end	= INT_CFI,
		.flags	= IORESOURCE_IRQ,
	},
};

static struct platform_device emxx_cfi_device = {
	.name = "emxx_cfi_ide",
	.id = -1,
	.num_resources	= ARRAY_SIZE(emxx_cfi_resources),
	.resource	= emxx_cfi_resources,
	
};

static struct platform_device *platform_devs[] __initdata = {
	&dma_device,
	&spi_device,
	&i2c0_device,
#ifdef CONFIG_I2C_EMXX_ENABLE_CH2
	&i2c1_device,
#endif
	&serial8250_device,
	&pcm_device,
	&pcm1_device,
	&emxx_fb_device,
	&emxx_image_device,
	&emxx_dsp_device,
#ifndef CONFIG_SNSC_EMXX_PLATFORM
	&emxx_udc_device,
	&emxx_ehci_device,
	&emxx_ohci_device,
	&emxx_sdio0_device,
	&emxx_sdio1_device,
	&emxx_sdc_device,
#endif /* !CONFIG_SNSC_EMXX_PLATFORM */
	&emxx_cfi_device,
};

void __init emxx_map_io(void)
{
	iotable_init(standard_io_desc, ARRAY_SIZE(standard_io_desc));
}

void emxx_serial_init(int *ports)
{
	int i;
	unsigned int clk, rst, val;

	for (i = 0; i < 4; i++) {
		if (ports[i] == 0) {
			serial_platform_data[i].flags = 0;
			continue;
		}
		switch (i) {
		case 0:
			clk = EMXX_CLK_USIA_U0_P | EMXX_CLK_USIA_U0_S;
			rst = EMXX_RST_USIA_U0_A;
			writel(SMU_PLLSEL_PLL3 | SMU_DIV(2),
				SMU_USIAU0SCLKDIV);
			break;
		case 1:
			clk = EMXX_CLK_USIB_U1_P | EMXX_CLK_USIB_U1_S;
			rst = EMXX_RST_USIB_U1_A;
			val = readl(SMU_USIB2SCLKDIV) & ~0xffff;
			writel(val | SMU_PLLSEL_PLL3 | SMU_DIV(2),
				SMU_USIB2SCLKDIV);
			break;
		case 2:
			clk = EMXX_CLK_USIB_U2_P | EMXX_CLK_USIB_U2_S;
			rst = EMXX_RST_USIB_U2_A;
			val = readl(SMU_USIB2SCLKDIV) & ~0xffff0000;
			writel(val | ((SMU_PLLSEL_PLL3 | SMU_DIV(2)) << 16),
				SMU_USIB2SCLKDIV);
			break;
		case 3:
			clk = EMXX_CLK_USIB_U3_P | EMXX_CLK_USIB_U3_S;
			rst = EMXX_RST_USIB_U3_A;
			writel(SMU_PLLSEL_PLL3 | SMU_DIV(2), SMU_USIB3SCLKDIV);
			break;
		default:
			return;
		}
		/* Unreset U7x */
		emxx_open_clockgate(clk);
		emxx_unreset_device(rst);
	}
}

static int __init emxx_init(void)
{
#ifdef CONFIG_OVERCOMMIT_ALWAYS
	extern int sysctl_overcommit_memory;
#endif

#ifdef CONFIG_OVERCOMMIT_ALWAYS
	sysctl_overcommit_memory = OVERCOMMIT_ALWAYS;
#endif

	return platform_add_devices(platform_devs, ARRAY_SIZE(platform_devs));
}

arch_initcall(emxx_init);
