﻿/*
 *  arch/arm/mach-emxx/kmsg_que.c
 *  implementation for msg queue management in kernel.
 *
 *  Copyright 2011 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/jiffies.h>
#include <linux/slab.h>
#include <linux/hardirq.h>
#include <asm/system.h>

#include "kmsg_que.h"

#define circ_empty(circ)	((circ)->head == (circ)->tail)
#define circ_clear(circ)	((circ)->head = (circ)->tail = 0)


/* Return count in buffer.  */
#define CIRC_CNT_02(head,tail,size)     \
        (head >= tail) ? (head - tail) : (head + size - tail)

/* Return space available, 0..size-1.  We always leave one free char
   as a completely full buffer has head == tail, which is the same as
   empty.  */
#define CIRC_SPACE_02(head,tail,size)   \
        (head >= tail) ? (tail + size - head - 1) : (tail - head)



/*
 * The core wakeup function.  Non-exclusive wakeups (nr_exclusive == 0) just
 * wake everything up.  If it's an exclusive wakeup (nr_exclusive == small +ve
 * number) then we wake all the non-exclusive tasks and one exclusive task.
 *
 * There are circumstances in which we can try to wake a task which has already
 * started to run but is not in state TASK_RUNNING.  try_to_wake_up() returns
 * zero in this (rare) case, and we handle it by continuing to scan the queue.
 */
static void __wake_up_common_02(wait_queue_head_t *q, unsigned int mode,
                                int nr_exclusive, int sync, void *key)
{
    struct list_head *tmp, *next;

    list_for_each_safe(tmp, next, &q->task_list)
    {
        wait_queue_t *curr = list_entry(tmp, wait_queue_t, task_list);

        /* attention... */
        if (curr && curr->func)
        {
            unsigned flags = curr->flags;
            if (curr->func(curr, mode, sync, key) &&
                    (flags & WQ_FLAG_EXCLUSIVE) && !--nr_exclusive)
                break;
        }
        else
        {
            kmsg_error("curr:0x%0x or curr->func is invalid. \n", (unsigned int)curr);
            return;
        }
    }
}



/* create a kmsg queue. */
int  kmsg_que_create (kmsg_que_id_t *msg_que_id,
                      unsigned long msg_num,
                      unsigned long msg_len)
{
    struct kmsg_que_t *p_kmsg_que = NULL;

    if (NULL == msg_que_id)
    {
        kmsg_error("msg_id is invalid. \n");
        return KMSG_QUE_ERROR;
    }

    /* the number of msg must be 2^n. */
    if ( (KMSG_QUE_MIN_MSG > msg_num)
            || (KMSG_QUE_MAX_MSG < msg_num)
            || (KMSG_QUE_MSG_MAX_LEN < msg_len)
       )
    {
        kmsg_error(" msg_num or msg_len is invalid, please check them...\n");
        return KMSG_QUE_LEN_ERR;
    }

    /* whether is it aligned by 2^n. */
    if (msg_num & (msg_num - 1))
    {
        kmsg_error(" msg_num must be aligned by 2^n now.\n");
        return KMSG_QUE_LEN_ERR;
    }

    /* is it in isr context?? */
    if (in_interrupt())
    {
        kmsg_error(" you should NOT use it when in_interrupt()==true.\n");
        return KMSG_QUE_IRQ_ERR;
    }

    p_kmsg_que = kzalloc(sizeof(struct kmsg_que_t) + (msg_num * msg_len), GFP_KERNEL);
    if (NULL == p_kmsg_que)
    {
        kmsg_error("kmalloc for kmsg_que failed. \n");
        return KMSG_QUE_ERROR;
    }

    p_kmsg_que->msg_size = msg_len;
    p_kmsg_que->total_msg = msg_num;
    p_kmsg_que->msg_data = (p_kmsg_que + 1);
    p_kmsg_que->reserved = KMSG_QUE_MAGIC;

    init_waitqueue_head(&p_kmsg_que->mq_full_wait);

    init_waitqueue_head(&p_kmsg_que->mq_empty_wait);

    spin_lock_init(&p_kmsg_que->mq_lock);

    /* attention: */
    *msg_que_id = (kmsg_que_id_t)&(p_kmsg_que->reserved);

    return KMSG_QUE_OK;
}
EXPORT_SYMBOL(kmsg_que_create);


/* destroy a specific kmsg queue. */
int  __sched kmsg_que_destroy (kmsg_que_id_t *msg_que_id )
{
    struct kmsg_que_t *p_kmsg_que = NULL;
    int ret = KMSG_QUE_ERROR;
    unsigned long flags;
    int count;

    if (NULL == msg_que_id)
    {
        kmsg_error(":msg_id is invalid. \n");
        return KMSG_QUE_ERROR;
    }

    if (KMSG_QUE_MAGIC != *(unsigned long *)(*msg_que_id))
    {
        kmsg_error(":magic number is modified. it means there are some errors in your code. \n");
    }

    p_kmsg_que = container_of((unsigned long *)(*msg_que_id), struct kmsg_que_t, reserved);


    spin_lock_irqsave(&p_kmsg_que->mq_lock, flags);


    /* clean up. */

    /* get the msg-count in queue. */
    count = CIRC_CNT(p_kmsg_que->head,
                     p_kmsg_que->tail,
                     p_kmsg_que->total_msg);
    while (count > 0)
    {
        /* update the head position. */
        p_kmsg_que->tail = (p_kmsg_que->tail + 1) & (p_kmsg_que->total_msg - 1);

        /* wake up one of the threads in the wait-queue. */
        __wake_up_common_02(&p_kmsg_que->mq_empty_wait,
                            TASK_UNINTERRUPTIBLE | TASK_INTERRUPTIBLE,
                            1,
                            1,
                            NULL);
        count--;
    }

    *msg_que_id = (kmsg_que_id_t)0;

    ret = KMSG_QUE_OK;

    spin_unlock_irqrestore(&p_kmsg_que->mq_lock, flags);

    kfree(p_kmsg_que);

    return ret;
}
EXPORT_SYMBOL(kmsg_que_destroy);


/* send a msg to the specific queue.
 * option: no-wait.
 *         wait_forever
 *         some milliseconds.
 * return: =0 --> OK --attention....
 *         <0 --> error
 *         >0 --> timeout
 */
int __sched kmsg_que_send (const kmsg_que_id_t *msg_que_id,
                           void *msg_data,
                           int msg_len,
                           int option)
{
    struct kmsg_que_t *p_kmsg_que = NULL;
    int count;
    int ret;
    unsigned long flags;

    if ((NULL == msg_que_id) || (NULL == msg_data))
    {
        kmsg_error(":msg_id is invalid. \n");
        return KMSG_QUE_ERROR;
    }

    if (KMSG_QUE_MAGIC != *(unsigned long *)(*msg_que_id))
    {
        kmsg_error(":magic number is modified exceptionally. it means there are some errors in your code. \n");
        return KMSG_QUE_ERROR;
    }

    /* is it in isr context?? */
    if (in_interrupt())
    {
        /* attention: you should NOT block in isr(). */
        if (NO_WAIT != option)
        {
            kmsg_error(": you should NOT use 'WAIT_FOREVER' in ISR().\n");
            return KMSG_QUE_IRQ_ERR;
        }
    }

    p_kmsg_que = container_of((unsigned long *)(*msg_que_id), struct kmsg_que_t, reserved);

    /* check the msg-length. */
    if (msg_len > p_kmsg_que->msg_size)
    {
        msg_len = p_kmsg_que->msg_size;
    }

    might_sleep();

    spin_lock_irqsave(&p_kmsg_que->mq_empty_wait.lock, flags);

    /* get the space-count in queue. */
    count = CIRC_SPACE(p_kmsg_que->head,
                       p_kmsg_que->tail,
                       p_kmsg_que->total_msg);

    if (!count)
    {
        DECLARE_WAITQUEUE(wait, current);

        wait.flags |= WQ_FLAG_EXCLUSIVE;
        __add_wait_queue_tail(&p_kmsg_que->mq_empty_wait, &wait);

        do
        {
            if (signal_pending(current))
            {
                ret = -ERESTARTSYS;
                __remove_wait_queue(&p_kmsg_que->mq_empty_wait, &wait);
                goto send_out;
            }
            __set_current_state(TASK_INTERRUPTIBLE);

            spin_unlock_irqrestore(&p_kmsg_que->mq_empty_wait.lock, flags);

            if (NO_WAIT == option)
            {
                return KMSG_QUE_ERROR;
            }
            else if ((int)WAIT_FOREVER == option)
            {
                /* schedule-yield. */
                schedule();

                /* lock again... */
                spin_lock_irqsave(&p_kmsg_que->mq_empty_wait.lock, flags);
            }
            else
            {
                /* schedule-yield timeout. */
                option = schedule_timeout(option);

                /* lock again... */
                spin_lock_irqsave(&p_kmsg_que->mq_empty_wait.lock, flags);

                if (!option)
                {
                    ret = KMSG_QUE_TIMEOUT_ERR;
                    __remove_wait_queue(&p_kmsg_que->mq_empty_wait, &wait);
                    goto send_out;
                }
            }

        } while (!(count = CIRC_SPACE(p_kmsg_que->head,
                                      p_kmsg_que->tail,
                                      p_kmsg_que->total_msg)));

        /* get a space in the queue... */
        __remove_wait_queue(&p_kmsg_que->mq_empty_wait, &wait);
    }


    /* save the msg data. */
    memcpy(((char *)(p_kmsg_que->msg_data) + p_kmsg_que->head * p_kmsg_que->msg_size),
           msg_data, msg_len);

    /* update the head position. */
    p_kmsg_que->head = (p_kmsg_que->head + 1) % p_kmsg_que->total_msg;


    /* wake up one of the threads in the wait-queue. */
    __wake_up_common_02(&p_kmsg_que->mq_full_wait,
                        TASK_UNINTERRUPTIBLE | TASK_INTERRUPTIBLE,
                        1,
                        1,
                        NULL);
    ret = KMSG_QUE_OK;

send_out:
    spin_unlock_irqrestore(&p_kmsg_que->mq_empty_wait.lock, flags);

    /* attention to the return-value, 0-->KMSG_QUE_OK. */
    return ret;
}
EXPORT_SYMBOL(kmsg_que_send);


/* for sysconif driver */
int  __sched kmsg_que_send_urgent (kmsg_que_id_t *msg_que_id,
                                   void *msg_data,
                                   int msg_len,
                                   int option)
{
    struct kmsg_que_t *p_kmsg_que = NULL;
    int count;
    int ret;
    unsigned long flags;
    unsigned long tail_tmp;

    if ((NULL == msg_que_id) || (NULL == msg_data))
    {
        kmsg_error(":msg_id is invalid. \n");
        return KMSG_QUE_ERROR;
    }

    if (KMSG_QUE_MAGIC != *(unsigned long *)(*msg_que_id))
    {
        kmsg_error(":magic number is modified exceptionally. it means there are some errors in your code. \n");
        return KMSG_QUE_ERROR;
    }

    /* is it in isr context?? */
    if (in_interrupt())
    {
        /* attention: you should NOT block in isr(). */
        if (NO_WAIT != option)
        {
            kmsg_error(": you should NOT use 'WAIT_FOREVER' in ISR().\n");
            return KMSG_QUE_IRQ_ERR;
        }
    }

    p_kmsg_que = container_of((unsigned long *)(*msg_que_id), struct kmsg_que_t, reserved);

    /* check the msg-length. */
    if (msg_len > p_kmsg_que->msg_size)
    {
        msg_len = p_kmsg_que->msg_size;
    }

    might_sleep();

    spin_lock_irqsave(&p_kmsg_que->mq_empty_wait.lock, flags);

    /* get the space-count in queue. */
    count = CIRC_SPACE(p_kmsg_que->head,
                       p_kmsg_que->tail,
                       p_kmsg_que->total_msg);
    if (!count)
    {
        DECLARE_WAITQUEUE(wait, current);

        wait.flags |= WQ_FLAG_EXCLUSIVE;
        __add_wait_queue_tail(&p_kmsg_que->mq_empty_wait, &wait);

        do
        {
            if (signal_pending(current))
            {
                ret = -ERESTARTSYS;
                __remove_wait_queue(&p_kmsg_que->mq_empty_wait, &wait);
                goto send_urgent_out;
            }
            __set_current_state(TASK_INTERRUPTIBLE);

            /* unlock... */
            spin_unlock_irqrestore(&p_kmsg_que->mq_empty_wait.lock, flags);

            if (NO_WAIT == option)
            {
                return KMSG_QUE_ERROR;
            }
            else if ((int)WAIT_FOREVER == option)
            {
                /* schedule-yield. */
                schedule();

                /* lock again... */
                spin_lock_irqsave(&p_kmsg_que->mq_empty_wait.lock, flags);
            }
            else
            {
                /* schedule-yield timeout. */
                option = schedule_timeout(option);

                /* lock again... */
                spin_lock_irqsave(&p_kmsg_que->mq_empty_wait.lock, flags);

                if (!option)
                {
                    ret = KMSG_QUE_TIMEOUT_ERR;
                    __remove_wait_queue(&p_kmsg_que->mq_empty_wait, &wait);
                    goto send_urgent_out;
                }
            }

        } while (!(count = CIRC_SPACE(p_kmsg_que->head,
                                      p_kmsg_que->tail,
                                      p_kmsg_que->total_msg)));

        /* get a space in the queue... */
        __remove_wait_queue(&p_kmsg_que->mq_empty_wait, &wait);
    }

    /* calculate the head-position.. */
    if(0 == p_kmsg_que->tail)
    {
        tail_tmp = p_kmsg_que->total_msg - 1;
    }
    else
    {
        tail_tmp = p_kmsg_que->tail - 1;
    }

    /* save the msg data. */
    memcpy(((char *)(p_kmsg_que->msg_data) + tail_tmp * p_kmsg_que->msg_size),
           msg_data, msg_len);

    /* update the tail-pointer. */
    p_kmsg_que->tail = tail_tmp;

    /* wake up one of the threads in the wait-queue. */
    __wake_up_common_02(&p_kmsg_que->mq_full_wait,
                        TASK_UNINTERRUPTIBLE | TASK_INTERRUPTIBLE,
                        1,
                        1,
                        NULL);
    ret = KMSG_QUE_OK;

send_urgent_out:
    spin_unlock_irqrestore(&p_kmsg_que->mq_empty_wait.lock, flags);

    /* attention to the return-value, 0-->KMSG_QUE_OK. */
    return ret;
}
EXPORT_SYMBOL(kmsg_que_send_urgent);


/* receive a msg from the specific queue.
 * option: no-wait.
 *         wait_forever
 *         some milliseconds.
 */
int  __sched kmsg_que_receive (const kmsg_que_id_t *msg_que_id,
                               void *msg_data,
                               int msg_len,
                               int option)
{
    struct kmsg_que_t *p_kmsg_que = NULL;
    int count;
    int ret;
    unsigned long flags;

    if ((NULL == msg_que_id) || (NULL == msg_data))
    {
        kmsg_error(":msg_id is invalid. \n");
        return KMSG_QUE_ERROR;
    }

    if (KMSG_QUE_MAGIC != *(unsigned long *)(*msg_que_id))
    {
        kmsg_error(":magic number is modified exceptionally. it means there are some errors in your code. \n");
        return KMSG_QUE_ERROR;
    }

    /* is it in isr context?? */
    if (in_interrupt())
    {
        /* attention: you should NOT block in isr(). */
        if (NO_WAIT != option)
        {
            kmsg_error(": you should NOT use 'WAIT_FOREVER' in ISR().\n");
            return KMSG_QUE_IRQ_ERR;
        }
    }

    p_kmsg_que = container_of((unsigned long *)(*msg_que_id), struct kmsg_que_t, reserved);

    /* attention: no-matter is it empty or full lock. */
    spin_lock_irqsave(&p_kmsg_que->mq_full_wait.lock, flags);


    /* get the msg-count in queue. */
    count = CIRC_CNT(p_kmsg_que->head,
                     p_kmsg_que->tail,
                     p_kmsg_que->total_msg);
    if (!count)
    {
        DECLARE_WAITQUEUE(wait, current);

        wait.flags |= WQ_FLAG_EXCLUSIVE;
        __add_wait_queue_tail(&p_kmsg_que->mq_full_wait, &wait);

        do
        {
            if (signal_pending(current))
            {
                ret = -ERESTARTSYS;
                __remove_wait_queue(&p_kmsg_que->mq_full_wait, &wait);
                goto recv_out;
            }
            __set_current_state(TASK_INTERRUPTIBLE);

            /* unlock... */
            spin_unlock_irqrestore(&p_kmsg_que->mq_full_wait.lock, flags);

            if (NO_WAIT == option)
            {
                return KMSG_QUE_ERROR;
            }
            else if ((int)WAIT_FOREVER == option)
            {
                /* schedule-yield. */
                schedule();

                /* lock again... */
                spin_lock_irqsave(&p_kmsg_que->mq_full_wait.lock, flags);
            }
            else
            {
                /* schedule-yield timeout. */
                option = schedule_timeout(option);

                /* lock again... */
                spin_lock_irqsave(&p_kmsg_que->mq_full_wait.lock, flags);

                if (!option)
                {
                    ret = KMSG_QUE_TIMEOUT_ERR;
                    __remove_wait_queue(&p_kmsg_que->mq_full_wait, &wait);
                    goto recv_out;
                }
            }

        } while (!(count = CIRC_CNT(p_kmsg_que->head,
                                    p_kmsg_que->tail,
                                    p_kmsg_que->total_msg)));

        /* get a space in the queue... */
        __remove_wait_queue(&p_kmsg_que->mq_full_wait, &wait);
    }

    /* get the msg-data. */
    memcpy(msg_data,
           ((char *)(p_kmsg_que->msg_data) + p_kmsg_que->tail * p_kmsg_que->msg_size),
           msg_len);

    /* update the head position. */
    p_kmsg_que->tail = (p_kmsg_que->tail + 1) % p_kmsg_que->total_msg;


    /* wake up one of the threads in the wait-queue. */
    __wake_up_common_02(&p_kmsg_que->mq_empty_wait,
                        TASK_UNINTERRUPTIBLE | TASK_INTERRUPTIBLE,
                        1,
                        1,
                        NULL);

    ret = KMSG_QUE_OK;

recv_out:
    spin_unlock_irqrestore(&p_kmsg_que->mq_full_wait.lock, flags);

    /* attention to the return-value, 0-->KMSG_QUE_OK. */
    return ret;
}
EXPORT_SYMBOL(kmsg_que_receive);


/* get the number of messages in the specific queue now.
 * msg_que_id: .
 */
int  kmsg_que_num_msgs (const kmsg_que_id_t *msg_que_id)
{
    struct kmsg_que_t *p_kmsg_que = NULL;
    int count;
    unsigned long flags;

    if (NULL == msg_que_id)
    {
        kmsg_error(":msg_id is invalid. \n");
        return KMSG_QUE_ERROR;
    }

    if (KMSG_QUE_MAGIC != *(unsigned long *)(*msg_que_id))
    {
        kmsg_error(":magic number is modified exceptionally. it means there are some errors in your code. \n");
        return KMSG_QUE_ERROR;
    }

    p_kmsg_que = container_of((unsigned long *)(*msg_que_id), struct kmsg_que_t, reserved);

    spin_lock_irqsave(&p_kmsg_que->mq_lock, flags);

    /* get the msg-count in queue. */
    count = CIRC_CNT(p_kmsg_que->head,
                     p_kmsg_que->tail,
                     p_kmsg_que->total_msg);
    spin_unlock_irqrestore(&p_kmsg_que->mq_lock, flags);

    return count;
}
EXPORT_SYMBOL(kmsg_que_num_msgs);


/* clean up all the messages in the specific queue now.
 * msg_que_id: .
 * attention: it would be a little dangerous,
 *            please refer to tiger for more details on its usage.
 */
int  __sched kmsg_que_cleanup (const kmsg_que_id_t *msg_que_id)
{
    struct kmsg_que_t *p_kmsg_que = NULL;
    int count;
    unsigned long flags;

    if (NULL == msg_que_id)
    {
        kmsg_error(":msg_id is invalid. \n");
        return KMSG_QUE_ERROR;
    }

    if (KMSG_QUE_MAGIC != *(unsigned long *)(*msg_que_id))
    {
        kmsg_error(":magic number is modified exceptionally. it means there are some errors in your code. \n");
        return KMSG_QUE_ERROR;
    }


    p_kmsg_que = container_of((unsigned long *)(*msg_que_id), struct kmsg_que_t, reserved);


    spin_lock_irqsave(&p_kmsg_que->mq_lock, flags);

    /* get the msg-count in queue. */
    count = CIRC_CNT(p_kmsg_que->head,
                     p_kmsg_que->tail,
                     p_kmsg_que->total_msg);
    while (count > 0)
    {
        /* update the head position. */
        p_kmsg_que->tail = (p_kmsg_que->tail + 1) & (p_kmsg_que->total_msg - 1);

        /* wake up one of the threads in the wait-queue. */
        __wake_up_common_02(&p_kmsg_que->mq_empty_wait,
                            TASK_UNINTERRUPTIBLE | TASK_INTERRUPTIBLE,
                            1,
                            1,
                            NULL);
        count--;
    }

    spin_unlock_irqrestore(&p_kmsg_que->mq_lock, flags);

    return KMSG_QUE_OK;
}
EXPORT_SYMBOL(kmsg_que_cleanup);

