﻿/*
 *  arch/arm/mach-emxx/kmsg_que.h
 *  implementation for msg queue management in kernel.
 *
 *  Copyright 2011 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 */

#ifndef _KMSG_QUEUE_H_
#define _KMSG_QUEUE_H_

#include <linux/circ_buf.h>
#include <linux/spinlock.h>
#include <linux/completion.h>


#define kmsg_error(fmt, args...)      \
         printk(KERN_ERR "%s: " fmt, __func__, ## args)


/* the minium total number of messages in queue,
 * and must be 2^n aligned.
 */
#define     KMSG_QUE_MIN_MSG        2


/* the maximum total number of messages in queue,
 * and must be 2^n aligned.
 */
#define     KMSG_QUE_MAX_MSG        512

/* the maxium number of bytes for one message.
 * attention: if you want to transit a big-chunk-data,
 * then you should use the data-pointer.
 */
#define     KMSG_QUE_MSG_MAX_LEN    256


#define     KMSG_QUE_MAGIC          0xAA55AA55


/*-----------------------------------------------------------------*/
/* ok */
#define     KMSG_QUE_OK         0

/* general error. */
#define     KMSG_QUE_ERROR      -1

/* length error. */
#define     KMSG_QUE_LEN_ERR    -2

/* it means you should NOT block in irq handler. */
#define     KMSG_QUE_IRQ_ERR    -3

/* attention: timeout error. */
#define     KMSG_QUE_TIMEOUT_ERR    1

/*-----------------------------------------------------------------*/


#ifndef NO_WAIT
#define NO_WAIT                 0
#endif


#ifndef WAIT_FOREVER
#define WAIT_FOREVER        LONG_MAX
//#define WAIT_FOREVER      0xffffffff
#endif

/* FASTCALL stuff */
#define FASTCALL(x)	x
#define fastcall

typedef unsigned long kmsg_que_id_t;


struct kmsg_que_t
{
    spinlock_t          mq_lock;

    wait_queue_head_t   mq_full_wait;
    wait_queue_head_t   mq_empty_wait;

    unsigned long       head;
    unsigned long       tail;

    unsigned long       total_msg;

    unsigned long       msg_size;

    void                *msg_data;

    unsigned long       reserved;   /* magic number for checking. 0xaa55aa55. */
};


/* create a kmsg queue. */
extern int FASTCALL(kmsg_que_create (kmsg_que_id_t *msg_que_id,
                                     unsigned long msg_num,
                                     unsigned long msg_len));

/* destroy a specific kmsg queue. */
extern int FASTCALL(kmsg_que_destroy (kmsg_que_id_t *msg_que_id /* , int option */));

/* send a msg to the specific queue. */
extern int FASTCALL(kmsg_que_send (const kmsg_que_id_t *msg_que_id,
                                   void *msg_data,
                                   int msg_len,
                                   int option));

/* for sysconif driver */
extern int FASTCALL(kmsg_que_send_urgent (kmsg_que_id_t *msg_que_id,
                    void *msg_data,
                    int msg_len,
                    int option));

/* receive a msg from the specific queue. */
extern int FASTCALL(kmsg_que_receive (const kmsg_que_id_t *msg_que_id,
                                      void *msg_data,
                                      int msg_len,
                                      int option));

/* get the number of messages in the specific queue now.
 * msg_que_id:
 */
extern int FASTCALL(kmsg_que_num_msgs (const kmsg_que_id_t *msg_que_id));

/* clean up all the messages in the specific queue now.
 * msg_que_id: .
 * attention: it would be a little dangerous,
 *            please refer to tiger for more details on its usage.
 */
extern int FASTCALL(kmsg_que_cleanup (const kmsg_que_id_t *msg_que_id));


#endif /* end of _KMSG_QUEUE_H_. */



