/*
 *  arch/arm/mach-emxx/lctty/EMMA_lctty.c
 *  EMMA tty driver
 *
 *  Copyright 2012 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 */

#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/wait.h>
#include <linux/tty.h>
#include <linux/tty_driver.h>
#include <linux/tty_flip.h>
#include <linux/serial.h>
#include <asm/uaccess.h>
#include <net/sock.h>
#include <linux/netlink.h>
#include <linux/sched.h>
#include <linux/list.h>
#include <linux/signal.h>
#include <linux/delay.h>

#define KERNEL_TO_USER_SOCKET_ID (25)
#define MAX_MSGSIZE (2048)

#define DRIVER_VERSION "v2.0"
#define DRIVER_DESC "EMMA tty driver"

MODULE_DESCRIPTION( DRIVER_DESC );

static struct tty_driver* tiny_tty_driver = NULL;

static struct sock* netlink_socket = NULL;
static pid_t user_pid = 0;
static struct tty_struct* ttys = NULL;

static int tiny_open(struct tty_struct* tty, struct file* file) {
    tty->driver_data = tiny_tty_driver;
    ttys = tty;
    return 0;
}
static void tiny_close(struct tty_struct* tty, struct file* file) {
    wake_up_interruptible(&tty->read_wait);
    wake_up_interruptible(&tty->write_wait);
}

static int task_is_ok(pid_t pid) {
    int ret = 0;
    struct task_struct* task = NULL;
    if(!pid) {
        return 0;
    }
// w/a sometime nsproxy has null pointer 
    if(current->nsproxy == NULL){
        printk(KERN_ALERT "[ERR][EMMA_tty] no nsproxy.\n");
        return 1;
    }
    rcu_read_lock();
    task = find_task_by_vpid(pid);
    if(task) {
        ret = task->state < TASK_STOPPED;
    }
    rcu_read_unlock();
    return ret;
}
static int tiny_write(struct tty_struct* tty, const unsigned char* buffer, int count) {
    int i = 0;
    int size = 0;
    struct sk_buff* skb = NULL;
    struct nlmsghdr* nlh = NULL;
    unsigned char* point = NULL;
    int ret = 0;
//  printk(KERN_ALERT "[EMMA_tty]tiny write buffer:%d\n",count);
    if(!tty || tty->stopped) {
        return 0;
    }
    i = tty->receive_room;
    if(i > count) {
        i = count;
    }
    if(!task_is_ok(user_pid)) {
        printk(KERN_ALERT "[ERR][EMMA_tty]user_pid failed\n");
        return 0;
    }
    size = NLMSG_SPACE(count + 4);
    if(in_interrupt()) {
        skb = alloc_skb(size, GFP_ATOMIC);
    } else {
        skb = alloc_skb(size, GFP_KERNEL);
    }
    nlh = NLMSG_PUT(skb, 0, 0, 0, size - sizeof(*nlh));
    nlh->nlmsg_pid = 0;
    nlh->nlmsg_flags = 0;
    nlh->nlmsg_len = NLMSG_SPACE(count + 4);
    NETLINK_CB(skb).pid = 0;
    NETLINK_CB(skb).dst_group = 0;
    point = (unsigned char*)NLMSG_DATA(nlh);
    memset(point, 0, count + 4);
    *(int*)point = count;
    memcpy(point + 4, buffer, count);
    ret = netlink_unicast(netlink_socket, skb, user_pid, MSG_DONTWAIT);
    if(ret <= 0) {
        printk(KERN_ALERT "[ERR][EMMA_tty]Sent to user Failed\n");
        return 0;
    }
    return i;
nlmsg_failure:
    kfree_skb(skb);
    return 0;
}

static int tiny_write_room(struct tty_struct* tty) {
    if(!tty || tty->stopped) {
        return 0;
    }
    return tty->receive_room;
}

static void tiny_flush_buffer(struct tty_struct* tty) {
    if(!tty) {
        return ;
    }
    if(tty->ldisc.ops->flush_buffer) {
        tty->ldisc.ops->flush_buffer(tty);
    }
    if(tty->packet) {
        tty->ctrl_status |= TIOCPKT_FLUSHWRITE;
        wake_up_interruptible(&tty->read_wait);
    }
}

static int tiny_chars_in_buffer(struct tty_struct* tty) {
    int count = 0;
    if(!tty || !tty->ldisc.ops->chars_in_buffer) {
        return 0;
    }
    count = tty->ldisc.ops->chars_in_buffer(tty);
    return count;
}


static void tiny_set_termios(struct tty_struct* tty, struct ktermios* old_termios) {
    return;
}


static struct tty_operations serial_ops = {
    .open = tiny_open,
    .close = tiny_close,
    .write = tiny_write,
    .write_room = tiny_write_room,
    .flush_buffer = tiny_flush_buffer,
    .chars_in_buffer = tiny_chars_in_buffer,
    .set_termios = tiny_set_termios,
};

static void netlink_receive(struct sk_buff* skb) {
    struct nlmsghdr* nlh = NULL;
    unsigned char* point = NULL;
    int i = 0;
    int j = 0;
    if(skb->len >= NLMSG_SPACE(0)) {
        nlh = (struct nlmsghdr*)skb->data;
    } else {
        return ;
    }
    if(strcmp((char*)NLMSG_DATA(nlh), "handlertiny") == 0) {
        user_pid = nlh->nlmsg_pid;
        printk(KERN_ALERT "[EMMA_tty][netlink_receive]user_pid = %d\n", user_pid);
    } else {
        point = (unsigned char*)NLMSG_DATA(nlh);
        j = *(int*)point;
        for(i = 0; i < j; i++) {
            tty_insert_flip_char(ttys, *(point + 4 + i), TTY_NORMAL);
        }
        tty_flip_buffer_push(ttys);
    }
}

static void netlink_rcv_msg(struct sk_buff* __skb) {
    struct sk_buff* skb = NULL;
    skb = skb_get(__skb);
    netlink_receive(skb);
    kfree_skb(skb);
}

static __init int tty_init(void) {
    int retval = 0;
    struct device *tty_dev = NULL;
    tiny_tty_driver = alloc_tty_driver(1);
    if (!tiny_tty_driver) {
        return -ENOMEM;
    }
    tiny_tty_driver->owner = THIS_MODULE;
    tiny_tty_driver->driver_name = "EMMA_tty";
    tiny_tty_driver->name = "EMMA_tty";
    tiny_tty_driver->major = 300;
    tiny_tty_driver->type = TTY_DRIVER_TYPE_SERIAL;
    tiny_tty_driver->subtype = SERIAL_TYPE_NORMAL;
    tiny_tty_driver->flags = TTY_DRIVER_REAL_RAW | TTY_DRIVER_DYNAMIC_DEV;
    tiny_tty_driver->init_termios = tty_std_termios;
    tiny_tty_driver->init_termios.c_cflag = B9600 | CS8 | CREAD | HUPCL | CLOCAL;
    tty_set_operations(tiny_tty_driver, &serial_ops);
    retval = tty_register_driver(tiny_tty_driver);
    if (retval) {
        printk(KERN_ALERT "[EMMA_tty]failed to register tiny tty driver\n");
        put_tty_driver(tiny_tty_driver);
        return retval;
    }
    tty_dev = tty_register_device(tiny_tty_driver, 0, NULL);
    if(IS_ERR(tty_dev)) {
        printk(KERN_ERR "[EMMA_tty]Cannot register LCtty device\n");
        tty_unregister_driver(tiny_tty_driver);
        put_tty_driver(tiny_tty_driver);
        return IS_ERR(tty_dev);
    }
    //netlink to be correct
    netlink_socket = netlink_kernel_create(&init_net, KERNEL_TO_USER_SOCKET_ID, 0, netlink_rcv_msg, NULL, THIS_MODULE);
    printk(KERN_ALERT "[EMMA_tty]tiny_init successful\n");
    return retval;
}


static __exit void tty_exit(void) {
    tty_unregister_device(tiny_tty_driver, 0);
    tty_unregister_driver(tiny_tty_driver);
    put_tty_driver(tiny_tty_driver);
    //netlink
    sock_release(netlink_socket->sk_socket);
    printk(KERN_ALERT "[EMMA_tty]tiny_exit successful\n");
}
module_init(tty_init);
module_exit(tty_exit);

