/*
 *  File Name       : arch/arm/mach-emxx/pm.c
 *  Function        : emxx_board
 *  Release Version : Ver 1.00e
 *  Release Date    : 2010/06/23
 *
 *  Copyright (C) Renesas Electronics 2010
 *
 *  This program is free software;you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by Free
 *  Softwere Foundation; either version 2 of License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warrnty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 *  more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; If not, write to the Free Software Foundation, Inc., 59
 *  Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include <linux/version.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/suspend.h>
#include <linux/device.h>
#include <linux/delay.h>
#include <linux/pm.h>
#include <linux/interrupt.h>
#ifdef CONFIG_PROC_FS
#include <linux/proc_fs.h>
#endif
#include <linux/io.h>
#include <linux/uaccess.h>
#include <linux/irq.h>

#include <asm/irq.h>
#include <asm/mach/time.h>

#include <mach/system.h>
#include <mach/smu.h>
#include <mach/pm.h>
#include <linux/em_export.h>

#ifdef CONFIG_SNSC_SSBOOT
#include <linux/ssboot.h>
#include <asm/pgalloc.h>
#include <asm/hardware/cache-l2x0.h>
#include "power.h"
#ifdef CONFIG_EMXX_SSBOOT_ADJUST_SCHED_CLOCK
#include "timer.h"
#endif
#endif

int emxx_sleep_while_idle;
EXPORT_SYMBOL(emxx_sleep_while_idle);

#ifdef CONFIG_PM
#include <linux/cpu.h>
#include <linux/suspend.h>
#include <mach/dma.h>
#include "pm_pmu.h"

static unsigned long pm_loops_per_jiffy;
static int sleep_while_idle_enable;
static unsigned int sleep_while_idle_count;
static unsigned int pm_idle_count;
static unsigned int skip_ktimer_idle_count;
static unsigned int suspend_enable;
static int emxx_can_sleep(void);
static int emxx_enter_state(suspend_state_t state);
static int emxx_suspend_prepare(suspend_state_t state);
static int emxx_suspend_enter(suspend_state_t state);
static void emxx_suspend_finish(suspend_state_t state);
static int emxx_pm_enter(suspend_state_t state);
static int emxx_pm_valid(suspend_state_t state);
static void emxx_pm_finish(void);
static void variable_init(void);
/* This is just an arbitrary number */
#define FREE_PAGE_NUMBER (100)

static void variable_init(void)
{
	/* initialize static variable */
	emxx_sleep_while_idle = 0;
	pm_loops_per_jiffy = 0;
	sleep_while_idle_enable = 0;
	sleep_while_idle_count = 0;
	pm_idle_count = 0;
	skip_ktimer_idle_count = 0;
	suspend_enable = 0;
}

/* refered by emxx_target() */
void emxx_change_normalx(unsigned int normal_div)
{
	ulong auto_frq_change;
	ulong syncset;

	if ((inl(SMU_CLK_MODE_SEL) & 0x00000F00) == (normal_div << 8))
		return;

	syncset = readl(SMU_CPUCLK_SYNCSET);
	if (!(syncset & 0x01)) {
		if (normal_div == NORMAL_A) {
			/* case ASYNC, set to PLL1 */
			writel((readl(SMU_CPUCLK_ASYNC_MODE) & 0xFFFFFFFC)
			 | 0x01, SMU_CPUCLK_ASYNC_MODE);
		} else {
			/* case ASYNC, set to domain_clk */
			writel(readl(SMU_CPUCLK_ASYNC_MODE) & 0xFFFFFFFC,
			 SMU_CPUCLK_ASYNC_MODE);
		}
		/* re-set ASYNC  */
		writel(syncset | 0x01000000 , SMU_CPUCLK_SYNCSET);
	}

	/* auto freq change off */
	auto_frq_change = inl(SMU_CKRQ_MODE);
	outl((auto_frq_change & ~0x1), SMU_CKRQ_MODE);

	outl(normal_div, SMU_CLK_MODE_SEL);
	while ((inl(SMU_CLK_MODE_SEL) & 0x00000F00) != (normal_div << 8))
		;

	/* restore auto freq change */
	outl(auto_frq_change, SMU_CKRQ_MODE);
}

static int idle_enable = 0;
module_param_named(idle, idle_enable, bool, S_IRUSR|S_IWUSR);

static void emxx_pm_do_idle(void)
{
	int suspend_req = 0;

	pm_idle_count++;

	/* for PCM Direct */
	if (suspend_enable) {
		if (!(inl(SMU_POWER_STATUS)
		 & PD_STATE_BIT(PMU_PW_POWERDOWN))) {
			/* DSP power is off, (indicates PCM
			   and pdma have switched to suspend mode) */
			suspend_req = 1;
		}
	}
	if (suspend_req) {
		emxx_sleep_while_idle = 1;
		if (emxx_can_sleep()) {
			emxx_enter_state(PM_SUSPEND_STANDBY);	/* L1 OFF */
			emxx_sleep_while_idle = 0;
			sleep_while_idle_count++;
			local_fiq_enable();
			local_irq_enable();
			return;
		}
		emxx_sleep_while_idle = 0;
	} else if (sleep_while_idle_enable) {
		emxx_sleep_while_idle = 1;
		if (emxx_can_sleep()) {
			emxx_enter_state(PM_SUSPEND_STANDBY);
			emxx_sleep_while_idle = 0;
			sleep_while_idle_count++;
			local_fiq_enable();
			local_irq_enable();
			return;
		}
		emxx_sleep_while_idle = 0;
	}

	arch_idle();

}

void emxx_pm_idle(void)
{
	local_irq_disable();
	local_fiq_disable();
	if (need_resched()) {
		local_fiq_enable();
		local_irq_enable();
		return;
	}

	if (idle_enable)
		emxx_pm_do_idle();

	local_fiq_enable();
	local_irq_enable();
}
#if 1
#define GCLKSDIO_CHECK_BIT	0x07
#define GCLKUSB_CHECK_BIT	0x0B
#define LCD_BASE		IO_ADDRESS(EMXX_LCD_BASE)
#define LCD_LCDOUT		(LCD_BASE + 0x10)

static int check_transfer_state(void)
{
	if (inl(LCD_LCDOUT))
		return 1;
	if (inl(SMU_SDIO0GCLKCTRL) & GCLKSDIO_CHECK_BIT)
		return 1;
	if (inl(SMU_SDIO1GCLKCTRL) & GCLKSDIO_CHECK_BIT)
		return 1;
	if (inl(SMU_SDIO2GCLKCTRL) & GCLKSDIO_CHECK_BIT)
		return 1;
	if (inl(SMU_SDCGCLKCTRL) & GCLKSDIO_CHECK_BIT)
		return 1;
	if (inl(SMU_USBGCLKCTRL) & GCLKUSB_CHECK_BIT)
		return 1;
	if (emxx_dma_busy())
		return 1;

	return 0;
}
#endif

#define EMXX_SLEEP_THRESHOLD 1

#ifdef CONFIG_EMXX_ANDROID
#include <linux/wakelock.h>
static int emxx_can_sleep(void)
{
	int ret = 0;

	if (!mutex_trylock(&pm_mutex))
		return 0;

	if (check_transfer_state())
		goto Unlock;

	if (has_wake_lock(WAKE_LOCK_IDLE) == 0)
		ret = 1;

Unlock:
	mutex_unlock(&pm_mutex);
	return ret;
}
#else
static int emxx_can_sleep(void)
{
	struct pm_message message = { .event = DEV_SUSPEND_IDLE_1, };
#ifdef CONFIG_NO_HZ
	unsigned long last_jiffies, delta_jiffies;
#endif
	int ret = 0;

	if (!mutex_trylock(&pm_mutex))
		return 0;

#ifdef CONFIG_NO_HZ
	/* check kernel timer */
	last_jiffies = jiffies;
	delta_jiffies = get_next_timer_interrupt(last_jiffies) - last_jiffies;
	if (delta_jiffies <= EMXX_SLEEP_THRESHOLD) {
		skip_ktimer_idle_count++;
		goto Unlock;
	}
#endif

	if (check_transfer_state())
		goto Unlock;

	if (!device_suspend(message))
		ret = 1;

	device_resume(PMSG_RESUME); /* restore dpm_list */

 Unlock:
	mutex_unlock(&pm_mutex);
	return ret;
}
#endif

static int emxx_enter_state(suspend_state_t state)
{
	int error;

	if (!mutex_trylock(&pm_mutex))
		return -EBUSY;

	error = emxx_suspend_prepare(state);
	if (error)
		goto Unlock;

	error = emxx_suspend_enter(state);

	emxx_suspend_finish(state);
 Unlock:
	mutex_unlock(&pm_mutex);
	return error;
}

static int emxx_suspend_prepare(suspend_state_t state)
{
	int error;
	unsigned int free_pages;

	free_pages = global_page_state(NR_FREE_PAGES);
	if ((free_pages) < FREE_PAGE_NUMBER) {
		pr_debug("PM: free some memory\n");
		shrink_all_memory(FREE_PAGE_NUMBER - free_pages);
		if (nr_free_pages() < FREE_PAGE_NUMBER) {
			error = -ENOMEM;
			printk(KERN_ERR "PM: No enough memory\n");
			goto Thaw;
		}
	}

	if (emxx_pm_valid(state) != 1) {
		error = -EINVAL;
		goto Thaw;
	}

	error = device_suspend(PMSG_SUSPEND);
	if (error)
		goto Resume_devices;

	error = disable_nonboot_cpus();
	if (!error)
		return 0;

	enable_nonboot_cpus();
 Resume_devices:
	emxx_pm_finish();
	device_resume(PMSG_RESUME);
 Thaw:
	return error;
}


static int emxx_suspend_enter(suspend_state_t state)
{
	int error = 0;

	error = device_power_down(PMSG_SUSPEND);
	if (error) {
		printk(KERN_ERR "Some devices failed to power down\n");
		goto Done;
	}

	error = emxx_pm_enter(state);
	device_power_up(PMSG_RESUME);
 Done:
	return error;
}

static void emxx_suspend_finish(suspend_state_t state)
{
	enable_nonboot_cpus();
	emxx_pm_finish();
	device_resume(PMSG_RESUME);
}

static ssize_t sleep_while_idle_show(struct kobject *kobj,
			struct kobj_attribute *attr, char *buf)
{
	return sprintf(buf, "%d %u %u %u\n",
		sleep_while_idle_enable,
		sleep_while_idle_count,
		skip_ktimer_idle_count,
		pm_idle_count
		);
}

static ssize_t sleep_while_idle_store(struct kobject *kobj,
		struct kobj_attribute *attr, const char *buf, size_t n)
{
	int value;
	if (sscanf(buf, "%d", &value) != 1 ||
	    (value != 0 && value != 1)) {
		printk(KERN_ERR "idle_sleep_store: Invalid value\n");
		return -EINVAL;
	}
	sleep_while_idle_enable = value;
	return n;
}

static struct kobj_attribute sleep_while_idle_attr =
	__ATTR(sleep_while_idle, 0644, sleep_while_idle_show,
						sleep_while_idle_store);



/* this table's value = div*2 */
static int div_table[12] = {
	2, 4, 6, 8, 12, 16, 24, 32, 3, 5, 10, 20
};

/*
 return value:
   "div*2"
*/
static int pm_get_acpu_div(int div_id)
{
	ulong div;

	switch (div_id) {
	case NORMAL_A:
		div = (inl(SMU_NORMALA_DIV) & 0x0F);
		break;
	case NORMAL_B:
		div = (inl(SMU_NORMALB_DIV) & 0x0F);
		break;
	case NORMAL_C:
		div = (inl(SMU_NORMALC_DIV) & 0x0F);
		break;
	case NORMAL_D:
	default:
		div = (inl(SMU_NORMALD_DIV) & 0x0F);
		break;
	}

	if (div < 12)
		div = div_table[div];
	else
		div = 2;

	return div;
}

void pm_change_normalA(void)
{
	emxx_change_normalx(NORMAL_A);

	loops_per_jiffy = pm_loops_per_jiffy;
}

void pm_change_normalB(void)
{
	/* set clock to div3(177MHz), or div4(133MHz) of Mode A */
	/* outl(0x04646172, SMU_NORMALB_DIV); */	/* 177MHz */
	outl(0x05757173, SMU_NORMALB_DIV);		/* 133MHz */
	emxx_change_normalx(NORMAL_B);

	/* div*2 */
	loops_per_jiffy = pm_loops_per_jiffy/(pm_get_acpu_div(NORMAL_B)/2);
}

#ifdef CONFIG_SNSC_SSBOOT

/* function prototypes */
extern int  emxx_suspend(void *func);
extern void emxx_resume(void);

/* buffer to save cpu context */
u_int32_t suspend_context_area[CONTEXT_SIZE_BYTES / sizeof(u_int32_t)];

/*
 * list of virtual addresses which must be mapped as VA=PA while
 * booting by Snapshot Boot.
 */

static void *lpmap_addr[] = {
	suspend_context_area,
	emxx_resume,
};

static struct saved_pmd {
	pmd_t *pmd;
#ifndef CONFIG_CPU_AFE
	pmd_t value[2];
#else
	pmd_t value[4];
#endif
} saved_pmd[ARRAY_SIZE(lpmap_addr)];

static int
emxx_ssboot_prepare_pgd(void)
{
	pmd_t *pmd_p, *pmd_v, *pmd_sv;
	unsigned long virt, phys;
	int i;

	for (i = 0; i < ARRAY_SIZE(lpmap_addr); i++) {
		virt = (unsigned long)lpmap_addr[i];
		phys = virt_to_phys((void *)virt);
		pmd_v = pmd_offset(pgd_offset(current->mm, virt), virt);
		pmd_p = pmd_offset(pgd_offset(current->mm, phys), phys);
		pmd_sv = (pmd_t *)&saved_pmd[i].value;
		saved_pmd[i].pmd = pmd_p;
		copy_pmd(pmd_sv, pmd_p);
		copy_pmd(pmd_p, pmd_v);
	}

	return 0;
}

static int
emxx_ssboot_restore_pgd(void)
{
	pmd_t *pmd_p, *pmd_sv;
	int i;

	for (i = ARRAY_SIZE(lpmap_addr) - 1; i >= 0; i--) {
		pmd_p = saved_pmd[i].pmd;
		pmd_sv = (pmd_t *)&saved_pmd[i].value;
		copy_pmd(pmd_p, pmd_sv);
	}

	flush_tlb_all();
	__flush_icache_all();

	return 0;
}

extern int gic_suspend(void);
extern int gic_resume(void);
extern int emxx_gio_suspend(void);
extern int emxx_gio_resume(void);
#ifdef CONFIG_MACH_EVSY
extern int evsy_prepare_l2x0_resume(void);
extern int evsy_board_resume(void);
#endif
static void
emxx_save_state(void)
{
#ifdef CONFIG_EMXX_SSBOOT_ADJUST_SCHED_CLOCK
	emxx_sched_clock_suspend();
#endif
	l2x0_suspend();
	gic_suspend();
	emxx_gio_suspend();
	emxx_ssboot_prepare_pgd();
}

static void
emxx_restore_state(void)
{
	emxx_ssboot_restore_pgd();
	emxx_gio_resume();
	gic_resume();
#ifdef CONFIG_MACH_EVSY
	if (ssboot_is_resumed())
		evsy_prepare_l2x0_resume();
#endif
	l2x0_resume();
#ifdef CONFIG_MACH_EVSY
	if (ssboot_is_resumed())
		evsy_board_resume();
#endif
#ifdef CONFIG_EMXX_SSBOOT_ADJUST_SCHED_CLOCK
	emxx_sched_clock_resume();
#endif
}
#endif /* CONFIG_SNSC_SSBOOT */

static void emxx_pm_suspend(suspend_state_t state)
{
	unsigned int sleep_mode;

	switch (state) {
	case PM_SUSPEND_STANDBY:
		sleep_mode = EMXX_PMU_CLK_SLEEP;
		break;
	case PM_SUSPEND_MEM:
	default:
		sleep_mode = EMXX_PMU_CLK_DEEPSLEEP;
		break;
	}
	emxx_pmu_sleep(sleep_mode);
}

static int emxx_pm_valid(suspend_state_t state)
{
	switch (state) {
	case PM_SUSPEND_STANDBY:
	case PM_SUSPEND_MEM:
#ifdef CONFIG_SNSC_SSBOOT
	case PM_SUSPEND_SNAPSHOT:
#endif
		return 1;

	default:
		return 0;
	}
}

static int emxx_pm_enter(suspend_state_t state)
{
	switch (state) {
	case PM_SUSPEND_STANDBY:
	case PM_SUSPEND_MEM:
		emxx_pm_suspend(state);
		break;
#ifdef CONFIG_SNSC_SSBOOT
	case PM_SUSPEND_SNAPSHOT:
	{
		int ret;
		emxx_save_state();
		ret = emxx_suspend(ssboot_snapshot);
		emxx_restore_state();
		return ret;
	}
#endif

	default:
		return -EINVAL;
	}

	return 0;
}

static void emxx_pm_finish(void)
{
}

static struct platform_suspend_ops emxx_pm_ops = {
	.valid = emxx_pm_valid,
	.enter = emxx_pm_enter,
	.finish = emxx_pm_finish,
};

void emxx_pm_power_off(void)
{
	unsigned long irq_flags;

	local_irq_save(irq_flags);

	/* poweroff setting function*/
	emxx_pm_do_poweroff();

	printk(KERN_INFO "%s(): AFTER PMU POWER OFF.\n", __func__);

	local_irq_restore(irq_flags);	  /* not call */
}

int emxx_pm_sleep(unsigned int mode)
{
	suspend_state_t state;
	int ret;

	switch (mode) {
	case PM_SLEEP_MODE_S1:
	case PM_SLEEP_MODE_S2:
		state = PM_SUSPEND_STANDBY;
		break;
	case PM_SLEEP_MODE_S3:
		state = PM_SUSPEND_MEM;
		break;
	default:
		return -EINVAL;
	}

	ret = pm_suspend(state);

	return ret;
}
EXPORT_SYMBOL(emxx_pm_sleep);

void emxx_pm_pdma_suspend_enable(void)
{
	suspend_enable = 1;
}
EXPORT_SYMBOL(emxx_pm_pdma_suspend_enable);

void emxx_pm_pdma_suspend_disable(void)
{
	suspend_enable = 0;
}
EXPORT_SYMBOL(emxx_pm_pdma_suspend_disable);

unsigned int emxx_pm_pdma_suspend_status(void)
{
	return suspend_enable;
}
EXPORT_SYMBOL(emxx_pm_pdma_suspend_status);


#ifdef CONFIG_PROC_FS
/*
 * Writing to /proc/pm puts the CPU in sleep mode
 */
static ssize_t emxx_pm_write_proc(struct file *file, const char *buffer,
				   size_t count, loff_t *ppos)
{
	char *buf;
	char *tok;
	char *s;
	char *str;
	char *whitespace = " \t\r\n";
	int ret = 0;

	if (current_uid() != 0)
		return -EPERM;
	if (count == 0)
		return 0;

	buf = kmalloc(count + 1, GFP_KERNEL);
	if (buf == NULL) {
		ret = -ENOMEM;
		goto exit_2;
	}
	if (copy_from_user(buf, buffer, count) != 0) {
		ret = -EFAULT;
		goto exit_1;
	}
	buf[count] = '\0';
	s = buf + strspn(buf, whitespace);
	tok = strsep(&s, whitespace);

	str = "HIGHCLK";
	if (strnicmp(tok, str, strlen(str) + 1) == 0) {
		printk(KERN_INFO "533MHz...\n");
		pm_change_normalA();
		goto exit_1;
	}
	str = "LOWCLK";
	if (strnicmp(tok, str, strlen(str) + 1) == 0) {
		printk(KERN_INFO "133MHz...\n");
		pm_change_normalB();
		goto exit_1;
	}

	str = "S1";
	if (strnicmp(tok, str, strlen(str) + 1) == 0) {
		emxx_pm_sleep(PM_SLEEP_MODE_S1);
		goto exit_1;
	}
	str = "S2";
	if (strnicmp(tok, str, strlen(str) + 1) == 0) {
		emxx_pm_sleep(PM_SLEEP_MODE_S2);
		goto exit_1;
	}
	str = "S3";
	if (strnicmp(tok, str, strlen(str) + 1) == 0) {
		emxx_pm_sleep(PM_SLEEP_MODE_S3);
		goto exit_1;
	}
	str = "PD";
	if (strnicmp(tok, str, strlen(str) + 1) == 0) {
		emxx_pm_power_off();
		goto exit_1;
	}
	ret = -EINVAL;

exit_1:
	kfree(buf);
exit_2:
	if (ret == 0)
		return count;

	return ret;
}

static int emxx_pm_read_proc(char *page, char **start, off_t off, int count,
			      int *eof, void *data)
{
	return 0;
}
#endif	/* CONFIG_PROC_FS */

#ifdef CONFIG_SNSC_SSBOOT
/*
 * Snapshot Boot support
 */

#define SSBOOT_TARGETID_EVSY 0x0D20

static void __init
emxx_ssboot_init(void)
{
#if defined(CONFIG_MACH_EVSY)
	ssboot_set_target_id(SSBOOT_TARGETID_EVSY);
#else
#error Please add target ID setup routine.
#endif
}
#endif

int __init emxx_pm_init(void)
{
	int ret;

	printk(KERN_INFO "Power Management for EMXX.\n");

	/* initialize static variable */
	variable_init();

	pm_idle = emxx_pm_idle;
	suspend_set_ops(&emxx_pm_ops);
	pm_loops_per_jiffy = loops_per_jiffy;

	ret = sysfs_create_file(power_kobj, &sleep_while_idle_attr.attr);
	if (ret)
		printk(KERN_ERR "subsys_create_file failed: %d\n", ret);

#ifdef CONFIG_PROC_FS
	{
		struct proc_dir_entry *entry;
		entry = create_proc_read_entry("pm", S_IWUSR | S_IRUGO, NULL,
					       emxx_pm_read_proc, 0);
		if (entry == NULL)
			return -ENOENT;
		entry->write_proc = (write_proc_t *)emxx_pm_write_proc;
	}
#endif

	pm_power_off = emxx_pm_power_off;

#ifdef CONFIG_SNSC_SSBOOT
	emxx_ssboot_init();
#endif
	return 0;
}

device_initcall(emxx_pm_init);

void em_reboot(int force)
{
#if 0
	pm_machine_reset(force);
#endif
}
#endif
