﻿/*
 *  arch/arm/mach-emxx/pwrctrl/pwrctrl_drv.c
 *  pwrctrl_drv interface
 *
 *  Copyright 2011 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/poll.h>
#include <linux/delay.h>
#include <linux/dma-mapping.h>
#include <linux/interrupt.h>
#include <linux/cdev.h>
#include <linux/platform_device.h>
#include <linux/vmalloc.h>
#include <linux/sched.h>
#include <linux/io.h>
#include <linux/gpio.h>

#include <mach/dma.h>
#include <mach/smu.h>
#include <mach/pmu.h>
#include <mach/spi.h>

#include "pwrctrl_drv.h"

#ifdef CONFIG_SNSC_HSS
#include <linux/snsc_hss.h>
#endif /* #ifdef CONFIG_SNSC_HSS */

#define DISABLE_GIO19_PULL(n) (n) = inl(CHG_PULL0); \
    (n) &= 0xffffffef;\
    outl((n),CHG_PULL0)

#define PWCTL_PACKET_SIZE 128

/*
 * @brief dma伝送
 */
struct spi_Buf_Dma {
    dma_addr_t dma_addr;    /* dma伝送元アドレス (物理) */
    unsigned int top;      /* バッファートップアドレス (仮想) */
    unsigned int len;      /* バッファー長さアドレス (仮想) */
};

/*
 * @brief spi バッファープロテクトポインタ
 */
struct spi_wr_Dma_Buf {
    struct spi_Buf_Dma DBufW;
    struct spi_Buf_Dma DBufR;
};

static int pwrctrl_major;
static dev_t t_dev;
static int onceIndex = 0;

static struct cdev pwrctrl_cdev;
static struct spi_wr_Dma_Buf spi_wr_Dma_Buf_local;

static SPI_CONFIG device0_config = {
    .dev    = SPI_DEV_SP0,
    .cs_sel = SPI_CS_SEL_CS0,
    .m_s    = SPI_M_S_MASTER,
    .dma    = SPI_DMA_ON,
    .pol    = SPI_CSW_8CLK | SPI_CK_DLY_OFF | SPI_CK_POL_NEG | SPI_CS_POL_NEG,
    .tiecs  = SPI_TIECS_FIXED,
    .nbw    = SPI_NB_8BIT,
    .nbr    = SPI_NB_8BIT,
    .sclk   = SPI_SCLK_1250KHZ,
    .mode   = SPI_CPHA_NORMAL,
};

#ifdef CONFIG_SNSC_HSS
struct hss_pwrctrl_info {
    struct hss_node *node;
};
struct hss_pwrctrl_info *p_hss_pwrctrl_info;
#endif  /* #ifdef CONFIG_SNSC_HSS */

/*
 * @brief       spiアロケートバッファー
 * @param[OUT]  なし
 * @param[IN]   size: アロケートバッファーサイズ
 * @return      ｛spiデバイ状態｝spiデバイの状態を戻す
 * @pre         なし
 * @post        なし
 * @attention   なし
 */
static int spi_allocate_buffer(unsigned int size)
{
    dma_addr_t phys;
    void* virt;

    if (0 == spi_wr_Dma_Buf_local.DBufW.dma_addr) {

        virt = dma_alloc_coherent(NULL, size, &phys, 0);
        if (NULL == virt) {
            printk(KERN_ERR "%s: memory allocation error\n",
                   PWRCTRL_NAME);
            return -ENOMEM;
        }

        spi_wr_Dma_Buf_local.DBufW.dma_addr = phys;
        spi_wr_Dma_Buf_local.DBufW.top = (unsigned long)virt;
        spi_wr_Dma_Buf_local.DBufW.len = size;

    }
    if(0 == spi_wr_Dma_Buf_local.DBufR.dma_addr) {

        virt = dma_alloc_coherent(NULL, size, &phys, 0);
        if (NULL == virt) {
            printk(KERN_ERR "%s: memory allocation error\n",
                   PWRCTRL_NAME);
            return -ENOMEM;
        }

        spi_wr_Dma_Buf_local.DBufR.dma_addr = phys;
        spi_wr_Dma_Buf_local.DBufR.top = (unsigned long)virt;
        spi_wr_Dma_Buf_local.DBufR.len = size;
    }

    return 0;
}

/*
 * @brief       spiフリーバッファー
 * @param[OUT]  なし
 * @param[IN]   なし
 * @return      なし
 * @pre         なし
 * @post        なし
 * @attention   なし
 */
static void spi_free_buffer(void)
{
    if (spi_wr_Dma_Buf_local.DBufW.dma_addr != 0) {
        dma_free_coherent(NULL,
                          spi_wr_Dma_Buf_local.DBufW.len,
                          (void*)(spi_wr_Dma_Buf_local.DBufW.top),
                          spi_wr_Dma_Buf_local.DBufW.dma_addr);

        spi_wr_Dma_Buf_local.DBufW.dma_addr = 0;
        spi_wr_Dma_Buf_local.DBufW.top = 0;
        spi_wr_Dma_Buf_local.DBufW.len = 0;
    }
    if (spi_wr_Dma_Buf_local.DBufR.dma_addr != 0) {
        dma_free_coherent(NULL,
                          spi_wr_Dma_Buf_local.DBufR.len,
                          (void*)(spi_wr_Dma_Buf_local.DBufR.top),
                          spi_wr_Dma_Buf_local.DBufR.dma_addr);

        spi_wr_Dma_Buf_local.DBufR.dma_addr = 0;
        spi_wr_Dma_Buf_local.DBufR.top = 0;
        spi_wr_Dma_Buf_local.DBufR.len = 0;
    }
    return ;
}

/*
 * @brief       pwrctrl伝送パケット
 * @param[OUT]  なし
 * @param[IN]   count: パケットサイズ
 * @return      実際伝送パケットサイズ
 * @pre         なし
 * @post        なし
 * @attention   なし
 */
static int  pwrctrl_sendPacket(size_t count)
{
    int ret = 0;
    int status = 0;
    int fix_val = -1;

    fix_val = 0;
    status = spi_force_cs(&device0_config, fix_val);
    if (status < 0) {
        status = -EFAULT;
        printk (KERN_ERR "SPI_CMD_SET_FIXCS force cs error\n");
        return status;
    }

    ret = spi_fdx_rw(&device0_config, spi_wr_Dma_Buf_local.DBufW.dma_addr, spi_wr_Dma_Buf_local.DBufR.dma_addr, count, 0);
    if (ret < 0) {
        printk(KERN_ERR "%s(): spi_write error\n", __func__);
        return ret;
    }

    fix_val = 1;
    status = spi_force_cs(&device0_config, fix_val);
    if (status < 0) {
        status = -EFAULT;
        printk (KERN_ERR "SPI_CMD_SET_FIXCS force cs error\n");
        return status;
    }
    return ret;
}

/** read (SPI0)
 * @brief ｛spi1デバイス読み出し操作｝spi_read_func関数をコールして、spi1デバイスの読み出し操作を行う。
 * @param[OUT] なし
 * @param[IN] <<file>> 標準のlinuxデバイスドライバーパラメーター
 * @param[IN] <<buf>> ユーザースペースにいるデータバッファエリア
 * @param[IN] <<count>> 読み出すべきバイトの数
 * @param[IN] <<ppos>> システムメンテナンス
 * @return ｛もう読み出したssize_t型バイトの数｝もう読み出したssize_t型バイトの数を戻す
 * @pre なし
 * @post なし
 * @attention なし
 */
static ssize_t pwrctrl_read(struct file* file, char* buf, size_t count,
                            loff_t* ppos)
{
    return 0;
}

/** write (SPI0)
 * @brief ｛spi1デバイス書き込み操作｝spi_write_func関数をコールして、spi1デバイスの書き込み操作を行う。
 * @param[OUT] なし
 * @param[IN] <<file>> 標準のlinuxデバイスドライバーパラメーター
 * @param[IN] <<buf>> ユーザースペースにいるデータバッファエリア
 * @param[IN] <<count>> バイトの数
 * @param[IN] <<ppos>> システムメンテナンス
 * @return ｛もう書き込んだssize_t型バイトの数｝もう書き込んだssize_t型のバイトの数を戻す
 * @pre なし
 * @post なし
 * @attention なし
 */
static ssize_t pwrctrl_write(struct file* file, const char* buf, size_t count,
                             loff_t* ppos)
{
    int writeSize = 0;

    if(count > PWCTL_PACKET_SIZE) {
        printk (KERN_ERR "Count size biger than  PWCTL_PACKET_SIZE! Error\n");
        return -EINVAL;
    }

    if (!copy_from_user((char*)spi_wr_Dma_Buf_local.DBufW.top, (char*)buf, count))
        writeSize = pwrctrl_sendPacket(count);
    if(writeSize < 0) {
        printk(KERN_ERR "%s(): %d pwrctrl_sendPacket error! \n", __func__, __LINE__);
        return writeSize;
    }
    copy_to_user((char*)buf, (char*)spi_wr_Dma_Buf_local.DBufR.top, writeSize);

    /* return copy size */
    return writeSize;
}

/** ioctl (SPI0)
 * @brief　 {spi1デバイスioctl操作｝spi_ioctl関数をコールして、spi1デバイスのioctl操作を行う。　
 * @param[OUT] なし
 * @param[IN] <<inode>> 標準のlinuxデバイスドライバーパラメーター
 * @param[IN] <<file>> 標準のlinuxデバイスドライバーパラメーター
 * @param[IN] <<cmd>> ユーザー定義する命令
 * @param[IN] <<arg>> ユーザー定義する命令パラメーター
 * @return ｛spiデバイ状態｝spiデバイの状態を戻す
 * @pre なし
 * @post なし
 * @attention なし
 */
static int pwrctrl_ioctl(struct inode* inode, struct file* file,
                         unsigned int cmd, unsigned long arg)
{
    int ret = 0;
    int fix_val = -1;

    switch (cmd) {
    case SPI_CMD_GET_MODE:
        if (copy_to_user((char*)arg, (char*)&device0_config,
                         sizeof(SPI_CONFIG))) {
            ret = -EFAULT;
        }
        break;

    case SPI_CMD_SET_MODE:
        if (copy_from_user((char*)&device0_config, (char*)arg,
                           sizeof(SPI_CONFIG))) {
            ret = -EFAULT;
        }
        if(device0_config.tiecs != SPI_TIECS_FIXED) {
            printk (KERN_ERR "This device.tiecs must be set as SPI_TIECS_FIXED! %s(): %d\n", __func__, __LINE__);
            device0_config.tiecs = SPI_TIECS_FIXED;
            ret = -EFAULT;
        }
        break;

    case PWRCTRL_IOC_PWRCTRL_NORMALCOM_POWEROFF:
        gpio_set_value(GPIO_P19, 0);
        break;
    default:/* error */
        printk (KERN_ERR "No such commands! %s(): %d\n", __func__, __LINE__);
        ret = -EINVAL;
        break;
    }
    return ret;
}

/** open (SPI0)
 * @brief  {spi1デバイスオーペン}　spi_open関数をコールして、spi1デバイスをオーペンする。
 * @param[OUT] なし
 * @param[IN] <<inode>> 標準のlinuxデバイスドライバーのパラメーター
 * @param[IN] <<file>> 標準のlinuxデバイスドライバーのパラメーター
 * @return ｛spiデバイ状態｝spiデバイの状態を戻す
 * @pre なし
 * @post なし
 * @attention なし
 */
static int pwrctrl_open(struct inode* inode, struct file* file)
{
    return 0;
}

/** release (SPI0)
 * @brief  ｛spiデバイス資源解放｝spi_release関数をコールして、spiデバイス資源解放を行う。
 * @param[OUT]なし
 * @param[IN] <<inode>> 標準のlinuxデバイスドライバーパラメーター
 * @param[IN] <<file>> 標準のlinuxデバイスドライバーパラメーター
 * @return ｛spiデバイス状態｝spiデバイスの状態を戻す
 * @pre 　　 spiデバイスが開いている
 * @post なし
 * @attention なし
 */
static int pwrctrl_release(struct inode* inode, struct file* file)
{
    return 0;
}

/* ファイル操作情報 (SPI0) */
static const struct file_operations pwrctrl_fops_pwrctrl = {
    .read = pwrctrl_read,
    .write = pwrctrl_write,
    .ioctl = pwrctrl_ioctl,
    .open = pwrctrl_open,
    .release = pwrctrl_release,
    .owner = THIS_MODULE,
};

/*
 * @brief       pwrctrlのgpioを初期化する
 * @param[OUT]  なし
 * @param[IN]   なし
 * @return      状態
 * @pre         なし
 * @post        なし
 * @attention   なし
 */
static int pwrctrl_gpio_init(void)
{
    unsigned int val = 0;

    gpio_direction_output(GPIO_P19, 1);
    gpio_set_value(GPIO_P19, 1);
    DISABLE_GIO19_PULL(val);

    return 0;
}

#ifdef CONFIG_SNSC_HSS
static int pwrctrl_hss_node_suspend(struct hss_node *node)
{
    /* TENTATIVE : no procedure */
    return 0;
}

static int pwrctrl_hss_node_resume(struct hss_node *node)
{
    /* TENTATIVE : initialize GPIO */
    pwrctrl_gpio_init();

    return 0;
}

static struct hss_node_ops hss_pwrctrl_node_ops = {
    .suspend = pwrctrl_hss_node_suspend,
    .resume = pwrctrl_hss_node_resume,
};

static int pwrctrl_hss_node_init(void)
{
    struct hss_node *node;
    int error = 0;

    /* alloc for pwrctrl info */
    p_hss_pwrctrl_info = kzalloc(sizeof(*p_hss_pwrctrl_info), GFP_KERNEL);
    if (!p_hss_pwrctrl_info) {
        return -ENOMEM;
    }

    /*
     * alloc the HSS node
     */
    node = hss_alloc_node(PWRCTRL_E_HSS_NODE_NAME);
    if (!node) {
        kfree(p_hss_pwrctrl_info);
        return -ENOMEM;
    }

    /*
     * save struct to HSS
     */
    hss_set_private(node, p_hss_pwrctrl_info);
    p_hss_pwrctrl_info->node = node;

    /*
     * Register the node.  If you want to get the callbacks to be called
     * before the specified node's one, specify the third parameter with
     * the name of the parent node. NULL means for no parent needed.
     */

    /* register HSS Platform for suspend/resume */
    error = hss_register_node(node, &hss_pwrctrl_node_ops, PWRCTRL_E_HSS_NODE_PARENT);

    if (error) {
        hss_free_node(node);
        kfree(p_hss_pwrctrl_info);
    }

    return error;
}

static void pwrctrl_hss_node_exit(void)
{
    hss_unregister_node(p_hss_pwrctrl_info->node);
    hss_free_node(p_hss_pwrctrl_info->node);
    kfree(p_hss_pwrctrl_info);
}
#endif /* #ifdef CONFIG_SNSC_HSS */

/** initialize
 * @brief  {ドライバー初期化}ドライバーを初期化する。
 * @param[OUT] なし
 * @param[IN] なし
 * @return {ドライバー初期化した状態}　ドライバーの初期化した状態を戻す
 * @pre 初めてドライバーを初期化しました
 * @post なし
 * @attention なし
 */
static int __init pwrctrl_init(void)
{
    int result = 0;
    int minor_num_start = 0;
    int dev_count = 1; /* SPI0 */
    spi_allocate_buffer(PWCTL_PACKET_SIZE);

    pwrctrl_gpio_init();

    if (pwrctrl_major) {
        t_dev  = MKDEV(pwrctrl_major, minor_num_start);
        result = register_chrdev_region(t_dev, dev_count, PWRCTRL_NAME);
    } else {
        result = alloc_chrdev_region(&t_dev, minor_num_start,
                                     dev_count, PWRCTRL_NAME);
        pwrctrl_major  = MAJOR(t_dev);
    }
    if (result < 0) {
        printk(KERN_ERR "pwrctrl: can't get major number.\n");
        goto fail_get_major;
    }

    cdev_init(&pwrctrl_cdev, &pwrctrl_fops_pwrctrl);
    pwrctrl_cdev.owner = THIS_MODULE;
    result = cdev_add(&pwrctrl_cdev, t_dev, dev_count);
    if (result) {
        goto fail_cdev_add;
    }

#ifdef CONFIG_SNSC_HSS
    /* register driver to hss */
    result = pwrctrl_hss_node_init();
    if(result != 0) {
        cdev_del(&pwrctrl_cdev);
        printk(KERN_ERR "pwrctrl: can't init hss.\n");
        goto fail_cdev_add;
    }
#endif /* #ifdef CONFIG_SNSC_HSS */

    printk(KERN_INFO "pwrctrl: registered device pwrctrl [TestTimerlist]\n");

    goto success;

fail_cdev_add:
    unregister_chrdev(pwrctrl_major, PWRCTRL_NAME);
fail_get_major:
    spi_free_buffer();
success:

    return result;
}

/** exit
 * @brief  ｛ドライバーアンロード｝ドライバーをアンロードする。
 * @param[OUT] なし
 * @param[IN] なし
 * @return なし
 * @pre pwrctrl もうドライバーを初期化しました
 * @post なし
 * @attention なし
 */
static void __exit pwrctrl_exit(void)
{
#ifdef CONFIG_SNSC_HSS
    /* unregister driver from hss */
    pwrctrl_hss_node_exit();
#endif /* #ifdef CONFIG_SNSC_HSS */
    cdev_del(&pwrctrl_cdev);
    unregister_chrdev_region(t_dev, 1);
    spi_free_buffer();
    return ;
}

module_init(pwrctrl_init);
module_exit(pwrctrl_exit);
MODULE_LICENSE("GPL");
