/* 2016-12-10: File added and changed by Sony Corporation */
/*
 *  File Name	    : arch/arm/mach-emxx/spi0.c
 *  Function	    : SPI0 interface
 *  Release Version : Ver 1.02
 *  Release Date    : 2011/01/21
 *
 * Copyright (C) 2010 Renesas Electronics Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1335, USA.
 */

#include <linux/delay.h>
#include <linux/dma-mapping.h>
#include <linux/interrupt.h>
#include <linux/platform_device.h>
#include <mach/dma.h>
#include <mach/smu.h>
#include <mach/pmu.h>

#include "spi.h"

#define SPI_PIO_POLLING

#define SPI_TMO_SEC 10
#define SPI_TMO_JIFFIES msecs_to_jiffies(SPI_TMO_SEC * MSEC_PER_SEC)

#define EPR(spi,fmt,args...) printk(KERN_ERR "spi%d: " fmt, (spi)->config->dev, ##args)

static struct spi_data *spi_private = 0;

static unsigned int spi_clock_table[] = {
	[SPI_SCLK_930KHZ]	= 937,		/* (930KHz) */
	[SPI_SCLK_1000KHZ]	= 955,		/* (955KHz) */
	[SPI_SCLK_1250KHZ]	= 1247,		/* (1247KHz) */
	[SPI_SCLK_1500KHZ]	= 1433,         /* (1434KHz) */
	[SPI_SCLK_3MHZ]		= 2867,         /* (2.87MHz) */
	[SPI_SCLK_6MHZ]		= 5734,         /* (5.73MHz) */
	[SPI_SCLK_6_0MHZ]	= 6036,         /* (6.03MHz) */
	[SPI_SCLK_12MHZ]	= 11468,        /* (11.47MHz) */
	[SPI_SCLK_24MHZ]	= 22937,        /* (22.94MHz) */
	[SPI_SCLK_48MHZ]	= 38229,        /* (38.23MHz) */
};

static unsigned int spi_plldiv_table[] = {
	[SPI_SCLK_930KHZ]	= SMU_DIV(255),	/*  930KHz(930KHz) */
	[SPI_SCLK_1000KHZ]	= SMU_DIV(240),	/* 1000KHz(1000KHz) */
	[SPI_SCLK_1250KHZ]	= SMU_DIV(188),	/* 1247KHz(1247KHz) */
	[SPI_SCLK_1500KHZ]	= SMU_DIV(160), /* 1500KHz(1434KHz) */
	[SPI_SCLK_3MHZ]		= SMU_DIV(80),  /* 3MHz   (2.87MHz) */
	[SPI_SCLK_6MHZ]		= SMU_DIV(40),  /* 6MHz   (5.73MHz) */
	[SPI_SCLK_6_0MHZ]	= SMU_DIV(38),  /* 6.03MHz (6.03MHz) */
	[SPI_SCLK_12MHZ]	= SMU_DIV(20),  /* 12MHz  (11.47MHz) */
	[SPI_SCLK_24MHZ]	= SMU_DIV(10),  /* 24MHz  (22.94MHz) */
	[SPI_SCLK_48MHZ]	= SMU_DIV(6),   /* 48MHz  (38.23MHz) */
};

static void spi_sft_reset(struct spi_data *spi)
{
	spi->regs->control |= SPx_CONTROL_RST;
	udelay((1000 * 4 / spi_clock_table[spi->config->sclk]) + 1);
	spi->regs->control &= ~(SPx_CONTROL_RST);
}

static void spi_power_on(struct spi_data *spi)
{
	emxx_open_clockgate(spi->smu->pclk);
	emxx_open_clockgate(spi->smu->sclk);
	emxx_unreset_device(spi->smu->reset);
	emxx_clkctrl_on(spi->smu->pclk_ctrl);
	emxx_clkctrl_on(spi->smu->sclk_ctrl);
}

static void spi_power_off(struct spi_data *spi)
{
	emxx_clkctrl_off(spi->smu->sclk_ctrl);
	emxx_clkctrl_off(spi->smu->pclk_ctrl);
	emxx_reset_device(spi->smu->reset);
	emxx_close_clockgate(spi->smu->sclk);
	emxx_close_clockgate(spi->smu->pclk);
}

static void spi_set_sclk_div(struct spi_data *spi, unsigned int sclk)
{
	unsigned int val;

	val = readl(spi->smu->sclk_div) & ~(0xffff << spi->smu->sclk_div_shift);
	val |= (spi_plldiv_table[sclk] | SMU_PLLSEL_PLL3) << spi->smu->sclk_div_shift;
	writel(val, spi->smu->sclk_div);
}

static int spi_unit(unsigned int bit)
{
	BUG_ON(!bit);

	if (bit <= SPI_NB_8BIT)
		return 1;
	else if (bit <= SPI_NB_16BIT)
		return 2;
	else if ((bit <= SPI_NB_24BIT))
		return 3;
	else
		return 4;
}

static void spi_lock(struct spi_data *spi)
{
	spin_lock_irqsave(&spi->spinlock, spi->lockflags);
}

static void spi_unlock(struct spi_data *spi)
{
	spin_unlock_irqrestore(&spi->spinlock, spi->lockflags);
}

static int spi_invalid_config(SPI_CONFIG *p)
{
	if (!p)
		return 1;
	if (p->dev >= SPI_DEV_NUM)
		return 1;
	if ((p->pol & SPI_CSW_MASK) > SPI_CSW_16CLK)
		return 1;
	if (p->sclk > SPI_SCLK_48MHZ)
		return 1;
	if (p->mode != SPI_CPHA_NORMAL && p->mode != SPI_CPHA_CHANGE)
		return 1;
	if ((p->nbr < SPI_NB_8BIT) || (SPI_NB_32BIT < p->nbr))
		return 1;
	if ((p->nbw < SPI_NB_8BIT) || (SPI_NB_32BIT < p->nbw))
		return 1;
	if (p->cs_sel > SPI_CS_SEL_CS7)
		return 1;
	if ((p->m_s != SPI_M_S_MASTER) && (p->m_s != SPI_M_S_SLAVE))
		return 1;
	if ((p->dma != SPI_DMA_OFF) && (p->dma != SPI_DMA_ON))
		return 1;

	return 0;
}

int spi_force_cs(SPI_CONFIG *config, int val)
{
	struct spi_data *spi;
	unsigned int tiecs = 0;
	int ret = 0;

	if (spi_invalid_config(config))
		return -EINVAL;

	if (val & ~1UL)
		return -EINVAL;

	if (!spi_private)
		return -EPERM;

	if (SPI_TIECS_TYPE(config->tiecs) == SPI_TIECS_FIXED) {
		tiecs = (1 << (config->cs_sel + 16));
		tiecs |= (val << config->cs_sel);
	}

	spi = &spi_private[config->dev];

	ret = wait_event_interruptible_timeout(spi->wq, !spi->busy, SPI_TMO_JIFFIES);
	if (ret < 0)
		return ret;
	if (!ret)
		return -ETIMEDOUT;

	ret = 0;
	spi_lock(spi);

	if (!spi->ready) {
		ret = -EPERM;
		goto out;
	}
	spi->busy = 1;

	/* set SPx_TIECS */
	spi->regs->tiecs = tiecs;
out:
	spi->busy = 0;
	spi_unlock(spi);

	wake_up(&spi->wq);

	return ret;
}
EXPORT_SYMBOL(spi_force_cs);

static void spi_cleanup(struct spi_data *spi)
{
#if 0 /* instead of spi_force_cs */
	SPI_CONFIG *config = spi->config;

	if (SPI_TIECS_TYPE(config->tiecs) == SPI_TIECS_FIXED) {
		unsigned int tiecs;
		tiecs = (1 << (config->cs_sel + 16));
		tiecs |= (!SPI_TIECS_ACTIVE(config->tiecs) << config->cs_sel);

		spi->regs->tiecs = tiecs;
	}
#endif
	spi->busy = 0;
	wake_up(&spi->wq);
}

static void spi_dump_regs(struct spi_data *spi)
{
	struct spi_regs *reg = spi->regs;

	EPR(spi, "mode=0x%08x, pol=0x%08x, control=0x%08x\n",
		reg->mode, reg->pol, reg->control);
	EPR(spi, "tx_data=0x%08x, rx_data=0x%08x\n",
		reg->tx_data, reg->rx_data);
	EPR(spi, "status=0x%08x, raw_status=0x%08x\n",
		reg->status, reg->raw_status);
	EPR(spi, "enset=0x%08x, enclr=0x%08x, ffclr=0x%08x\n",
		reg->enset, reg->enclr, reg->ffclr);
	EPR(spi, "control2=0x%08x, tiecs=0x%08x\n",
		reg->control2, reg->tiecs);
}

static int spi_xfer_has_finished(struct spi_data *spi)
{
	struct spi_status *rx = &spi->rx;
	struct spi_status *tx = &spi->tx;

	if (rx->err || tx->err)
		return 1;

	if (!rx->left && !tx->left)
		return 1;

	return 0;
}

static int spi_xfer_has_finished_one(struct spi_data *spi)
{
	struct spi_status *rx = &spi->rx;
	struct spi_status *tx = &spi->tx;

	if (!rx->size && !tx->size)
		return 1;

	return 0;
}

static void spi_update_stat(struct spi_status *stat, int err)
{
	unsigned int size;

	stat->err = err;

	if (err)
		return;

	size = stat->size;

	stat->size = 0;
	stat->buf += size;
	stat->left -= size;
}

static void spi_rx_done(struct spi_data *spi, int err)
{
	struct spi_status *rx = &spi->rx;
	struct spi_regs *regs = spi->regs;
	unsigned int data;

	BUG_ON(rx->size > sizeof(data));

	data = regs->rx_data;
	memcpy(rx->buf, &data, rx->size);

	if (err) {
		EPR(spi, "rx error detect: err=0x%08x\n", err);
		spi_dump_regs(spi);
	}

	spi_update_stat(rx, err);
}

static void spi_tx_done(struct spi_data *spi, int err)
{
	if (err) {
		EPR(spi, "tx error detect: err=0x%08x\n", err);
		spi_dump_regs(spi);
	}

	spi_update_stat(&spi->tx, err);
}

static void spi_setup_xfer(struct spi_status *stat, int size, void (*done)(struct spi_data *spi, int err))
{
	stat->size = size;
	stat->done = done;
}

static void spi_setup_rx(struct spi_data *spi)
{
	spi_setup_xfer(&spi->rx, spi_unit(spi->config->nbr), spi_rx_done);
}

static void spi_setup_tx(struct spi_data *spi)
{
	struct spi_status *tx = &spi->tx;
	unsigned int data;

	spi_setup_xfer(tx, spi_unit(spi->config->nbw), spi_tx_done);

	BUG_ON(tx->size > sizeof(data));

	memcpy(&data, tx->buf, tx->size);
	spi->regs->tx_data = data;
}

static void spi_start_rx(struct spi_data *spi, unsigned int control2, unsigned int enset)
{
	struct spi_regs *regs = spi->regs;

	regs->control2 = control2;
	regs->enset = (enset | SPx_ENSET_RX_ALLERR_EN);
	regs->control = (SPx_CONTROL_RD | SPx_CONTROL_START);
}

static void spi_start_tx(struct spi_data *spi, unsigned int control2, unsigned int enset)
{
	struct spi_regs *regs = spi->regs;

	regs->control2 = control2;
	regs->enset = (enset | SPx_ENSET_TX_ALLERR_EN);
	regs->control = (SPx_CONTROL_WRT | SPx_CONTROL_START);
}

static void spi_start_fdx(struct spi_data *spi, unsigned int control2, unsigned int enset)
{
	struct spi_regs *regs = spi->regs;

	regs->control2 = control2;
	regs->enset = (enset | SPx_ENSET_RX_ALLERR_EN | SPx_ENSET_TX_ALLERR_EN);
	regs->control =	(SPx_CONTROL_RD | SPx_CONTROL_WRT | SPx_CONTROL_START);
}

static void spi_rx_pio(struct spi_data *spi, int inten)
{
	if (inten)
		inten = SPx_ENSET_RDV_EN;

	spi_setup_rx(spi);
	spi_start_rx(spi, 0, inten);
}

static void spi_tx_pio(struct spi_data *spi, int inten)
{
	if (inten)
		inten = SPx_ENSET_END_EN;

	spi_setup_tx(spi);
	spi_start_tx(spi, 0, inten);
}

static void spi_fdx_pio(struct spi_data *spi, int inten)
{
	if (inten)
		inten = SPx_ENSET_RDV_EN | SPx_ENSET_END_EN;

	spi_setup_rx(spi);
	spi_setup_tx(spi);
	spi_start_fdx(spi, 0, inten);
}

static void __spi_xfer_pio(struct spi_data *spi, int inten)
{
	SPI_CONFIG *config = spi->config;

	if (config->dir == SPI_READ)
		spi_rx_pio(spi, inten);
	else if (config->dir == SPI_WRITE)
		spi_tx_pio(spi, inten);
	else
		spi_fdx_pio(spi, inten);
}

static void spi_xfer_done(struct spi_data *spi, int err)
{
	spi_cleanup(spi);

	if (spi->done) {
		if (!err && (spi->rx.err || spi->tx.err))
			err = -EIO;
		spi->done(spi->priv, err);
		spi->done = 0;
	}
}

#ifdef SPI_PIO_POLLING
static void spi_xfer_pio_done(struct spi_data *spi)
{
	struct spi_regs *reg = spi->regs;
	unsigned int stat = reg->raw_status;

	if (stat & (SPx_RAW_STATUS_RDV_RAW | SPx_RAW_STATUS_RX_ALLERR_RAW)) {
		if (spi->rx.done)
			spi->rx.done(spi, stat & SPx_RAW_STATUS_RX_ALLERR_RAW);
	}

	if (stat & (SPx_RAW_STATUS_END_RAW | SPx_RAW_STATUS_TX_ALLERR_RAW)) {
		if (spi->tx.done)
			spi->tx.done(spi, stat & SPx_RAW_STATUS_TX_ALLERR_RAW);
	}
}

static int spi_xfer_wait(struct spi_data *spi)
{
	struct spi_regs *reg = spi->regs;
	unsigned long tmo = jiffies + SPI_TMO_JIFFIES;
	unsigned int v;
	int err = 0;

	while (!spi_xfer_has_finished_one(spi)) {
		v = reg->control;
		if (!(v & SPx_CONTROL_START))
			spi_xfer_pio_done(spi);

		spi_unlock(spi);

		if (time_after(jiffies, tmo)) {
			EPR(spi, "polling timedout: %d sec\n", SPI_TMO_SEC);
			err = -ETIMEDOUT;
			spi_lock(spi);
			break;
		}

		cond_resched();
		spi_lock(spi);
	}

	return err;
}

static void spi_xfer_pio_poll(struct spi_data *spi)
{
	int err = 0;

	while (!spi_xfer_has_finished(spi)) {
		__spi_xfer_pio(spi, 0);

		err = spi_xfer_wait(spi);
		if (err)
			break;
	}

	spi_xfer_done(spi, err);
}

# define spi_xfer_pio(spi) spi_xfer_pio_poll(spi)
#else
# define spi_xfer_pio(spi) __spi_xfer_pio(spi, 1)
#endif /* SPI_PIO_POLLING */

static void spi_setup_dma_mode(dma_regs_t *regs, unsigned int nb)
{
	if (nb <= SPI_NB_8BIT)
		regs->mode = EMXX_DMAC_DEFMODE_8BIT;
	else if (nb <= SPI_NB_16BIT)
		regs->mode = EMXX_DMAC_DEFMODE_16BIT;
	else
		regs->mode = EMXX_DMAC_DEFMODE_32BIT;
}

static int spi_start_dma(struct spi_dma *dma, dma_addr_t src, dma_addr_t dst, int inten)
{
	return emxx_start_dma(dma->lch, src, 0, dst, inten | EMXX_DMAC_INT_ERROR_EN);
}

static void spi_stop_dma(struct spi_dma *dma)
{
	emxx_stop_dma(dma->lch);
}

static void spi_rx_dma_done(struct spi_data *spi, int err)
{
	struct spi_status *rx = &spi->rx;
	struct spi_regs *regs = spi->regs;
	SPI_CONFIG *config = spi->config;
	unsigned int v;

	if (err) {
		EPR(spi, "rx dma error detect: err=0x%08x\n", err);
		spi_dump_regs(spi);
	}

	if (config->m_s == SPI_M_S_MASTER) {
		v = regs->control;
		if (err && (v & SPx_CONTROL_START)) {
			v |= SPx_CONTROL_STOP;
			regs->control = v;
			udelay((1000 * config->nbr / spi_clock_table[config->sclk]) + 1);
		}
		v = regs->control2;
		v &= ~(SPx_CONTROL2_RX_STOP_MODE | SPx_CONTROL2_RX_FIFO_FULL_MASK);
		regs->control2 = v;
	}
	spi_stop_dma(spi->dma_rx);
	regs->enclr = SPx_ENCLR_RX_ALL_MASK;

	if (err)
		spi_sft_reset(spi);

	spi_update_stat(rx, err);
}

static void spi_tx_dma_done(struct spi_data *spi, int err)
{
	struct spi_status *tx = &spi->tx;
	struct spi_regs *regs = spi->regs;
	SPI_CONFIG *config = spi->config;
	unsigned int v;

	if (err) {
		EPR(spi, "tx dma error detect: err=0x%08x\n", err);
		spi_dump_regs(spi);
	}

	if (config->m_s == SPI_M_S_MASTER) {
		v = regs->control;
		if (err && (v & SPx_CONTROL_START)) {
			v |= SPx_CONTROL_STOP;
			regs->control = v;
			udelay((1000 * config->nbw / spi_clock_table[config->sclk]) + 1);
		}
		v = regs->control2;
		v &= ~SPx_CONTROL2_TX_STOP_MODE;
		regs->control2 = v;
	}
	spi_stop_dma(spi->dma_tx);
	regs->enclr = SPx_ENCLR_TX_ALL_MASK;

	if (err)
		spi_sft_reset(spi);

	spi_update_stat(tx, err);
}

#define dma_size2blksize(size) (((size) > SPI_DMA_BLOCK_MAXSIZE) ? SPI_DMA_BLOCK_MAXSIZE : (size))

static unsigned int spi_setup_rx_dma(struct spi_data *spi, int inten)
{
	struct spi_status *rx = &spi->rx;
	struct spi_dma *dma = spi->dma_rx;
	dma_regs_t *regs = dma->regs;
	SPI_CONFIG *config = spi->config;
	int size = rx->left;
	unsigned int control2 = 0;
	int err;

	if (size > SPI_DMA_MAXSIZE)
		size = SPI_DMA_MAXSIZE;

	dma_sync_single_for_device(0, (dma_addr_t)rx->buf, size, DMA_FROM_DEVICE);

	spi_setup_xfer(rx, size, spi_rx_dma_done);
	spi_setup_dma_mode(regs, config->nbr);

	regs->boff = 0;
	regs->bsize = dma_size2blksize(size);
	regs->bsize_count = 0;
	regs->leng = size;

	err = spi_start_dma(dma, dma->data, (dma_addr_t)rx->buf, inten);
	if (err) {
		EPR(spi, "rx: spi_start_dma failed: %d\n", err);
		BUG();
	}

	if (config->m_s == SPI_M_S_MASTER) {
		unsigned int full = ((size / spi_unit(spi->config->nbr)) - 1);
		if (full < 0x10000)
			control2 = (((full & 0xff00) << 8) | (full & 0xff) | SPx_CONTROL2_RX_STOP_MODE);
	}

	return control2;
}

static unsigned int spi_setup_tx_dma(struct spi_data *spi, int inten)
{
	struct spi_status *tx = &spi->tx;
	struct spi_dma *dma = spi->dma_tx;
	dma_regs_t *regs = dma->regs;
	SPI_CONFIG *config = spi->config;
	int size = tx->left;
	unsigned int control2 = 0;
	int err = 0;

	if (size > SPI_DMA_MAXSIZE)
		size = SPI_DMA_MAXSIZE;

	dma_sync_single_for_device(0, (dma_addr_t)tx->buf, size, DMA_TO_DEVICE);

	spi_setup_xfer(tx, size, spi_tx_dma_done);
	spi_setup_dma_mode(regs, config->nbw);

	regs->aoff = 0;
	regs->asize = dma_size2blksize(size);
	regs->asize_count = 0;
	regs->leng = size;

	err = spi_start_dma(dma, (dma_addr_t)tx->buf, dma->data, inten);
	if (err) {
		EPR(spi, "tx: spi_start_dma failed: %d\n", err);
		BUG();
	}

	if (config->m_s == SPI_M_S_MASTER)
		control2 = SPx_CONTROL2_TX_STOP_MODE;

	return control2;
}

static void spi_rx_dma(struct spi_data *spi)
{
	unsigned int control2;

	/* as rx, should detect to complete with DMA interrupt (spi->dma) */
	control2 = spi_setup_rx_dma(spi, EMXX_DMAC_INT_LENG_EN);
	spi_start_rx(spi, control2, 0);
}

static void spi_tx_dma(struct spi_data *spi)
{
	unsigned int control2;

	/* as tx, should detect to complete with SPI interrupt (dma->spi) */
	control2 = spi_setup_tx_dma(spi, 0);
	spi_start_tx(spi, control2, SPx_ENSET_TX_STOP_EN);
}

static void spi_fdx_dma(struct spi_data *spi)
{
	unsigned int control2;

	/* as rx, should detect to complete with DMA interrupt (spi->dma) */
	/* as tx, should detect to complete with SPI interrupt (dma->spi) */
	control2 = spi_setup_rx_dma(spi, EMXX_DMAC_INT_LENG_EN);
	control2 |= spi_setup_tx_dma(spi, 0);

	spi_start_fdx(spi, control2, SPx_ENSET_TX_STOP_EN);
}

static void spi_xfer_dma(struct spi_data *spi)
{
	SPI_CONFIG *config = spi->config;

	if (config->dir == SPI_READ)
		spi_rx_dma(spi);
	else if (config->dir == SPI_WRITE)
		spi_tx_dma(spi);
	else
		spi_fdx_dma(spi);
}

static void spi_xfer(struct spi_data *spi)
{
	SPI_CONFIG *config = spi->config;

	if (config->dma == SPI_DMA_ON)
		spi_xfer_dma(spi);
	else
		spi_xfer_pio(spi);
}

static int spi_config(struct spi_data *spi, SPI_CONFIG *config)
{
	unsigned int mode = 0;
	unsigned int pol = 0;
	unsigned int pol_bit = 0;
	unsigned int csw = 0;
	unsigned int csw_pol = 0;
	unsigned int nb = 0;
	unsigned int tiecs = 0;

	/* bit length */
	if (config->dir == SPI_READ)
		nb = config->nbr;
	else if (config->dir == SPI_WRITE || config->dir == SPI_RW)
		nb = config->nbw;
	else
		nb = SPI_NB_16BIT;

	mode |= ((nb - 1) << 8);
	if (config->mode == SPI_CPHA_CHANGE)
		mode |= (1 << 13);	/* SPI_MODE_1 or SPI_MODE_3 */

	/* chip select */
	mode |= (config->cs_sel << 4);

	/* master/slave */
	mode |= (config->m_s << 1);

	/* dma on/off */
	mode |= (config->dma << 0);

	/* tiecs */
#ifdef CONFIG_EMEV_KZM9D
	if (SPI_TIECS_TYPE(config->tiecs) == SPI_TIECS_FIXED)
		tiecs = 1 << (config->cs_sel + 16);
#endif
#if 0 /* instead of spi_force_cs */
	if (SPI_TIECS_TYPE(config->tiecs) == SPI_TIECS_FIXED) {
		tiecs = (1 << (config->cs_sel + 16));
		tiecs |= (SPI_TIECS_ACTIVE(config->tiecs) << config->cs_sel);
	}
#endif
	/* pol */
	if (config->cs_sel < SPI_CS_SEL_CS4)
		pol_bit = config->cs_sel * 3;
	else
		pol_bit = 16 + ((config->cs_sel - 4) * 3);

	csw = (config->pol & SPI_CSW_MASK);
	pol = (config->pol & SPI_POL_MASK);

	csw_pol = spi->pol;
	csw_pol &= ~(SPI_CSW_MASK | (SPI_POL_MASK << pol_bit));
	csw_pol |= (csw | (pol << pol_bit));

	/* set sclk */
	spi_set_sclk_div(spi, config->sclk);

	/* set SPx_MODE */
	spi->regs->mode = mode;

	/* set SPx_TIECS */
	spi->regs->tiecs = tiecs;

	/* set SPx_POL */
	spi->regs->pol = csw_pol;
	spi->pol = csw_pol;

	/* software reset */
	spi_sft_reset(spi);

	memcpy(spi->config, config, sizeof(SPI_CONFIG));

	return 0;
}

static void spi_init_status(struct spi_status *stat, char *buf, unsigned int size)
{
	stat->buf = buf;
	stat->left = size;
	stat->size = 0;
	stat->err = 0;
	stat->done = 0;
}

static int spi_init_xfer(struct spi_data *spi, SPI_CONFIG *config, struct spi_transfer *trans,
				void (*done)(void *, int), void *priv)
{
	if (!spi->ready)
		return -EPERM;

	if (spi->busy)
		return -EBUSY;

	spi->busy = 1;
	spi->done = done;
	spi->priv = priv;

	spi_init_status(&spi->rx, trans->rx_buf, trans->rx_len);
	spi_init_status(&spi->tx, trans->tx_buf, trans->tx_len);

	spi_config(spi, config);

	return 0;
}

static int spi_async(SPI_CONFIG *config, struct spi_transfer *trans, void (*done)(void *, int), void *priv)
{
	struct spi_data *spi;
	int err;

	if (!spi_private)
		return -EPERM;

	spi = &spi_private[config->dev];

	err = wait_event_interruptible_timeout(spi->wq, !spi->busy, SPI_TMO_JIFFIES);
	if (err < 0)
		return err;
	if (!err)
		return -ETIMEDOUT;

	spi_lock(spi);

	err = spi_init_xfer(spi, config, trans, done, priv);
	if (err)
		goto out;

	spi_xfer(spi);

	/* tentative for CSI */
	if (!done)
		spi_cleanup(spi);
out:
	spi_unlock(spi);

	return err;
}

struct spi_sync_data {
	struct completion com;
	int err;
};

static void spi_sync_done(void *priv, int err)
{
	struct spi_sync_data *data = (struct spi_sync_data *)priv;

	if (data) {
		data->err = err;
		complete(&data->com);
	}
}

static void spi_err_cleanup(SPI_CONFIG *config, int err)
{
	struct spi_data *spi;

	if (!spi_private)
		return;

	spi = &spi_private[config->dev];
	EPR(spi, "error detect: err=%d\n", err);

	spi_lock(spi);

	spi_cleanup(spi);

	spi_unlock(spi);
}

static int spi_sync(SPI_CONFIG *config, struct spi_transfer *trans)
{
	int err;
	struct spi_sync_data data = {
		.com = COMPLETION_INITIALIZER(data.com),
		.err = 0,
	};

	err = spi_async(config, trans, spi_sync_done, &data);
	if (err)
		return err;

	err = wait_for_completion_interruptible_timeout(&data.com, SPI_TMO_JIFFIES);
	if (err > 0) {
		err = data.err;
	} else {
		if (!err)
			err = -ETIMEDOUT;
		spi_err_cleanup(config, err);
	}

	return err;
}

static void spi_done(struct spi_data *spi)
{
	if (spi_xfer_has_finished(spi))
		spi_xfer_done(spi, 0);
	else if (spi_xfer_has_finished_one(spi))
		spi_xfer(spi);
}

static irqreturn_t spi_interrupt(int irq, void *dev_id)
{
	struct spi_data *spi;
	struct spi_regs *regs;
	unsigned int stat;

	if (!dev_id)
		return IRQ_NONE;

	spi = (struct spi_data *)dev_id;
	spi_lock(spi);

	regs = spi->regs;

	stat = regs->status;
	regs->ffclr = stat;

	if (stat & (SPx_STATUS_RX_STOP | SPx_STATUS_RDV | SPx_STATUS_RX_ALLERR)) {
		if (spi->rx.done)
			spi->rx.done(spi, stat & SPx_STATUS_RX_ALLERR);
	}

	if (stat & (SPx_STATUS_TX_STOP | SPx_STATUS_END | SPx_STATUS_TX_ALLERR)) {
		if (spi->tx.done)
			spi->tx.done(spi, stat & SPx_STATUS_TX_ALLERR);
	}

	spi_done(spi);

	spi_unlock(spi);

	return IRQ_HANDLED;
}

static void spi_interrupt_dma(struct spi_data *spi, int intsts, int dir)
{

	if (!spi)
		return;

	spi_lock(spi);

	if (intsts & EMXX_DMAC_INT_LENG_WR) {
		intsts &= ~EMXX_DMAC_INT_LENG_WR;
		if ((dir == SPI_READ) && spi->rx.done)
			spi->rx.done(spi, intsts);
		else if (spi->tx.done)
			spi->tx.done(spi, intsts);
	}

	spi_done(spi);

	spi_unlock(spi);
}

static void spi_interrupt_dma_rx(void *data, int intsts, int intrawsts)
{
	spi_interrupt_dma((struct spi_data *)data, intsts, SPI_READ);
}

static void spi_interrupt_dma_tx(void *data, int intsts, int intrawsts)
{
	spi_interrupt_dma((struct spi_data *)data, intsts, SPI_WRITE);
}

static inline void spi_transfer_init(struct spi_transfer *trans, char *rx_buf, unsigned int rx_len,
					char *tx_buf, unsigned int tx_len)
{
	trans->rx_buf = rx_buf;
	trans->rx_len = rx_len;
	trans->tx_buf = tx_buf;
	trans->tx_len = tx_len;
}

int spi_read(SPI_CONFIG *config, char *buf, unsigned long phys,
		unsigned int size, unsigned int flags)
{
	struct spi_transfer trans;
	int ret;

	if (spi_invalid_config(config))
		return -EINVAL;

	if (config->dma == SPI_DMA_ON)
		buf = (char *)phys;

	if (!buf || !size)
		return -EINVAL;

	if (size % spi_unit(config->nbr))
		return -EINVAL;

	spi_transfer_init(&trans, buf, size, 0, 0);

	config->dir = SPI_READ;

	ret = spi_sync(config, &trans);
	if (!ret)
		ret = size;

	return ret;
}
EXPORT_SYMBOL(spi_read);

int spi_write(SPI_CONFIG *config, char *buf, unsigned long phys,
		unsigned int size, unsigned int flags)
{
	struct spi_transfer trans;
	int ret;

	if (spi_invalid_config(config))
		return -EINVAL;

	if (config->dma == SPI_DMA_ON)
		buf = (char *)phys;

	if (!buf || !size)
		return -EINVAL;

	if (size % spi_unit(config->nbw))
		return -EINVAL;

	spi_transfer_init(&trans, 0, 0, buf, size);

	config->dir = SPI_WRITE;

	ret = spi_sync(config, &trans);
	if (!ret)
		ret = size;

	return ret;
}
EXPORT_SYMBOL(spi_write);

static int spi_fdx_invalid_param(SPI_CONFIG *config, unsigned long wphys, unsigned long rphys, unsigned int size)
{
	if (spi_invalid_config(config))
		return 1;

	if (!wphys || !rphys || !size)
		return 1;

	if (size % spi_unit(config->nbr))
		return 1;

	if (size % spi_unit(config->nbw))
		return 1;

	return 0;
}

int spi_fdx_rw(SPI_CONFIG *config, unsigned long wphys, unsigned long rphys,
		unsigned int size, unsigned int flags)
{
	struct spi_transfer trans;
	int ret;

	if (spi_fdx_invalid_param(config, wphys, rphys, size))
		return -EINVAL;

	spi_transfer_init(&trans, (char *)rphys, size, (char *)wphys, size);

	config->dir = SPI_RW;

	ret = spi_sync(config, &trans);
	if (!ret)
		ret = size;

	return ret;
}
EXPORT_SYMBOL(spi_fdx_rw);

int spi_fdx_rw_async(SPI_CONFIG *config, unsigned long wphys,
		unsigned long rphys, unsigned int size)
{
	struct spi_transfer trans;

	if (spi_fdx_invalid_param(config, wphys, rphys, size))
		return -EINVAL;

	spi_transfer_init(&trans, (char *)rphys, size, (char *)wphys, size);

	config->dir = SPI_RW;

	return spi_async(config, &trans, 0, 0);
}
EXPORT_SYMBOL(spi_fdx_rw_async);

static int spi_device_init(struct spi_data *spi)
{
	int err;
	int ch = spi->config->dev;

	spi_lock(spi);

	spi_power_on(spi);

	writel(SIO_SWITCHEN_ENABLE, SIO_SWITCHEN(ch));
	writel(SIO_MODE_SWITCH_SPI, SIO_MODE_SWITCH(ch));
	writel(SIO_SWITCHEN_DISABLE, SIO_SWITCHEN(ch));

	err = spi_config(spi, spi->config);
	if (err) {
		spi_power_off(spi);
		goto out;
	}

	spi->ready = 1;
out:
	spi_unlock(spi);

	return err;
}

static int spi_device_cleanup(struct spi_data *spi)
{

	spi_lock(spi);

	spi->ready = 0;
	spi_power_off(spi);

	spi_unlock(spi);

	return 0;
}

static int spi_request_dma(struct spi_data *spi)
{
	int ret;
	struct spi_dma *rx = spi->dma_rx;
	struct spi_dma *tx = spi->dma_tx;

	ret = emxx_request_dma(rx->lch, rx->name, spi_interrupt_dma_rx, spi, &rx->regs);
	if (ret < 0) {
		EPR(spi, "request_dma rx: ret=%d\n", ret);
		return ret;
	}

	ret = emxx_request_dma(tx->lch, tx->name, spi_interrupt_dma_tx, spi, &tx->regs);
	if (ret < 0) {
		EPR(spi, "request_dma tx: ret=%d\n", ret);
		emxx_free_dma(rx->lch);
		return ret;
	}

	return 0;
}

static void spi_free_dma(struct spi_data *spi)
{
	emxx_free_dma(spi->dma_tx->lch);
	emxx_free_dma(spi->dma_rx->lch);
}

static int __spi_probe(struct spi_data *spi)
{
	int ret;

	ret = spi_request_dma(spi);
	if (ret < 0) {
		EPR(spi, "request dma failed: %d\n", ret);
		return ret;
	}

	ret = request_irq(spi->int_spi, spi_interrupt, IRQF_DISABLED, spi->name, spi);
	if (ret < 0) {
		EPR(spi, "request irq failed: %d\n", ret);
		spi_free_dma(spi);
		return ret;
	}

	ret = spi_device_init(spi);
	if (ret < 0) {
		EPR(spi, "invalid configure %d\n", ret);
		free_irq(spi->int_spi, spi);
		spi_free_dma(spi);
		return ret;
	}

	return 0;
}

static inline struct spi_data *spi_get_data(struct platform_device *dev)
{
	return (struct spi_data *)dev->dev.platform_data;
}

static int spi_probe(struct platform_device *dev)
{
	struct spi_data *spi = spi_get_data(dev);
	int err = 0;
	int i;

	for (i = 0; i < SPI_DEV_NUM; i++) {
		err = __spi_probe(&spi[i]);
		if (err)
			return err;
	}
	spi_private = spi;

	return err;
}

static int __spi_remove(struct spi_data *spi)
{
	spi_device_cleanup(spi);
	free_irq(spi->int_spi, (void *)spi);
	spi_free_dma(spi);

	return 0;
}

static int spi_remove(struct platform_device *dev)
{
	struct spi_data *spi = spi_get_data(dev);
	int err = 0;
	int i;

	spi_private = 0;
	for (i = 0; i < SPI_DEV_NUM; i++) {
		err = __spi_remove(&spi[i]);
		if (err)
			return err;
	}

	return err;
}

static int __spi_suspend(struct spi_data *spi, pm_message_t state)
{
	int tmo;

	switch (state.event) {
	case PM_EVENT_SUSPEND:
		tmo = wait_event_timeout(spi->wq, !spi->busy, SPI_TMO_JIFFIES);
		if (!tmo)
			EPR(spi, "suspend timedout: %d sec\n", SPI_TMO_SEC);
		break;
	default:
		break;
	}
#ifdef CONFIG_SNSC_SSBOOT
	spi_device_cleanup(spi);
#endif
	spi_free_dma(spi);

	return 0;
}

static int spi_suspend(struct platform_device *dev, pm_message_t state)
{
	struct spi_data *spi = spi_get_data(dev);
	int err = 0;
	int i;

	for (i = 0; i < SPI_DEV_NUM; i++) {
		err = __spi_suspend(&spi[i], state);
		if (err)
			return err;
	}

	return err;
}

static int __spi_resume(struct spi_data *spi)
{
	int ret;

	ret = spi_request_dma(spi);

#ifdef CONFIG_SNSC_SSBOOT
	ret = spi_device_init(spi);
	if (ret < 0)
		EPR(spi, "spi_probe() failed %d\n", ret);
#endif
	return 0;
}

static int spi_resume(struct platform_device *dev)
{
	struct spi_data *spi = spi_get_data(dev);
	int err = 0;
	int i;

	for (i = 0; i < SPI_DEV_NUM; i++) {
		err = __spi_resume(&spi[i]);
		if (err)
			return err;
	}

	return err;
}

static struct platform_driver spi_drv = {
	.driver = {
		.name = SPI_NAME,
		.owner = THIS_MODULE,
	},
	.probe = spi_probe,
	.remove = spi_remove,
	.suspend = spi_suspend,
	.resume = spi_resume,
};

static int __init spi_init(void)
{
	return platform_driver_register(&spi_drv);
}

static void __exit spi_exit(void)
{
	platform_driver_unregister(&spi_drv);
}

module_init(spi_init);
module_exit(spi_exit);
MODULE_LICENSE("GPL");
