/*
 *  arch/arm/mach-emxx/thermal_sensor/thermal_sensor.c
 *
 *  Copyright 2011 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 */

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <memory.h>
#include <sys/ioctl.h>
#include <stdbool.h>
#include <sys/syscall.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <pthread.h>
#include <unistd.h>
#include "../include/mach/thermal_sensor.h"

#define I2C_OK			(0)
#define I2C_NG			(-1)
#define I2C_READ_MAX	(32)
#define I2C_WRITE_MAX	(32)

#define TEMP_SLAVE_ADDRESS		(0x48)
#define TEMP_SUB_ADDRESS		(0x00)

#define I2C_RETRIES		(0x0701)
#define I2C_TIMEOUT		(0x0702)
#define I2C_SLAVE			(0x0703)
#define I2C_SLAVE_FORCE	(0x0706)
#define I2C_TENBIT			(0x0704)
#define I2C_FUNCS			(0x0705)
#define I2C_RDWR			(0x0707)
#define I2C_PEC				(0x0708)
#define I2C_SMBUS			(0x0720)
#define I2C_READ			(0x0709)
#define I2C_WRITE			(0x0710)
#define I2C_SET_POS		(0x0711)

#define PHYADR_IOCWRITE    _IO('k', 0)
#define PHYADR_IOCREAD     _IO('k', 1)
#define PHYADR_IOCDUMP     _IO('k', 2)
#define PHYADR_IOCMALLOC   _IO('k', 3)
#define PHYADR_IOCFREE     _IO('k', 4)

#define TEMP_SIGN_POSITION (0x8000)
#define TEMP_D14_POSITION  (1280)
#define TEMP_D13_POSITION  (640)
#define TEMP_D12_POSITION  (320)
#define TEMP_D11_POSITION  (160)
#define TEMP_D10_POSITION  (80)
#define TEMP_D09_POSITION  (40)
#define TEMP_D08_POSITION  (20)
#define TEMP_D07_POSITION  (10)
#define TEMP_D06_POSITION  (5)
#define TEMP_D05_POSITION  (3)
#define TEMP_D04_POSITION  (1)
#define TEMP_D03_POSITION  (1)

#define TEMP_POSITION_MAX  (13)

#define TEMP_SIGN_PLUS  (1)
#define TEMP_SIGN_MINUS (0)

unsigned short temp_position[TEMP_POSITION_MAX] = {
	TEMP_SIGN_POSITION,
	TEMP_D14_POSITION,
	TEMP_D13_POSITION,
	TEMP_D12_POSITION,
	TEMP_D11_POSITION,
	TEMP_D10_POSITION,
	TEMP_D09_POSITION,
	TEMP_D08_POSITION,
	TEMP_D07_POSITION,
	TEMP_D06_POSITION,
	TEMP_D05_POSITION,
	TEMP_D04_POSITION
};

#define WCHTEMP_THERMAL_REG_DATA            (0x00)                          // $B%G!<%?%l%8%9%?(B
#define WCHTEMP_THERMAL_REG_CONFIG          (0x01)                          // $B%3%s%U%#%0%l!<%7%g%s%l%8%9%?(B
#define WCHTEMP_THERMAL_REG_UP_LIMIT        (0x02)                          // $B29EY!!(BUpper $B%j%_%C%H%l%8%9%?(B
#define WCHTEMP_THERMAL_REG_LOW_LIMIT       (0x03)                          // $B29EY!!(BLower $B%j%_%C%H%l%8%9%?(B
#define WCHTEMP_THERMAL_REG_CONTROL         (0x04)                          // $B%3%s%H%m!<%k%l%8%9%?(B
#define WCHTEMP_THERMAL_REG_STATUS          (0x04)                          // $B%9%F!<%?%9%l%8%9%?(B
#define WCHTEMP_THERMAL_REG_IDENTIFICATION  (0x07)                          // IDENTIFICATION $B%l%8%9%?(B

#define WCHTEMP_THERMAL_REG_DATA_SIZE       (2)                             // $B%G!<%?%l%8%9%?$N%"%/%;%9%5%$%:(B
#define WCHTEMP_THERMAL_REG_CONFIG_SIZE     (1)                             // $B%3%s%U%#%0%l!<%7%g%s%l%8%9%?$N%"%/%;%9%5%$%:(B
#define WCHTEMP_THERMAL_REG_UP_LIMIT_SIZE   (2)                             // $B29EY!!(BUpper $B%j%_%C%H%l%8%9%?$N%"%/%;%9%5%$%:(B
#define WCHTEMP_THERMAL_REG_LOW_LIMIT_SIZE  (2)                             // $B29EY!!(BLower $B%j%_%C%H%l%8%9%?$N%"%/%;%9%5%$%:(B
#define WCHTEMP_THERMAL_REG_CONTROL_SIZE    (1)                             // $B%3%s%H%m!<%k%l%8%9%?$N%"%/%;%9%5%$%:(B

#define WCHTEMP_THERMAL_DATA_OFFSET         (0x80)                          // $B29EY%G!<%?%*%U%;%C%H(B
#define WCHTEMP_THERMAL_DATA_MAX            (150)                           // $B29EY%G!<%?$N:GBgCM(B
#define WCHTEMP_THERMAL_DATA_MIN            (-40)                           // $B29EY%G!<%?$N:G>.CM(B
#define WCHTEMP_THERMAL_DATA_LIMIT_OVER     (151)                           // $B29EY%G!<%?$N:GBgCMHO0O30(B
#define WCHTEMP_THERMAL_DATA_LIMIT_UNDER    (-41)                           // $B29EY%G!<%?$N:G>.CMHO0O30(B
#define WCHTEMP_THERMAL_DATA_INIT           (1000)                          // $B29EY%G!<%?$N=i4|CM!JHO0O30$NCM!K(B

#define WCHTEMP_THERMAL_REG_CONFIG_INIT     (0x4000)                        // $B%3%s%U%#%0%l!<%7%g%s%l%8%9%?=i4|@_Dj(B
#define WCHTEMP_THERMAL_REG_UP_LIMIT_INIT   (0x7fe0)                        // $B29EY!!(BUpper $B%j%_%C%H%l%8%9%?=i4|@_Dj(B
#define WCHTEMP_THERMAL_REG_LOW_LIMIT_INIT  (0x8000)                        // $B29EY!!(BLower $B%j%_%C%H%l%8%9%?=i4|@_Dj(B
#define WCHTEMP_THERMAL_REG_CONTROL_INIT    (0x0000)                        // $B%3%s%H%m!<%k%l%8%9%?=i4|@_Dj(B
                                                                            // 0.25$B!k(BC/LSB, 11-bit word
typedef struct t_wchtempInitTable
{
	unsigned char       ucSubAdrs;
	unsigned char       ucSize;
	unsigned short      usData;
} t_wchtempInitTable;

struct stPhyadr_drv_cmd{
	unsigned long address;
	unsigned long value;
	unsigned long mask;
	unsigned long length;
	unsigned char * buf;
};
#define PHYADR_CMD_INFO struct stPhyadr_drv_cmd

PHYADR_CMD_INFO phyadr_info;

int  i2c_fd;

void if_temperature_init(void);
void if_temperature_end(void);
int if_temperature_get( signed int *temp);
int temp_read(unsigned char *buf, unsigned int size);
int temp_write(unsigned char sub, unsigned short data, int size);
int temp_calc(unsigned char *data);
void i2c_speed_set(void);


void if_temperature_init( void)
{
    static const t_wchtempInitTable     initTable[] =
    {
        {WCHTEMP_THERMAL_REG_CONFIG,    WCHTEMP_THERMAL_REG_CONFIG_SIZE,    WCHTEMP_THERMAL_REG_CONFIG_INIT},
        {WCHTEMP_THERMAL_REG_UP_LIMIT,  WCHTEMP_THERMAL_REG_UP_LIMIT_SIZE,  WCHTEMP_THERMAL_REG_UP_LIMIT_INIT},
        {WCHTEMP_THERMAL_REG_LOW_LIMIT, WCHTEMP_THERMAL_REG_LOW_LIMIT_SIZE, WCHTEMP_THERMAL_REG_LOW_LIMIT_INIT},
        {WCHTEMP_THERMAL_REG_CONTROL,   WCHTEMP_THERMAL_REG_CONTROL_SIZE,   WCHTEMP_THERMAL_REG_CONTROL_INIT}
    };
    unsigned int    uiCount = (sizeof(initTable) / sizeof(t_wchtempInitTable));
    unsigned int    i;

	i2c_speed_set();
	i2c_fd = open("/dev/i2c", O_RDWR);

    // $B29EY%;%s%5!<=i4|@_Dj(B
    for (i = 0; i < uiCount; i++)
    {
		temp_write(initTable[i].ucSubAdrs, initTable[i].usData, initTable[i].ucSize);
    }
}

void if_temperature_end(void)
{
	close(i2c_fd);
}

int if_temperature_get( signed int *temp)
{
	unsigned int  count = 0;
	unsigned int  ret = I2C_OK;
	unsigned char read_buf[I2C_READ_MAX];

	memset(read_buf,  0x00, sizeof(read_buf));

	read_buf[0] = TEMP_SUB_ADDRESS;
	count = 2;

	/*** I2C Read ***/
	ret = temp_read(read_buf, count);
	temp[0] = temp_calc(&read_buf[0]);

	return ret;
}

int temp_calc(unsigned char *data)
{
	unsigned char  i;
	unsigned char  sign = TEMP_SIGN_PLUS;
	unsigned short temp = 0;
	unsigned short calc = 0x8000;
	         int   chg_temp = 0;

	temp = ((data[0] << 8) & 0xFF00);
	temp |= ((unsigned short)data[1] & 0x00F8);

	if((temp & calc) != 0) {
		sign = TEMP_SIGN_MINUS;
	}
	calc = calc >> 1;

	for(i = 1; i < TEMP_POSITION_MAX; i++) {
		if((temp & calc) != 0 ) {
			chg_temp += temp_position[i];
		}
		calc = calc >> 1;
	}

	if(sign == TEMP_SIGN_MINUS) {
		chg_temp -= 2560;
	}

	return chg_temp;
}

int temp_read(unsigned char *buf, unsigned int size)
{
	int ret = I2C_NG;

	ioctl(i2c_fd, I2C_TENBIT, (unsigned long)0x00);
	ioctl(i2c_fd, I2C_SLAVE_FORCE, (unsigned long)TEMP_SLAVE_ADDRESS);

	/*** Write Data Set ***/
	ret = write(i2c_fd, buf, 1);

	/*** Read Data Set ***/
	ret = read(i2c_fd, buf, size);

	if(ret < I2C_OK) {
		printf("\n\n*** Read Data Error ***\n\n");
		ret = I2C_NG;
	}
	return ret;
}

int temp_write(unsigned char sub, unsigned short data, int size)
{
	int	 ret = I2C_NG;
	unsigned char write_buf[I2C_WRITE_MAX];

	memset(write_buf,  0x00, sizeof(write_buf));

	ioctl(i2c_fd, I2C_TENBIT, (unsigned long)0x00);
	ioctl(i2c_fd, I2C_SLAVE_FORCE, (unsigned long)TEMP_SLAVE_ADDRESS);

	write_buf[0] = sub;
	write_buf[1] = (unsigned char)((data & 0xFF00) >> 8);
	write_buf[2] = (unsigned char)(data & 0x00FF);


	/*** Write Data Set ***/
	ret = write(i2c_fd, write_buf, (size+1));

	if(ret < I2C_OK) {
		printf("\n\n*** Write Data Error ***\n\n");
		ret = I2C_NG;
	}
	return ret;
}

void i2c_speed_set(void)
{
	int phy_fd = 0;
	unsigned char *phyadr_buf;

	phyadr_info.address = 0xe0110624;
	phyadr_info.value   = 0x002f002f;
	phyadr_info.mask    = 0x00ff00ff;
	phyadr_info.buf     = phyadr_buf;

	phy_fd = open("/dev/phyadr_drv", O_RDWR);
	ioctl(phy_fd, PHYADR_IOCWRITE, &phyadr_info);

	usleep(1000);

	close(phy_fd);
}
