/*
 *  File Name       : arch/arm/mach-emxx/time.c
 *  Function        : time
 *  Release Version : Ver 1.00
 *  Release Date    : 2010/02/05
 *
 *  Copyright (C) NEC Electronics Corporation 2010
 *
 *
 *  This program is free software;you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by Free
 *  Softwere Foundation; either version 2 of License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warrnty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 *  more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; If not, write to the Free Software Foundation, Inc., 59
 *  Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/irq.h>
#ifdef CONFIG_EMXX_USE_TIMER_TI1_FOR_SCHED_CLOCK
#include <linux/cnt32_to_63.h>
#endif /* CONFIG_EMXX_USE_TIMER_TI1_FOR_SCHED_CLOCK */
#include <linux/clocksource.h>
#include <linux/clockchips.h>
#include <linux/version.h>
#include <linux/io.h>

#include <asm/div64.h>
#include <asm/mach/time.h>

#include <mach/smu.h>
#include <mach/pmu.h>
#include <mach/timer.h>

#ifdef CONFIG_SNSC_SSBOOT
#include <linux/ssboot.h>
#endif

#include "timer.h"

static DEFINE_SPINLOCK(timer_spinlock);
static DEFINE_SPINLOCK(wdt_spinlock);

static volatile struct sti_reg_t *sti_timer =
	(volatile struct sti_reg_t *)(IO_ADDRESS(EMXX_STI_BASE));

static struct tm_param_t tm_param[] = {
	/* TIMER_TGn */
	{.usecs = TIMER_DEFAULT_TIME,},
	{.usecs = TIMER_DEFAULT_TIME,},
	{.usecs = TIMER_DEFAULT_TIME,},
	{.usecs = TIMER_DEFAULT_TIME,},
	{.usecs = TIMER_DEFAULT_TIME,},
	{.usecs = TIMER_DEFAULT_TIME,},
	/* TIMER_TIn */
	{.usecs = TIMER_DEFAULT_TIME,},
	{.usecs = TIMER_DEFAULT_TIME,},
	{.usecs = TIMER_DEFAULT_TIME,},
	{.usecs = TIMER_DEFAULT_TIME,},
	/* TIMER_TWn */
	{.usecs = TW_DEFAULT_TIME,},
	{.usecs = TIMER_DEFAULT_TIME,},
	{.usecs = TIMER_DEFAULT_TIME,},
	{.usecs = TIMER_DEFAULT_TIME,},
};

static const struct tm_reg_t tm_reg[] = {
	/* TIMER_TGn */
	{
	 .irq    = INT_TG0,
	 .reg = (volatile struct timer_reg_t *)IO_ADDRESS(EMXX_TG0_BASE),
	 .clkdev = EMXX_CLK_TG0,
	 .rstdev = EMXX_RST_TG0,
	},
	{
	 .irq    = INT_TG1,
	 .reg = (volatile struct timer_reg_t *)IO_ADDRESS(EMXX_TG1_BASE),
	 .clkdev = EMXX_CLK_TG1,
	 .rstdev = EMXX_RST_TG1,
	},
	{
	 .irq    = INT_TG2,
	 .reg = (volatile struct timer_reg_t *)IO_ADDRESS(EMXX_TG2_BASE),
	 .clkdev = EMXX_CLK_TG2,
	 .rstdev = EMXX_RST_TG2,
	},
	{
	 .irq    = INT_TG3,
	 .reg = (volatile struct timer_reg_t *)IO_ADDRESS(EMXX_TG3_BASE),
	 .clkdev = EMXX_CLK_TG3,
	 .rstdev = EMXX_RST_TG3,
	},
	{
	 .irq    = INT_TG4,
	 .reg = (volatile struct timer_reg_t *)IO_ADDRESS(EMXX_TG4_BASE),
	 .clkdev = EMXX_CLK_TG4,
	 .rstdev = EMXX_RST_TG4,
	},
	{
	 .irq    = INT_TG5,
	 .reg = (volatile struct timer_reg_t *)IO_ADDRESS(EMXX_TG5_BASE),
	 .clkdev = EMXX_CLK_TG5,
	 .rstdev = EMXX_RST_TG5,
	},
	/* TIMER_TIn */
	{
	 .irq    = INT_TIMER0,
	 .reg = (volatile struct timer_reg_t *)IO_ADDRESS(EMXX_TIMER0_BASE),
	 .clkdev = EMXX_CLK_TI0,
	 .rstdev = EMXX_RST_TI0,
	},
	{
	 .irq    = INT_TIMER1,
	 .reg = (volatile struct timer_reg_t *)IO_ADDRESS(EMXX_TIMER1_BASE),
	 .clkdev = EMXX_CLK_TI1,
	 .rstdev = EMXX_RST_TI1,
	},
	{
	 .irq    = INT_TIMER2,
	 .reg = (volatile struct timer_reg_t *)IO_ADDRESS(EMXX_TIMER2_BASE),
	 .clkdev = EMXX_CLK_TI2,
	 .rstdev = EMXX_RST_TI2,
	},
	{
	 .irq    = INT_TIMER3,
	 .reg = (volatile struct timer_reg_t *)IO_ADDRESS(EMXX_TIMER3_BASE),
	 .clkdev = EMXX_CLK_TI3,
	 .rstdev = EMXX_RST_TI3,
	},
	/* TIMER_TWn */
	{
	 .irq     = INT_WDT0,
	 .reg = (volatile struct timer_reg_t *)IO_ADDRESS(EMXX_WDT0_BASE),
	 .clkdev  = EMXX_CLK_TW0,
	 .rstdev  = EMXX_RST_TW0,
	 .tin_sel = SMU_TWI0TIN_SEL,
	},
	{
	 .irq     = INT_WDT1,
	 .reg = (volatile struct timer_reg_t *)IO_ADDRESS(EMXX_WDT1_BASE),
	 .clkdev  = EMXX_CLK_TW1,
	 .rstdev  = EMXX_RST_TW1,
	 .tin_sel = SMU_TWI1TIN_SEL,
	},
	{
	 .irq     = INT_WDT2,
	 .reg = (volatile struct timer_reg_t *)IO_ADDRESS(EMXX_WDT2_BASE),
	 .clkdev  = EMXX_CLK_TW2,
	 .rstdev  = EMXX_RST_TW2,
	 .tin_sel = SMU_TWI2TIN_SEL,
	},
	{
	 .irq     = INT_WDT3,
	 .reg = (volatile struct timer_reg_t *)IO_ADDRESS(EMXX_WDT3_BASE),
	 .clkdev  = EMXX_CLK_TW3,
	 .rstdev  = EMXX_RST_TW3,
	 .tin_sel = SMU_TWI3TIN_SEL,
	},
	{
	 .irq     = INT_WDT4,
	 .reg = (volatile struct timer_reg_t *)IO_ADDRESS(EMXX_WDT4_BASE),
	 .clkdev  = EMXX_CLK_TW4,
	 .rstdev  = EMXX_RST_TW4,
	 .tin_sel = SMU_TW4TIN_SEL,
	},
};

/* inline function */
inline int check_validity_channel(unsigned int timer)
{
	if ((TIMER_MIN <= timer) && (timer <= TIMER_MAX))
		return 0;
	else
		return -ENODEV;
}

inline unsigned int get_count_value(unsigned timer, unsigned int usecs)
{
	uint64_t counts;

	if (usecs == TIMER_MAX_COUNT)
		return TIMER_MAX_COUNT;

	if (!(readl(SMU_TGNTIN_SEL) & (0x3 << (timer * 4))))
		counts = (uint64_t)usecs * TIMER_CLOCK_TICK_RATE_PLL3;
	else
		counts = (uint64_t)usecs * TIMER_CLOCK_TICK_RATE_32K;

	do_div(counts, 1000000);
	counts--;

	return (unsigned int)counts;
}
static inline unsigned int
get_tw_count_value(unsigned timer, unsigned int usecs)
{
	uint64_t counts;

	if (timer != TIMER_TW4) {
		if (!(readl(tm_reg[timer].tin_sel) & MASK_TWNTIN_SEL))
			counts = (uint64_t)usecs * TIMER_CLOCK_TICK_RATE_PLL3;
		else
			counts = (uint64_t)usecs * TIMER_CLOCK_TICK_RATE_32K;
	} else {
		if (!(readl(tm_reg[timer].tin_sel) & 0x03))
			counts = (uint64_t)usecs * TIMER_CLOCK_TICK_RATE_PLL3;
		else
			counts = (uint64_t)usecs * TIMER_CLOCK_TICK_RATE_32K;
	}

	do_div(counts, 1000000);
	counts--;

	return (unsigned int)counts;
}


inline unsigned int get_usec_value(unsigned timer, unsigned int counts)
{
	uint64_t usecs;

	usecs = (uint64_t)counts * 1000000;

	if (!(readl(SMU_TGNTIN_SEL) & (0x3 << (timer * 4))))
		do_div(usecs, TIMER_CLOCK_TICK_RATE_PLL3);
	else
		do_div(usecs, TIMER_CLOCK_TICK_RATE_32K);

	return (unsigned int)usecs;
}

static void timer_set_mode(enum clock_event_mode mode,
			   struct clock_event_device *clk)
{
	volatile struct timer_reg_t *reg = tm_reg[TIMER_SYSTEM].reg;

	switch (mode) {
	case CLOCK_EVT_MODE_PERIODIC:
		reg->tm_set = TIMER_INTERVAL_PLL3;
		reg->tm_op = TO_EN | TSTART | TM_EN;
		break;
	case CLOCK_EVT_MODE_ONESHOT:
		reg->tm_op = TO_EN;
		break;
	case CLOCK_EVT_MODE_UNUSED:
	case CLOCK_EVT_MODE_SHUTDOWN:
	default:
		reg->tm_op = TSTOP;
		break;
	}
}

static int timer_set_next_event(unsigned long evt,
		struct clock_event_device *unused)
{
	volatile struct timer_reg_t *reg = tm_reg[TIMER_SYSTEM].reg;

	if (reg->tm_op & 0x03)
		reg->tm_op = TSTOP;

	if (evt < COUNTER_MAXVAL / 256)
		evt = COUNTER_MAXVAL / 256;

	reg->tm_set = evt;
	reg->tm_op = TO_EN | TSTART | TM_EN;

	return 0;
}

static struct clock_event_device timer0_clockevent = {
	.name		= "timer0",
	.shift		= 32,
	.features	= CLOCK_EVT_FEAT_PERIODIC | CLOCK_EVT_FEAT_ONESHOT,
	.set_mode	= timer_set_mode,
	.set_next_event	= timer_set_next_event,
	.rating		= 200,
	.cpumask	= cpu_all_mask,
};

static irqreturn_t emxx_system_timer_interrupt(int irq, void *dev_id)
{
	struct clock_event_device *evt = &timer0_clockevent;
	evt->event_handler(evt);

	return IRQ_HANDLED;
}

static void __init emxx_clockevents_init(unsigned int timer_irq)
{
	unsigned int cpu = smp_processor_id();

	timer0_clockevent.cpumask = &cpumask_of_cpu(cpu);
	timer0_clockevent.irq = timer_irq;
	timer0_clockevent.mult = div_sc(TIMER_CLOCK_TICK_RATE_PLL3,
		NSEC_PER_SEC, timer0_clockevent.shift);
	timer0_clockevent.max_delta_ns =
		clockevent_delta2ns(0xffffffff, &timer0_clockevent);
	timer0_clockevent.min_delta_ns =
		clockevent_delta2ns(0xf, &timer0_clockevent);

	clockevents_register_device(&timer0_clockevent);
}

#ifdef CONFIG_SNSC_SSBOOT
static void emxx_clocksource_resume(void);
#endif

#ifdef CONFIG_EMXX_USE_TG_TIMER_FOR_CLOCKSOURCE
static cycle_t emxx_tg_clksrc_get_cycles(void)
{
	volatile struct timer_reg_t *reg;
	cycle_t ticks;

	reg = tm_reg[TIMER_TG0 + TIMER_CLOCKSOURCE].reg;
	ticks = reg->tm_rcr;

	return ticks;
}

static struct clocksource clocksource_tg_clksrc = {
	.name 		= "tg_clksrc",
	.rating		= 256,
	.read		= emxx_tg_clksrc_get_cycles,
	.mask		= CLOCKSOURCE_MASK(32),
	.shift 		= 10,
	.flags		= CLOCK_SOURCE_IS_CONTINUOUS,
#ifdef CONFIG_SNSC_SSBOOT
	.resume = emxx_clocksource_resume,
#endif
};
#endif /* CONFIG_EMXX_USE_TG_TIMER_FOR_CLOCKSOURCE */

#ifdef CONFIG_EMXX_USE_ST_TIMER_FOR_CLOCKSOURCE
static cycle_t emxx_get_cycles(void)
{
	cycle_t ticks;

	ticks = (cycle_t)(sti_timer->count_h & 0xffff) << 32ULL;
	ticks |= (cycle_t)sti_timer->count_l;
	return ticks;
}

static struct clocksource clocksource_emxx = {
	.name	= "sti",
	.rating	= 250,
	.read	= emxx_get_cycles,
	.mask	= CLOCKSOURCE_MASK(48),
	.shift	= 10,
	.flags	= CLOCK_SOURCE_IS_CONTINUOUS,
#ifdef CONFIG_SNSC_SSBOOT
	.resume = emxx_clocksource_resume,
#endif
};
#endif /* CONFIG_EMXX_USE_ST_TIMER_FOR_CLOCKSOURCE */

#ifdef CONFIG_EMXX_USE_TIMER_TI1_FOR_SCHED_CLOCK
static int emxx_sched_timer_init = 0;
static struct clocksource clocksource_ti;
#endif /* CONFIG_EMXX_USE_TIMER_TI1_FOR_SCHED_CLOCK */

unsigned long long notrace sched_clock(void);

static void __init emxx_clocksource_init(void)
{
	/* Setup system timer */
	writel(0, SMU_STI_CLKSEL);	/* 32768 */
	emxx_open_clockgate(EMXX_CLK_STI_P | EMXX_CLK_STI_S);
	emxx_unreset_device(EMXX_RST_STI);

	sti_timer->set_h    = 0x80000000;
	sti_timer->set_l    = 0x00000000;
	sti_timer->intenclr = 0x3;
	sti_timer->intffclr = 0x3;

#ifdef CONFIG_EMXX_USE_ST_TIMER_FOR_CLOCKSOURCE
	clocksource_emxx.mult =
		clocksource_hz2mult(32768, clocksource_emxx.shift);
	clocksource_register(&clocksource_emxx);
#endif /* CONFIG_EMXX_USE_ST_TIMER_FOR_CLOCKSOURCE */

#ifdef CONFIG_EMXX_USE_TIMER_TI1_FOR_SCHED_CLOCK
	clocksource_register(&clocksource_ti);
#endif /* CONFIG_EMXX_USE_TIMER_TI1_FOR_SCHED_CLOCK */

#ifdef CONFIG_EMXX_USE_TG_TIMER_FOR_CLOCKSOURCE
	/*
	 * TG[0-5] timer is used for clocksource.
	 * TG[0-5] can be driven by C32_P0CLK or DIV_TIM_CLK (PLL3).
	 * Configured in TIMCLKDIV register for using DIV_TIM_CLK (PLL3) source at 5.7344Mhz.
	 * TG[0-5] timer is initalized and enabled .
	 */
	emxx_timer_start(TIMER_CLOCKSOURCE);
	clocksource_tg_clksrc.mult =
		clocksource_hz2mult(TIMER_CLOCK_TICK_RATE_PLL3,
				    clocksource_tg_clksrc.shift);
	clocksource_register(&clocksource_tg_clksrc);
#endif /* CONFIG_EMXX_USE_TG_TIMER_FOR_CLOCKSOURCE */

#ifdef CONFIG_EMXX_USE_TIMER_TI1_FOR_SCHED_CLOCK
	emxx_sched_timer_init = 1;
#endif
}

static struct irqaction emxx_system_timer_irq = {
	.name    = "system_timer",
	.flags   = IRQF_DISABLED | IRQF_TIMER,
	.handler = emxx_system_timer_interrupt
};

#ifdef CONFIG_EMXX_USE_TIMER_TI1_FOR_SCHED_CLOCK
/*
 * Set sched_clock(For showing boottime)
 */

/*
 * This is the sched_clock implementation for EM/EV platform.
 *
 * EM/EV platform uses the onboard TIMER_TG0 which provides an 32 bit value.
 * The counter increments at 5.7344MHZ frequency so its
 * resolution of 174.3ns, and the 32-bit counter will overflow in 748.61 Sec.
 */
#define CYC2NS_SCALE_FACTOR 		10

/* make sure COUNTER_CLOCK_TICK_RATE % TICK_RATE_SCALE_FACTOR == 0 */
#define TICK_RATE_SCALE_FACTOR 		1000
#define COUNTER_CLOCK_TICK_RATE		(TIMER_CLOCK_TICK_RATE_PLL3)

/* cycles to nsec conversions taken from arch/arm/mach-omap1/time.c,
 * convert from cycles(64bits) => nanoseconds (64bits)
 */
#define CYC2NS_SCALE (((NSEC_PER_SEC / TICK_RATE_SCALE_FACTOR) << CYC2NS_SCALE_FACTOR) \
                      / (COUNTER_CLOCK_TICK_RATE / TICK_RATE_SCALE_FACTOR))

static inline unsigned long long notrace cycles_2_ns(unsigned long long cyc)
{
#if (CYC2NS_SCALE & 0x1)
	return (cyc * CYC2NS_SCALE << 1) >> (CYC2NS_SCALE_FACTOR + 1);
#else
	return (cyc * CYC2NS_SCALE) >> CYC2NS_SCALE_FACTOR;
#endif
}

static inline unsigned long read_freerun(void)
{
	volatile struct timer_reg_t *p = tm_reg[TIMER_TI1].reg;
	return p->tm_rcr;
}

unsigned long long notrace emxx_timer_get_nsec(void)
{
	cycle_t cyc;

	if(!emxx_sched_timer_init)
		return 0;
	cyc = cnt32_to_63(read_freerun());
	return cycles_2_ns(cyc);
}

static cycle_t emxx_ti_get_cycles(void)
{
	return read_freerun();
}

static struct clocksource clocksource_ti = {
	.name 		= "TI1-timer",
	.rating		= 256,
	.read		= emxx_ti_get_cycles,
	.mask		= CLOCKSOURCE_MASK(32),
	.flags		= CLOCK_SOURCE_IS_CONTINUOUS,
#if (CYC2NS_SCALE & 0x1)
	.shift 		= CYC2NS_SCALE_FACTOR + 1,
	.mult		= CYC2NS_SCALE << 1,
#else
	.shift 		= CYC2NS_SCALE_FACTOR,
	.mult		= CYC2NS_SCALE,
#endif
};

#ifdef CONFIG_EMXX_SSBOOT_ADJUST_SCHED_CLOCK
static unsigned long long sched_clock_saved;
static unsigned long long sched_clock_adjust = 0;

void emxx_sched_clock_suspend(void)
{
	/* save current sched_clock */
	sched_clock_saved = sched_clock();
}

void emxx_sched_clock_resume(void)
{
	/* restoration is required only at resume */
	if (!ssboot_is_resumed())
		return;

	/* calculate adjust value */
	sched_clock_adjust = sched_clock_saved - emxx_timer_get_nsec();
}
#endif /*CONFIG_EMXX_SSBOOT_ADJUST_SCHED_CLOCK */

unsigned long long notrace sched_clock(void)
{
#ifdef CONFIG_EMXX_SSBOOT_ADJUST_SCHED_CLOCK
	return emxx_timer_get_nsec() + sched_clock_adjust;
#else
	return emxx_timer_get_nsec();
#endif
}

#endif /* CONFIG_EMXX_USE_TIMER_TI1_FOR_SCHED_CLOCK */

/*
 * Set up timer interrupt, and return the current time in seconds.
 */
static void __init emxx_init_timer(void)
{
	timer_set_clock(TIMER_INIT);

#ifdef CONFIG_LOCAL_TIMERS
	twd_base = __io_address(EMXX_INTA_TIM_BASE + 0x600);
	/* Set clock 32000Hz */
	writel(1, SMU_INTA_CLKSEL);
#endif

#ifdef CONFIG_GENERIC_CLOCKEVENTS_BROADCAST
	local_timer_setup();
#endif

	setup_irq(tm_reg[TIMER_SYSTEM].irq, &emxx_system_timer_irq);

	emxx_clocksource_init();
	emxx_clockevents_init(tm_reg[TIMER_SYSTEM].irq);
}

#ifdef CONFIG_SNSC_SSBOOT
static void emxx_clocksource_resume(void)
{
	if (!ssboot_is_resumed())
		return;

	timer_set_clock(TIMER_INIT);

#ifdef CONFIG_LOCAL_TIMERS
	twd_base = __io_address(EMXX_INTA_TIM_BASE + 0x600);
	/* Set clock 32000Hz */
	writel(1, SMU_INTA_CLKSEL);
#endif

	setup_irq(tm_reg[TIMER_SYSTEM].irq, &emxx_system_timer_irq);

	/* Setup system timer */
	writel(0, SMU_STI_CLKSEL);	/* 32768 */
	emxx_open_clockgate(EMXX_CLK_STI_P | EMXX_CLK_STI_S);
	emxx_unreset_device(EMXX_RST_STI);

	sti_timer->set_h    = 0x80000000;
	sti_timer->set_l    = 0x00000000;
	sti_timer->intenclr = 0x3;
	sti_timer->intffclr = 0x3;

#ifdef CONFIG_EMXX_USE_TG_TIMER_FOR_CLOCKSOURCE

	emxx_timer_start(TIMER_CLOCKSOURCE);

#endif /* CONFIG_EMXX_USE_TG_TIMER_FOR_CLOCKSOURCE */

	return;
}
#endif

struct sys_timer emxx_timer = {
	.init = emxx_init_timer,
};

static unsigned int tg_tm_op[] = { 0, 0, 0, 0, 0, 0 };

void timer_set_clock(unsigned int mode)
{
	int i;
	unsigned int count;
	unsigned int tm_delay = 0;
	volatile struct timer_reg_t *reg;

	/* delay value set */
	tm_delay = TIMER_DELAY(TIMER_CLOCK_TICK_RATE_32K);

	switch (mode) {
	case TIMER_SUSPEND:
		/* close clockgate */
		emxx_close_clockgate(tm_reg[TIMER_SYSTEM].clkdev);
		emxx_close_clockgate(tm_reg[TIMER_WDT].clkdev);
		emxx_close_clockgate(tm_reg[TIMER_DSP].clkdev);
		for (i = 0; i < TIMER_TG_MAX_NUM; i++) {
			if (tg_tm_op[i] == 1)
				emxx_close_clockgate(
					tm_reg[TIMER_TG0 + i].clkdev);
		}
		break;
	case TIMER_RESUME:
		emxx_open_clockgate(tm_reg[TIMER_SYSTEM].clkdev);
		emxx_open_clockgate(tm_reg[TIMER_WDT].clkdev);
		emxx_open_clockgate(tm_reg[TIMER_DSP].clkdev);
		for (i = 0; i < TIMER_TG_MAX_NUM; i++) {
			if (tg_tm_op[i] == 1)
				emxx_open_clockgate(
					tm_reg[TIMER_TG0 + i].clkdev);
		}
		break;
	case TIMER_INIT:
		emxx_open_clockgate(EMXX_CLK_TIM_P);

		emxx_open_clockgate(tm_reg[TIMER_SYSTEM].clkdev);
		emxx_open_clockgate(tm_reg[TIMER_DSP].clkdev);
		emxx_unreset_device(tm_reg[TIMER_SYSTEM].rstdev);
		emxx_unreset_device(tm_reg[TIMER_DSP].rstdev);

		for (i = 0; i < TIMER_TG_MAX_NUM; i++)
			emxx_unreset_device(tm_reg[TIMER_TG0 + i].rstdev);

		reg = tm_reg[TIMER_SYSTEM].reg;
		reg->tm_op = TSTOP;
		reg->tm_clr = TCR_CLR;

		reg = tm_reg[TIMER_WDT].reg;
		reg->tm_op = TSTOP;
		reg->tm_clr = TCR_CLR;

		reg = tm_reg[TIMER_DSP].reg;
		reg->tm_op = TSTOP;
		reg->tm_clr = TCR_CLR;
		reg->tm_set = TIMER_INTERVAL_DSP;

		/* select TIN clock */
		writel(TINTIN_SEL_PLL3 | TWNTIN_SEL_32K, SMU_TWI0TIN_SEL);
		writel(TINTIN_SEL_PLL3 | TWNTIN_SEL_PLL3, SMU_TWI1TIN_SEL);
		writel(TINTIN_SEL_PLL3 | TWNTIN_SEL_PLL3, SMU_TWI2TIN_SEL);
		writel(TINTIN_SEL_32K | TWNTIN_SEL_PLL3, SMU_TWI3TIN_SEL);
		writel(TINTIN_SEL_32K, SMU_TW4TIN_SEL);
		writel(TGNTIN_SEL_PLL3, SMU_TGNTIN_SEL);

		/* DIVTIMTIN Register set */
		writel(SMU_PLLSEL_PLL3 | SMU_DIV(40), SMU_TIMCLKDIV);

		for (i = 0; i < TIMER_TG_MAX_NUM; i++) {
			count = get_count_value(i,
				tm_param[TIMER_TG0 + i].usecs);
			reg  = tm_reg[TIMER_TG0 + i].reg;
			reg->tm_set = count;
		}
		break;
	default:
		printk(KERN_INFO "%s(): set clock error mode = %d.\n",
			__func__, mode);
		break;
	}
}


int emxx_timer_start(unsigned int timer)
{
	int ret = 0;
	unsigned long flags;
	volatile struct timer_reg_t *reg;

	ret = check_validity_channel(timer);
	if (ret != 0)
		return ret;

	reg = tm_reg[TIMER_TG0 + timer].reg;

	spin_lock_irqsave(&timer_spinlock, flags);

	emxx_open_clockgate(tm_reg[TIMER_TG0 + timer].clkdev);
	tg_tm_op[timer] = 1;

	/* Timer start */
	reg->tm_op = TSTART | TM_EN | TO_EN;

	spin_unlock_irqrestore(&timer_spinlock, flags);

	return 0;
}
EXPORT_SYMBOL(emxx_timer_start);

int emxx_timer_stop(unsigned int timer)
{
	int ret = 0;
	unsigned int tm_delay = 0;
	unsigned long flags;
	volatile struct timer_reg_t *reg;

	ret = check_validity_channel(timer);
	if (ret != 0)
		return ret;

	/* delay value set */
	tm_delay = TIMER_DELAY(TIMER_CLOCK_TICK_RATE_PLL3);

	reg = tm_reg[TIMER_TG0 + timer].reg;

	spin_lock_irqsave(&timer_spinlock, flags);

	/* Timer stop */
	reg->tm_op = TSTOP;

	udelay(tm_delay);

	emxx_close_clockgate(tm_reg[TIMER_TG0 + timer].clkdev);
	tg_tm_op[timer] = 0;

	spin_unlock_irqrestore(&timer_spinlock, flags);

	return 0;
}
EXPORT_SYMBOL(emxx_timer_stop);

int emxx_timer_set_period(unsigned int timer, unsigned int usecs)
{
	int ret = 0, regval = 0;
	unsigned long	flags;
	unsigned int tm_delay = 0;
	volatile struct timer_reg_t *reg;

	ret = check_validity_channel(timer);
	if (ret != 0)
		return ret;

	if ((usecs < TIMER_MIN_USECS) || (TIMER_MAX_USECS < usecs))
		return -EINVAL;

	/* 4clk x2 wait value */
	tm_delay = TIMER_DELAY(TIMER_CLOCK_TICK_RATE_PLL3) * 2;
	reg = tm_reg[TIMER_TG0 + timer].reg;

	spin_lock_irqsave(&timer_spinlock, flags);

	if (reg->tm_op & TSTART) {
		spin_unlock_irqrestore(&timer_spinlock, flags);
		return -EBUSY;
	}

	emxx_open_clockgate(tm_reg[TIMER_TG0 + timer].clkdev);
	udelay(tm_delay);

	tm_param[TIMER_TG0 + timer].usecs = usecs;

	/* set counter */
	regval = get_count_value(timer, usecs);
	reg->tm_set = regval;

	emxx_close_clockgate(tm_reg[TIMER_TG0 + timer].clkdev);
	spin_unlock_irqrestore(&timer_spinlock, flags);

	return 0;
}
EXPORT_SYMBOL(emxx_timer_set_period);

int emxx_timer_get_period(unsigned int timer, unsigned int *usecs)
{
	int ret = 0;

	ret = check_validity_channel(timer);
	if (ret != 0)
		return ret;

	if (usecs == NULL)
		return -EINVAL;

	*usecs = tm_param[TIMER_TG0 + timer].usecs;

	return 0;
}
EXPORT_SYMBOL(emxx_timer_get_period);

int emxx_timer_get_usecs(unsigned int timer, unsigned int *usecs)
{
	unsigned int regval = 0;
	int ret = 0;
	volatile struct timer_reg_t *reg;

	ret = check_validity_channel(timer);
	if (ret != 0)
		return ret;

	if (usecs == NULL)
		return -EINVAL;

	reg = tm_reg[TIMER_TG0 + timer].reg;
	regval = reg->tm_rcr;
	*usecs = get_usec_value(timer, regval);

	return 0;
}
EXPORT_SYMBOL(emxx_timer_get_usecs);

int emxx_timer_register_cb(unsigned int timer, char *devname,
			irqreturn_t(*cb) (int, void *), void *dev_id)
{
	unsigned int irq = 0;
	int ret = 0;
	volatile struct timer_reg_t *reg;

	ret = check_validity_channel(timer);
	if (ret != 0)
		return ret;

	reg = tm_reg[TIMER_TG0 + timer].reg;

	if (reg->tm_op & TSTART)
		return -EBUSY;

	irq = tm_reg[TIMER_TG0 + timer].irq;
	ret = request_irq(irq, cb, IRQF_SHARED, devname, dev_id);
	if (ret != 0)
		return ret;

	return 0;
}
EXPORT_SYMBOL(emxx_timer_register_cb);

int emxx_timer_unregister_cb(unsigned int timer, void *dev_id)
{
	unsigned int irq = 0;
	int ret = 0;
	volatile struct timer_reg_t *reg;

	ret = check_validity_channel(timer);
	if (ret != 0)
		return ret;

	reg = tm_reg[TIMER_TG0 + timer].reg;

	if (reg->tm_op & TSTART)
		return -EBUSY;

	irq = tm_reg[TIMER_TG0 + timer].irq;
	free_irq(irq, dev_id);

	return 0;
}
EXPORT_SYMBOL(emxx_timer_unregister_cb);


/*
 * WatchDog Timer Function
 */
int emxx_wdt_set_timeout(unsigned int usecs)
{
	int ret = 0, val_tm_op, regval;
	unsigned int tm_delay = 0;
	volatile struct timer_reg_t *reg = tm_reg[TIMER_WDT].reg;

	if ((usecs < TW_MIN_USECS) || (TW_MAX_USECS < usecs))
		return -EINVAL;
	tm_delay = TIMER_DELAY(TW_CLOCK_TICK_RATE) * 2;

	spin_lock(&wdt_spinlock);

	/* get tm_op */
	val_tm_op = reg->tm_op;

	/* timer stop */
	reg->tm_op = TSTOP;

	/* set counter */
	regval = get_tw_count_value(TIMER_WDT, usecs);

	udelay(tm_delay);
	reg->tm_set = regval;

	/* timer start (restore tm_op) */
	reg->tm_op = val_tm_op;

	spin_unlock(&wdt_spinlock);

	return ret;
}
EXPORT_SYMBOL(emxx_wdt_set_timeout);

/* Clear watchdog timer. */
void emxx_wdt_ping(void)
{
	unsigned int tm_delay = TIMER_DELAY(TW_CLOCK_TICK_RATE);
	volatile struct timer_reg_t *reg = tm_reg[TIMER_WDT].reg;

	spin_lock(&wdt_spinlock);
	reg->tm_clr = TCR_CLR;
	udelay(tm_delay);
	spin_unlock(&wdt_spinlock);
}
EXPORT_SYMBOL(emxx_wdt_ping);

/* Start watchdog timer counting. */
void emxx_wdt_enable(void)
{
	volatile struct timer_reg_t *reg = tm_reg[TIMER_WDT].reg;

	spin_lock(&wdt_spinlock);
	reg->tm_op = TM_EN | TSTART | TO_EN;
	spin_unlock(&wdt_spinlock);
}
EXPORT_SYMBOL(emxx_wdt_enable);

/* Stop watchdog timer counting. */
void emxx_wdt_disable(void)
{
	unsigned int tm_delay = TIMER_DELAY(TW_CLOCK_TICK_RATE);
	volatile struct timer_reg_t *reg = tm_reg[TIMER_WDT].reg;

	spin_lock(&wdt_spinlock);
	reg->tm_op = TSTOP;
	udelay(tm_delay);
	spin_unlock(&wdt_spinlock);
}
EXPORT_SYMBOL(emxx_wdt_disable);

/* Setup watchdog timer. */
void emxx_wdt_setup(void)
{
	unsigned int regval;
	unsigned int tm_delay = 0;
	volatile struct timer_reg_t *reg = tm_reg[TIMER_WDT].reg;

	tm_delay = TIMER_DELAY(TW_CLOCK_TICK_RATE) * 2;
	spin_lock(&wdt_spinlock);

	writel(readl(SMU_WDT_INT_RESET) | 0x01003003, SMU_WDT_INT_RESET);

	emxx_open_clockgate(EMXX_CLK_TIM_P);
	emxx_open_clockgate(tm_reg[TIMER_WDT].clkdev);
	emxx_unreset_device(tm_reg[TIMER_WDT].rstdev);

	/* stop timer for sure. */
	reg->tm_op = TSTOP;

	/* set counter */
	regval = get_tw_count_value(TIMER_WDT, tm_param[TIMER_WDT].usecs);
	udelay(tm_delay);
	reg->tm_set = regval;

	spin_unlock(&wdt_spinlock);
}
EXPORT_SYMBOL(emxx_wdt_setup);
