/*
 * i.MX25 generic GPIO handling
 *
 * Copyright (c) 2009 Atmark Techno, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 */

#include <linux/module.h>
#include <linux/seq_file.h>
#include <linux/gpio.h>
#include <asm/io.h>

struct mxc_gpio_chip {
	struct gpio_chip chip;
	int port_num;
};

#define to_mxc_gpio_chip(c) container_of(c, struct mxc_gpio_chip, chip)

#define GPIO_TO_IOMUX(port, num) (((port) << MUX_IO_P) | ((num) << MUX_IO_I))

#if defined(DEBUG)
#define dprintf(...) do{pr_dbg(__func__": "__VA_ARGS__);}while(0)
#else
#define dpeintf(...) do{}while(0)
#endif

struct gpio_port {
	u32 num;		/*!< gpio port number */
	u32 base;		/*!< gpio port base VA */
#ifdef MXC_GPIO_SPLIT_IRQ_2
	u16 irq_0_15, irq_16_31;
#else
	u16 irq;		/*!< irq number to the core */
#endif
	u16 virtual_irq_start;	/*!< virtual irq start number */
	u32 reserved_map;	/*!< keep track of which pins are in use */
	u32 irq_is_level_map;	/*!< if a pin's irq is level sensitive. default is edge */
	u32 suspend_wakeup;
	u32 saved_wakeup;
	spinlock_t lock;	/*!< lock when operating on the port */
};
static struct gpio_port gpio_port[GPIO_PORT_NUM];

/*
 * Find the pointer to the gpio_port for a given pin.
 * @param gpio		a gpio pin number
 * @return		pointer to \b struc \b gpio_port
 */
static inline struct gpio_port *get_gpio_port(u32 gpio)
{
	return &gpio_port[GPIO_TO_PORT(gpio)];
}

/*
 * Check if a gpio pin is within [0, MXC_MAX_GPIO_LINES -1].
 * @param gpio		a gpio pin number
 * @return		0 if the pin number is valid; -1 otherwise
 */
static int check_gpio(u32 gpio)
{
	if (gpio >= MXC_MAX_GPIO_LINES) {
		printk(KERN_ERR "mxc-gpio: invalid GPIO %d\n", gpio);
		dump_stack();
		return -1;
	}
	return 0;
}

/*
 * Set a GPIO pin's direction
 * @param port		pointer to a gpio_port
 * @param index		gpio pin index value (0~31)
 * @param is_input	0 for output; non-zero for input
 */
static void _set_gpio_direction(struct gpio_port *port, u32 index, int is_input)
{
	u32 reg = port->base + GPIO_GDIR;
	u32 l;

	l = __raw_readl(reg);
	if (is_input)
		l &= ~(1 << index);
	else
		l |= 1 << index;
	__raw_writel(l, reg);
}

/*!
 * Exported function to set a GPIO pin's direction
 * @param pin		a name defined by \b iomux_pin_name_t
 * @param is_input	1 (or non-zero) for input; 0 for output
 */
void mxc_set_gpio_direction(iomux_pin_name_t pin, int is_input)
{
	struct gpio_port *port;
	u32 gpio = IOMUX_TO_GPIO(pin);

	if (check_gpio(gpio) < 0)
		return;
	port = get_gpio_port(gpio);
	spin_lock(&port->lock);
	_set_gpio_direction(port, GPIO_TO_INDEX(gpio), is_input);
	spin_unlock(&port->lock);
}

/*!
 * Return the GPIO pin's direction.
 * @param pin	a name defined by \b iomux_pin_name_t
 * @return 	value (0 or 1) of the direction; -1 if pass in invalid pin
 */
int mxc_get_gpio_direction(iomux_pin_name_t pin)
{
	struct gpio_port *port;
	u32 gpio = IOMUX_TO_GPIO(pin);
	u32 val;

	if (check_gpio(gpio) < 0)
		return -1;

	port = get_gpio_port(gpio);
	val = (__raw_readl(port->base + GPIO_GDIR) >> GPIO_TO_INDEX(gpio) & 1);
	return (val ? 0 : 1);
}

/*
 * Set a GPIO pin's data output
 * @param port		pointer to a gpio_port
 * @param index		gpio pin index value (0~31)
 * @param data		value to be set (only 0 or 1 is valid)
 */
static void _set_gpio_dataout(struct gpio_port *port, u32 index, u32 data)
{
	u32 reg = port->base + GPIO_DR;
	u32 l = 0;

	l = (__raw_readl(reg) & (~(1 << index))) | (data << index);
	__raw_writel(l, reg);
}

/*!
 * Exported function to set a GPIO pin's data output
 * @param pin		a name defined by \b iomux_pin_name_t
 * @param data		value to be set (only 0 or 1 is valid)
 */
void mxc_set_gpio_dataout(iomux_pin_name_t pin, u32 data)
{
	struct gpio_port *port;
	u32 gpio = IOMUX_TO_GPIO(pin);

	if (check_gpio(gpio) < 0)
		return;

	port = get_gpio_port(gpio);
	spin_lock(&port->lock);
	_set_gpio_dataout(port, GPIO_TO_INDEX(gpio), (data == 0) ? 0 : 1);
	spin_unlock(&port->lock);
}

/*!
 * Return the data value of a GPIO signal.
 * @param pin	a name defined by \b iomux_pin_name_t
 *
 * @return 	value (0 or 1) of the GPIO signal; -1 if pass in invalid pin
 */
int mxc_get_gpio_datain(iomux_pin_name_t pin)
{
	struct gpio_port *port;
	u32 gpio = IOMUX_TO_GPIO(pin);

	if (check_gpio(gpio) < 0)
		return -1;

	port = get_gpio_port(gpio);

	return (__raw_readl(port->base + GPIO_DR) >> GPIO_TO_INDEX(gpio)) & 1;
}

static int mx25_gpio_direction_input(struct gpio_chip *chip, unsigned offset)
{
	struct mxc_gpio_chip *mxc_gpio = to_mxc_gpio_chip(chip);
	u32 pin = GPIO_TO_IOMUX(mxc_gpio->port_num, offset);

	mxc_set_gpio_direction(pin, 1);

	return 0;
}

static int mx25_gpio_direction_output(struct gpio_chip *chip,
				      unsigned offset, int val)
{
	struct mxc_gpio_chip *mxc_gpio = to_mxc_gpio_chip(chip);
	u32 pin = GPIO_TO_IOMUX(mxc_gpio->port_num, offset);

	mxc_set_gpio_dataout(pin, val);
	mxc_set_gpio_direction(pin, 0);

	return 0;
}

static int mx25_gpio_get(struct gpio_chip *chip, unsigned offset)
{
	struct mxc_gpio_chip *mxc_gpio = to_mxc_gpio_chip(chip);
	u32 pin = GPIO_TO_IOMUX(mxc_gpio->port_num, offset);

	return mxc_get_gpio_datain(pin);
}

static void mx25_gpio_set(struct gpio_chip *chip, unsigned offset, int val)
{
	struct mxc_gpio_chip *mxc_gpio = to_mxc_gpio_chip(chip);
	u32 pin = GPIO_TO_IOMUX(mxc_gpio->port_num, offset);

	mxc_set_gpio_dataout(pin, val);
}

static void mx25_gpio_dbg_show(struct seq_file *s, struct gpio_chip *chip)
{
	struct mxc_gpio_chip *mxc_gpio = to_mxc_gpio_chip(chip);
	u32 pin;
	int val, dir;
	int i;

	for (i = 0; i < chip->ngpio; i++) {
		pin = GPIO_TO_IOMUX(mxc_gpio->port_num, i);
		val = mxc_get_gpio_datain(pin);
		dir = mxc_get_gpio_direction(pin);

                 seq_printf(s, "GPIO %s%d: %s %s\n",
			    chip->label, i,
                            (val == 1) ? "set" : (val == 0) ? "clear" : "error",
                            (dir == 1) ? "in" : (dir == 0) ? "out" : "error");
	}
}

#define MXC_GPIO_BANK(name, nport, base_gpio)                            \
         {                                                               \
                 .chip = {                                               \
                         .label            = name,                       \
                         .direction_input  = mx25_gpio_direction_input,  \
                         .direction_output = mx25_gpio_direction_output, \
                         .get              = mx25_gpio_get,              \
                         .set              = mx25_gpio_set,              \
                         .dbg_show         = mx25_gpio_dbg_show,         \
                         .base             = base_gpio,                  \
                         .ngpio            = GPIO_NUM_PIN,               \
                 },                                                      \
		 .port_num = nport,                                      \
         }

static struct mxc_gpio_chip mx25_gpio_banks[] = {
	MXC_GPIO_BANK("1", 0, MXC_GPIO_LINE_1_BASE),
	MXC_GPIO_BANK("2", 1, MXC_GPIO_LINE_2_BASE),
	MXC_GPIO_BANK("3", 2, MXC_GPIO_LINE_3_BASE),
	MXC_GPIO_BANK("4", 3, MXC_GPIO_LINE_4_BASE),
};

void mx25_generic_gpio_init(void)
{
	int i;

	for (i = 0; i < ARRAY_SIZE(mx25_gpio_banks); i++)
		gpiochip_add(&mx25_gpio_banks[i].chip);
}
EXPORT_SYMBOL(mx25_generic_gpio_init);
