/*
 * arch/arm/mm/cache-l2x0.c - L210/L220 cache controller support
 *
 * Copyright (C) 2007 ARM Limited
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
#include <linux/init.h>
#include <linux/spinlock.h>
#include <linux/io.h>
#ifdef CONFIG_SNSC_SSBOOT
#include <linux/ssboot.h>
#endif

#include <asm/cacheflush.h>
#include <asm/hardware/cache-l2x0.h>

#define CACHE_LINE_SIZE		32

#ifdef CONFIG_SNSC_OPROFILE_ARM11_EVTMON
#include <linux/module.h>
#define L2X0_ENABLE_BIT       	0x1
#define L2X0_EVTBUS_BIT		(1 << 20)
#define L2X0_CTL_REG      	(l2x0_base + L2X0_CTRL)
#define L2X0_AUX_REG		(l2x0_base + L2X0_AUX_CTRL)
#endif

static void __iomem *l2x0_base;
bool l2x0_disabled;
static uint32_t l2x0_way_mask;	/* Bitmask of active ways */

static inline void cache_wait_always(void __iomem *reg, unsigned long mask)
{
	/* wait for the operation to complete */
	while (readl(reg) & mask)
		;
}

#ifdef CONFIG_CACHE_PL310

static inline void cache_wait(void __iomem *reg, unsigned long mask)
{
	/* cache operations are atomic */
}

#define _l2x0_lock(lock, flags)		((void)(flags))
#define _l2x0_unlock(lock, flags)	((void)(flags))

#define block_end(start, end)		(end)

#define L2CC_TYPE			"PL310/L2C-310"

#else	/* !CONFIG_CACHE_PL310 */

#define cache_wait			cache_wait_always

static DEFINE_RAW_SPINLOCK(l2x0_lock);
#define _l2x0_lock(lock, flags)		spin_lock_irqsave(lock, flags)
#define _l2x0_unlock(lock, flags)	spin_unlock_irqrestore(lock, flags)

#define block_end(start, end)		((start) + min((end) - (start), 4096UL))

#define L2CC_TYPE			"L2x0"

#endif	/* CONFIG_CACHE_PL310 */

static inline void cache_sync(void)
{
	void __iomem *base = l2x0_base;
#ifdef CONFIG_ARM_ERRATA_484863
	unsigned long val = 0;
	asm volatile("swp %0, %0, [%1]\n"
		     : "+r" (val) : "r" (base + L2X0_CACHE_SYNC));
#else
	writel(0, base + L2X0_CACHE_SYNC);
#endif
	cache_wait(base + L2X0_CACHE_SYNC, 1);
}

static inline void l2x0_clean_line(unsigned long addr)
{
	void __iomem *base = l2x0_base;
	cache_wait(base + L2X0_CLEAN_LINE_PA, 1);
	writel(addr, base + L2X0_CLEAN_LINE_PA);
}

static inline void l2x0_inv_line(unsigned long addr)
{
	void __iomem *base = l2x0_base;
	cache_wait(base + L2X0_INV_LINE_PA, 1);
	writel(addr, base + L2X0_INV_LINE_PA);
}

static inline void l2x0_flush_line(unsigned long addr)
{
	void __iomem *base = l2x0_base;
	cache_wait(base + L2X0_CLEAN_INV_LINE_PA, 1);
	writel(addr, base + L2X0_CLEAN_INV_LINE_PA);
}

static void l2x0_cache_sync(void)
{
	unsigned long flags;

	_l2x0_lock(&l2x0_lock, flags);
	cache_sync();
	_l2x0_unlock(&l2x0_lock, flags);
}

static inline void l2x0_inv_all(void)
{
	unsigned long flags;

	/* invalidate all ways */
	_l2x0_lock(&l2x0_lock, flags);
	writel(l2x0_way_mask, l2x0_base + L2X0_INV_WAY);
	cache_wait_always(l2x0_base + L2X0_INV_WAY, l2x0_way_mask);
	cache_sync();
	_l2x0_unlock(&l2x0_lock, flags);
}

static void l2x0_flush_all(void)
{
	unsigned long flags;

	/* invalidate all ways */
	_l2x0_lock(&l2x0_lock, flags);
	writel(l2x0_way_mask, l2x0_base + L2X0_CLEAN_INV_WAY);
	cache_wait_always(l2x0_base + L2X0_CLEAN_INV_WAY, l2x0_way_mask);
	cache_sync();
	_l2x0_unlock(&l2x0_lock, flags);
}

static void l2x0_inv_range(unsigned long start, unsigned long end)
{
	void __iomem *base = l2x0_base;
	unsigned long flags;

#ifdef CONFIG_L2_CACHE_UNALIGN_INV_BACKTRACE
 	if((start & (CACHE_LINE_SIZE -1)) || (end & (CACHE_LINE_SIZE -1))){
 		printk("Unalign Invalidate L2 Cache: "
 		       "start = 0x%lx end = 0x%lx size = %ld\n",
 		       start,end,end-start);
 		__backtrace();
 	}
#endif

	_l2x0_lock(&l2x0_lock, flags);
	if (start & (CACHE_LINE_SIZE - 1)) {
		start &= ~(CACHE_LINE_SIZE - 1);
		l2x0_flush_line(start);
		start += CACHE_LINE_SIZE;
	}

	if ((end & (CACHE_LINE_SIZE - 1)) && (end > start)) {
		end &= ~(CACHE_LINE_SIZE - 1);
		l2x0_flush_line(end);
	}

	while (start < end) {
		unsigned long blk_end = block_end(start, end);

		while (start < blk_end) {
			l2x0_inv_line(start);
			start += CACHE_LINE_SIZE;
		}

		if (blk_end < end) {
			_l2x0_unlock(&l2x0_lock, flags);
			_l2x0_lock(&l2x0_lock, flags);
		}
	}
	cache_wait(base + L2X0_INV_LINE_PA, 1);
	cache_sync();
	_l2x0_unlock(&l2x0_lock, flags);
}

static void l2x0_clean_range(unsigned long start, unsigned long end)
{
	void __iomem *base = l2x0_base;
	unsigned long flags;

	_l2x0_lock(&l2x0_lock, flags);
	start &= ~(CACHE_LINE_SIZE - 1);
	while (start < end) {
		unsigned long blk_end = block_end(start, end);

		while (start < blk_end) {
			l2x0_clean_line(start);
			start += CACHE_LINE_SIZE;
		}

		if (blk_end < end) {
			_l2x0_unlock(&l2x0_lock, flags);
			_l2x0_lock(&l2x0_lock, flags);
		}
	}
	cache_wait(base + L2X0_CLEAN_LINE_PA, 1);
	cache_sync();
	_l2x0_unlock(&l2x0_lock, flags);
}

static void l2x0_flush_range(unsigned long start, unsigned long end)
{
	void __iomem *base = l2x0_base;
	unsigned long flags;

	_l2x0_lock(&l2x0_lock, flags);
	start &= ~(CACHE_LINE_SIZE - 1);
	while (start < end) {
		unsigned long blk_end = block_end(start, end);

		while (start < blk_end) {
			l2x0_flush_line(start);
			start += CACHE_LINE_SIZE;
		}

		if (blk_end < end) {
			_l2x0_unlock(&l2x0_lock, flags);
			_l2x0_lock(&l2x0_lock, flags);
		}
	}
	cache_wait(base + L2X0_CLEAN_INV_LINE_PA, 1);
	cache_sync();
	_l2x0_unlock(&l2x0_lock, flags);
}

#ifdef CONFIG_SNSC_OPROFILE_ARM11_EVTMON
/*!
 * Enable the EVTBUS to monitor L2 cache events
 */
void l2x0_evtbus_enable(void)
{
	unsigned int flags;

	local_irq_save(flags);

	/* If L2 cache is enabled then disable L2 cache,
	 * enable L2 evtbus, re-enable L2 cache
	 */
	if ((readl(L2X0_CTL_REG) & L2X0_ENABLE_BIT) != 0) {
		writel(0, L2X0_CTL_REG);
		writel((readl(L2X0_AUX_REG) | L2X0_EVTBUS_BIT), L2X0_AUX_REG);
		writel(L2X0_ENABLE_BIT, L2X0_CTL_REG);
	} else {
		writel((readl(L2X0_AUX_REG) | L2X0_EVTBUS_BIT), L2X0_AUX_REG);
	}
	local_irq_restore(flags);
}

/*!
 * Disable the EVTBUS
 */
void l2x0_evtbus_disable(void)
{
	unsigned int flags;

	local_irq_save(flags);

	/* If L2 cache is enabled then disable L2 cache,
	 * disable L2 evtbus, re-enable L2 cache
 	 */
	if ((readl(L2X0_CTL_REG) & L2X0_ENABLE_BIT) != 0) {
		writel(0, L2X0_CTL_REG);
		writel((readl(L2X0_AUX_REG) & ~L2X0_EVTBUS_BIT), L2X0_AUX_REG);
		writel(L2X0_ENABLE_BIT, L2X0_CTL_REG);
	} else {
		writel((readl(L2X0_AUX_REG) & ~L2X0_EVTBUS_BIT), L2X0_AUX_REG);
	}
	local_irq_restore(flags);
}

EXPORT_SYMBOL(l2x0_evtbus_enable);
EXPORT_SYMBOL(l2x0_evtbus_disable);

#endif /* CONFIG_SNSC_OPROFILE_ARM11_EVTMON */

void __init l2x0_init(void __iomem *base, __u32 aux_val, __u32 aux_mask)
{
	__u32 aux;
	__u32 cache_id;
	int ways;
	const char *type;

	if (l2x0_disabled) {
		printk(KERN_INFO "L2X0 cache controller disabled\n");
		return;
	}

	l2x0_base = base;

	cache_id = readl(l2x0_base + L2X0_CACHE_ID);
	aux = readl(l2x0_base + L2X0_AUX_CTRL);

	aux &= aux_mask;
	aux |= aux_val;

	/* Determine the number of ways */
	switch (cache_id & L2X0_CACHE_ID_PART_MASK) {
	case L2X0_CACHE_ID_PART_L310:
		if (aux & (1 << 16))
			ways = 16;
		else
			ways = 8;
		type = "L310";
		break;
	case L2X0_CACHE_ID_PART_L210:
		ways = (aux >> 13) & 0xf;
		type = "L210";
		break;
	default:
		/* Assume unknown chips have 8 ways */
		ways = 8;
		type = "L2x0 series";
		break;
	}

	l2x0_way_mask = (1 << ways) - 1;

	/*
	 * Check if l2x0 controller is already enabled.
	 * If you are booting from non-secure mode
	 * accessing the below registers will fault.
	 */
	if (!(readl(l2x0_base + L2X0_CTRL) & 1)) {

		/* l2x0 controller is disabled */
		writel(aux, l2x0_base + L2X0_AUX_CTRL);

		l2x0_inv_all();

		/* enable L2X0 */
		writel(1, l2x0_base + L2X0_CTRL);
	}

	outer_cache.inv_range = l2x0_inv_range;
	outer_cache.clean_range = l2x0_clean_range;
	outer_cache.flush_range = l2x0_flush_range;
	outer_cache.flush_all = l2x0_flush_all;
	outer_cache.sync = l2x0_cache_sync;

	printk(KERN_INFO "%s cache controller enabled\n", type);
	printk(KERN_INFO "l2x0: %d ways, CACHE_ID 0x%08x, AUX_CTRL 0x%08x\n",
			 ways, cache_id, aux);
}

static int __init l2x0_disable(char *unused)
{
	l2x0_disabled = 1;
	return 0;
}
early_param("nol2x0", l2x0_disable);

#if defined(CONFIG_ARCH_EMXX) || defined(CONFIG_SNSC_SSBOOT)

#ifdef CONFIG_SNSC_SSBOOT
static __u32 saved_aux_ctrl;
#endif

static inline void l2x0_clean_all(void)
{
	/* invalidate all ways */
	writel(l2x0_way_mask, l2x0_base + L2X0_CLEAN_WAY);
	cache_wait_always(l2x0_base + L2X0_CLEAN_WAY, l2x0_way_mask);
	cache_sync();
}

void l2x0_suspend(void)
{
	if (l2x0_base != NULL) {
		if (readl(l2x0_base + L2X0_CTRL) & 1) {
#ifdef CONFIG_SNSC_SSBOOT
			if (ssboot_is_writing()) {
				/* save AUX_CTRL register */
				saved_aux_ctrl =
					readl(l2x0_base + L2X0_AUX_CTRL);
				return;
			}
#endif
			/* disable L2X0 */
			flush_cache_all();
			asm("dsb");
			writel(0, l2x0_base + L2X0_CTRL);
			l2x0_clean_all();
		}
	}
}

void l2x0_resume(void)
{
#ifdef CONFIG_SNSC_SSBOOT
	if (ssboot_is_writing())
		return;
#endif

	if (l2x0_base != NULL) {
		if (!(readl(l2x0_base + L2X0_CTRL) & 1)) {
#ifdef CONFIG_SNSC_SSBOOT
			if (ssboot_is_resumed()) {
				writel(saved_aux_ctrl,
				       l2x0_base + L2X0_AUX_CTRL);
			}
#endif
			/* enable L2X0 */
			l2x0_inv_all();
			writel(1, l2x0_base + L2X0_CTRL);
		}
	}
}
#endif

