/*
 *  linux/arch/arm/mm/pgd.c
 *
 *  Copyright (C) 1998-2005 Russell King
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */
#include <linux/mm.h>
#include <linux/highmem.h>

#include <asm/pgalloc.h>
#include <asm/page.h>
#include <asm/tlbflush.h>

#include "mm.h"

#define FIRST_KERNEL_PGD_NR	(FIRST_USER_PGD_NR + USER_PTRS_PER_PGD)

#ifndef CONFIG_EJ_SPLIT_PAGETABLE
#define PGD_ORDER 2
#else /* CONFIG_EJ_SPLIT_PAGETABLE */
#if CONFIG_EJ_SPLIT_SHIFT < 3
#define PGD_ORDER (2 - CONFIG_EJ_SPLIT_SHIFT)
#else
#define PGD_SIZE (PAGE_SIZE)/(1 << (CONFIG_EJ_SPLIT_SHIFT - 2))
#endif
#endif

#ifdef PGD_ORDER
#define PGD_SIZE (PAGE_SIZE * (1 << PGD_ORDER))
#define ALLOC_PGD() (pgd_t *)__get_free_pages(GFP_KERNEL, PGD_ORDER)
#define FREE_PGD(p) free_pages((unsigned long)(p), PGD_ORDER)
#else
#define ALLOC_PGD() ({BUG_ON(!pgd_cachep);(pgd_t *)kmem_cache_alloc(pgd_cachep, GFP_KERNEL);})
#define FREE_PGD(p) kmem_cache_free(pgd_cachep, p)

static struct kmem_cache *pgd_cachep;
static int pgd_init(void)
{
    pgd_cachep = kmem_cache_create("pgd", PGD_SIZE, PGD_SIZE, SLAB_PANIC, NULL);
    return 0;
}

late_initcall(pgd_init);
#endif

/*
 * need to get a 16k page for level 1
 */
pgd_t *get_pgd_slow(struct mm_struct *mm)
{
	pgd_t *new_pgd, *init_pgd;
	pmd_t *new_pmd, *init_pmd;
	pte_t *new_pte, *init_pte;
#ifdef CONFIG_SNSC_PREVENT_MIGRATION_IN_CRITICAL_SECTION
	migration_t saved;
#endif

	new_pgd = ALLOC_PGD();
	if (!new_pgd)
		goto no_pgd;

#ifdef CONFIG_SNSC_PREVENT_MIGRATION_IN_CRITICAL_SECTION
	migration_disable(&saved);
#endif
	memset(new_pgd, 0, FIRST_KERNEL_PGD_NR * sizeof(pgd_t));

	/*
	 * Copy over the kernel and IO PGD entries
	 */
	init_pgd = pgd_offset_k(0);
#ifndef CONFIG_EJ_SPLIT_PAGETABLE
	memcpy(new_pgd + FIRST_KERNEL_PGD_NR, init_pgd + FIRST_KERNEL_PGD_NR,
		       (PTRS_PER_PGD - FIRST_KERNEL_PGD_NR) * sizeof(pgd_t));
#endif

	clean_dcache_area(new_pgd, PGD_SIZE);
#ifdef CONFIG_SNSC_PREVENT_MIGRATION_IN_CRITICAL_SECTION
	migration_enable(saved);
#endif

	if (!vectors_high()) {
		/*
		 * On ARM, first page must always be allocated since it
		 * contains the machine vectors.
		 */
		new_pmd = pmd_alloc(mm, new_pgd, 0);
		if (!new_pmd)
			goto no_pmd;

		new_pte = pte_alloc_map(mm, new_pmd, 0);
		if (!new_pte)
			goto no_pte;

		init_pmd = pmd_offset(init_pgd, 0);
		init_pte = pte_offset_map_nested(init_pmd, 0);
		set_pte_ext(new_pte, *init_pte, 0);
		pte_unmap_nested(init_pte);
		pte_unmap(new_pte);
	}

	return new_pgd;

no_pte:
	pmd_free(mm, new_pmd);
no_pmd:
	FREE_PGD(new_pgd);
no_pgd:
	return NULL;
}

void free_pgd_slow(struct mm_struct *mm, pgd_t *pgd)
{
	pmd_t *pmd;
	pgtable_t pte;

	if (!pgd)
		return;

	/* pgd is always present and good */
	pmd = pmd_off(pgd, 0);
	if (pmd_none(*pmd))
		goto free;
	if (pmd_bad(*pmd)) {
		pmd_ERROR(*pmd);
		pmd_clear(pmd);
		goto free;
	}

	pte = pmd_pgtable(*pmd);
	pmd_clear(pmd);
	pte_free(mm, pte);
	pmd_free(mm, pmd);
free:
	FREE_PGD(pgd);
}
