/*
 *  /proc/mx31_reg to read MX31 registers.
 *
 *  Copyright 2003-2007 Sony Corporation.
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/init.h>
#include <linux/proc_fs.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include <asm/system.h>
#include <mach/hardware.h>


/* ======================================================================= *
 * Constants                                                               *
 * ======================================================================= */

#define MX31_REG_PROC_MODULE          "mx31_reg_proc"
#define MX31_REG_PROC_DIRNAME         "mx31_reg"
#define MX31_REG_PROC_MODULE_FILENAME "all"
#define MX31_REG_PROC_MAX_NAMELEN     38
#define MX31_REG_PROC_BUF_SIZE	      (MX31_REG_PROC_MAX_NAMELEN + 32)

MODULE_AUTHOR("Sony");
MODULE_DESCRIPTION("Freescale MX31 register viewer for proc filesystem");
MODULE_LICENSE("GPL");


/* ======================================================================= *
 * Debug print function                                                    *
 *   set permutations of:                                                  *
 *     0 : disables all output including load/unload message.              *
 *     1 : Print output. (Default)                                         *
 *     2 : Verbose, dump registers                                         *
 *     4 : Print entry of functions.                                       *
 * ======================================================================= */
#define REGPROC_DEBUG 0x1

#if REGPROC_DEBUG & 1
#define PRINTK(format, args...)                    \
          printk("" format ,                    \
		 ##args)
#else
#define PRINTK(format, args...)
#endif

#if REGPROC_DEBUG & 2
#define DEBUG(format, args...)                   \
          printk("%s():%d " format ,            \
		 __FUNCTION__ , __LINE__ , ##args)
#else
#define DEBUG(format, args...)
#endif

#if REGPROC_DEBUG & 4
#define ENTRY()                                    \
          printk("%s():%d\n",                   \
		 __FUNCTION__ , __LINE__ )
#else
#define ENTRY()
#endif

#define ERROR(format, args...)                     \
          printk(KERN_ERR "%s(): " format ,     \
		 __FUNCTION__ , ##args)

/* Class of registers */
enum reg_type {
	REG_MMIO_32,
	REG_MMIO_16,
	REG_MMIO_8,
 	REG_COP_RO,
 	REG_COP_RW,
 	REG_COP_WO,
};

/* Common register class */
typedef struct regproc_reg_t {
	const char		*name;
	enum reg_type		 type;
	unsigned int		 low_ino;
 	unsigned long		 offset;
} regproc_reg_t;

/* Module class contains table of registers in on-chip peripheral
 * modules and additional infomations (if any). */
typedef struct regproc_module_t {
	const char		*name;
 	regproc_reg_t		*regs;
	void __iomem		*base;
	struct proc_dir_entry	*dir;
	unsigned int		 low_ino;
} regproc_module_t;


/* ======================================================================= *
 * Prototype declaration                                                   *
 * ======================================================================= */

static int mx31_reg_proc_init(void);
static void mx31_reg_proc_exit(void);
static ssize_t mx31_reg_proc_module_read(struct file *filp, char *buf,
					   size_t count, loff_t *f_pos);
static ssize_t mx31_reg_proc_reg_read(struct file *filp, char *buf,
					size_t count, loff_t *f_pos);
static ssize_t mx31_reg_proc_reg_write(struct file * filp, const char * buf,
					 size_t count, loff_t *ppos);


/* ======================================================================= *
 * Global variable                                                         *
 * ======================================================================= */

/* Per-processor definitions */
#ifdef CONFIG_ARCH_MX3
#include "mx31_regproc.h"
#else
#error Unknwon processor.
#endif

static struct proc_dir_entry  *mx31_reg_proc_root;


/* ======================================================================= *
 * Linux driver interface                                                  *
 * ======================================================================= */

static struct file_operations mx31_reg_proc_module_ops = {
	read:	mx31_reg_proc_module_read,
};

static struct file_operations mx31_reg_proc_reg_ops = {
	read:	mx31_reg_proc_reg_read,
	write:	mx31_reg_proc_reg_write,
};

static int
mx31_reg_proc_init(void)
{
	int i, j = 0, res = 0;
	regproc_reg_t *regs;
	struct proc_dir_entry   *dir;
	struct proc_dir_entry   *ent;
	regproc_module_t	*mod;

	ENTRY();

	PRINTK("Registering /proc/%s\n", MX31_REG_PROC_DIRNAME);
	mx31_reg_proc_root = proc_mkdir(MX31_REG_PROC_DIRNAME, NULL);
	if (mx31_reg_proc_root == NULL) {
		ERROR("can't create /proc/%s\n", MX31_REG_PROC_DIRNAME);
		res = -ENOMEM;
		goto err1;
	}

	for (i = 0; mx31_reg_proc_modules[i].name != NULL; i++) {
		mod = mx31_reg_proc_modules+i;

		dir = proc_mkdir(mod->name, mx31_reg_proc_root);

		DEBUG("Registering /proc/%s/%s/\n",
		      MX31_REG_PROC_DIRNAME, mod->name);
		if (dir == NULL) {
			ERROR("can't create /proc/%s/%s\n",
			      MX31_REG_PROC_DIRNAME, mod->name);
			res = -ENOMEM;
			goto err2;
		}
		mod->dir = dir;
		ent = create_proc_entry(MX31_REG_PROC_MODULE_FILENAME,
					S_IRUSR | S_IRGRP | S_IROTH, dir);

		DEBUG("Registering /proc/%s/%s/%s\n",
		      MX31_REG_PROC_DIRNAME,
		      mod->name, MX31_REG_PROC_MODULE_FILENAME);
		if (ent == NULL) {
			ERROR("can't create /proc/%s/%s/%s\n",
			      MX31_REG_PROC_DIRNAME, mod->name,
			      MX31_REG_PROC_MODULE_FILENAME);
			res = -ENOMEM;
			goto err3;
		}
		mod->low_ino = ent->low_ino;
		ent->proc_fops = &mx31_reg_proc_module_ops;

		regs = mod->regs;
		for (j = 0; regs[j].name != NULL; j++) {
			ent = create_proc_entry(regs[j].name,
						S_IWUSR | S_IRUSR | S_IRGRP |
						S_IROTH, dir);

			if (ent == NULL) {
				ERROR("can't create /proc/%s/%s/%s\n",
				      MX31_REG_PROC_DIRNAME,
				      mod->name, regs[j].name);
				res = -ENOMEM;
				goto err4;
			}
			regs[j].low_ino = ent->low_ino;
			ent->proc_fops = &mx31_reg_proc_reg_ops;

			DEBUG("Registering /proc/%s/%s/%s type=%d, offset=0x%lx, inode=0x%08x\n",
			      MX31_REG_PROC_DIRNAME,
			      mod->name, regs[j].name,
			      regs[j].type, regs[j].offset, ent->low_ino);
		}
	}

	return 0;

	/* handle errors */
	do {
		regs = mod->regs;
		do {
			remove_proc_entry(regs[j].name, mod->dir);
err4:
			j--;
		} while (j >= 0);

		remove_proc_entry(MX31_REG_PROC_MODULE_FILENAME,
				  mod->dir);
err3:
		remove_proc_entry(mod->name,
				  mx31_reg_proc_root);
err2:
		i--;
	} while (i >= 0);
	remove_proc_entry(MX31_REG_PROC_DIRNAME, NULL);
err1:
	return res;
}

static void
mx31_reg_proc_exit(void)
{
	regproc_module_t	*mod;
	regproc_reg_t		*regs;
	int i, j;

	ENTRY();

	for (i = 0; mx31_reg_proc_modules[i].name != NULL; i++) {
		mod = mx31_reg_proc_modules + i;
		regs = mx31_reg_proc_modules[i].regs;

		for (j = 0; regs[j].name != NULL; j++)
			remove_proc_entry(regs[j].name, mod->dir);

		remove_proc_entry(MX31_REG_PROC_MODULE_FILENAME, mod->dir);
		remove_proc_entry(mod->name,  mx31_reg_proc_root);
	}
	remove_proc_entry(MX31_REG_PROC_DIRNAME, NULL);
	PRINTK("Unregistering /proc/%s\n", MX31_REG_PROC_DIRNAME);
	return ;
}

module_init(mx31_reg_proc_init);
module_exit(mx31_reg_proc_exit);


static ssize_t
mx31_reg_proc_module_read(struct file *filp, char *buf,
			      size_t count, loff_t *f_pos)
{
	regproc_module_t	*mod;
	regproc_reg_t	*regs;
	ssize_t		nwritten;
	loff_t		tmp_pos;
	unsigned int	val;
	int		i, j, res, tmp_len, ncopy;
	unsigned int	low_ino;
	char		tmp_buf[MX31_REG_PROC_BUF_SIZE];

	res = access_ok(VERIFY_WRITE, buf, count);
	if (!res)
		return -EFAULT;

	if (count == 0)
		return 0;

	low_ino = filp->f_dentry->d_inode->i_ino;
	for (i = 0; mx31_reg_proc_modules[i].name != NULL; i++) {
		if (mx31_reg_proc_modules[i].low_ino == low_ino) {
			goto module_found;
		}
	}
	return -ENXIO;

module_found:
	mod = mx31_reg_proc_modules + i;
	regs = mx31_reg_proc_modules[i].regs;

	nwritten = 0;
	tmp_pos = 0;
	val = 0xdeadbeef;
	for (j = 0; regs[j].name != NULL; j++) {
		unsigned long flags;
		local_irq_save(flags);
		switch(regs[j].type) {
		case REG_MMIO_32:
			res = snprintf(tmp_buf, MX31_REG_PROC_BUF_SIZE,
				       "%-38s = 0x%08x\n",
				       regs[j].name,
				       readl(mod->base + regs[j].offset));
			break;
		case REG_MMIO_16:
			res = snprintf(tmp_buf, MX31_REG_PROC_BUF_SIZE,
				       "%-38s = 0x%04x\n",
				       regs[j].name,
				       readw(mod->base + regs[j].offset));
			break;
		case REG_MMIO_8:
			res = snprintf(tmp_buf, MX31_REG_PROC_BUF_SIZE,
				       "%-38s = 0x%02x\n",
				       regs[j].name,
				       readb(mod->base + regs[j].offset));
			break;
		case REG_COP_RO:
		case REG_COP_RW:
			res = snprintf(tmp_buf, MX31_REG_PROC_BUF_SIZE,
				       "%-38s = 0x%08x\n",
				       regs[j].name,
				       do_read_coprocessor(regs[j].offset));
			break;
		case REG_COP_WO:
			local_irq_restore(flags);
			continue;
		default:
			local_irq_restore(flags);
			return -EINVAL;
		}
		local_irq_restore(flags);

		tmp_len = res;
		if (tmp_pos == *f_pos) {
			ncopy = tmp_len;
			if (ncopy > count) {
				ncopy = count;
			}
			res = copy_to_user(buf + nwritten, tmp_buf, ncopy);
			if (res != 0) {
				return -EFAULT;
			}
			*f_pos += ncopy;
			nwritten += ncopy;
			count -= ncopy;
			if (count == 0) {
				break;
			}
		} else if (tmp_pos < *f_pos && tmp_pos + tmp_len > *f_pos) {
			ncopy = tmp_pos + tmp_len - *f_pos;
			if (ncopy > count) {
				ncopy = count;
			}
			res = copy_to_user(buf + nwritten,
					   tmp_buf + (*f_pos - tmp_pos),
					   ncopy);
			if (res != 0) {
				return -EFAULT;
			}
			*f_pos += ncopy;
			nwritten += ncopy;
			count -= ncopy;
			if (count == 0) {
				break;
			}
		}
		tmp_pos += tmp_len;
	}
	return nwritten;
}


static ssize_t
mx31_reg_proc_reg_read(struct file *filp, char *buf,
			   size_t count, loff_t *f_pos)
{
	regproc_module_t	*mod;
	regproc_reg_t	*regs;
	ssize_t		nwritten;
	unsigned long	flags;
	int		i, j, res, tmp_len, ncopy;
	unsigned int	low_ino;
	char		tmp_buf[MX31_REG_PROC_MAX_NAMELEN + 22];

	ENTRY();

	res = access_ok(VERIFY_WRITE, buf, count);
	if (!res)
		return -EFAULT;

	if (count == 0)
		return 0;

	low_ino = filp->f_dentry->d_inode->i_ino;
	DEBUG("Searching 0x%08x\n", low_ino);
	for (i = 0; mx31_reg_proc_modules[i].name != NULL; i++) {
		mod = mx31_reg_proc_modules + i;
		regs = mx31_reg_proc_modules[i].regs;
		for (j = 0; regs[j].name != NULL; j++) {
			DEBUG("%s:%s (%08x)\n", mod->name, regs[j].name,regs[j].low_ino);
			if (regs[j].low_ino == low_ino) {
				goto found;
			}
		}
	}
	PRINTK("Not registered in table\n");
	return -ENXIO;

found:
	nwritten = 0;
	DEBUG("%s(0x%lx) type=%d\n", regs[j].name, regs[j].offset, regs[j].type);

	local_irq_save(flags);
	switch(regs[j].type) {
	case REG_MMIO_32:
		res = snprintf(tmp_buf, MX31_REG_PROC_BUF_SIZE,
			       "%-38s = 0x%08x\n",
			       regs[j].name,
			       readl(mod->base + regs[j].offset));
		break;
	case REG_MMIO_16:
		res = snprintf(tmp_buf, MX31_REG_PROC_BUF_SIZE,
			       "%-38s = 0x%04x\n",
			       regs[j].name,
			       readw(mod->base + regs[j].offset));
		break;
	case REG_MMIO_8:
		res = snprintf(tmp_buf, MX31_REG_PROC_BUF_SIZE,
			       "%-38s = 0x%02x\n",
			       regs[j].name,
			       readb(mod->base + regs[j].offset));
		break;
	case REG_COP_RO:
	case REG_COP_RW:
		res = snprintf(tmp_buf, MX31_REG_PROC_BUF_SIZE,
			       "%-38s = 0x%08x\n",
			       regs[j].name,
			       do_read_coprocessor(regs[j].offset));
		break;
	case REG_COP_WO:
	default:
		local_irq_restore(flags);
		return -EINVAL;
	}
	local_irq_restore(flags);

	tmp_len = res;
	if (*f_pos == 0) {
		ncopy = tmp_len;
		if (ncopy > count) {
			ncopy = count;
		}
		res = copy_to_user(buf, tmp_buf, ncopy);
		if (res != 0) {
			ERROR("copy_to_user() failed\n");
			return -EFAULT;
		}
		*f_pos += ncopy;
		nwritten += ncopy;
		count -= ncopy;
	} else if (0 < *f_pos && tmp_len > *f_pos) {
		ncopy = tmp_len - *f_pos;
		if (ncopy > count) {
			ncopy = count;
		}
		res = copy_to_user(buf, tmp_buf + *f_pos, ncopy);
		if (res != 0) {
			ERROR("copy_to_user() failed\n");
			return -EFAULT;
		}
		*f_pos += ncopy;
		nwritten += ncopy;
		count -= ncopy;
	}

	return nwritten;
}


static ssize_t
mx31_reg_proc_reg_write(struct file * filp, const char * buf,
			  size_t count, loff_t *ppos)
{
	regproc_module_t	*mod;
	regproc_reg_t	*regs;
	unsigned long	flags, newRegValue;
	int		i, j, res;
	unsigned int	low_ino;
	char *endp;

	ENTRY();

	res = access_ok(VERIFY_WRITE, buf, count);
	if (!res)
		return -EFAULT;

	if (count == 0)
		return 0;

	low_ino = filp->f_dentry->d_inode->i_ino;
	for (i = 0; mx31_reg_proc_modules[i].name != NULL; i++) {
		mod = mx31_reg_proc_modules + i;
		regs = mx31_reg_proc_modules[i].regs;
		for (j = 0; regs[j].name != NULL; j++) {
			if (regs[j].low_ino == low_ino) {
				goto found;
			}
		}
	}
	PRINTK("Not registered in table\n");
	return -ENXIO;

found:
	newRegValue = simple_strtoul(buf,&endp,0);

	DEBUG("%s(%ld) type=%d <= %lx\n",
	      regs[j].name, regs[j].offset, regs[j].type, newRegValue);

	local_irq_save(flags);
	switch(regs[j].type) {
	case REG_MMIO_32:
		writel(newRegValue, mod->base + regs[j].offset);
		break;
	case REG_MMIO_16:
		writew(newRegValue, mod->base + regs[j].offset);
		break;
	case REG_MMIO_8:
		writeb(newRegValue, mod->base + regs[j].offset);
		break;
	case REG_COP_RW:
	case REG_COP_WO:
		do_write_coprocessor(newRegValue, regs[j].offset);
		break;
	case REG_COP_RO:
	default:
		local_irq_restore(flags);
		return -EINVAL;
	}
	local_irq_restore(flags);
	return (count+endp-buf);


}

