/*
 * linux/block/k3d.c
 * Copyright 2007 Sony Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
/* k3d is based on linux/fs/xvfat/rdchk.c */

#include "k3d_priv.h"

#define K3D_BASE_INTERVAL	1
static unsigned int k3d_default_check_interval;

/* k3d lists and semaphores */
LIST_HEAD(k3d_info_list);
struct rw_semaphore k3d_info_list_sem;
static LIST_HEAD(k3d_info_chk_list);
static spinlock_t k3d_chk_list_lock;

/* k3d control */
static pid_t k3d_running = 0;
static struct rw_semaphore k3d_running_sem;
static DECLARE_WAIT_QUEUE_HEAD(k3d_wait);
static struct semaphore k3d_finish_sem;

/**************************************************************************
 * Methods to handle block_device I/O error notification.
 **************************************************************************/
static LIST_HEAD(bioe_list);
static spinlock_t bioe_list_lock = SPIN_LOCK_UNLOCKED;

struct blkdev_io_error *bioe_get_entry(struct gendisk *disk)
{
	unsigned long flags;
	struct blkdev_io_error *bioe;
	dev_t dev;

	if (!disk) {
		k3d_printk0("%s NULL disk\n", __FUNCTION__);
		return NULL;
	}

	dev = MKDEV(disk->major, disk->first_minor + 0);

	spin_lock_irqsave(&bioe_list_lock, flags);
	list_for_each_entry(bioe, &bioe_list, e_list) {
		if (bioe->e_dev == dev) {
  			spin_unlock_irqrestore(&bioe_list_lock, flags);
  			return bioe;
  		}
	}
	spin_unlock_irqrestore(&bioe_list_lock, flags);
	return NULL;
}

static struct blkdev_io_error *bioe_register(struct gendisk *disk)
{
	struct blkdev_io_error *bioe;
	unsigned long flags;
	dev_t dev;

	bioe = bioe_get_entry(disk);
	if (bioe) {
		atomic_set(&bioe->e_count, 0);
		return bioe;
	}
	bioe = (struct blkdev_io_error *)
		kzalloc(sizeof(struct blkdev_io_error), GFP_KERNEL);
	if (!bioe)
		return NULL;
	dev = MKDEV(disk->major, disk->first_minor + 0);
	bioe->e_dev = dev;
	atomic_set(&bioe->e_count, 0);
	spin_lock_irqsave(&bioe_list_lock, flags);
	list_add(&bioe->e_list, &bioe_list);
	spin_unlock_irqrestore(&bioe_list_lock, flags);
	return bioe;
}

#ifdef NEVER_K3D_MODULE
static void bioe_unregister(struct gendisk *disk)
{
	struct blkdev_io_error *bioe;
	unsigned long flags;

	bioe = bioe_get_entry(disk);
	if (bioe) {
		spin_lock_irqsave(&bioe_list_lock, flags);
		list_del(&bioe->e_list);
		spin_unlock_irqrestore(&bioe_list_lock, flags);
		bioe->e_dev = 0;
		atomic_set(&bioe->e_count, 0);
		kfree(bioe);
	}
}
#endif

/**************************************************************************
 * k3d methods
 **************************************************************************/

/*
 * get k3d_info chage or insert staus internally.
 */
static int get_disk_change_status(struct gendisk *disk)
{
	const struct block_device_operations *bdops;

	if (disk == NULL)
		return 0;
	bdops = disk->fops;
	if (bdops == NULL)
		return 0;
	if (bdops->media_changed == NULL)
		return 0;
	if (!bdops->media_changed(disk))
		return 0;
	return 1;
}

static int get_disk_insert_status(struct gendisk *disk)
{
	const struct block_device_operations *bdops;

	if (disk == NULL)
		return 0;
	bdops = disk->fops;
	if (bdops == NULL)
		return 0;
	if (bdops->media_inserted == NULL)
		return 1;  /* Not cared medium inserted status */
	if (!bdops->media_inserted(disk))
		return 0;
	return 1;
}


/**************************************************************************/

/*
 * support methods to set and get k3d_info state or capability.
 */
static void k3di_set_state(struct k3d_info *k3di, int type, int val)
{
#ifdef CONFIG_SNSC_BLOCK_K3D_STATE_CHANGE_COUNT
	unsigned long state_save = k3di->state;
	unsigned long mask = (K3D_S_CHANGE_BIT|K3D_S_INSERT_BIT);
#endif
	if (val)
		set_bit(type, &k3di->state);
	else
		clear_bit(type, &k3di->state);
#ifdef CONFIG_SNSC_BLOCK_K3D_STATE_CHANGE_COUNT
	if ((state_save & mask) != (k3di->state & mask))
		k3di->state_change_count++;
#endif
}

static void k3di_set_capability(struct k3d_info *k3di, int val)
{
	if (val < 0)
		val = 0;
	k3di->capability = val;
}

static void k3di_set_capability_disabled(struct k3d_info *k3di)
{
	k3di_set_capability(k3di, 0);
	/* sync previous k3d info to current one */
	k3di->prev_state = k3di->state;
	k3di->prev_cap = k3di->capability;
}

/*
 * check state and capability difference and notify to proc and uevent.
 */
static int diff_state(unsigned long statea, unsigned long stateb,
		      unsigned long capa, unsigned long capb)
{
	unsigned long mask =
		(K3D_S_CHANGE_BIT|K3D_S_INSERT_BIT|K3D_S_NOTCHECKED_BIT);

	if ((statea & mask) != (stateb & mask) ||
	    (capa & mask) != (capb & mask))
		return 1;
	else
		return 0;
}

static void diff_state_notify(struct k3d_info *k3di)
{
	k3d_printk("%s:dev=%s (cap,sta)=(%lx,%lx)->(%lx,%lx)\n",
		     __FUNCTION__, k3di->name,
		   k3di->prev_cap, k3di->prev_state,
		   k3di->capability, k3di->state);

	if (diff_state(k3di->state, k3di->prev_state,
		       k3di->capability, k3di->prev_cap)) {
#ifdef CONFIG_PROC_FS
		/* notify "/proc/removabledisk/{state,event}" */
		k3d_proc_notify(k3di);
#endif
		/* notify uevent */
		if(k3di->disk)
			kobject_uevent(&disk_to_dev(k3di->disk)->kobj,
				       KOBJ_CHANGE);

		k3di->prev_state = k3di->state;
		k3di->prev_cap = k3di->capability;
	}
}

/*
 * support method to check disk status.
 */
static int
check_blkdev_capability(struct gendisk *disk)
{
	const struct block_device_operations *bdops;
	int cap;

	if (disk == NULL)
		return K3D_NOT_CHECKED;
	bdops = disk->fops;
	if (!bdops)
		return K3D_NO_METHOD;
	cap = 0;
	if (bdops->media_changed)
		cap |= K3D_S_CHANGE_BIT;
	if (bdops->media_inserted)
		cap |= K3D_S_INSERT_BIT;
	return cap;
}

static struct k3d_info *find_k3dinfo(struct gendisk *disk)
{
	struct k3d_info *e;
	dev_t dev;

	if (!disk)
		return NULL;

	dev = MKDEV(disk->major, disk->first_minor + 0);

	list_for_each_entry(e, &k3d_info_list, dev_list) {
		if (e->bioe->e_dev == dev)
  			return e;
  	}
  	return NULL;
}

struct k3d_info *find_k3dinfo_by_name(const char* name)
{
	struct k3d_info *e;

	if (!name)
		return NULL;

	list_for_each_entry(e, &k3d_info_list, dev_list) {
		if (!strcmp(name, e->name))
  			return e;
  	}
  	return NULL;
}


static void check_status_with_bioe(struct k3d_info *k3di);
static int k3d_delay_invalidate(struct gendisk *disk);

/* k3d checks the disk status with bioe. If k3d found the disk status
 * changed, k3d sends a quiery to the device later. */
static void check_status_with_bioe(struct k3d_info *k3di)
{
	if (!k3di_capability(k3di, K3D_S_CHANGE))
		return;
	if (k3di_state(k3di, K3D_S_CHANGE))
		return;
	if (atomic_read(&k3di->bioe->e_count) <= 0)
		return;

	k3d_printk("%s bioe error occurred\n", __FUNCTION__);
	k3di_set_state(k3di, K3D_S_CHANGE, 1);
	k3d_delay_invalidate(k3di->disk);

	/* add this disk to check list. */
	spin_lock(&k3d_chk_list_lock);
	if (k3di->interval > 0 && list_empty(&k3di->chk_list))
		list_add(&k3di->chk_list, &k3d_info_chk_list);
	spin_unlock(&k3d_chk_list_lock);
}

/*
 * get and set k3d_info change/insert/state_change_count/interval.
 */
int get_k3dinfo_change(struct k3d_info *k3di)
{
	int rval;

	if (k3di_state(k3di, K3D_S_NOTCHECKED)) {
		rval = K3D_NOT_CHECKED;
		goto end_func;
	}
	if (!k3di_capability(k3di, K3D_S_CHANGE)) {
		rval = K3D_NO_METHOD;
		goto end_func;
	}
	if (k3di_state(k3di, K3D_S_CHANGE))
		rval = 1;
	else
		rval = 0;
 end_func:
	return rval;
}

/* arg. call_daemon is obsoleted. k3d thread is waken up later */
int k3d_get_disk_change(struct gendisk *disk, int call_daemon)
{
	struct k3d_info *k3di;
	int rval;

	k3di = find_k3dinfo(disk);
	if (k3di == NULL) {
		rval = K3D_NOT_CHECKED;
		goto end_func;
	}
	if (k3di->disk != disk) {
		rval = 1;
		goto end_func;
	}

	check_status_with_bioe(k3di);

	rval = get_k3dinfo_change(k3di);
end_func:
	return rval;
}
EXPORT_SYMBOL(k3d_get_disk_change);

int get_k3dinfo_insert(struct k3d_info *k3di)
{
	int rval;

	if (k3di_state(k3di, K3D_S_NOTCHECKED)) {
		rval = K3D_NOT_CHECKED;
		goto end_func;
	}
	if (!k3di_capability(k3di, K3D_S_INSERT)) {
		rval = K3D_NO_METHOD;
		goto end_func;
	}
	if (k3di_state(k3di, K3D_S_INSERT))
		rval = 1;
	else
		rval = 0;
end_func:
	return rval;
}

/* arg. call_daemon is obsoleted. k3d thread is waken up later */
int k3d_get_disk_insert(struct gendisk *disk, int call_daemon)
{
	struct k3d_info *k3di;
	int rval;

	k3di = find_k3dinfo(disk);
	if (k3di == NULL) {
		rval = K3D_NOT_CHECKED;
		goto end_func;
	}

	check_status_with_bioe(k3di);

	rval = get_k3dinfo_insert(k3di);
end_func:
	return rval;
}
EXPORT_SYMBOL(k3d_get_disk_insert);

/* arg. call_daemon is obsoleted. k3d thread is waken up later */
int k3d_get_disk_state_change_count(struct gendisk *disk,
				    unsigned int *ret, int call_daemon)
{
	struct k3d_info *k3di;
	int rval;

	k3di = find_k3dinfo(disk);
	if (k3di == NULL) {
		rval = K3D_NOT_CHECKED;
		goto end_func;
	}

	check_status_with_bioe(k3di);

	*ret = k3di->state_change_count;
	rval = 0;
 end_func:
	return rval;
}
EXPORT_SYMBOL(k3d_get_disk_state_change_count);

int k3d_get_disk_check_interval(struct gendisk *disk, unsigned int *interval)
{
	struct k3d_info *k3di;
	int rval;

	k3di = find_k3dinfo(disk);
	if (k3di == NULL) {
		rval = -EINVAL;
	} else {
 		*interval = k3di->interval;
		rval = 0;
	}
	return rval;
}
EXPORT_SYMBOL(k3d_get_disk_check_interval);

int set_k3dinfo_interval(struct k3d_info *k3di, unsigned int val)
{
	int cap;
	unsigned int prev_interval;

	prev_interval = k3di->interval;
	k3di->interval = val;
	k3di->interval_count = k3di->interval;

	if (val == 0) {
		k3di_set_state(k3di, K3D_S_NOTCHECKED, 1);
		diff_state_notify(k3di);
		k3di_set_capability_disabled(k3di);
	}
	else if (prev_interval == 0 && val > 0) {
		/* re-start disk check */
		cap = check_blkdev_capability(k3di->disk);
		k3di_set_capability(k3di, cap);
		diff_state_notify(k3di);
		//wake_up_interruptible(&k3d_wait);
	}
	return 0;
}

int k3d_set_disk_check_interval(struct gendisk *disk, unsigned int interval)
{
	struct k3d_info *k3di;
	int rval;

	k3di = find_k3dinfo(disk);
	if (k3di == NULL) {
		rval = -EINVAL;
	} else {
		rval = set_k3dinfo_interval(k3di, interval);
	}
	return rval;
}
EXPORT_SYMBOL(k3d_set_disk_check_interval);

/*
 * count up or down reference to the disk.
 */
int k3d_get_disk(struct gendisk *disk)
{
	struct k3d_info *k3di;
	int ret = 0;

	k3di = find_k3dinfo(disk);
	if (k3di == NULL) {
		printk(KERN_ERR "%s: not found device (dev=%s)\n",
		       __FUNCTION__, disk->disk_name);
		ret = -1;
		goto out;
	}
	atomic_inc(&k3di->ref_count);
	if (k3di_capability(k3di, K3D_S_CHANGE) &&
	    atomic_read(&k3di->bioe->e_count) == 0) {
		/* When mount file system, parhaps media not changed */
		if (k3di->interval > 0)
			k3di_set_state(k3di, K3D_S_NOTCHECKED, 0);
		k3di_set_state(k3di, K3D_S_CHANGE, 0);
		diff_state_notify(k3di);
	}
 out:
 	return ret;
}
EXPORT_SYMBOL(k3d_get_disk);

int k3d_put_disk(struct gendisk *disk)
{
	struct k3d_info *k3di;
 	int ret = 0;

	k3di = find_k3dinfo(disk);
	if (k3di == NULL) {
		printk(KERN_ERR "%s: not found device (dev=%s)\n",
		       __FUNCTION__, disk->disk_name);
 		ret = -1;
		goto out;
	}
	atomic_dec(&k3di->ref_count);
 out:
 	return ret;
}
EXPORT_SYMBOL(k3d_put_disk);

#ifdef CONFIG_SNSC_BLOCK_CHECK_DISK
/*
 * save or load a disk flag.
 */
int k3d_save_disk_flag(struct gendisk *disk, int flag)
{
	struct k3d_info *k3di;
	int ret = 0;

	k3di = find_k3dinfo(disk);
	if (k3di == NULL) {
		ret = -1;
		goto out;
	}
	k3di->flag = flag;
 out:
	return ret;
}
EXPORT_SYMBOL(k3d_save_disk_flag);

int k3d_load_disk_flag(struct gendisk *disk, int *flag)
{
	struct k3d_info *k3di;
	int ret = 0;

	k3di = find_k3dinfo(disk);
	if (k3di == NULL) {
		ret = -1;
		goto out;
	}
	if (flag != NULL)
		*flag = k3di->flag;
 out:
	return ret;
}
EXPORT_SYMBOL(k3d_load_disk_flag);

/*
 * check the disk is registered.
 */
int k3d_disk_is_registered(struct gendisk *disk)
{
	struct k3d_info *k3di;

	k3di = find_k3dinfo(disk);
	if (k3di)
		return 1;
	else
		return 0;
}
EXPORT_SYMBOL(k3d_disk_is_registered);
#endif

/*
 * register and unregister the disk.
 */
int k3d_register_disk(struct gendisk *disk)
{
	struct k3d_info *k3di;
	int cap;
	int ret = 0;

	if (!(disk->flags & GENHD_FL_REMOVABLE))
		return 1;

	cap = check_blkdev_capability(disk);
	if (cap < 0)
		ret = -ENODEV;

	k3di = find_k3dinfo(disk);
	if (k3di) {
		k3d_printk("%s re-registration\n", __FUNCTION__);

		if (k3di->disk != disk)
			k3di->disk = disk;

		if (cap & K3D_S_CHANGE_BIT) {
			if (atomic_read(&k3di->ref_count) == 0) {
				k3di_set_state(k3di, K3D_S_CHANGE, 0);
				atomic_set(&k3di->bioe->e_count, 0);
			} else
				k3di_set_state(k3di, K3D_S_CHANGE, 1);
		}
		k3di->interval_count = k3di->interval;
	} else {
		k3d_printk("%s first registration\n", __FUNCTION__);
		k3di = (struct k3d_info *)
			kzalloc(sizeof(struct k3d_info), GFP_KERNEL);
		if (k3di == NULL) {
			printk("%s: NOMEM k3di\n", __FUNCTION__);
			return -ENOMEM;
		}
		k3di->disk = disk;
		disk_name(disk, 0, k3di->name);
		if (disk->major == 2) { /* Floppy block device */
			k3di->state = 0;
			k3di->interval = 0;
			printk("%s: not polling floppy drive (dev:%s)\n",
			       __FUNCTION__, k3di->name);
		} else
			k3di->interval = k3d_default_check_interval;
		atomic_set(&k3di->ref_count, 0);
		atomic_set(&k3di->flag_delay_inv, 0);
		atomic_set(&k3di->flag_wait_unregister, 0);
		k3di->bioe = bioe_register(disk);
		if (k3di->bioe == NULL) {
			printk("%s: NOMEM k3di->bioe\n", __FUNCTION__);
			kfree(k3di);
			return -ENOMEM;
		}
		INIT_LIST_HEAD(&k3di->chk_list);
		down_write(&k3d_info_list_sem);
		list_add(&k3di->dev_list, &k3d_info_list);
		up_write(&k3d_info_list_sem);
	}
	k3di_set_state(k3di, K3D_S_NOTCHECKED, 1);
	k3di_set_capability(k3di, cap);
	diff_state_notify(k3di);
	return ret;
}

static void k3d_unregister(struct k3d_info *k3di)
{
	k3di_set_state(k3di, K3D_S_NOTCHECKED, 1);
	k3di_set_state(k3di, K3D_S_INSERT, 0);
	diff_state_notify(k3di);
	while(atomic_read(&k3di->flag_wait_unregister)){
		msleep(1);
	}
	k3di->disk = NULL;
	k3di_set_capability_disabled(k3di);

	k3d_printk0("%s: k3di->ref_count=%d\n",
		    __FUNCTION__, atomic_read(&k3di->ref_count));
}

#ifdef NEVER_K3D_MODULE
static void k3d_unregister_all(void)
{
	struct k3d_info *k3di, *nk3di;

	down_write(&k3d_info_list_sem);
	list_for_each_entry_safe(k3di, nk3di, &k3d_info_list, dev_list) {
		bioe_unregister(k3di->disk);
		k3d_unregister(k3di);
		list_del(&k3di->dev_list);
		kfree(k3di);
	}
	up_write(&k3d_info_list_sem);
}
#endif

void k3d_unregister_disk(struct gendisk* disk)
{
	struct k3d_info *k3di;

	k3di = find_k3dinfo(disk);
	if (k3di == NULL)
	  	goto out;
	k3d_unregister(k3di);
out:
	return;
}

/**************************************************************************/

/*
 * invalidate the disk, when the medium has been ejected.
 */
static int k3d_delay_invalidate(struct gendisk *disk)
{
	struct k3d_info *k3di;
	int rval;

	k3di = find_k3dinfo(disk);
	if (k3di == NULL) {
		rval = -EINVAL;
	} else {
		atomic_set(&k3di->flag_delay_inv, K3D_F_DELAY_INV);
		wake_up_interruptible(&k3d_wait);
		rval = 0;
	}
	return rval;
}

static void k3d_invalidate_disk_partition(struct gendisk *disk, int index)
{
	struct block_device *bdev;

	bdev = bdget_disk(disk, index);
	if (bdev) {
		__invalidate_device(bdev);
		bdput(bdev);
	}
}

static void k3d_invalidate_disk(struct gendisk *disk)
{
	int p;

	if (disk == NULL)
		return ;

	for (p = disk->minors - 1; p > 0; p--) {
		k3d_invalidate_disk_partition(disk, p);
	}
	k3d_invalidate_disk_partition(disk, 0);
	k3d_printk("%s: disk=%s\n", __FUNCTION__, disk->disk_name);
}

/*
 * revalidate the disk, when the new medium is not refered by anyone.
 */
static void k3d_notify_revalidate_internal(struct gendisk *disk)
{
	struct k3d_info *k3di;

	k3di = find_k3dinfo(disk);
	k3d_printk("%s: diskname=%s k3di=%08x\n", __FUNCTION__,
		     k3di ? k3di->name : "???", (unsigned)k3di);
	if (k3di) {
		/*
		if (k3di_state(k3di, K3D_S_CHANGE)) {
			k3di_set_state(k3di, K3D_S_CHANGE, 0);
			diff_state_notify(k3di);
		}
		*/
		wake_up_interruptible(&k3d_wait);
	}
}

int k3d_revalidate_disk(struct gendisk *disk, int flag)
{
	const struct block_device_operations *bdops;
	struct block_device *bdev;

	if (disk == NULL)
		return K3D_NOT_CHECKED;
	bdops = disk->fops;
	if (bdops == NULL)
		return K3D_NO_METHOD;

	/* revalidate_disk executes MBR read (i.e. index = 0) */
	bdev = bdget_disk(disk, 0);
	if (bdev) {
		mutex_lock_nested(&bdev->bd_mutex, 0);
		if (bdops->revalidate_disk && flag)
			bdops->revalidate_disk(disk);
		mutex_unlock(&bdev->bd_mutex);
		bdput(bdev);
	}

	k3d_notify_revalidate_internal(disk);

	return 0;
}


/*
 * k3d checks media for each disks.
 */
static void k3d_chk_one(struct k3d_info *k3di)
{
	int chg, ins, cap, e_count;

	atomic_inc(&k3di->flag_wait_unregister);
	chg = ins = 0;
	cap = check_blkdev_capability(k3di->disk);
	k3di_set_capability(k3di, cap);
	lock_kernel();
	if (atomic_cmpxchg(&k3di->flag_delay_inv, K3D_F_DELAY_INV, 0)) {
		k3d_invalidate_disk(k3di->disk);
	}
	e_count = atomic_read(&k3di->bioe->e_count);
	if (k3di_capability(k3di, K3D_S_CHANGE))
		chg = get_disk_change_status(k3di->disk);
	if (k3di_capability(k3di, K3D_S_INSERT)) {
		ins = get_disk_insert_status(k3di->disk);
		k3di_set_state(k3di, K3D_S_INSERT, ins);
	}
	if (k3di_state(k3di, K3D_S_NOTCHECKED) && k3di->capability > 0)
		k3di_set_state(k3di, K3D_S_NOTCHECKED, 0);
	k3d_printk("%s: chg=%d ins=%d\n", __FUNCTION__, chg, ins);
	if (!k3di_capability(k3di, K3D_S_CHANGE)) {
		diff_state_notify(k3di);
		unlock_kernel(); /* FES patch */
		atomic_dec(&k3di->flag_wait_unregister);
		return;
	}
	if (chg == 1) {
		if (!k3di_state(k3di, K3D_S_CHANGE)) {
			k3d_invalidate_disk(k3di->disk);
			k3d_printk("%s: changed, device %s\n", __FUNCTION__,
			       k3di->name);
		}
#ifdef NEVER
		else if (atomic_read(&k3di->ref_count) == 0) {
			k3d_invalidate_disk(k3di->disk);
			k3d_revalidate_disk(k3di->disk, 1);
		}
#endif
		k3di_set_state(k3di, K3D_S_CHANGE, 1);
	} else {
		if (k3di_state(k3di, K3D_S_CHANGE) &&
		    atomic_read(&k3di->ref_count) == 0) {/* mount count == 0 */
			k3di_set_state(k3di, K3D_S_CHANGE, 0);
			k3d_invalidate_disk(k3di->disk);
			k3d_revalidate_disk(k3di->disk, 1);
		}
	}
	if (e_count > 0) {
		/* clear I/O error count */
		if (atomic_add_negative(-e_count, &k3di->bioe->e_count))
			BUG();
	}
	unlock_kernel();

	diff_state_notify(k3di);
	atomic_dec(&k3di->flag_wait_unregister);
}

static int k3d(void *startup)
{
	daemonize("k3d");
	set_freezable();
	down_write(&k3d_running_sem);
	k3d_running = 1;
	up_write(&k3d_running_sem);

#if defined(CONFIG_SNSC_BLOCK_K3D_SCHED_OPTION)
	if (set_thread_priority(CONFIG_SNSC_BLOCK_K3D_SCHED_POLICY,
				CONFIG_SNSC_BLOCK_K3D_SCHED_PRIORITY) < 0) {
		k3d_printk("%s: set schedule & priority failed(%d:%d)."
		       " Continue to run\n",
		       __FUNCTION__,
		       CONFIG_SNSC_BLOCK_K3D_SCHED_POLICY,
		       CONFIG_SNSC_BLOCK_K3D_SCHED_PRIORITY);
	}
#endif
	complete((struct completion *)startup);

	k3d_printk("%s: start daemon [%d:%s]\n", __FUNCTION__,
		     current->pid, current->comm);

	for (;;) {
		struct k3d_info *k3di, *n;
		long t;

		LIST_HEAD(work_list);

		t = wait_event_interruptible_timeout(
			k3d_wait,
			!list_empty(&k3d_info_chk_list),
			K3D_BASE_INTERVAL * HZ
			);
		try_to_freeze();

		/* move items to work_list */
		spin_lock(&k3d_chk_list_lock);
		list_splice(&k3d_info_chk_list, &work_list);
		INIT_LIST_HEAD(&k3d_info_chk_list);
		spin_unlock(&k3d_chk_list_lock);

		/* check for periodic */
		if (t == 0) {
			k3d_printk2("k3d: check .. k3d_info_list\n");
			list_for_each_entry(k3di, &k3d_info_list, dev_list) {
				if (!k3di->disk || k3di->interval == 0)
					continue;

				if (k3di->interval_count > 0)
					--k3di->interval_count;

				if (k3di->interval_count != 0)
					continue;

				k3d_printk("%s %s interval=%u\n", __FUNCTION__,
					   k3di->name,k3di->interval);
				spin_lock(&k3d_chk_list_lock);
				if (list_empty(&k3di->chk_list))
					list_add(&k3di->chk_list, &work_list);
				spin_unlock(&k3d_chk_list_lock);
			}
		}

		list_for_each_entry_safe(k3di, n, &work_list, chk_list){
			k3d_chk_one(k3di);
			if (k3di->interval_count == 0)
				k3di->interval_count = k3di->interval;
			spin_lock(&k3d_chk_list_lock);
			list_del_init(&k3di->chk_list);
			spin_unlock(&k3d_chk_list_lock);
		}

#ifdef NEVER_K3D_MODULE
		down_read(&k3d_running_sem);
		if (!k3d_running) {
			up_read(&k3d_running_sem);
			break;
		}
		up_read(&k3d_running_sem);
#endif
	}
	up(&k3d_finish_sem);
	return 0;
}

/**************************************************************************/

/*
 * k3d init and exit methods.
 */
static int __init
k3d_thread_init(void)
{
	static struct completion startup __initdata
		= COMPLETION_INITIALIZER(startup);

	kernel_thread(k3d, &startup, 0);
	wait_for_completion(&startup);
	return 0;
}

#ifdef NEVER_K3D_MODULE
static void
k3d_finish(void)
{
	down_write(&k3d_running_sem);
	k3d_running = 0;
	up_write(&k3d_running_sem);
	k3d_printk("%s[%d:%s]: wake up k3d\n", __FUNCTION__,
		     current->pid, current->comm);
	wake_up_interruptible(&k3d_wait);
	down(&k3d_finish_sem);
	return;
}
#endif

int __init k3d_init(void)
{
	k3d_printk("%s[%d:%s]: start\n", __FUNCTION__,
		     current->pid, current->comm);

	/* set default disk check interval */
	k3d_default_check_interval = CONFIG_SNSC_BLOCK_K3D_INTERVAL;
	if (k3d_default_check_interval < K3D_BASE_INTERVAL) {
		printk("%s: default check interval (%u) was changed to "
		       "base interval",
		       __FUNCTION__, k3d_default_check_interval);
		k3d_default_check_interval = K3D_BASE_INTERVAL;
	}

	spin_lock_init(&k3d_chk_list_lock);
	init_rwsem(&k3d_info_list_sem);
	init_rwsem(&k3d_running_sem);
	sema_init(&k3d_finish_sem, 0);
#ifdef CONFIG_PROC_FS
	k3d_proc_init();
#endif
	k3d_thread_init();
	return 0;
}

#ifdef NEVER_K3D_MODULE
void __exit k3d_exit(void)
{
	k3d_printk("%s[%d:%s]:\n", __FUNCTION__, current->pid, current->comm);
	k3d_unregister_all();
	k3d_printk("%s[%d:%s]: stop daemon\n", __FUNCTION__,
		     current->pid, current->comm);
	k3d_finish();
#ifdef CONFIG_PROC_FS
	k3d_printk("%s[%d:%s]: stop proc FS service\n", __FUNCTION__,
		     current->pid, current->comm);
	k3d_proc_exit();
#endif
	k3d_printk("%s[%d:%s]: end\n", __FUNCTION__,
		     current->pid, current->comm);
}
#endif

