/*
 * Cryptographic API.
 *
 * LZ77 algorithm
 *
 * Copyright 2010 Sony Corporation
 *
 * This program is free software; you can redistribute  it and/or modify it
 * under  the terms of  the GNU General  Public License as published by the
 * Free Software Foundation;  version 2 of the  License.
 *
 * THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 * WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 * NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 * USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * You should have received a copy of the  GNU General Public License along
 * with this program; if not, write  to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include <linux/init.h>
#include <linux/lz77.h>
#include <linux/module.h>
#include <crypto/internal/compress.h>

#define LZ77_HEADER_SIZE	2
#define LZ77_DEFLATE_BLOCK	4096
/* an LZ77 compression record is at most 4100 (4K+4) bytes long */
#define LZ77_MAX_RECORD_SIZE	(LZ77_DEFLATE_BLOCK + 4)

#define LZ77_READ_HEADER(p)	(*(unsigned char *)(p)			\
				 + (*((unsigned char *)(p) + 1) << 8))

struct lz77_ctx {
	char *decomp_buffer;
	unsigned buffered_bytes;
	unsigned record_size;
	int has_split_header;
	unsigned char split_header[2];
};

static int
lz77_init(struct crypto_tfm *tfm)
{
	return 0;
}

static void
lz77_exit(struct crypto_tfm *tfm)
{
	struct lz77_ctx *dctx = crypto_tfm_ctx(tfm);

	if (dctx->decomp_buffer != NULL) {
		kfree(dctx->decomp_buffer);
		dctx->decomp_buffer = NULL;
	}
	dctx->buffered_bytes = 0;
	dctx->record_size = 0;
	dctx->has_split_header = 0;
}

static int
lz77_compress_setup(struct crypto_pcomp *tfm, void *params,
		    unsigned int len)
{
	/* not supported */
	return -EINVAL;
}

static int
lz77_compress_init(struct crypto_pcomp *tfm)
{
	/* not supported */
	return -EINVAL;
}

static int
lz77_compress_update(struct crypto_pcomp *tfm,
		     struct comp_request *req)
{
	/* not supported */
	return -EINVAL;
}

static int
lz77_compress_final(struct crypto_pcomp *tfm,
		    struct comp_request *req)
{
	/* not supported */
	return -EINVAL;
}

static int
lz77_decompress_setup(struct crypto_pcomp *tfm, void *params,
		      unsigned int len)
{
	struct lz77_ctx *dctx = crypto_tfm_ctx(crypto_pcomp_tfm(tfm));

	dctx->decomp_buffer = kmalloc(LZ77_MAX_RECORD_SIZE, GFP_KERNEL);
	if (dctx->decomp_buffer == NULL) {
		pr_debug("could not allocate decomp buffer.\n");
		return -ENOMEM;
	}

	dctx->buffered_bytes = 0;
	dctx->record_size = 0;
	dctx->has_split_header = 0;

	return 0;
}

static int
lz77_decompress_init(struct crypto_pcomp *tfm)
{
	struct lz77_ctx *dctx = crypto_tfm_ctx(crypto_pcomp_tfm(tfm));

	dctx->buffered_bytes = 0;
	dctx->record_size = 0;
	dctx->has_split_header = 0;

	return 0;
}

static int
lz77_do_decompress(struct crypto_pcomp *tfm,
		   struct comp_request *req)
{
	struct lz77_ctx *dctx = crypto_tfm_ctx(crypto_pcomp_tfm(tfm));

	pr_debug("next_in = %lx, next_out = %lx, avail_in = %u, avail_out = %u\n",
		 (unsigned long)req->next_in, (unsigned long)req->next_out,
		 req->avail_in, req->avail_out);

	while (req->avail_in > 0 && req->avail_out > 0) {
		int consumed_input, output_len;

		consumed_input = output_len = 0;

		if (dctx->has_split_header) {
			/* the second byte of a record header */
			dctx->split_header[1] = *(unsigned char *)req->next_in;
			dctx->record_size = LZ77_READ_HEADER(dctx->split_header);

			if (dctx->record_size > LZ77_MAX_RECORD_SIZE) {
				pr_debug("invalid record size: %d\n",
					 dctx->record_size);
				return -EINVAL;
			}

			dctx->has_split_header = 0;
			consumed_input = 1;
		} else if (dctx->record_size > 0) {
			/* decomp_buffer is not empty */
			int copy_size;

			/* append this input to the decomp_buffer */
			copy_size = min(dctx->record_size - dctx->buffered_bytes,
					req->avail_in);
			memcpy(dctx->decomp_buffer + dctx->buffered_bytes,
			       req->next_in, copy_size);

			consumed_input = copy_size;
			dctx->buffered_bytes += copy_size;

			/* decomp_buffer contains a complete record */
			if (dctx->buffered_bytes == dctx->record_size) {
				if (req->avail_out < LZ77_DEFLATE_BLOCK)
					return -EINVAL;

				output_len = lz77_inflate(dctx->decomp_buffer,
							  dctx->buffered_bytes,
							  req->next_out,
							  req->avail_out, NULL);
				if (output_len < 0)
					return -EINVAL;

				dctx->buffered_bytes = 0;
				dctx->record_size = 0;
			}
		} else if (req->avail_in == 1) {
			/* the first byte of a record header */
			dctx->split_header[0] = *(unsigned char *)req->next_in;
			dctx->has_split_header = 1;
			consumed_input = 1;
		} else {
			int record_size = LZ77_READ_HEADER(req->next_in);

			if (record_size > LZ77_MAX_RECORD_SIZE) {
				pr_debug("invalid record size: %d\n",
					 record_size);
				return -EINVAL;
			}

			/* a complete record is available */
			if (req->avail_in >= LZ77_HEADER_SIZE + record_size) {
				if (req->avail_out < LZ77_DEFLATE_BLOCK)
					return -EINVAL;

				output_len = lz77_inflate((void *)(req->next_in +
								   LZ77_HEADER_SIZE),
							  record_size,
							  req->next_out,
							  req->avail_out, NULL);
				if (output_len < 0)
					return -EINVAL;

				consumed_input = LZ77_HEADER_SIZE + record_size;
			} else { /* partial record */
				memcpy(dctx->decomp_buffer,
				       req->next_in + LZ77_HEADER_SIZE,
				       req->avail_in - LZ77_HEADER_SIZE);
				dctx->buffered_bytes = req->avail_in
					- LZ77_HEADER_SIZE;
				dctx->record_size = record_size;

				consumed_input = req->avail_in;
			}
		}
		req->next_in += consumed_input;
		req->avail_in -= consumed_input;
		req->next_out += output_len;
		req->avail_out -= output_len;
	}

	return 0;
}

static int
lz77_decompress_update(struct crypto_pcomp *tfm,
		       struct comp_request *req)
{
	return lz77_do_decompress(tfm, req);
}

static int
lz77_decompress_final(struct crypto_pcomp *tfm,
		      struct comp_request *req)
{
	return lz77_do_decompress(tfm, req);
}

static struct pcomp_alg lz77_alg = {
	.compress_setup         = lz77_compress_setup,
	.compress_init          = lz77_compress_init,
	.compress_update        = lz77_compress_update,
	.compress_final         = lz77_compress_final,
	.decompress_setup       = lz77_decompress_setup,
	.decompress_init        = lz77_decompress_init,
	.decompress_update      = lz77_decompress_update,
	.decompress_final       = lz77_decompress_final,

	.base                   = {
		.cra_name       = "lz77",
		.cra_flags      = CRYPTO_ALG_TYPE_PCOMPRESS,
		.cra_ctxsize    = sizeof(struct lz77_ctx),
		.cra_module     = THIS_MODULE,
		.cra_init       = lz77_init,
		.cra_exit       = lz77_exit,
	}
};

static int __init
lz77_mod_init(void)
{
	return crypto_register_pcomp(&lz77_alg);
}

static void __exit
lz77_mod_fini(void)
{
	crypto_unregister_pcomp(&lz77_alg);
}

module_init(lz77_mod_init);
module_exit(lz77_mod_fini);

MODULE_LICENSE("GPL v2");
MODULE_DESCRIPTION("LZ77 Compression Algorithm");
MODULE_AUTHOR("Sony Corporation");
