/*
 *  LZO de/compression driver
 *
 *  Copyright 2007 Sony Corporation.
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <linux/cdev.h>
#include <linux/lzo.h>
#include <linux/snsc_lzo_dev.h>
#include <asm/uaccess.h>

#define LZO_DEV_NAME		"lzo_dev"
#define LZO_DEV_ERR(...)	printk(KERN_ERR LZO_DEV_NAME": " __VA_ARGS__)

/* as character device */
#define LZO_DEV_MAJOR		62
#define LZO_DEV_MINOR		0
static struct cdev		lzod_cdev;
static struct semaphore		lzod_sem;
static int			lzod_open = 0;

/* buffers */
#define LZO_DEV_MAXSIZE		(CONFIG_SNSC_LZO_DEV_MAXSIZE * 1024)
static size_t			max_size;
static unsigned char		*src_buf;
static unsigned char		*dst_buf;
static unsigned char		*work_buf;

/*
 * data de/compression
 */
static int
lzo_dev_compress(lzo_dev_param_t *arg)
{
	lzo_dev_param_t param;
	size_t dst_len;
	int ret;

	/* copy parameters from user */
	ret = copy_from_user(&param, arg, sizeof(lzo_dev_param_t));
	if (ret != 0) {
		return -EFAULT;
	}

	/* check src data size */
	if (param.src_len > max_size) {
		return -ENOSPC;
	}

	/* copy src data from user */
	ret = copy_from_user(src_buf, param.src, param.src_len);
	if (ret != 0) {
		return -EFAULT;
	}

	/* compress data */
	dst_len = max_size;
	memset(work_buf, 0, LZO1X_MEM_COMPRESS);
	ret = lzo1x_1_compress(src_buf, param.src_len,
			       dst_buf, &dst_len, work_buf);
	if (ret < 0) {
		LZO_DEV_ERR("failed LZO1X compression: %d\n", ret);
		return -EIO;
	}

	/* check dst data size */
	if (param.dst_len < dst_len) {
		return -ENOSPC;
	}
	param.dst_len = dst_len;

	/* copy dst data to user */
	ret = copy_to_user(param.dst, dst_buf, dst_len);
	if (ret != 0) {
		return -EFAULT;
	}

	/* copy parameters to user */
	ret = copy_to_user(arg, &param, sizeof(lzo_dev_param_t));
	if (ret != 0) {
		return -EFAULT;
	}

	return 0;
}

static int
lzo_dev_decompress(lzo_dev_param_t *arg)
{
	lzo_dev_param_t param;
	size_t dst_len;
	int ret;

	/* copy parameters from user */
	ret = copy_from_user(&param, arg, sizeof(lzo_dev_param_t));
	if (ret != 0) {
		return -EFAULT;
	}

	/* check src data size */
	if (param.src_len > max_size) {
		return -ENOSPC;
	}

	/* copy src data from user */
	ret = copy_from_user(src_buf, param.src, param.src_len);
	if (ret != 0) {
		return -EFAULT;
	}

	/* decompress data */
	dst_len = max_size;
	ret = lzo1x_decompress_safe(src_buf, param.src_len,
				    dst_buf, &dst_len);
	if (ret < 0) {
		LZO_DEV_ERR("failed LZO1X decompression: %d\n", ret);
		return -EIO;
	}

	/* check dst data size */
	if (param.dst_len < dst_len) {
		return -ENOSPC;
	}
	param.dst_len = dst_len;

	/* copy dst data to user */
	ret = copy_to_user(param.dst, dst_buf, dst_len);
	if (ret != 0) {
		return -EFAULT;
	}

	/* copy parameters to user */
	ret = copy_to_user(arg, &param, sizeof(lzo_dev_param_t));
	if (ret != 0) {
		return -EFAULT;
	}

	return 0;
}

/*
 * character device interface
 */
static int
lzo_dev_open(struct inode *inode, struct file *filp)
{
	int ret;

	if (down_interruptible(&lzod_sem))
		return -ERESTARTSYS;

	if (lzod_open == 0) {
		/* allocate buffers */
		src_buf = kmalloc(max_size, GFP_KERNEL);
		if (src_buf == NULL) {
			LZO_DEV_ERR("cannot allocate src buffer\n");
			ret = -ENOMEM;
			goto err_src;
		}
		dst_buf = kmalloc(max_size, GFP_KERNEL);
		if (dst_buf == NULL) {
			LZO_DEV_ERR("cannot allocate dst buffer\n");
			ret = -ENOMEM;
			goto err_dst;
		}
		work_buf = kmalloc(LZO1X_MEM_COMPRESS, GFP_KERNEL);
		if (work_buf == NULL) {
			LZO_DEV_ERR("cannot allocate work buffer\n");
			ret = -ENOMEM;
			goto err_work;
		}
	}

	lzod_open++;

	up(&lzod_sem);
	return 0;

 err_work:
	kfree(dst_buf);
 err_dst:
	kfree(src_buf);
 err_src:
	up(&lzod_sem);
	return ret;
}

static int
lzo_dev_release(struct inode *inode, struct file *filp)
{
	if (down_interruptible(&lzod_sem))
		return -ERESTARTSYS;

	lzod_open--;

	if (lzod_open <= 0) {
		/* free buffers */
		kfree(work_buf);
		kfree(dst_buf);
		kfree(src_buf);
		lzod_open = 0;
	}

	up(&lzod_sem);
	return 0;
}

static int
lzo_dev_ioctl(struct inode *inode, struct file *filp,
	      unsigned int cmd, unsigned long arg)
{
	int ret;

	if (down_interruptible(&lzod_sem))
		return -ERESTARTSYS;

	switch(cmd) {
	case LZO_DEV_IOC_COMPRESS:
		ret = lzo_dev_compress((lzo_dev_param_t *)arg);
		break;
	case LZO_DEV_IOC_DECOMPRESS:
		ret = lzo_dev_decompress((lzo_dev_param_t *)arg);
		break;
	default:
		ret = -ENOTTY;
		break;
	}

	up(&lzod_sem);
	return ret;
}

static struct file_operations lzo_dev_fops = {
	.owner   = THIS_MODULE,
	.open    = lzo_dev_open,
	.ioctl   = lzo_dev_ioctl,
	.release = lzo_dev_release,
};

/*
 * module entry
 */
static int lzo_dev_init(void)
{
	dev_t devno;
	int ret;

	/* register device */
	devno = MKDEV(LZO_DEV_MAJOR, LZO_DEV_MINOR);
	ret = register_chrdev_region(devno, 1, LZO_DEV_NAME);
	if (ret < 0) {
		LZO_DEV_ERR("register_chrdev_region() failed\n");
		return ret;
	}

	/* setup cdev */
	cdev_init(&lzod_cdev, &lzo_dev_fops);
	lzod_cdev.owner = THIS_MODULE;
	ret = cdev_add(&lzod_cdev, devno, 1);
	if (ret < 0) {
		LZO_DEV_ERR("cdev_add() failed: %d\n", ret);
		goto err;
	}

	/* initialize semaphore */
	init_MUTEX(&lzod_sem);

	/* set maximum data size */
	max_size = lzo1x_worst_compress(LZO_DEV_MAXSIZE);

	return 0;

 err:
	unregister_chrdev_region(devno, 1);
	return ret;
}

static void lzo_dev_exit(void)
{
	/* delete and unregister device */
	cdev_del(&lzod_cdev);
	unregister_chrdev_region(MKDEV(LZO_DEV_MAJOR, LZO_DEV_MINOR), 1);

	return;
}

module_init(lzo_dev_init);
module_exit(lzo_dev_exit);

MODULE_AUTHOR("Sony Corporation");
MODULE_DESCRIPTION("LZO de/compression driver");
MODULE_LICENSE("GPL");
